/* pdu-ip.c
 
   PDU builder for IPv4 packets

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-ip.h"

static const struct pdu_dict protocol_mappings[] = {
    {"udp", (uint8_t []) {IP_PROTO_UDP} },
    {"tcp", (uint8_t []) {IP_PROTO_TCP} },
    {"ospf", (uint8_t []) {IP_PROTO_OSPF} },
    {"ospf-hello", (uint8_t []) {IP_PROTO_OSPF} },
    {"icmp", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-echo", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-echoreply", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-redirect", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-srcquench", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-parmprob", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-timeexceed", (uint8_t []) {IP_PROTO_ICMP} },
    {"icmp-unreachable", (uint8_t []) {IP_PROTO_ICMP} },
    {"gre", (uint8_t []) {IP_PROTO_GRE} },
    {"igmp", (uint8_t []) {IP_PROTO_IGMP} },
    {NULL, NULL}
};

static GHashTable *protos;

static void
build(const GNode *pdu, void *dest)
{
    struct ip_hdr *ip;
    intf_t *i;
    struct intf_entry iface;
    struct addr dest_ip;
    uint16_t u16;
    ip_addr_t ipaddr;
    void *field;
    uint8_t *proto;
    const GNode *next_pdu;
    size_t opts_len;

    ip = dest;

    ip->ip_v = (field = _pb_pdata(pdu, "version") ) ? num_next(field) : 4;
    ip->ip_ttl = (field = _pb_pdata(pdu, "ttl") ) ? num_next(field) : IP_TTL_DEFAULT;
    ip->ip_tos = num_next(_pb_pdata(pdu, "tos") );

    /*
     * If no IP ID has been specified we can't use 0 as the default
     * because the kernel (at least the Linux kernel) will fill in a
     * value that will then render invalid our calculated checksum.
     *
     * Yes, I am aware that using "1" instead of "htons(1)" will yield
     * an IP ID of 256 on little endian architectures and an IP ID of
     * 1 on big endian architectures.
     */
    u16 = (field = _pb_pdata(pdu, "id") ) ? htons(num_next(field) ) : 1;
    SSVAL(ip, offsetof(struct ip_hdr, ip_id), u16);

    if (  !(field = _pb_pdata(pdu, "proto") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * User didn't specify the IP protocol so we try to put in the right
	 * value. For this we multiplex based on the next PDU, i.e. if the next
	 * PDU is ICMP then the protocol field is 1, etc.
	 */

	proto = g_hash_table_lookup(protos, pb_getname(next_pdu) );
	if (proto)
	    ip->ip_p = *proto;
    } else
	/*
	 * Use for IP protocol whatever the user specified. Note that if
	 * there is no user-specified IP protocol *and* there is no
	 * next PDU then we end up here, but that's okay because
	 * num_next(NULL) is 0.
	 */
	ip->ip_p = num_next(field);

    /*
     * Put in destination IP address before source IP address so source
     * IP address can be determined based on the IP address of the egress
     * interface.
     */
    ipaddr = ip_next(_pb_pdata(pdu, "dst") );
    SIVAL(ip, offsetof(struct ip_hdr, ip_dst), ipaddr);

    if (!(field = _pb_pdata(pdu, "src") ) ) {
	/*
	 * The source IP address was not specified in the PDU definition; use
	 * the IP address of the egress interface as the source IP address.
	 */
	i = intf_open();
	if (i != NULL) {
	    memset(&dest_ip, 0, sizeof(dest_ip) );
	    dest_ip.addr_type = ADDR_TYPE_IP;
	    dest_ip.addr_bits = IP_ADDR_BITS;
	    dest_ip.addr_ip = ip->ip_dst;

	    /*
	     * Don't forget to do this or bad things will happen (dumbnet(3)
	     * says that this is required for all intf_get_*() functions.)
	     */
	    iface.intf_len = sizeof(struct intf_entry);

	    if (intf_get_dst(i, &iface, &dest_ip) == 0)
		SIVAL(ip, offsetof(struct ip_hdr, ip_src),
		      iface.intf_addr.addr_ip);

	    intf_close(i);
	}
    } else {
	ipaddr = ip_next(field);
	SIVAL(ip, offsetof(struct ip_hdr, ip_src), ipaddr);
    }

    /*
     * If an Internet header length (IHL) was specified we use
     * that, otherwise we use the correct (calculated) length.
     */
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;

    ip->ip_hl = (field = _pb_pdata(pdu, "ihl") )
		? num_next(field) : (sizeof(struct ip_hdr)+ opts_len)/4;

    u16  = _pb_pdata(pdu, "df") ? IP_DF : 0;
    u16 += _pb_pdata(pdu, "mf") ? IP_MF : 0;
    u16 += _pb_pdata(pdu, "rf") ? IP_RF : 0;
    u16 += num_next(_pb_pdata(pdu, "fragoff") ) & IP_OFFMASK;
    u16 = htons(u16);
    SSVAL(ip, offsetof(struct ip_hdr, ip_off), u16);
}

static void
postbuild(const GNode *pdu, void *dest, void *prev_pdu_hdr _U_)
{
    struct ip_hdr *ip;
    uint16_t u16;
    void *field;
    size_t opts_len, payload_len;

    ip = dest;

    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    u16 = (field = _pb_pdata(pdu, "length") ) ? htons(num_next(field) )
			    : htons(sizeof(struct ip_hdr) + opts_len + payload_len);
    SSVAL(ip, offsetof(struct ip_hdr, ip_len), u16);

    if ( (field = _pb_pdata(pdu, "cksum") ) )
	u16 = htons(num_next(field) );
    else {
	/* Needs to be 0 before calculating the cksum per RFC 791 */
	SSVAL(ip, offsetof(struct ip_hdr, ip_sum), 0);

	u16 = _pb_cksum(dest, sizeof(struct ip_hdr) + opts_len);
    }
    SSVAL(ip, offsetof(struct ip_hdr, ip_sum), u16);
}

#if 0
static void
pdu_ipv4hdr_dumper(pdu_t *p, const char *prefix)
{
    struct ipv4hdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Internet header length: %s\n", prefix,
	   hdr_data->ihl ? num_info(hdr_data->ihl) : "automatic");
    printf("%s    ID: %s\n", prefix, num_info(hdr_data->ip_id) );
    printf("%s    Time-to-live: %s\n", prefix, num_info(hdr_data->ip_ttl) );
    printf("%s    Protocol: %s\n", prefix, num_info(hdr_data->ip_proto) );
    printf("%s    TOS: %s\n", prefix, num_info(hdr_data->ip_tos) );
    printf("%s    Total length: %s\n", prefix,
	   hdr_data->tot_len ? num_info(hdr_data->tot_len) : "automatic");
    printf("%s    Source address: %s\n", prefix,
	   num_info(hdr_data->source_ip) );
    printf("%s    Destination address: %s\n", prefix,
	   num_info(hdr_data->destination_ip) );
    printf("%s    IP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
    printf("%s    Fragment offset: %s\n", prefix,
	   num_info(hdr_data->fragoff) );
    printf("%s    IP flags set: ", prefix);
    if (hdr_data->rf || hdr_data->df || hdr_data->mf) {
	if (hdr_data->rf)
	    printf("Reserved ");
	if (hdr_data->df)
	    printf("DF ");
	if (hdr_data->mf)
	    printf("MF");
	printf("\n");
    } else
	printf("none set\n");
}
#endif

/********************************************************************
 *			       Options                              *
 ********************************************************************/

static void
ipv4opt_raw(const GNode *option, void *dest, void *curr_pdu_hdr _U_,
	    void *prev_pdu _U_)
{
    struct payload *raw_option;

    raw_option = ( (struct node_data *) option->data)->_data_parm.data;

    memcpy(dest, raw_option->data, raw_option->len);
}

static void
ipv4opt_eol(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	    void *prev_pdu _U_)
{
    *(uint8_t *) dest = IP_OPT_EOL;
}

static void
ipv4opt_nop(const GNode *option _U_, void *dest,
	    void *curr_pdu_hdr _U_, void *prev_pdu _U_)
{
    *(uint8_t *) dest = IP_OPT_NOP;
}

static void
ipv4opt_rr(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	   void *prev_pdu _U_)
{
    GNode *child;
    struct node_data *node_data;
    uint8_t *len_ptr;
    const char *opt_name;
    numspec_t *field;

    opt_name = ( (struct node_data *) option->data)->name;

    if (!strcmp(opt_name, "rr") )
	*(uint8_t *) dest++ = IP_OPT_RR;
    else if (!strcmp(opt_name, "lsrr") )
	*(uint8_t *) dest++ = IP_OPT_LSRR;
    else
	*(uint8_t *) dest++ = IP_OPT_SSRR;

    len_ptr = (uint8_t *) dest++; /* Skip putting in a length for now */

    *(uint8_t *) dest++ = num_next(_pb_pdata(option, "ptr") );

    for (child = g_node_first_child(option);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	/* We only care about "router"; everything else ("length",
	 * "type") is handled above.
	 */
	if (!strcmp(node_data->name, "router") ) {
	    SIVAL(dest, 0, ip_next(node_data->_data_parm.data) );
	    dest += sizeof(ip_addr_t);
	}
    }

    if ( (field = _pb_pdata(option, "length") ) )
	/* Put in fake length */
	*(uint8_t *) len_ptr = num_next(field);
    else 
	/* Calculate real length */
	*(uint8_t *) len_ptr = (uint8_t *) dest - len_ptr + 1;
}

/*
 * Returns size of IPv4 raw, rr, lsrr, or ssrr options. All other IPv4
 * options have fixed sizes and therefore don't need a .get_len method
 * in their option definition structures.
 */
static size_t
option_len(const GNode *option)
{
    struct payload *payload;
    const char *opt_name;
    size_t opt_size;
    struct node_data *node_data;
    GNode *child;

    opt_name = ( (struct node_data *) option->data)->name;

    if (!strcmp(opt_name, "raw") ) {
	payload = ( (struct node_data *) option->data)->_data_parm.data;
	opt_size = payload->len;
    } else {
	/* RR, LSRR, or SSRR */

	opt_size = 1 /* option type */ + 1 /* option length */
		   + 1 /* ptr */;

	for (child = g_node_first_child(option);
	     child;
	     child = g_node_next_sibling(child) ) {
	    node_data = child->data;

	    /* We only care about "router"; everything else ("length",
	     * "type", "ptr") is handled above.
	     */
	    if (!strcmp(node_data->name, "router") )
		opt_size += sizeof(ip_addr_t);
	}
    }

    return opt_size;
}


static const pdu_option_t pdu_option_ipv4raw = {
    .name = "raw",
    .description = "Raw option",
    .build = &ipv4opt_raw,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4eol = {
    .name = "eol",
    .description = "End of option list",
    .len = sizeof(uint8_t),
    .build = &ipv4opt_eol,
};

static const pdu_option_t pdu_option_ipv4nop = {
    .name = "nop",
    .description = "No operation (pad)",
    .len = sizeof(uint8_t),
    .build = &ipv4opt_nop,
};

static const pdu_option_t pdu_option_ipv4rr = {
    .name = "rr",
    .description = "IPv4 record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4lsrr = {
    .name = "lsrr",
    .description = "IPv4 loose route, record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4ssrr = {
    .name = "ssrr",
    .description = "IPv4 strict route, record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_t pdu_ipv4 = {
    .name = "ip",
    .description = "IPv4 packet",
    .documented_in = "RFC 791",
    .len = sizeof(struct ip_hdr),
    .fields = (field_t []) {
	{.name = "version", .type = PDU_FTYPE_UINT8},
	{.name = "ttl", .type = PDU_FTYPE_UINT8},
	{.name = "proto", .type = PDU_FTYPE_UINT8},
	{.name = "src", .type = PDU_FTYPE_IP},
	{.name = "dst", .type = PDU_FTYPE_IP},
	{.name = "id", .type = PDU_FTYPE_UINT16},
	{.name = "tos", .type = PDU_FTYPE_UINT8},
	{.name = "length", .type = PDU_FTYPE_UINT16},
	{.name = "fragoff", .type = PDU_FTYPE_UINT16},
	{.name = "cksum", .type = PDU_FTYPE_UINT16},
	{.name = "df", .type = PDU_FTYPE_BIT},
	{.name = "mf", .type = PDU_FTYPE_BIT},
	{.name = "rf", .type = PDU_FTYPE_BIT},
	{.name = "ihl", .type = PDU_FTYPE_UINT8},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "nop", .type = PDU_FTYPE_BIT}, 
	    {.name = "eol", .type = PDU_FTYPE_BIT},
	    {.name = "raw", .type = PDU_FTYPE_DATA},
	    {.name = "lsrr", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "length", .type = PDU_FTYPE_UINT8}, 
		{.name = "ptr", .type = PDU_FTYPE_UINT8}, 
		{.name = "router", .type = PDU_FTYPE_IP}, 
		{.name = NULL} }, .defval = NULL},
	    {.name = "ssrr", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "length", .type = PDU_FTYPE_UINT8}, 
		{.name = "ptr", .type = PDU_FTYPE_UINT8}, 
		{.name = "router", .type = PDU_FTYPE_IP}, 
		{.name = NULL} }, .defval = NULL},
	    {.name = "rr", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "length", .type = PDU_FTYPE_UINT8}, 
		{.name = "ptr", .type = PDU_FTYPE_UINT8}, 
		{.name = "router", .type = PDU_FTYPE_IP}, 
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "ip",
    .build = &build,
    .postbuild = &postbuild,
};

void
_pb_register_ipv4(void)
{
    /*
     * Create hash table for protocol name to protocol value
     * mappings. This is used by the 1st pass PDU builder.
     */
    protos = _pb_new_hash(protocol_mappings);

    _pb_register_protocol(&pdu_ipv4);

    _pb_register_protocol_option("ip", &pdu_option_ipv4raw);
    _pb_register_protocol_option("ip", &pdu_option_ipv4eol);
    _pb_register_protocol_option("ip", &pdu_option_ipv4nop);
    _pb_register_protocol_option("ip", &pdu_option_ipv4rr);
    _pb_register_protocol_option("ip", &pdu_option_ipv4lsrr);
    _pb_register_protocol_option("ip", &pdu_option_ipv4ssrr);
}
