/* util.c
 
   Utility functions for the packets module

   Copyright (C) 2007-2014 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include "missing.h"

#include "xmalloc.h"
#include "xstrdup.h"
#include "packets-priv.h"

/*
 * Returns a pointer to a character string that represents an "absolute"
 * variable name.
 *
 * The returned string is dynamically allocated via g_malloc(), so the caller
 * must call g_free() to free it.
 */
char *
_pkt_varname(proto_node *node, const char *ns_name)
{
    field_info *fi;
    GString *varname;
    proto_tree *root;
    proto_node *parent;
    const char *node_name;

    root = proto_tree_get_root(node);

    fi = PITEM_FINFO(node);

    varname = g_string_new(fi->hfinfo->abbrev);

    for (parent = proto_item_get_parent(node);
	 parent != root;
	 parent = proto_item_get_parent(parent) ) {
	node_name = fi->hfinfo->abbrev;

	fi = PITEM_FINFO(parent);

	if (strstr(varname->str, fi->hfinfo->abbrev) == varname->str
	    && strcmp(node_name, fi->hfinfo->abbrev) != 0)
	    /*
	     * The original name of the node we want to create a variable name
	     * for starts with the name of the parent node, and the name of the
	     * current node is not the same as the name of the parent node. In
	     * this case we do not prepend the name of the parent node to the
	     * original node name. This handles the case of an ICMP error
	     * generated for an ICMP message since without the second condition
	     * we would end up with something like "icmp.type" twice instead of
	     * "icmp.type" and "icmp.icmp.type".
	     */
	    continue;

	g_string_prepend_c(varname, '.');
	g_string_prepend(varname, fi->hfinfo->abbrev);
    }

    /* Prepend the name of the Tcl namespace, if we have one */
    if (ns_name) {
	g_string_prepend(varname, "::");
	g_string_prepend(varname, ns_name);
    }

    return g_string_free(varname, FALSE);
}

void
_pkt_createvar(Tcl_Interp *interp, const char *name, Tcl_Obj *value,
	       GHashTable *dissection_vars)
{
    Tcl_Obj *list_obj;

    /*
     * Store the names of Tcl variables we create. Create Tcl lists for
     * dissection results that use the same field name to avoid
     * overwriting fields that have the same name.
     */
    if (g_hash_table_lookup(dissection_vars, name) ) {
	/*
	 * A Tcl variable with this same name already exists.
	 * Treat variable as a list an append to it.
	 */
	list_obj = Tcl_GetVar2Ex(interp, name, NULL, 0);
	Tcl_ListObjAppendElement(interp, list_obj, value);
    } else {
	/*
	 * A Tcl variable with this same name does not exist
	 * yet; create variable and add to the hash table.
	 */
	Tcl_SetVar2Ex(interp, name, NULL, value, 0);
	g_hash_table_insert(dissection_vars, (gpointer) name, (gpointer) 1);
    }
}

/*
 * Looks in the "pdu(list)" Tcl list for the name of the layer
 * of the nth element in the list.
 */
char *
_pkt_layername(Tcl_Interp *interp, const char *ns_name, int n)
{
    Tcl_Obj *listObj, *obj;
    int llength;
    char *varname;

    varname = _pkt_prepend_ns_name(ns_name, "pdu(list)");
    listObj = Tcl_GetVar2Ex(interp, varname, NULL, 0);
    free(varname);
    Tcl_ListObjLength(interp, listObj, &llength);

    if (n < 0 || n >= llength)
	return NULL;
    else {
	Tcl_ListObjIndex(interp, listObj, n, &obj);
	return Tcl_GetStringFromObj(obj, NULL);
    }
}

/*
 * Prepend namespace n to variable name v if n is not NULL.
 */
char *
_pkt_prepend_ns_name(const char *n, const char *v)
{
    char *new;
    size_t len;

    if (n) {
	len = strlen(n) + sizeof("::") + strlen(v);
	new = xmalloc(len);
	strlcpy(new, n, len);
	strlcat(new, "::", len);
	strlcat(new, v, len);
	return new;
    } else
	return xstrdup(v);
}
