#ifndef NOMAD_JUKEBOX_H
#define NOMAD_JUKEBOX_H

#include <glib-object.h>

#include <glib/gqueue.h>
#include <glib/gthread.h>

#include <semaphore.h>

#include <libnjb.h>

#include "nomad-jukebox-track.h"

#define NOMAD_TYPE_JUKEBOX  (nomad_jukebox_get_type())
#define NOMAD_JUKEBOX(obj)  (G_TYPE_CHECK_INSTANCE_CAST( ( obj ), \
                              NOMAD_TYPE_JUKEBOX, \
                              NomadJukebox ) )
#define NOMAD_IS_JUKEBOX(obj)  (G_TYPE_CHECK_INSTANCE_TYPE( ( obj ), \
                                 NOMAD_TYPE_JUKEBOX ) )

typedef struct {
	GObjectClass parent_spot;

	/* track signals */
	void (*scanning_tracks)();
	void (*track_add)( guint total_tracks, const NomadTrack *track );
	void (*scanned_tracks)( guint total );

	void (*transfer)( const gchar *uri,
			  gboolean upload, guint64 sent, guint64 total );
	void (*upload_complete)();
	void (*download_complete)();
	void (*downloaded_track)( const gchar *uri, NomadTrack *track );

	void (*track_remove)( const NomadTrack *track );

	void (*track_changed)( const NomadTrack *orig,
			       const NomadTrack *track );

	/* playlist signals */
	void (*scanning_playlists)();
	void (*playlist_add)( guint total_tracks,
			      const gchar *name );
	void (*scanned_playlists)( guint total );
	void (*playlist_track_erased)( const gchar *playlist,
				       const gchar *artist,
				       const gchar *album,
				       const gchar *title );
	void (*playlist_tracks_erased)( const gchar *playlist );
	void (*playlist_remove)( const gchar *name );
	void (*playlist_renamed)( const gchar *orignal, 
				  const gchar *new_name );

	/* playback signals */
	void (*play_begin)( gpointer track );
	void (*play_progress)( guint progress, gboolean changed,
			       gpointer track );
	void (*play_finished)();


	/* return value signals */
	void (*time)( const gchar *time_string );
	void (*usage)( guint64 total, guint64 free, guint64 used );

	void (*eax)( gpointer eax );

	void (*error)( const gchar *message );

} NomadJukeboxClass;


typedef struct  {
	GObject parent_object;
	
	gboolean connected;

	njb_t *njb;
	njbid_t *njbid;

	gchar *firmware;

	gchar *owner;

	GHashTable *tracks;
	GHashTable *playlists;
	GHashTable *playlistids;

	gchar *transfer_file;

	GThread *transfer_thread;
	GQueue  *job_queue;
	GMutex  *job_queue_mutex;
	
	sem_t transfer_semaphore;

	NomadTrack *playing;

	gint abort;

	GMutex *lock;
} NomadJukebox;


GType  nomad_jukebox_get_type( void );
NomadJukebox *nomad_jukebox_new( njb_t *njb );

void nomad_jukebox_lock( NomadJukebox *jukebox );
void nomad_jukebox_unlock( NomadJukebox *jukebox );

gboolean nomad_jukebox_aquire( NomadJukebox *jukebox );
void nomad_jukebox_release( NomadJukebox *jukebox );

const gchar *nomad_jukebox_get_idstring( NomadJukebox *jukebox );
const gchar *nomad_jukebox_get_firmware( NomadJukebox *jukebox );
const gchar *nomad_jukebox_get_prodname( NomadJukebox *jukebox );

guint nomad_jukebox_get_num_tracks( NomadJukebox *jukebox );
guint nomad_jukebox_get_num_playlists( NomadJukebox *jukebox );
guint nomad_jukebox_get_num_data_files( NomadJukebox *jukebox );

gboolean nomad_jukebox_get_power( NomadJukebox *jukebox );
void nomad_jukebox_get_time( NomadJukebox *jukebox );

void nomad_jukebox_build_tracklist( NomadJukebox *jukebox );
void nomad_jukebox_build_playlist( NomadJukebox *jukebox );

void nomad_jukebox_delete_tracks( NomadJukebox *jukebox, GList *list );

void nomad_jukebox_delete_tracks_from_playlist( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name );

void nomad_jukebox_delete_files( NomadJukebox *jukebox, GList *list );

void nomad_jukebox_set_metadata( NomadJukebox *jukebox,
				    const NomadTrack *track );

void nomad_jukebox_create_playlist( NomadJukebox *jukebox,
                                       const gchar *name );
void nomad_jukebox_delete_playlist( NomadJukebox *jukebox, 
				       const gchar *name );
void nomad_jukebox_rename_playlist( NomadJukebox *jukebox, 
				       const gchar *orig,
                                       const gchar *name );
void nomad_jukebox_add_tracks_to_playlist( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist );

const gchar *nomad_jukebox_get_ownerstring( NomadJukebox *jukebox );
void nomad_jukebox_set_ownerstring( NomadJukebox *jukebox, 
				       const gchar *owner );

void nomad_jukebox_getusage( NomadJukebox *jukebox );

void nomad_jukebox_get_eax( NomadJukebox *jukebox );

void nomad_jukebox_set_volume( NomadJukebox *jukebox, guint8 volume );
void nomad_jukebox_set_muting( NomadJukebox *jukebox, 
				  gboolean mutingstatus );
void nomad_jukebox_set_eq_active( NomadJukebox *jukebox, gboolean eqactive );
void nomad_jukebox_set_bass( NomadJukebox *jukebox, gint8 bass );
void nomad_jukebox_set_midrange( NomadJukebox *jukebox, gint8 midrange );
void nomad_jukebox_set_treble( NomadJukebox *jukebox, gint8 treble );
void nomad_jukebox_set_effamt( NomadJukebox *jukebox, gint8 effamt );
void nomad_jukebox_set_midfreq( NomadJukebox *jukebox, gint8 freq );
void nomad_jukebox_set_effect( NomadJukebox *jukebox, gint8 effect );
void nomad_jukebox_set_headphone( NomadJukebox *jukebox, gint8 hpmode );
void nomad_jukebox_set_rearmode( NomadJukebox *jukebox, gint8 rearmode );

void nomad_jukebox_upload( NomadJukebox *jukebox, GList *list );
void nomad_jukebox_download( NomadJukebox *jukebox, 
				const gchar *dest,
				const gchar *format,
				GList *list );


void nomad_jukebox_play_track( NomadJukebox *jukebox, guint trackid );
void nomad_jukebox_play_elapsed( NomadJukebox *jukebox );
void nomad_jukebox_play_stop( NomadJukebox *jukebox );

void nomad_jukebox_abort( NomadJukebox *jukebox );

GHashTable *nomad_jukebox_request_tracks( NomadJukebox *jukebox );
GHashTable *nomad_jukebox_request_playlists( NomadJukebox *jukebox );

NomadTrack *nomad_jukebox_request_track( NomadJukebox *jukebox,
					       guint id );

#endif
