/*  Neutrino:  neutrino-id3.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <errno.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#endif

#include <glib/gconvert.h>
#include <glib/gunicode.h>

#include <glib/gtypes.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/types.h>

#include "neutrino-id3.h"
#include "neutrino-genre.h"

#include "mp3file.h"

#include "neutrino-util.h"
#include "nomad-util.h"

#define ID3V1_TITLE_SIZE 30
#define ID3V1_ARTIST_SIZE 30
#define ID3V1_ALBUM_SIZE 30
#define ID3V1_YEAR_SIZE 4
#define ID3V1_COMMENT_SIZE 30
#define ID3V1_DOT_1_COMMENT_SIZE 28
typedef struct {
	gchar title[ ID3V1_TITLE_SIZE ];
	gchar artist[ ID3V1_ARTIST_SIZE ];
	gchar album[ ID3V1_ALBUM_SIZE ];
	gchar year[ ID3V1_YEAR_SIZE ];
	union {
		gchar comment[ ID3V1_COMMENT_SIZE ];
		struct {
			gchar comment1dot_1[ ID3V1_DOT_1_COMMENT_SIZE ];
			gchar empty;
			gchar track;
		} __attribute((packed)) V1dot1;
	} comment;
	gchar genre;
} __attribute((packed)) ID3V1;


typedef struct {
	gchar id[ 3 ];
	gshort version;
	gchar  flags;
	glong  size;
} __attribute((packed)) ID3V2Header;
static const int ID3V2HeaderSize = sizeof( ID3V2Header );

typedef struct {
	gchar id[ 4 ];
	glong size;
	gshort flags;
} __attribute((packed)) ID3V2Frame;

typedef struct {
	gchar encoding;
	gchar string[];
} __attribute((packed)) ID3V2TextFrame;

typedef struct {
	ID3V2Header header;
	gboolean hasExtended;
	void *frames;
} ID3V2;

struct NeutrinoID3Details {
	gchar *uri;
	
	guint64 size;
	
	ID3V1 *v1tag;
	ID3V2 *v2tag;
};


static gboolean neutrino_string_is_empty( const gchar *field, guint field_size );
static gboolean neutrino_id3_scan_id3v1( NeutrinoID3 *id3 );
static gboolean neutrino_id3_scan_id3v2( NeutrinoID3 *id3 );

static void *neutrino_id3_frame_get_data( const ID3V2Frame *frame, gulong *length );
static ID3V2Frame *neutrino_id3_get_frame( const NeutrinoID3 *id3, 
					   const gchar *frame_name );

static gchar *neutrino_id3_text_frame_get_string( const ID3V2Frame *frame,
						  gchar *encoding );
static gulong neutrino_id3_fill_text_frame( ID3V2Frame *frame,
					    const gchar *text,
					    guint length,
					    const gchar *fname );

static guint neutrino_id3_to_v2_length( guint length );
static guint neutrino_id3_from_v2_length( guint length );

static void neutrino_id3_class_init( NeutrinoID3Class *klass );
static void neutrino_id3_init( NeutrinoID3 *id3 );
static void neutrino_id3_finalize( GObject *object );
static void neutrino_id3_set_prop( GObject *object, guint prop_id, 
				   const GValue *value, GParamSpec *spec );
static void neutrino_id3_get_prop( GObject *object, guint prop_id, 
				   GValue *value, GParamSpec *spec );

NeutrinoID3 *neutrino_id3_new( const gchar *uri )
{
	NeutrinoID3 *id3;
	
	id3 = NEUTRINO_ID3( g_object_new( neutrino_id3_get_type(), NULL ) );
	
	id3->details->uri = g_strdup( uri );
	
	if( ! neutrino_id3_scan_id3v2( id3 ) ) {
		if( ! neutrino_id3_scan_id3v1( id3 ) ) {
			/* no id3 tag */
			g_object_unref( G_OBJECT( id3 ) );
			id3 = NULL;
		}
	}

	return id3;
}

gchar *neutrino_id3_get_filesize( NeutrinoID3 *id3 )
{
	gchar *size;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	size = g_strdup_printf( "%llu", id3->details->size );

	return size;
}

gchar *neutrino_id3_get_length( NeutrinoID3 *id3 )
{
	ID3V2Frame *frame;
	gchar *length;
	
	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );
	
	length = NULL;

	if( id3->details->v2tag ) {
		frame = neutrino_id3_get_frame( id3, "TLEN" );
		
		if( frame ) {
			gchar *text;
			gulong millisec;
			
			text = neutrino_id3_text_frame_get_string( frame,
								   NULL );
			
			millisec = strtoul( text, NULL, 10 );
			millisec /= 1000;
			g_free( text );
			length = seconds_to_mmss( millisec );
		}
	} else {
		frame = NULL;
	}

	if( ! frame ) {
		gchar *size;

		size = neutrino_id3_get_filesize( id3 );

		length = length_from_file( id3->details->uri, size );

		g_free( size );
	}

	return length;
}


gchar *neutrino_id3_get_title( NeutrinoID3 *id3 )
{
	gchar *title;
	gchar *temp;
	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	title = NULL;

	if( id3->details->v1tag && 
	    ! neutrino_string_is_empty( id3->details->v1tag->title, ID3V1_TITLE_SIZE ) ) {

		title = g_strndup( id3->details->v1tag->title, 
				   ID3V1_TITLE_SIZE );
		temp = neutrino_util_charset_convert( title );
		if( temp ) {
			g_free( title );
			title = temp;
		}

		title = g_strstrip( title );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = neutrino_id3_get_frame( id3, "TIT2" );
		if( frame ) {
			title = neutrino_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return title;
}

gchar *neutrino_id3_get_artist( NeutrinoID3 *id3 )
{
	gchar *artist;
	gchar *temp;
	
	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	artist = NULL;

	if( id3->details->v1tag && 
	    ! neutrino_string_is_empty( id3->details->v1tag->artist,
					ID3V1_ARTIST_SIZE )  ) {
		artist = g_strndup( id3->details->v1tag->artist, 
				    ID3V1_ARTIST_SIZE);
		temp = neutrino_util_charset_convert( artist );
		if( temp ) {
			g_free( artist );
			artist = temp;
		}

		artist = g_strstrip( artist );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = neutrino_id3_get_frame( id3, "TPE1" );
		if( ! frame ) {
			frame = neutrino_id3_get_frame( id3, "TPE2" );
		}
		if( ! frame ) {
			frame = neutrino_id3_get_frame( id3, "TPE3" );
		}
		if( ! frame ) {
			frame = neutrino_id3_get_frame( id3, "TPE4" );
		}
		if( frame ) {
			artist = neutrino_id3_text_frame_get_string( frame,
								     NULL );
		}
	}

	return artist;
}

gchar *neutrino_id3_get_album( NeutrinoID3 *id3 )
{
	gchar *album;
	gchar *temp;
	
	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	album = NULL;

	if( id3->details->v1tag && 
	    ! neutrino_string_is_empty( id3->details->v1tag->album, ID3V1_ALBUM_SIZE ) ) {
		album = g_strndup( id3->details->v1tag->album, 
				   ID3V1_ALBUM_SIZE );
		temp = neutrino_util_charset_convert( album );
		if( temp ) {
			g_free( album );
			album = temp;
		}

		album = g_strstrip( album );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = neutrino_id3_get_frame( id3, "TALB" );
		if( frame ) {
			album = neutrino_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return album;
}

gchar *neutrino_id3_get_year( NeutrinoID3 *id3 )
{
	gchar *year;
	gchar *temp;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	year = NULL;

	if( id3->details->v1tag && 
	    ! neutrino_string_is_empty( id3->details->v1tag->year, ID3V1_YEAR_SIZE ) ) {
		year = g_strndup( id3->details->v1tag->year, ID3V1_YEAR_SIZE );
		temp = neutrino_util_charset_convert( year );
		if( temp ) {
			g_free( year );
			year = temp;
		}

		year = g_strstrip( year );
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = neutrino_id3_get_frame( id3, "TYER" );

		if( ! frame ) {
			frame = neutrino_id3_get_frame( id3, "TPRO" );
		}
		if( ! frame ) {
			frame = neutrino_id3_get_frame( id3, "TCOP" );
		}
		if( frame ) {
			year = neutrino_id3_text_frame_get_string( frame,
								   NULL ); 
		}
	}


	return year;
}

gchar *neutrino_id3_get_comment( NeutrinoID3 *id3 )
{
	gchar *comment;
	gchar *temp;
	
	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	comment = NULL;

	if( id3->details->v1tag && 
	    ! neutrino_string_is_empty( id3->details->v1tag->comment.comment,
					ID3V1_COMMENT_SIZE )  ) {
		comment = g_strndup( id3->details->v1tag->comment.comment,
				     ID3V1_COMMENT_SIZE );
		temp = neutrino_util_charset_convert( comment );
		if( temp ) {
			g_free( comment );
			comment = temp;
		}

		comment = g_strstrip( comment );
	}

	return comment;
}

gchar *neutrino_id3_get_track( NeutrinoID3 *id3 )
{
	gchar *track;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	track = NULL;

	if( id3->details->v1tag ) {
	
		if( id3->details->v1tag->comment.V1dot1.empty == '\0' &&
		    id3->details->v1tag->comment.V1dot1.track != '\0' ) {
			track = g_strdup_printf("%i",
						id3->details->v1tag->comment.V1dot1.track);
		}
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;

		frame = neutrino_id3_get_frame( id3, "TRCK" );
		if( frame ) {
			track = neutrino_id3_text_frame_get_string( frame,
								    NULL );
		}
	}

	return track;
}

gchar *neutrino_id3_get_genre( NeutrinoID3 *id3 )
{
	gchar *genre;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), NULL );

	genre = NULL;

	if( id3->details->v1tag ) {
		const gchar *genre_string;

		guint i;

		i = id3->details->v1tag->genre;
		if( i <= genre_count ) {
			genre_string = neutrino_id3_genre_table[ i ];
			
			if( *genre_string ) {
				genre = g_strdup( genre_string );
			}
		}
	} else if( id3->details->v2tag ) {
		ID3V2Frame *frame;
		
		frame = neutrino_id3_get_frame( id3, "TCON" );
		if( frame ) {
			genre = neutrino_id3_text_frame_get_string( frame,
								    NULL );
		}
	}
	
	
	return genre;
}


void neutrino_id3_tag_v1dot1( const gchar *uri,
			      const gchar *title,
			      const gchar *artist,
			      const gchar *album,
			      const gchar *year,
			      const gchar *track,
			      const gchar *genre )
{
	ID3V1 *tag;
	gulong tracknum;

#ifndef HAVE_GNOME_VFS
	FILE *handle;
	gboolean ok;
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
#endif
	
	tag = g_new0( ID3V1, 1 );
       
	if( title ) {
		strcpy( tag->title, title );
	}
	if( artist ) {
		strcpy( tag->artist, artist );
	}
	if( album ) {
		strcpy( tag->album, album );
	}
	if( year ) {
		strcpy( tag->year, year );
	}

	if( track ) {
		tracknum = strtoul( track, NULL, 10 );
		if( tracknum > 255 ) {
			tracknum = 0;
		}
	} else {
		tracknum = 0;
	}
	tag->comment.V1dot1.track = tracknum;

	if( genre ) {
		guint index;

		for( index = 0; index < genre_count; ++ index ) {
			if( ! strcmp( genre, neutrino_id3_genre_table[ index ] ) ) {
				tag->genre = index + 1;
			}
		}
		if( tag->genre == 0 ) {
			tag->genre = 255;
		}
	} else {
		tag->genre = 255;
	}

	/* tag built, now write it */
#ifndef HAVE_GNOME_VFS
	handle = fopen( uri, "a+" );
	ok = ( handle != NULL );
	if( ok ) {
		fwrite( "TAG", 1, 3, handle );
		fwrite( tag, 1, sizeof( ID3V1 ), handle );
		fclose( handle );
	}
#else
	info = NULL;
	handle = NULL;
	result = gnome_vfs_open( &handle, uri,
				 GNOME_VFS_OPEN_READ | 
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );
	if( result == GNOME_VFS_OK ) {
		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
	}
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle, GNOME_VFS_SEEK_START,
					 info->size );
	}
	if( result == GNOME_VFS_OK ) {
		/* write the tag */
		GnomeVFSFileSize wrote;
		result = gnome_vfs_write( handle, "TAG", 3, &wrote );
		if( result == GNOME_VFS_OK ) {
			result = gnome_vfs_write( handle, tag,
						  sizeof( ID3V1 ), &wrote );
		}
	}

	if( info ) {
		gnome_vfs_file_info_unref( info );
	}
	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif
	g_free( tag );
}

void neutrino_id3_tag_v2( const gchar *uri,
			  const gchar *title,
			  const gchar *artist,
			  const gchar *album,
			  const gchar *year,
			  const gchar *track,
			  const gchar *genre,
			  const gchar *length )
{
	ID3V2Header *header;
	glong header_size;
	void *frames;
	ID3V2Frame *frame;
	guint size;
#ifndef HAVE_GNOME_VFS
	FILE *handle;
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
#endif
	gchar *tempuri;
	gboolean ok;
	gsize written;

	gint index;
	gchar *fullgenre;

	guint seconds;
	gchar *len;
	
	header = g_new0( ID3V2Header, 1 );

	strncpy( header->id, "ID3", 3 );

	header->version = 0x0003;
	header->flags = 0;

	/* calc header size */
	header_size = sizeof( ID3V2Frame ) * 6;
	/* +1 for encoding */

	header_size += strlen( title ) + 1;
	header_size += strlen( artist ) + 1;
	header_size += strlen( album ) + 1;
	header_size += strlen( year ) + 1;
	header_size += strlen( track ) + 1;

	/* we need to convert genre to "genre number" NIL "genre name" NIL */
	for( index = 0; index < genre_count; ++ index ) {
		if( ! strcmp( genre, neutrino_id3_genre_table[ index ] ) ) {
			break;
		}
	}
	if( index == genre_count ) {
		index = -1;
	}
	fullgenre = g_strdup_printf( "%i%c%s", index, '\0', genre );
	/* fullgenre strlen will stop at the embedded \0, so we +1 for that
	   and add on the length of genre, + it's \0 and of course the byte
	   for the encoding type */
	header_size += strlen( fullgenre ) + 1 + strlen( genre ) + 1;

	/* we need to convert length to milliseconds */
	seconds = mmss_to_seconds( length );
	if( seconds != 0 ) {
		len = g_strdup_printf( "%u", seconds * 1000 );
		header_size += strlen( len ) + 1;
		/* add on the size of another frame, as by default we
		   only have 6 frames */
		header_size += sizeof( ID3V2Frame );
	} else {
		len = NULL;
	}

	frames = g_malloc0( header_size );

	header->size = neutrino_id3_to_v2_length( header_size );

	/* fill in frames data */
	frame = (ID3V2Frame*)frames;
	size = 0;

	size += neutrino_id3_fill_text_frame( frame, 
					      title, strlen( title ),
					      "TIT2" );
	frame = (ID3V2Frame*)( frames + size );

	size += neutrino_id3_fill_text_frame( frame, 
					      artist, strlen( artist ),
					      "TPE1" );
	frame = (ID3V2Frame*)( frames + size );

	size += neutrino_id3_fill_text_frame( frame, 
					      album, strlen( album ),
					      "TALB" );
	frame = (ID3V2Frame*)( frames + size );

	size += neutrino_id3_fill_text_frame( frame, 
					      year, strlen( year ),
					      "TYER" );
	frame = (ID3V2Frame*)( frames + size );

	size += neutrino_id3_fill_text_frame( frame, 
					      track, strlen( track ),
					      "TRCK" );

	frame = (ID3V2Frame*)( frames + size );

	size += neutrino_id3_fill_text_frame( frame, 
					      fullgenre, 
					      strlen( fullgenre ) +
					      1 +
					      strlen( genre ),
					      "TCON" );
	g_free( fullgenre );

	if( len ) {
		frame = (ID3V2Frame*)( frames + size );

		size += neutrino_id3_fill_text_frame( frame, len, 
						      strlen( len ),
						      "TLEN" );

		g_free( len );
	}
	frame = (ID3V2Frame*)( frames + size );

	/* write header, followed by frames, followed by data from uri */
	handle = NULL;
	ok = FALSE;

	tempuri = neutrino_util_create_backup_filename( uri );
#ifndef HAVE_GNOME_VFS
	handle = fopen( tempuri, "w+" );
	ok = ( handle != NULL );
	if( ok ) {
		fwrite( header, 1, sizeof( ID3V2Header ), handle );
		fwrite( frames, 1, header_size, handle );
	}
#else
	result = gnome_vfs_create( &handle, tempuri,
				   GNOME_VFS_OPEN_WRITE,
				   TRUE,
				   GNOME_VFS_PERM_USER_READ |
				   GNOME_VFS_PERM_USER_WRITE |
				   GNOME_VFS_PERM_GROUP_READ |
				   GNOME_VFS_PERM_OTHER_READ );
	if( result == GNOME_VFS_OK ) {
		/* write the tag */
		GnomeVFSFileSize wrote;
		result = gnome_vfs_write( handle, header,
					  sizeof( ID3V2Header ), &wrote );
		if( result == GNOME_VFS_OK ) {
			result = gnome_vfs_write( handle, frames,
						  header_size, &wrote );
			ok = TRUE;
		}
	}
#endif

	if( ok ) {
		if( neutrino_util_concat_file_to_handle( handle, uri,
							      0, &written ) ) {
#ifndef HAVE_GNOME_VFS
			fclose( handle );
			handle = fopen( uri, "w" );
			if( handle ) {
				neutrino_util_concat_file_to_handle( handle, tempuri, 0, &written );
				fclose( handle );
			}
#else
			gnome_vfs_close( handle );
			gnome_vfs_move( tempuri, uri, TRUE );
#endif
		}
	} else if( handle ) {
#ifndef HAVE_GNOME_VFS
		fclose( handle );
#else
		gnome_vfs_close( handle );
#endif
	}

	g_free( tempuri );

	g_free( frames );
	g_free( header );
}

void neutrino_id3_strip_v1dot1( const gchar *uri )
{
#ifndef HAVE_GNOME_VFS
	FILE *handle;
	gboolean ok;
	gsize size;
	gsize offset;
	gchar buffer[ 3 ];

	offset = 0;
	handle = fopen( uri, "r" );
	ok = ( handle != NULL );

	if( ok ) {
		ok = ( fseek( handle, -128, SEEK_END ) == 0 );
	}
	if( ok ) {
		offset = ftell( handle );
		strcpy( buffer, "\0\0\0" );
		size = fread( buffer, 1, 3, handle );
		ok = ( size == 3 );
	}
	if( handle != NULL ) {
		fclose( handle );
	}
	if( ok && ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
		truncate( uri, offset );	
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileSize size;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | 
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );

	size = 0;
	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new();
		
		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
		if( result == GNOME_VFS_OK ) {
			size = info->size;
		}
		gnome_vfs_file_info_unref( info );
	}
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle,
					 GNOME_VFS_SEEK_START,
					 size - 128 );
	}
	if( result == GNOME_VFS_OK ) {
		/* read 3 bytes, see if == 'TAG' */
		gchar buffer[ 3 ];
		GnomeVFSFileSize got;

		strcpy( buffer, "\0\0\0" );
		result = gnome_vfs_read( handle, buffer, 3, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
			/* we got and ID3V1 tag */
			gnome_vfs_truncate_handle( handle,
						   size - 128 );
		}
	}
	
	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif		
}

void neutrino_id3_strip_v2( const gchar *uri )
{
	ID3V2 *tag;

#ifndef HAVE_GNOME_VFS
	FILE *handle;
	gboolean ok;
	gchar buffer[ ID3V2HeaderSize ];
	gsize got;
	gchar *tempuri;
	
	tag = NULL;
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		/* read possible header */
		got = fread( buffer, 1, ID3V2HeaderSize, handle );
		ok = ( got == ID3V2HeaderSize );
	}
	if( ok && ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
		/* we got and ID3V2 tag */
		tag = g_new0( ID3V2, 1 );

		memcpy( tag, buffer, ID3V2HeaderSize );
	
		tag->header.size = 
			neutrino_id3_from_v2_length( tag->header.size);

		tag->hasExtended = ( tag->header.flags & 64 );

		tag->frames = g_new0( gchar, tag->header.size );
		/* read frames */

		got = fread( tag->frames, 1, tag->header.size, handle );
		ok = ( got == tag->header.size );
	}
	if( ok ) {
		/* got an id3 v2 tag */
		tempuri = neutrino_util_create_backup_filename( uri );
		ok = neutrino_util_concat_handle_to_file( handle,
						tempuri, 0, &got );
		fclose( handle );
		handle = NULL;
		if( ok ) {
			gchar *temp;
			gchar *temp2;
			
			temp = g_strconcat( "\"", uri, "\"", NULL );
			remove( temp );
			temp2 = g_strconcat( "\"", tempuri, "\"", NULL );
			rename( temp2, temp );
			
			g_free( temp );
			g_free( temp2 );
		}
		g_free( tempuri );
	}
	
	if( handle ) {
		fclose( handle );
	}
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	
	tag = NULL;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ |
				 GNOME_VFS_OPEN_RANDOM |
				 GNOME_VFS_OPEN_WRITE );

	if( result == GNOME_VFS_OK ) {
		/* read possible header */
		gchar buffer[ ID3V2HeaderSize ];
		GnomeVFSFileSize got;

		result = gnome_vfs_read( handle, buffer, ID3V2HeaderSize, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
			/* we got and ID3V2 tag */
			tag = g_new0( ID3V2, 1 );

			memcpy( tag, buffer, ID3V2HeaderSize );
	
			tag->header.size = 
				neutrino_id3_from_v2_length( tag->header.size);

			tag->hasExtended = ( tag->header.flags & 64 );

			tag->frames = g_new0( gchar, tag->header.size );
			/* read frames */
			result = gnome_vfs_read( handle,
						 tag->frames,
						 tag->header.size,
						 &got );
			if( result == GNOME_VFS_OK ) {
				/* got an id3 v2 tag */
				gchar *tempuri;
				GnomeVFSFileSize written;

				tempuri = neutrino_util_create_backup_filename( uri );
				neutrino_util_concat_handle_to_file( handle,
						tempuri, 0, &written );
				gnome_vfs_close( handle );
				handle = NULL;
				gnome_vfs_move( tempuri, uri, TRUE );
				g_free( tempuri );
			}
		}
	}

	if( handle ) {
		gnome_vfs_close( handle );
	}
#endif	
	if( tag ) {
		if( tag->frames ) {
			g_free( tag->frames );
		}
		g_free( tag );
	}

}


/* static stuff */

/* mp3info fills fields with spaces (at least in interactive mode),
   this sucks and is against the spec which states fields should be
   NIL padded */
static gboolean neutrino_string_is_empty( const gchar *field, 
					  guint field_size )
{
	gboolean empty = FALSE;
	int i = 0;

	while( i < field_size && field[ i ] != '\0' && 
	       g_ascii_isspace( field[ i ] ) ) {
		i ++;
	}

	if( i == field_size || ( i = 0 && field[ i ] != '\0' ) ) {
		empty = TRUE;
	}

	return empty;
}

static gboolean neutrino_id3_scan_id3v1( NeutrinoID3 *id3 )
{
#ifndef HAVE_GNOME_VFS
	const gchar *uri;
	FILE *handle;
	gboolean ok;
	gsize size;
	gchar buffer[ 4 ];

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), FALSE );
	g_return_val_if_fail( ! id3->details->v2tag, TRUE );

	uri = id3->details->uri;
	id3->details->v1tag = NULL;
	
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		ok = ( fseek( handle, -128, SEEK_END ) == 0 );
	}
	if( ok ) {
		id3->details->size = ftell( handle ) + 128;
		memset( buffer, 0, 4 ); 
		size = fread( buffer, 1, 3, handle );
		ok = ( size == 3 );
	}
	if( ok && ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
		/* found */
		id3->details->v1tag = g_new0( ID3V1, 1 );

		size = fread( id3->details->v1tag, 1, 125, handle );
		ok = ( size == 125 );
		if( ! ok ) {
			g_free( id3->details->v1tag );
			id3->details->v1tag = NULL;
		}
	}
	if( handle != NULL ) {
		fclose( handle );
	}

	return ok;
#else
	const gchar *uri;
	gboolean found;
	GnomeVFSHandle *handle;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), FALSE );
	g_return_val_if_fail( ! id3->details->v2tag, TRUE );

	uri = id3->details->uri;
	id3->details->v1tag = NULL;

	found = FALSE;
	info = NULL;
	handle = NULL;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | GNOME_VFS_OPEN_RANDOM );
	if( result == GNOME_VFS_OK ) {
		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
	}
	if( result == GNOME_VFS_OK ) {
		id3->details->size = info->size;
		result = gnome_vfs_seek( handle,
					 GNOME_VFS_SEEK_START,
					 info->size - 128 );
	}
	if( result == GNOME_VFS_OK ) {
		/* read 3 bytes, see if == 'TAG' */
		gchar buffer[ 3 ];
		GnomeVFSFileSize got;

		strcpy( buffer, "\0\0\0" );
		result = gnome_vfs_read( handle, buffer, 3, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "TAG", strlen( "TAG" ) ) ) {
			/* we got and ID3V1 tag */
			id3->details->v1tag = g_new0( ID3V1, 1 );

			result = gnome_vfs_read( handle, id3->details->v1tag,
						 125, &got );

		}
		if( result == GNOME_VFS_OK ) {
			found = TRUE;
		} else if( id3->details->v1tag ) {
			g_free( id3->details->v1tag );
			id3->details->v1tag = NULL;
		}
	}
	
	if( info ) {
		gnome_vfs_file_info_unref( info );
	}
	
	if( handle ) {
		gnome_vfs_close( handle );
	}
		
	return found;
#endif
}

static gboolean neutrino_id3_scan_id3v2( NeutrinoID3 *id3 )
{
#ifndef HAVE_GNOME_VFS
	const gchar *uri;
	FILE *handle;
	gboolean ok;
	gchar buffer[ ID3V2HeaderSize ];
	gsize got;
	gchar *tempuri;
	ID3V2 *tag;

	uri = id3->details->uri;
	
	tag = NULL;
	handle = fopen( uri, "r+" );
	ok = ( handle != NULL );

	if( ok ) {
		/* read possible header */
		got = fread( buffer, 1, ID3V2HeaderSize, handle );
		ok = ( got == ID3V2HeaderSize );
	}
	if( ok ) {
		ok = ( ! strncmp( buffer, "ID3", strlen( "ID3" ) ) );
	}
	if( ok ) {
		/* we got and ID3V2 tag */
		tag = id3->details->v2tag = g_new0( ID3V2, 1 );

		memcpy( tag, buffer, ID3V2HeaderSize );
	
		tag->header.size = 
			neutrino_id3_from_v2_length( tag->header.size);

		tag->hasExtended = ( tag->header.flags & 64 );

		tag->frames = g_new0( gchar, tag->header.size );
		/* read frames */

		got = fread( tag->frames, 1, tag->header.size, handle );
		ok = ( got == tag->header.size );
	}
	if( ok ) {
		ok = ( fseek( handle, 0, SEEK_END ) == 0 );
		if( ok ) {
			id3->details->size = ftell( handle );
		}
	}
	if( tag && ! ok ) {
		if( id3->details->v2tag->frames ) {
			g_free( id3->details->v2tag->frames );
		}
		g_free( id3->details->v2tag );
		id3->details->v2tag = NULL;
		tag = NULL;
	}
	return ( tag != NULL );
#else
	const gchar *uri;
	gboolean found;
	GnomeVFSHandle *handle;
	GnomeVFSResult result;

	g_return_val_if_fail( NEUTRINO_IS_ID3( id3 ), FALSE );

	uri = id3->details->uri;

	found = FALSE;

	result = gnome_vfs_open( &handle, uri, 
				 GNOME_VFS_OPEN_READ | GNOME_VFS_OPEN_RANDOM );

	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new();

		result = gnome_vfs_get_file_info_from_handle( handle,
							      info,
							      GNOME_VFS_FILE_FLAGS_NONE );
		if( result == GNOME_VFS_OK ) {
			id3->details->size = info->size;
		}
		gnome_vfs_file_info_unref( info );
	}

	if( result == GNOME_VFS_OK ) {
		/* read possible header */
		gchar buffer[ ID3V2HeaderSize ];
		GnomeVFSFileSize got;

		result = gnome_vfs_read( handle, buffer, ID3V2HeaderSize, &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( buffer, "ID3", strlen( "ID3" ) ) ) {
			/* we got and ID3V2 tag */
			ID3V2 *tag;

			tag = id3->details->v2tag = g_new0( ID3V2, 1 );

			memcpy( tag, buffer, ID3V2HeaderSize );

			/* FIXME: is right? */
			tag->header.flags = tag->header.flags;		
			tag->header.size = 
				neutrino_id3_from_v2_length(tag->header.size);

			tag->hasExtended = ( tag->header.flags & 64 );

			tag->frames = g_new0( gchar, tag->header.size );
			/* read frames */
			result = gnome_vfs_read( handle,
						 tag->frames,
						 tag->header.size,
						 &got );
			if( result == GNOME_VFS_OK ) {
				found = TRUE;
			}

		}
		if( ! found && id3->details->v2tag ) {
			if( id3->details->v2tag->frames ) {
				g_free( id3->details->v2tag->frames );
			}
			g_free( id3->details->v2tag );
			id3->details->v2tag = NULL;
		}
	}

	if( handle ) {
		gnome_vfs_close( handle );
	}

	return found;
#endif
}

static void *neutrino_id3_frame_get_data( const ID3V2Frame *frame, gulong *length )
{
	gshort flags;		
	void *ret;
	gulong dsize;

	flags = g_ntohs( frame->flags );

	ret = (void*)frame;
	ret += sizeof( ID3V2Frame );

	dsize = neutrino_id3_from_v2_length( frame->size );

	if( flags & 16384 ) {
		/* a - tag alter preservation */
	}
	if( flags & 8192 ) {
		/* b - file alter preservation */
	}
	if( flags & 4096 ) {
		/* c - read only */
	}
	if( flags & 128 ) {
		/* h - grouping id */
		ret ++;
	}
	if( flags & 8 ) {
		/* k - compression */
	}
	if( flags & 4 ) {
		/* m - encryption */
		ret ++;
	}
	if( flags & 2 ) {
		/* n - unsynchronisation */
	}
	if( flags & 1 ) {
		/* p - frame data length indicator -
		   the actual length of the data in the frame,
		   before decompression etc */
		dsize = neutrino_id3_from_v2_length( (*(long*)ret) );
		ret += 4;
	}

	if( length ) {
		*length = dsize;
	}

	return ret;
}

static ID3V2Frame *neutrino_id3_get_frame( const NeutrinoID3 *id3, 
					   const gchar *frame_name )
{
	ID3V2Frame *frame;
	void *start;
	void *end;
	glong size;
	
	start = id3->details->v2tag->frames;
	size = id3->details->v2tag->header.size;
	if( id3->details->v2tag->hasExtended ) {
		size -= *(long*)start;
		start += *(long*)start;
	}
	
	/* %0abc0000 %0h00kmnp */

	frame = NULL;
	end = start + size;
	/* start points to the start of our frames */
	while( start < end ) {
		glong real_size;

		frame = (ID3V2Frame*)start;

		if( ! strncmp( frame->id, frame_name, 4 ) ) {
			/* this is the frame we want */
			break;
		} else if( frame->id[ 0 ] == '\0' ) {
			frame = NULL;
			break;
		} else {
			void *data;
			
			data = neutrino_id3_frame_get_data( frame, 
							    &real_size );
			start = data + real_size;
			frame = NULL;
		}
	}

	return frame;
}

static gchar *neutrino_id3_text_frame_get_string( const ID3V2Frame *frame,
						  gchar *encoding )
{
	ID3V2TextFrame *text;
	gulong length;
	gchar *ret;
	const gchar *enc;

	text = (ID3V2TextFrame*) neutrino_id3_frame_get_data( frame, &length );

	if( encoding ) {
		*encoding = text->encoding;
	}
	
	switch( text->encoding ) {
	case 1:
		enc = "UTF-16";
		break;
	case 2:
		enc = "UTF-16BE";
		break;
	case 3:
		enc = "UTF-8";
		break;
	case 0:
	default:
		enc = "ISO-8859-1";
		break;
	}

	if( length > 1 ) {
		ret = g_new0( gchar, length );
		memcpy( ret, text->string, length - 1 );

		if( text->encoding != 3 ) {
			gchar *temp;
			gint len;

			temp = g_convert( ret, length - 1, "UTF-8",
					  enc, NULL, &len, NULL );
			g_free( ret );
			ret = temp;
		}
	} else {
		ret = NULL;
	}

	return ret;
}

static gulong neutrino_id3_fill_text_frame( ID3V2Frame *frame,
					    const gchar *text,
					    guint length,
					    const gchar *fname )
{
	ID3V2TextFrame *text_frame;

	text_frame = (ID3V2TextFrame*)((void*)frame) + sizeof( ID3V2Frame );

	text_frame->encoding = 3;

	/* memcpy so we can deal with embeded \0 */
	memcpy( text_frame->string, text, length );

	strncpy( frame->id, fname, 4 );
	/* +1 for encoding */
	frame->size = neutrino_id3_to_v2_length( length + 1 );

	return ( length + 1 + sizeof( ID3V2Frame ) );
}

static guint neutrino_id3_to_v2_length( guint length )
{
	if( length & 0x80 ) {
		length += 0x100;
		length ^= 0x80;
	}
	if( length & 0x8000 ) {
		length += 0x1000;
		length ^= 0x8000;
	}
	if( length & 0x800000 ) {
		length += 0x1000000;
		length ^= 0x800000;
	}
	if( length & 0x80000000 ) {
		length ^= 0x80000000;
	}

	return g_htonl( length );
}

static guint neutrino_id3_from_v2_length( guint length )
{
	length = g_ntohl( length );
	
	length = ( (length & 0xFF000000 ) >> 1 ) +
		( (length & 0xFF0000 ) >> 1 ) +
		( (length & 0xFF00) >> 1 ) + (length & 0xFF);

	return length;
}


/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void
neutrino_id3_class_init( NeutrinoID3Class *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = neutrino_id3_finalize;
	object_class->get_property = neutrino_id3_get_prop;
	object_class->set_property = neutrino_id3_set_prop;
}

static void
neutrino_id3_init( NeutrinoID3 *id3 )
{
	id3->details = g_new0( NeutrinoID3Details, 1 );
}

static void
neutrino_id3_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	NeutrinoID3 *id3;

	id3 = NEUTRINO_ID3( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
neutrino_id3_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	NeutrinoID3 *id3;

	id3 = NEUTRINO_ID3( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
neutrino_id3_finalize( GObject *object )
{
	NeutrinoID3 *id3;

	id3 = NEUTRINO_ID3( object );

	g_free( id3->details->uri );

	if( id3->details->v1tag ) {
		g_free( id3->details->v1tag );
	}

	if( id3->details->v2tag ) {
		g_free( id3->details->v2tag );
	}

	g_free( id3->details );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType neutrino_id3_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( NeutrinoID3Class ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)neutrino_id3_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( NeutrinoID3 ),
			0, /* n_preallocs */
			(GInstanceInitFunc)neutrino_id3_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "NeutrinoID3",
					       &info, 0 );
	}

	return type;
}
