/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// AsynchMsgs.h: interface for the AsynchMsgs class.
//
//////////////////////////////////////////////////////////////////////

#ifndef ASYNCHMSGS_H
#define ASYNCHMSGS_H

#include <openssl/x509.h>
#include <openssl/evp.h>
#include <mString.h>
#include <ASN1/Asn1Cert.h>
#include <ASN1/Asn1Conf.h>
#include <PKI_CERT.h>
#include "AsynchMsgs_ASN1.h"

#include "NewPKIStore.h"

/*!
	This class holds the basic functionnalities for a requester and a responder
*/
class AsynchMsgs : public NewPKIStore
{
public:
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param e [IN] The ENGINE, can be NULL.
	 *  \param Type [IN] The type.
	 */
	AsynchMsgs(const mString & EntityName, ENGINE * e, unsigned long Type);

	/*! \brief This is the destructor.
	 */
	virtual ~AsynchMsgs();

	/*! \brief Run the asynch messages handling.
	 *  \return true on success, false on failure.
	 */
	bool Run();

	bool CreateTables(const SQL_Connection * DbConn);

	/*! \brief This function converts the errors into a inter-entity response.
	 *  \param transactionID [IN] The transaction ID.
	 *  \param recipient [IN] The recipient.
	 */
	void ERR_to_STORED_NEWPKI_RESPONSE(const Asn1OctetString & transactionID, const X509_PUBKEY * recipient);

	/*! \brief This function is used to send a inter-entity-request, it will sent it to the repositories.
	 *  \param priv_attr [IN] A private value used by children.
	 *  \param Request [IN] The request.
	 *  \param Recipient [IN] The recipient for the request.
	 *  \return true on success, false on failure.
	 */
	bool InsertRequest(unsigned long priv_attr, const NewpkiRequest & Request, const X509_PUBKEY * Recipient) const;

	/*! \brief Return the priv attr that was associated with a request.
	 *  \param transactionID [IN] The transaction ID of the request.
	 *  \param priv_attr [IN] The priv attr.
	 *  \return true on success, false on failure.
	 */
	bool GetPrivAttr(const Asn1OctetString & transactionID, unsigned long & priv_attr);

	/*! \brief Return the type that was associated with a request.
	 *  \param transactionID [IN] The transaction ID of the request.
	 *  \param type [IN] The type.
	 *  \return true on success, false on failure.
	 */
	bool GetRequestType(const Asn1OctetString & transactionID, int & type);

	/*! \brief This function is used to send a inter-entity-response, it will sent it to the repositories.
	 *  \param transactionID [IN] The transaction ID.
	 *  \param Response [IN] The response.
	 *  \param Recipient [IN] The recipient.
	 *  \return true on success, false on failure.
	 */
	bool InsertResponse(const Asn1OctetString & transactionID, const NewpkiResponse & Response, const X509_PUBKEY * Recipient) const;

	/*! \brief This function sets the list of repositories.
	 *  \param Repositories [IN] The list.
	 *  \return true on success, false on failure.
	 */
	bool SetRepositories(const mVector<RepEntryInfo> & Repositories);

	/*! \brief Event triggered when a new conf has been seeked from a repository.
	 *  \param newConf [IN] The new conf.
	 *  \return true on success, false on failure.
	 */
	virtual bool OnNewConf(const EntityConfCrypted & newConf)=0;
	
	/*! \brief Event triggered when new request has been seeked from a repository. The event handler must validate the sender, to make sure the sender is allowed to request on us.
	 *  \param Request [IN] The request.
	 *  \param Requester [IN] The requester.
	 *  \param SenderName [OUT] The name of the entity.
	 *  \return true when the requester is allowed, false the requester is not allowed.
	 */
	virtual bool Responder_ValidateRequest(const NewpkiRequest & Request, const X509_PUBKEY * Requester, mString & SenderName)=0;

	/*! \brief Event triggered when a new request is ready to be processed by the responder. Requests are treated synchronously.
	 *  \param Request [IN] The request.
	 *  \param SenderName [IN] The name of the sender, previously returned by Responder_ValidateRequester.
	 *  \param Response [OUT] The response.
	 *  \return true on success, false on failure.
	 */
	virtual bool Responder_TreatRequest(const NewpkiRequest & Request, const mString & SenderName, NewpkiResponse & Response)=0;
	
	/*! \brief Event triggered when a new request is ready to be processed by the responder. Requests are treated asynchronously.
	 *  \param Request [IN] The request.
	 *  \param transactionId [IN] The response.
	 *  \param SenderName [IN] The name of the sender, previously returned by Responder_ValidateRequester.
	 *  \return true on success, false on failure.
	 */
	virtual bool Responder_TreatRequestAsynch(const NewpkiRequest & Request, const Asn1OctetString & transactionId, const mString & SenderName)=0;
	
	/*! \brief Event triggered when new response has been seeked from a repository. Responses are treated synchronously.
	 *  \param transactionID [IN] The transaction ID of the new response.
	 *  \param sender [IN] The response sender.
	 *  \param Response [IN] The new response.
	 *  \return true on success, false on failure.
	 */
	virtual bool Requester_OnNewResponse(const Asn1OctetString & transactionID, const X509_PUBKEY * sender, const NewpkiResponse & Response)=0;
protected:
	bool DoUpgrade(const char * Version);
	bool CreateRequester(const SQL_Connection * DbConn);
	bool CreateResponder(const SQL_Connection * DbConn);

private:
	#define RESPONDER_TABLE "responder"
	#define REQUESTER_TABLE "requester"
	
	unsigned long m_Type;

	bool Responder_SetStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS Status);
	static void ThreadRepositoriesSynchro(const NewpkiThread * Thread, void *param);
	bool DeleteResponse(const Asn1OctetString & transationID) const;
	bool SynchronizeWithRepositories(const NewpkiThread * Thread, EntityConfCrypted & myConf, CryptedNewpkiRequests & reqs, CryptedNewpkiResponses & resps, mVector<StackRequest> & m_waitingReqs, CryptedNewpkiResponses & m_waitingResps, int what);
	bool SynchronizeWithRepository(const NewpkiThread * Thread, EntityConfCrypted & myConf, CryptedNewpkiRequests & reqs, CryptedNewpkiResponses & resps, mVector<StackRequest> & m_waitingReqs, CryptedNewpkiResponses & m_waitingResps, const mString & RepName, const mString & RepAddress, unsigned int RepPort, const PKI_CERT & RepCert, PkiClient * ClientPki, mString & Err, int what);
	bool DoRepositoriesSynchro(const NewpkiThread * Thread, mVector<StackRequest> & m_waitingReqs, CryptedNewpkiResponses & m_waitingResps, int what);

	static void ThreadResponderWorker(const NewpkiThread * Thread, void *param);
	bool OnNewResponse(CryptedNewpkiResponse & response);
	bool OnNewRequest(const CryptedNewpkiRequest & request, CryptedNewpkiResponses & m_waitingResps, mString & SenderName);
	NewpkiThread hThreadRepositoriesSynchro;
	NewpkiThread hThreadResponderWorker;


	ReadersWriter m_RepositoriesLock;
	mVector<RepEntryInfo> m_Repositories;

	TransactionIds m_waitingDelResps;
	CriticalSection m_waitingDelRespsLock;

	#define SYNCH_CONF 1
	#define SYNCH_REQS 2
	#define SYNCH_RESPS 4
	#define SYNCH_ALL (SYNCH_CONF | SYNCH_REQS | SYNCH_RESPS)

	

	
	
	/*! \brief This function is used to get the list of known transactions.
	 *  \param transactionIDs [OUT] The transaction Ids.
	 *  \return true on success, false on failure.
	 */
	bool Responder_GetKnownTIDs(TransactionIds &transactionIDs);

	/*! \brief This function inserts a new request to be treated.
	 *  \param Request [IN] The request.
	 *  \return true on success, false on failure.
	 */
	bool Responder_InsertRequest(const WorkingRequest & Request);

	/*! \brief This function returns the requests that haven't been processed yet.
	 *  \param Requests [OUT] The un-processed requests.
	 *  \return true on success, false on failure.
	 */
	bool Responder_GetRequests(mVector<WorkingRequest> & Requests);

	/*! \brief This function returns the responses that haven't been sent yet.
	 *  \param Responses [OUT] The unsent responses.
	 *  \param index [IN] The position of datas to send.
	 *  \param max [IN] The maximum number of entries to return.
	 *  \return true on success, false on failure.
	 */
	bool Responder_GetUnsentResponses(mVector<CryptedNewpkiResponse> & Responses, int index = 0, int max = 0);

	/*! \brief This function sets a response.
	 *  \param Response [IN] The new response.
	 *  \return true on success, false on failure.
	 */
	bool Responder_SetResponse(const CryptedNewpkiResponse & Response) const;

	/*! \brief This function gets a response.
	 *  \param TransactionID [IN] The transaction ID.
	 *  \param Response [OUT] The response.
	 *  \return The response on success, NULL on failure.
	 */
	bool Responder_GetResponse(const Asn1OctetString & TransactionID, CryptedNewpkiResponse & Response);

	/*! \brief This function marks a response as sent.
	 *  \param TransactionID [IN] The ID of the sent response.
	 *  \return true on success, false on failure.
	 */
	bool Responder_ResponseSent(const Asn1OctetString & TransactionID);

	/*! \brief This function marks a request as 'On Hold'.
	 *  \param TransactionID [IN] The ID of the request.
	 *  \return true on success, false on failure.
	 */
	bool Responder_RequestOnHold(const Asn1OctetString & TransactionID);

	/*! \brief This function gets the status of a transaction.
	 *  \param TransactionID [IN] The transaction ID.
	 *  \param Status [OUT] The status of the transaction.
	 *  \return true on success, false on failure.
	 */
	bool Responder_GetTransactionStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS & Status);

	/*! \brief This function inserts a new request.
	 *  \param Request [IN/OUT] The request, the member internalID will be set.
	 *  \param priv_attr [IN] A private value used by children.
	 *  \param type [IN] The request type.
	 *  \return true on success, false on failure.
	 */
	bool Requester_InsertRequest(CryptedNewpkiRequest & Request, unsigned long priv_attr, int type) const;

	/*! \brief This functions returns the list of requests for which we're expecting a response.
	 *  \param transactionIDs [OUT] The un-responded transaction Ids.
	 *  \return true on success, false on failure.
	 */
	bool Requester_GetWaitingTIDs(TransactionIds &transactionIDs);
	
	/*! \brief This function marks the request as sent an assign it the transactionID given by the Respository.
	 *  \param req_id [IN] The internal ID of the request.
	 *  \return true on success, false on failure.
	 */
	bool Requester_RequestSent(unsigned long req_id);
	
	/*! \brief This function is used to delete a request.
	 *  \param req_id [IN] The internal ID of the request.
	 *  \return true on success, false on failure.
	 */
	bool Requester_DeleteTransaction(unsigned long req_id) const;
	
	/*! \brief This function is used to delete a request matching priv_attr.
	 *  \param priv_attr [IN] A private value used by children.
	 *  \return true on success, false on failure.
	 */
	bool Requester_DeleteTransactionPrivAttr(unsigned long priv_attr);
	
	/*! \brief This function returns the requests that haven't been sent yet.
	 *  \param Requests [OUT] The unsent requests.
	 *  \param index [IN] The position of datas to send.
	 *  \param max [IN] The maximum number of entries to return.
	 *  \return true on success, false on failure.
	 */
	bool Requester_GetUnsentRequests(mVector<StackRequest> & Requests, int index = 0, int max = 0);
	
	/*! \brief This function marks a request as having been processed by the responder.
	 *  \param TransactionID [IN] The processed request.
	 *  \return true on success, false on failure.
	 */
	bool Requester_RequestWasResponded(const Asn1OctetString & TransactionID);

	/*! \brief This function gets the status of a transaction.
	 *  \param TransactionID [IN] The transaction ID.
	 *  \param Status [OUT] The status of the transaction.
	 *  \return true on success, false on failure.
	 */
	bool Requester_GetTransactionStatus(const Asn1OctetString & TransactionID, TRANSACTION_STATUS & Status);

	#define ASYNCHMSGS_CREATE_1							"create table "RESPONDER_TABLE" (req_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, transactionID VARCHAR(41) NOT NULL, status INT UNSIGNED NOT NULL, type INT UNSIGNED NOT NULL, request LONGBLOB NOT NULL, response LONGBLOB NOT NULL, INDEX (transactionID), INDEX(status));"
	#define ASYNCHMSGS_CREATE_2							"create table "REQUESTER_TABLE" (req_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, transactionID VARCHAR(41) NOT NULL, priv_attr INT UNSIGNED NOT NULL, status INT UNSIGNED NOT NULL, type INT UNSIGNED NOT NULL, request LONGBLOB NOT NULL, INDEX (transactionID), INDEX(status));"

	#define RESPONDER_ASYNCHMSGS_INSERT_REQ				"INSERT INTO "RESPONDER_TABLE" (transactionID, status, request, response) VALUES ('%s', %d, '%s', '');"
	#define RESPONDER_ASYNCHMSGS_GET_TIDS				"SELECT transactionID FROM "RESPONDER_TABLE" WHERE status <> %d;"
	#define RESPONDER_ASYNCHMSGS_SET_RESPONSE			"UPDATE "RESPONDER_TABLE" SET status='%d', response='%s', request='' WHERE transactionID='%s';"
	#define RESPONDER_ASYNCHMSGS_GET_RESPONSE			"SELECT response FROM "RESPONDER_TABLE" WHERE transactionID='%s';"
	#define RESPONDER_ASYNCHMSGS_SET_STATUS				"UPDATE "RESPONDER_TABLE" SET status='%d', response='' WHERE transactionID='%s';"
	#define RESPONDER_ASYNCHMSGS_GET_STATUS				"SELECT status FROM "RESPONDER_TABLE" WHERE transactionID='%s';"
	#define RESPONDER_ASYNCHMSGS_GET_REQS_BY_STATE		"SELECT request FROM "RESPONDER_TABLE" WHERE status='%d' ORDER BY req_id DESC LIMIT 5;"
	#define RESPONDER_ASYNCHMSGS_GET_RESP_BY_STATE		"SELECT response FROM "RESPONDER_TABLE" WHERE status='%d' ORDER BY req_id ASC %s;"
	#define RESPONDER_ASYNCHMSGS_GET_TID_STATUS			"SELECT status FROM "RESPONDER_TABLE" WHERE transactionID='%s';"
	
	#define REQUESTER_ASYNCHMSGS_INSERT_REQ				"INSERT INTO "REQUESTER_TABLE" (transactionID, status, priv_attr, type) VALUES ('', %d, '%ld', '%d');"
	#define REQUESTER_ASYNCHMSGS_SET_REQ				"UPDATE "REQUESTER_TABLE" SET transactionID='%s', request='%s' WHERE req_id='%ld';"
	#define REQUESTER_ASYNCHMSGS_GET_TIDS_BY_STATE		"SELECT transactionID FROM "REQUESTER_TABLE" WHERE status='%d' ORDER BY req_id ASC LIMIT 0,100;"
	#define REQUESTER_ASYNCHMSGS_SET_SENT				"UPDATE "REQUESTER_TABLE" SET status='%d', request='' WHERE req_id='%ld';"
	#define REQUESTER_ASYNCHMSGS_SET_TID_STATUS			"UPDATE "REQUESTER_TABLE" SET status='%d' WHERE transactionID='%s';"
	#define REQUESTER_ASYNCHMSGS_DELETE_REQ				"DELETE FROM "REQUESTER_TABLE" WHERE req_id='%ld';"
	#define REQUESTER_ASYNCHMSGS_GET_REQS_BY_STATE		"SELECT req_id, request FROM "REQUESTER_TABLE" WHERE status='%d' ORDER BY req_id ASC %s;"
	#define REQUESTER_ASYNCHMSGS_SET_REQ_STATE			"UPDATE "REQUESTER_TABLE" SET status='%d' WHERE transactionID='%s';"
	#define REQUESTER_ASYNCHMSGS_DELETE_REQ_PRIV_ATTR	"DELETE FROM "REQUESTER_TABLE" WHERE priv_attr='%ld';"
	#define REQUESTER_ASYNCHMSGS_GET_TID_STATUS			"SELECT status FROM "REQUESTER_TABLE" WHERE transactionID='%s';"
	#define REQUESTER_ASYNCHMSGS_GET_TID_PRIV_ATTR		"SELECT priv_attr FROM "REQUESTER_TABLE" WHERE transactionID='%s';;"
	#define REQUESTER_ASYNCHMSGS_GET_TID_TYPE			"SELECT type FROM "REQUESTER_TABLE" WHERE transactionID='%s';;"
};

#endif
