/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef RASTORE_ASN1_H
#define RASTORE_ASN1_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <ASN1/Asn1Helper.h>

/*!
	This structure represents the signature of a profile.
*/
typedef struct st_RA_STORE_SIG
{
	ASN1_BIT_STRING * signature;		//!< The signature.
	X509_ALGOR * sig_alg;				//!< The algo.
}RA_STORE_SIG;
DECLARE_ASN1_ITEM(RA_STORE_SIG)


/*!
	This structure represents the private datas for a certificate request.
*/
typedef struct st_RA_STORE_PRIVDATAS
{
	ASN1_UTF8STRING * passwd;				//!< The PKCS#12 password.
	RSA * privkey;							//!< The private key.
}RA_STORE_PRIVDATAS;
DECLARE_ASN1_ITEM(RA_STORE_PRIVDATAS)



/*!
  This class is the representation of RA_STORE_SIG
*/
class RaStoreSig : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	RaStoreSig();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RaStoreSig(const RaStoreSig & other);

	/*! \brief This is the destructor.
	 */
	~RaStoreSig();


	/*! \brief This function converts a RA_STORE_SIG to a RaStoreSig.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const RA_STORE_SIG * Datas);

	/*! \brief This function converts a RaStoreSig to a RA_STORE_SIG.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(RA_STORE_SIG ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of RA_STORE_SIG.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_ALGOR::sig_alg.
	 *  \param c_sigAlg [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sigAlg(const X509_ALGOR * c_sigAlg);

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	const X509_ALGOR * get_sigAlg() const;

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	X509_ALGOR * get_sigAlg();


	/*! \brief This function sets ASN1_BIT_STRING::signature.
	 *  \param c_signature [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signature(const ASN1_BIT_STRING * c_signature);

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	const ASN1_BIT_STRING * get_signature() const;

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	ASN1_BIT_STRING * get_signature();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a RaStoreSig to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RaStoreSig & other);
	/*! \brief This member is an empty instance of RaStoreSig.
	 */
	static RaStoreSig EmptyInstance;

private:
	X509_ALGOR * m_sigAlg;
	ASN1_BIT_STRING * m_signature;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of RA_STORE_PRIVDATAS
*/
class RaStorePrivdatas : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	RaStorePrivdatas();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	RaStorePrivdatas(const RaStorePrivdatas & other);

	/*! \brief This is the destructor.
	 */
	~RaStorePrivdatas();


	/*! \brief This function converts a RA_STORE_PRIVDATAS to a RaStorePrivdatas.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const RA_STORE_PRIVDATAS * Datas);

	/*! \brief This function converts a RaStorePrivdatas to a RA_STORE_PRIVDATAS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(RA_STORE_PRIVDATAS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of RA_STORE_PRIVDATAS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::passwd.
	 *  \param c_passwd [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_passwd(const mString & c_passwd);

	/*! \brief This function returns ASN1_UTF8STRING::passwd.
	 *  \return The value of ASN1_UTF8STRING::passwd.
	 */
	const mString & get_passwd() const;

	/*! \brief This function returns ASN1_UTF8STRING::passwd.
	 *  \return The value of ASN1_UTF8STRING::passwd.
	 */
	mString & get_passwd();


	/*! \brief This function sets RSA::privkey.
	 *  \param c_privkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_privkey(const RSA * c_privkey);

	/*! \brief This function returns RSA::privkey.
	 *  \return The value of RSA::privkey.
	 */
	const RSA * get_privkey() const;

	/*! \brief This function returns RSA::privkey.
	 *  \return The value of RSA::privkey.
	 */
	RSA * get_privkey();


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a RaStorePrivdatas to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const RaStorePrivdatas & other);
	/*! \brief This member is an empty instance of RaStorePrivdatas.
	 */
	static RaStorePrivdatas EmptyInstance;

private:
	mString m_passwd;
	RSA * m_privkey;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef RASTORE_ASN1_H
