# -*- coding: utf-8 -*-
#
#  hanayu.py - a "花柚" compatible Saori module for ninix
#  Copyright (C) 2002-2010 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - usetime, line, radar 以外の形式のグラフへの対応.
# - タイトルの縦書き.(GTK+の対応待ち.)

import sys
import os
import math
import time

if 'DISPLAY' in os.environ:
    if 'gtk' not in sys.modules:
        try:
            import pygtk
            pygtk.require('2.0')
        except ImportError:
            pass
    import gtk
    import pango
    import ninix.pix
else:
    gtk = None

from ninix.dll import SAORI


class Saori(SAORI):

    __DBNAME = 'HANAYU.db'

    def __init__(self):
        SAORI.__init__(self)
        self.graphs = {}
        self.data = {}

    def check_import(self):
        if not gtk:
            return 0
        else:
            return 1

    def setup(self):
        self.dbpath = os.path.join(self.dir, self.__DBNAME)
        self.graphs = {}
        self.data = self.read_hanayu_txt(self.dir)
        if self.data:
            self.read_db()
            return 1
        else:
            return 0

    def read_hanayu_txt(self, dir):
        graphs = {}
        try:
            f = open(os.path.join(dir, 'hanayu.txt'), 'r')
            data = {}
            name = ''
            tmp_name = ''
            for line in f:
                line = line.strip()
                if not line:
                    continue
                if line.startswith('//'):
                    continue
                if line.startswith('['): # bln.txt like format
                    graphs[name] = data
                    data = {}
                    end = line.find(']')
                    if end < 0:
                        end = len(line)
                    name = line[1:end]
                elif line == '{': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                    name = tmp_name ## FIXME
                elif line == '}': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                    name = ''
                elif ',' in line:
                    key, value = [x.strip() for x in line.split(',', 1)]
                    data[key] = value
                elif not name:
                    tmp_name = line
            if data:
                graphs[name] = data
            return graphs
        except:
            return None

    def finalize(self):
        for name in self.graphs.keys():
            self.graphs[name].destroy()
        self.graphs = {}
        self.data = {}
        self.write_db()
        return 1

    def time_to_key(self, secs, offset):
        year = int(time.strftime('%Y', time.localtime(secs)))
        month = int(time.strftime('%m', time.localtime(secs)))
        day = int(time.strftime('%d', time.localtime(secs)))
        target_time = time.mktime(
            (year, month, day + offset, 0, 0, 0, -1, -1, -1))
        year = int(time.strftime('%Y', time.localtime(target_time)))
        month = int(time.strftime('%m', time.localtime(target_time)))
        day = int(time.strftime('%d', time.localtime(target_time)))
        key = str(year * 10000 + month * 100 + day)
        return key, year, month, day

    def read_db(self):
        self.seven_days = []
        current_time = self.last_update = time.time()
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        try:
            f = open(self.dbpath)
        except:
            return
        else:
            ver = None
            for line in f:
                line = line.strip()
                if not line:
                    continue
                if ver is None:
                    if line == '# Format: v1.0':
                        ver = 1
                    continue
                if ',' not in line:
                    continue
                key, value = line.split(',', 1)
                for index in range(7):
                    if self.seven_days[index][0] == key:
                        self.seven_days[index][4] = float(value)
            f.close()

    def update_db(self):
        current_time = time.time()
        old_seven_days = []
        old_seven_days.extend(self.seven_days)
        self.seven_days = []
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        for i in range(7):
            key = old_seven_days[i][0]
            for j in range(7):
                if self.seven_days[j][0] == key:
                    self.seven_days[j][4] = old_seven_days[i][4]
                    if j == 6:
                        self.seven_days[j][4] = self.seven_days[j][4] + \
                                                (current_time - \
                                                 self.last_update) / \
                                                 (60.0 * 60.0)
        self.last_update = current_time

    def write_db(self):
        self.update_db()
        try:
            f = open(self.dbpath, 'w')
        except IOError:
            print 'HANAYU: cannot write database (ignored)'
        else:
            f.write('# Format: v1.0\n')
            for index in range(7):
                f.write('%s, %s\n' % \
                        (self.seven_days[index][0],
                         self.seven_days[index][4]))
            f.close()

    def execute(self, argument):
        if not argument:
            return self.RESPONSE[400]
        command = argument[0]
        if command == 'show':
            if len(argument) >= 2:
                name = argument[1]
                if name in self.graphs:
                    self.graphs[name].destroy()
            else:
                name = ''
            if name not in self.data:
                return self.RESPONSE[400]
            if 'graph' in self.data[name] and \
                   self.data[name]['graph'] in ['line', 'bar',
                                                'radar', 'radar2']:
                graph_type = self.data[name]['graph']
            else:
                graph_type = 'usetime'
            if graph_type == 'usetime':
                self.update_db()
                new_args = []
                for index in range(7):
                    date = ''.join((str(self.seven_days[index][2]),
                                    '/',
                                    str(self.seven_days[index][3])))
                    new_args.append(date)
                    hours = self.seven_days[index][4]
                    new_args.append(hours)
                self.graphs[name] = Line(
                    self.dir, self.data[name], new_args, 0, 24)
            elif graph_type == 'line':
                self.graphs[name] = Line(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'bar':
                self.graphs[name] = Bar(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'radar':
                self.graphs[name] = Radar(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'radar2':
                self.graphs[name] = Radar2(
                    self.dir, self.data[name], argument[2:])
        elif command == 'hide':
            if len(argument) >= 2:
                name = argument[1]
            else:
                name = ''
            if name in self.graphs:
                self.graphs[name].destroy()
            else:
                return self.RESPONSE[400]
        else:
            return self.RESPONSE[400]
        return self.RESPONSE[204]

class Graph:

    width = 450
    height = 340

    def __init__(self, dir, data, args=[], limit_min=None, limit_max=None):
        self.dir = dir
        self.data = data
        self.args = args
        self.min = limit_min
        self.max = limit_max
        self.create_window()
        self.draw_title()
        self.draw_frame()
        self.draw_graph()
        self.window.show()

    def create_window(self):
        self.window = gtk.Window()
        self.window.set_title('花柚') # UTF-8
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect('delete_event', self.delete)
        self.window.connect('button_press_event', self.button_press)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        self.x = left + scrn_w // 2
        self.y = top + scrn_h // 4
        self.window.move(self.x, self.y)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.set_size_request(self.width, self.height)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.layout = pango.Layout(self.darea.get_pango_context())
        pixbuf = None
        if 'background.filename' in self.data:
            path = os.path.join(
                self.dir, self.data['background.filename'].replace('\\', '/'))
            
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(path, is_pnr=0)
            except:
                pixbuf = None
        self.surface_pixmap = gtk.gdk.Pixmap(
            None, self.width, self.height, self.darea.get_visual().depth)
        cmap = self.darea.get_colormap()
        self.surface_gc = self.surface_pixmap.new_gc()
        self.surface_gc.foreground = cmap.alloc_color('#FFFFFF')
        self.surface_pixmap.draw_rectangle(self.surface_gc, True, 0, 0,
                                           self.width, self.height)
        if pixbuf:
            width = pixbuf.get_width()
            height = pixbuf.get_height()
            xoffset = (self.width - width) // 2
            yoffset = (self.height - height) // 2
            self.surface_pixmap.draw_pixbuf(self.surface_gc, pixbuf,
                                            0, 0, xoffset, yoffset,
                                            width, height,
                                            gtk.gdk.RGB_DITHER_NONE, 0, 0)
        self.darea.window.draw_drawable(self.surface_gc, self.surface_pixmap,
                                        0, 0, 0, 0, self.width, self.height)

    def draw_title(self):
        fontcolor_r = fontcolor_g = fontcolor_b = 0
        if 'font.color' in self.data:
            fontcolor_r = int(self.data['font.color'][:2], 16)
            fontcolor_g = int(self.data['font.color'][2:4], 16)
            fontcolor_b = int(self.data['font.color'][4:6], 16)
        else:
            if 'font.color.r' in self.data:
                fontcolor_r = int(self.data['font.color.r'])
            if 'font.color.g' in self.data:
                fontcolor_g = int(self.data['font.color.g'])
            if 'font.color.b' in self.data:
                fontcolor_b = int(self.data['font.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (fontcolor_r, fontcolor_g, fontcolor_b))
        if 'title' in self.data:
            self.title = unicode(self.data['title'], 'Shift_JIS', 'ignore')
        font_size = 12 # pixel
        self.font_desc = pango.FontDescription()
        self.font_desc.set_family('Sans') # FIXME
        self.font_desc.set_size(font_size * pango.SCALE)
        text = ''
        i = 0
        while i < len(self.title):
            if text:
                ''.join((text, '\n'))
            ''.join((text, self.title[i]))
            i += 1
        self.layout.set_font_description(self.font_desc)
        self.layout.set_wrap(pango.WRAP_CHAR)
        self.layout.set_alignment(pango.ALIGN_CENTER)
        self.layout.set_text(text)
        self.surface_pixmap.draw_layout(self.surface_gc, 20, 58, self.layout)

    def draw_frame(self):
        pass

    def draw_graph(self):
        pass

    def redraw(self, widget, event):
        x, y, w, h = event.area
        if self.surface_pixmap:
            self.darea.window.draw_drawable(
                self.surface_gc, self.surface_pixmap, x, y, x, y, w, h)

    def button_press(self, window, event):
        if event.type == gtk.gdk.BUTTON_PRESS:
            self.window.begin_move_drag(
                event.button, int(event.x_root), int(event.y_root),
                event.time)
        elif event.type == gtk.gdk._2BUTTON_PRESS: # double click
            self.destroy()
        return True

    def delete(self, window, event):
        return True

    def destroy(self):
        if self.window:
            self.window.destroy()
            self.window = None
            self.timeout_id = None

class Line(Graph):

    def draw_frame(self):
        framecolor_r = framecolor_g = framecolor_b = 0
        if 'frame.color' in self.data:
            framecolor_r = int(self.data['frame.color'][:2], 16)
            framecolor_g = int(self.data['frame.color'][2:4], 16)
            framecolor_b = int(self.data['frame.color'][4:6], 16)
        else:
            if 'frame.color.r' in self.data:
                framecolor_r = int(self.data['frame.color.r'])
            if 'frame.color.g' in self.data:
                framecolor_g = int(self.data['frame.color.g'])
            if 'frame.color.b' in self.data:
                framecolor_b = int(self.data['frame.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (framecolor_r, framecolor_g, framecolor_b))
        frame_width = 2
        if 'frame.width' in self.data:
            frame_width = int(self.data['frame.width'])
        self.surface_gc.line_width = frame_width
        self.surface_pixmap.draw_line(self.surface_gc,
                                      60, 48, 60, 260)
        self.surface_pixmap.draw_line(self.surface_gc,
                                      60, 260, 420, 260)

    def draw_graph(self):
        num = len(self.args) // 2
        step = 368 // num
        for index in range(num):
            self.layout.set_text(self.args[index * 2])
            w, h = self.layout.get_pixel_size()
            pos_x = 60 + index * step + step // 2 - w // 2
            pos_y = 268
            self.surface_pixmap.draw_layout(
                self.surface_gc, pos_x, pos_y, self.layout)
        if self.min is not None:
            limit_min = self.min
        else:
            limit_min = self.args[1]
            for index in range(2, num):
                if self.args[index * 2] < limit_min:
                    limit_min = self.args[index * 2]
        if self.max is not None:
            limit_max = self.max
        else:
            limit_max = self.args[1]
            for index in range(2, num):
                if self.args[index * 2] > limit_max:
                    limit_max = self.args[index * 2]
        linecolor_r = linecolor_g = linecolor_b = 0
        if 'line.color' in self.data:
            linecolor_r = int(self.data['line.color'][:2], 16)
            linecolor_g = int(self.data['line.color'][2:4], 16)
            linecolor_b = int(self.data['line.color'][4:6], 16)
        else:
            if 'line.color.r' in self.data:
                linecolor_r = int(self.data['line.color.r'])
            if 'line.color.g' in self.data:
                linecolor_g = int(self.data['line.color.g'])
            if 'line.color.b' in self.data:
                linecolor_b = int(self.data['line.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (linecolor_r, linecolor_g, linecolor_b))
        line_width = 2
        if 'line.width' in self.data:
            line_width = int(self.data['line.width'])
        self.surface_gc.line_width = line_width
        for index in range(1, num):
            src_x = 60 + (index - 1) * step + step // 2
            src_y = 220 - int(
                168 * self.args[(index - 1) * 2 + 1] / (limit_max - limit_min))
            dst_x = 60 + index * step + step // 2
            dst_y = 220 - int(168 * self.args[index * 2 + 1] / (limit_max - limit_min))
            self.surface_pixmap.draw_line(self.surface_gc,
                                          src_x, src_y, dst_x, dst_y)
        for index in range(num):
            image, mask = None, None
            if self.args[index * 2 + 1] == limit_min and \
               'mark0.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark0.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        image, mask = ninix.pix.create_pixmap_from_file(path)
                    except:
                        image, mask = None, None
            elif self.args[index * 2 + 1] == limit_max and \
                 'mark2.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark2.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        image, mask = ninix.pix.create_pixmap_from_file(path)
                    except:
                        image, mask = None, None
            elif 'mark1.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark1.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        image, mask = ninix.pix.create_pixmap_from_file(path)
                    except:
                        image, mask = None, None
            if image:
                w, h = image.get_size()
                x = 60 + index * step + step // 2 - w // 2
                y = 220 - int(
                    168 * self.args[index * 2 + 1] / (limit_max - limit_min)) - h / 2
                gc = self.surface_pixmap.new_gc()
                gc.set_clip_mask(mask)
                gc.set_clip_origin(x, y)
                self.surface_pixmap.draw_drawable(gc, image, 0, 0, x, y, w, h)

class Bar(Graph):

    def draw_graph(self):
        barcolor_r = barcolor_g = barcolor_b = 0
        if 'bar.color' in self.data:
            barcolor_r = int(self.data['bar.color'][:2], 16)
            barcolor_g = int(self.data['bar.color'][2:4], 16)
            barcolor_b = int(self.data['bar.color'][4:6], 16)
        ##else:
        ##    if 'bar.color.r' in self.data:
        ##        barcolor_r = int(self.data['bar.color.r'])
        ##    if 'bar.color.g' in self.data:
        ##        barcolor_g = int(self.data['bar.color.g'])
        ##    if 'bar.color.b' in self.data:
        ##        barcolor_b = int(self.data['bar.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (barcolor_r, barcolor_g, barcolor_b))
        if 'bar.width' in self.data:
            pass # FIXME
        ### NOT YET ###

class Radar(Graph):

    width = 288
    height = 288

    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)

    def draw_frame(self):
        framecolor_r = framecolor_g = framecolor_b = 0
        if 'frame.color' in self.data:
            framecolor_r = int(self.data['frame.color'][:2], 16)
            framecolor_g = int(self.data['frame.color'][2:4], 16)
            framecolor_b = int(self.data['frame.color'][4:6], 16)
        else:
            if 'frame.color.r' in self.data:
                framecolor_r = int(self.data['frame.color.r'])
            if 'frame.color.g' in self.data:
                framecolor_g = int(self.data['frame.color.g'])
            if 'frame.color.b' in self.data:
                framecolor_b = int(self.data['frame.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (framecolor_r, framecolor_g, framecolor_b))
        frame_width = 2
        if 'frame.width' in self.data:
            frame_width = int(self.data['frame.width'])
        self.surface_gc.line_width = frame_width
        num = len(self.args) // 2
        for index in range(num):
            x = 146 + int(math.cos(math.pi * (0.5 - 2.0 * index / num)) * 114)
            y = 146 - int(math.sin(math.pi * (0.5 - 2.0 * index / num)) * 114)
            self.surface_pixmap.draw_line(self.surface_gc,
                                          146, 146, x, y)

    def draw_graph(self):
        num = len(self.args) // 2
        for index in range(num):
            try:
                value = self.args[index * 2 + 1]
                self.args[index * 2 + 1] = float(value)
            except:
                self.args[index * 2 + 1] = 0.0
            if self.args[index * 2 + 1] < 0:
                self.args[index * 2 + 1] = 0.0
        limit_min = self.args[1]
        for index in range(num):
            if self.args[index * 2 + 1] < limit_min:
                limit_min = self.args[index * 2 + 1]
        limit_max = self.args[1]
        for index in range(num):
            if self.args[index * 2 + 1] > limit_max:
                limit_max = self.args[index * 2 + 1]
        linecolor_r = linecolor_g = linecolor_b = 0
        if 'line.color' in self.data:
            linecolor_r = int(self.data['line.color'][:2], 16)
            linecolor_g = int(self.data['line.color'][2:4], 16)
            linecolor_b = int(self.data['line.color'][4:6], 16)
        else:
            if 'line.color.r' in self.data:
                linecolor_r = int(self.data['line.color.r'])
            if 'line.color.g' in self.data:
                linecolor_g = int(self.data['line.color.g'])
            if 'line.color.b' in self.data:
                linecolor_b = int(self.data['line.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (linecolor_r, linecolor_g, linecolor_b))
        line_width = 2
        if 'line.width' in self.data:
            line_width = int(self.data['line.width'])
        self.surface_gc.line_width = line_width
        if limit_max > 0:
            value = self.args[(num - 1) * 2 + 1] / limit_max
        else:
            value = 1.0
        src_x = 146 + int(math.cos(
            math.pi * (0.5 - 2.0 * (num - 1) / num)) * value * 100)
        src_y = 146 - int(math.sin(
            math.pi * (0.5 - 2.0 * (num - 1) / num)) * value * 100)
        for index in range(num):
            if limit_max > 0:
                value = self.args[index * 2 + 1] / limit_max
            else:
                value = 1.0
            dst_x = 146 + int(
                math.cos(math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            dst_y = 146 - int(
                math.sin(math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            self.surface_pixmap.draw_line(self.surface_gc,
                                          src_x, src_y, dst_x, dst_y)
            src_x = dst_x
            src_y = dst_y
        font_size = 9 # pixel
        self.font_desc.set_size(font_size * pango.SCALE)
        self.layout.set_font_description(self.font_desc)
        fontcolor_r = fontcolor_g = fontcolor_b = 0
        if 'font.color' in self.data:
            fontcolor_r = int(self.data['font.color'][:2], 16)
            fontcolor_g = int(self.data['font.color'][2:4], 16)
            fontcolor_b = int(self.data['font.color'][4:6], 16)
        else:
            if 'font.color.r' in self.data:
                fontcolor_r = int(self.data['font.color.r'])
            if 'font.color.g' in self.data:
                fontcolor_g = int(self.data['font.color.g'])
            if 'font.color.b' in self.data:
                fontcolor_b = int(self.data['font.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            '#%02x%02x%02x' % (fontcolor_r, fontcolor_g, fontcolor_b))
        for index in range(num):
            if limit_max > 0:
                value = self.args[index * 2 + 1] / limit_max
            else:
                value = 1.0
            x = 146 + int(math.cos(
                math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            y = 146 - int(math.sin(
                math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            self.layout.set_text(self.args[index * 2])
            self.surface_pixmap.draw_layout(
                self.surface_gc, x, y + 12, self.layout)

class Radar2(Graph):

    width = 288
    height = 288

    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)


if __name__ == '__main__':
    USAGE= 'Usage: hanayu.py <dir>'
    if len(sys.argv) == 2:
        saori = Saori()
        data = saori.read_hanayu_txt(sys.argv[1])
        print data
    else:
        print USAGE
