/*
 * <spindown.c>
 *
 * Copyright (C) 2000 Daniel Kobras
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * $Id: spindown.c,v 1.9 2001/03/29 08:23:27 nold Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "disk_info.h"
#include "spindown.h"
#include "diskhelper.h"
#include "noflushd.h"

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <errno.h>

/* IDE knows two versions of several commands. Some versions work on
 * old drives, some on new ones. We keep trying until we run out of
 * options.
 */
static int spindown_ide(int fd)
{
	unsigned char pwrcheck[4]={WIN_CHECKPOWERMODE1, 0, 0, 0};
	unsigned char spindown[4]={WIN_STANDBYNOW1, 0, 0, 0};

	/* Has the user manually spun down the drive already? */
	if (ioctl(fd, HDIO_DRIVE_CMD, &pwrcheck) &&
	    (pwrcheck[0] = WIN_CHECKPOWERMODE2) &&
	    ioctl(fd, HDIO_DRIVE_CMD, &pwrcheck)) {
		if (errno == EIO && !pwrcheck[0] && !pwrcheck[1]) {
			DEBUG("Disk already sleeping");
			return 0;
		}
		DEBUG("Current disk state unknown");
	} else {
		if (pwrcheck[2] != 255) {
			DEBUG("Disk already standing by");
			return 0;
		}
		DEBUG("Disk was up and running");
	}
	
	if (ioctl(fd, HDIO_DRIVE_CMD, &spindown) &&
	    (spindown[0] = WIN_STANDBYNOW2) &&
	    ioctl(fd, HDIO_DRIVE_CMD, &spindown))
		return -1;

	return 0;
}

static int spindown_scsi(int fd)
{
	return ioctl(fd, SCSI_IOCTL_STOP_UNIT, NULL);
}

/*
 * Generic spindown dispatcher. Calls the spindown method in the
 * disk_info structure. Returns 0 on failure, 1 on success.
 */

int spindown(disk_info_t di)
{
	int fd, err=0;

	DEBUG("Sending spindown command to %s.", di->name);

	fd = open(di->name, O_RDONLY);
	if (fd == -1) {
		ERR("Could not open %s for spindown", di->name);
		goto _err;
	}
	if (di->spindown(fd)) {
		ERR("Spindown for %s failed", di->name);
		goto _out;
	}
	di->spundown_at = time(NULL);
	err=1;
_out:
	close(fd);
_err:
	return err;
	
}

spindown_t spindown_setup(int major, int minor)
{
	if (IDE_DISK_MAJOR(major))
		return spindown_ide;

	if (SCSI_DISK_MAJOR(major))
/*
 * XXX: Change the 0 below to 1 to make SCSI spindown work, but only do it if
 *
 * - you included a SCSI spinup patch like the one in contrib/ in your kernel,
 * - you are aware that the patch might have serious flaws, e.g. it quite
 *   certainly is not 100% SMP-safe,
 * - you generally know what you are doing and swear you won't get at me if
 *   it turns out you actually didn't.
 */
		
#if 0
		return spindown_scsi;
#else
	ERR("SCSI spindown is dangerous. Edit the noflushd source to make this work!");
#endif


	return NULL;

}
