/* Miscellaneous utility functions for ANSI C NoSQL operators.
 *
 * Copyright (c) 2003 Micah Stetson <micah@cnm-vra.com>
 * 
 * Permission is hereby granted, free of charge, to any
 * person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the
 * Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice
 * shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF
 * ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include "util.h"

char *argv0 = NULL;

/* Getfields() taken from Plan 9's u9fs.
 * Modified to be ASCII only.
 *
 * The authors of this software are Russ Cox, Sean Dorward, Eric Grosse, et al.
 * Copyright (c) 2002 by Lucent Technologies.
 * Permission to use, copy, modify, and distribute this software for any
 * purpose without fee is hereby granted, provided that this entire notice
 * is included in all copies of any software which is or includes a copy
 * or modification of this software and in all copies of the supporting
 * documentation for such software.
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR LUCENT MAKE ANY
 * REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY
 * OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */
int
getfields(char *str, char **args, int max, int mflag, char *set)
{
	char c;
	int intok, narg;

	if(max <= 0)
		return 0;

	narg = 0;
	args[narg] = str;
	if(!mflag)
		narg++;
	intok = 0;
	for(;; str++) {
		c = *str;
		if(c == 0)
			break;
		if(strchr(set, c)) {
			if(narg >= max)
				break;
			*str = 0;
			intok = 0;
			args[narg] = str + 1;
			if(!mflag)
				narg++;
		} else {
			if(!intok && mflag)
				narg++;
			intok = 1;
		}
	}
	return narg;
}
/* End Plan 9 stuff. */

static void
verror(char *fmt, va_list arg)
{
	char buf[1024];
	vsnprintf(buf, nelem(buf), fmt, arg);
	if (argv0)
		fprintf(stderr, "%s: ", argv0);
	fprintf(stderr, "%s", buf);
	if (fmt[strlen(fmt)-1] == ':')
		fprintf(stderr, " %s", strerror(errno));
	putc('\n', stderr);
}

void
warn(char *fmt, ...)
{
	va_list arg;
	va_start(arg, fmt);
	verror(fmt, arg);
	va_end(arg);
}

void
fatal(char *fmt, ...)
{
	va_list arg;
	va_start(arg, fmt);
	verror(fmt, arg);
	va_end(arg);
	exit(1);
}

void *
emalloc(int size)
{
	void *p = malloc(size);
	if (p == NULL)
		fatal("malloc(%d) failed", size);
	return p;
}

void *
erealloc(void *p, int size)
{
	void *q = realloc(p, size);
	if (q == NULL)
		fatal("realloc(%ulx, %d) failed", (unsigned long) p, size);
	return q;
}

char *
estrdup(const char *s)
{
	char *t = strdup(s);
	if (t == NULL)
		fatal("strdup(\"%s\") failed");
	return t;
}

FILE *
efopen(const char *path, const char *mode)
{
	FILE *f = fopen(path, mode);
	if (f == NULL)
		fatal("fopen(\"%s\", \"%s\") failed:", path, mode);
	return f;
}

FILE *
efreopen(const char *path, const char *mode, FILE *stream)
{
	FILE *f = freopen(path, mode, stream);
	if (f == NULL)
		fatal("freopen(\"%s\", \"%s\", %ulx) failed:", path, mode, (unsigned long) stream);
	return f;
}

/* Clone of Bio's Brdstr */
char *
frdstr(FILE *f, char delim, int nulldelim)
{
	char *buf;
	int size=0, got=0;
	int c;

	size = 128;
	buf = malloc(size);

	while ((c = getc(f)) != EOF) {
		if (got >= size-1) {	/* size-1 so we can add a NUL */
			size *= 1.5;
			buf = erealloc(buf, size);
		}
		buf[got++] = c;
		if (c == delim)
			break;
	}
	if (got == 0) {
		free(buf);
		if (ferror(f))
			fatal("read error:");
		return NULL;
	}
	if (nulldelim && buf[got-1] == delim)
		buf[got-1] = '\0';
	else
		buf[got] = '\0';

	return buf;
}
