/* genericups.c - support for generic contact-closure UPS models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <sys/ioctl.h>
#include <sys/termios.h>

#include "main.h"
#include "genericups.h"

	int	upstype = -1;

void upsdrv_initinfo(void)
{
	/* setup the basics */

	if (getval("mfr"))
		addinfo(INFO_MFR, getval("mfr"), 0, 0);
	else
		addinfo(INFO_MFR, upstab[upstype].mfr, 0, 0);

	if (getval("model"))
		addinfo(INFO_MODEL, getval("model"), 0, 0);
	else
		addinfo(INFO_MODEL, upstab[upstype].model, 0, 0);

	addinfo(INFO_STATUS, "", 0, 0);
}

/* normal idle loop - keep up with the current state of the UPS */
void upsdrv_updateinfo(void)
{
	int	flags, ol, bl, ret;
	char	temp[VALSIZE];

	ret = ioctl(upsfd, TIOCMGET, &flags);

	if (ret != 0) {
		upslog(LOG_INFO, "ioctl failed");
		return;
	}

	ol = ((flags & upstab[upstype].line_ol) == upstab[upstype].val_ol);
	bl = ((flags & upstab[upstype].line_bl) == upstab[upstype].val_bl);

	strcpy(temp, "");

	if (bl)
		strcat(temp, "LB ");	/* low battery */

	if (ol)
		strcat(temp, "OL");	/* on line */
	else
		strcat(temp, "OB");	/* on battery */

	setinfo(INFO_STATUS, "%s", temp);

	writeinfo();
}

/* show all possible UPS types */
static void listtypes(void)
{
	int	i;

	printf("Valid UPS types:\n\n");

	for (i = 0; upstab[i].mfr != NULL; i++)
		printf("%i: %s\n", i, upstab[i].desc);
}

/* set the flags for this UPS type */
static void set_ups_type(void)
{
	int	i;

	if (!getval("upstype"))
		fatalx("No upstype set - see help text / man page!\n");
	
	upstype = atoi(getval("upstype"));

	for (i = 0; upstab[i].mfr != NULL; i++)
		if (upstype == i) {
			printf("UPS type: %s\n", upstab[i].desc);
			return;
		}

	printf("\nFatal error: unknown UPS type number\n");
	listtypes();
	exit(1);
}			

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	int	flags, ret;

	if (upstype == -1)
		fatalx("No upstype set - see help text / man page!\n");

	open_serial_simple(device_path, 0, upstab[upstype].line_norm);

	upslogx(LOG_INFO, "Initiating UPS shutdown\n");

	flags = upstab[upstype].line_sd;

	if (flags == -1)
		fatalx("No shutdown command defined for this model!\n");

	if (flags == TIOCM_ST) {
#ifdef HAVE_TCSENDBREAK
		tcsendbreak (upsfd, 4901);
#else
		fatalx("Need to send a BREAK, but don't have tcsendbreak!\n");
#endif
	} else {
		ret = ioctl(upsfd, TIOCMSET, &flags);
		if (ret != 0)
			perror("ioctl");
	}
}

void upsdrv_help(void)
{
	listtypes();
}

void upsdrv_banner(void)
{
        printf("Network UPS Tools - Generic UPS driver 0.60 (%s)\n", UPS_VERSION);
}

void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "upstype", "Set UPS type (required)");
	addvar(VAR_VALUE, "mfr", "Override manufacturer name");
	addvar(VAR_VALUE, "model", "Override model name");
}

void upsdrv_initups(void)
{
	set_ups_type();

	open_serial_simple(device_path, 0, upstab[upstype].line_norm);
}

int upsdrv_infomax(void)
{
	return 8;
}
