/* sms.c - model specific routines for SMS Ltda ( Brazilian ) units

   This code is derived from Russell Kroll <rkroll@exploits.org>,
   Fenton UPS Driver   

   Copyright (C) 2001  Marcio Gomes  <tecnica@microlink.com.br>
   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


   2001/05/17 - Version 0.1 - Initial release
   2001/06/01 - Version 0.2 - Add Battery Informations in driver
   2001/06/04 - Version 0.3 - Updated Battery Volts range, to reflect a correct 
                              percent ( % ) 
   Microlink ISP contributed with MANAGER III 1300, MANAGER III 650 UPS
   for my tests
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "sms.h"
#include "common.h"

#define INFOMAX 20 
#define ENDCHAR 13	/* replies end with CR */

	int	shmok = 1, cap_upstemp = 0;
	float	lowvolt = 0, voltrange;
	int	lownorm, highnorm;
	int	sddelay = 90;	/* wait 90 seconds for shutdown */

void initinfo (void)
{
	create_info(INFOMAX, shmok);

	/* setup variables that should always exist */

	addinfo (INFO_OUTVOLT, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTVOLT, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
}

void guessmodel (char *raw)
{
	char	 mstr[256];

	printf ("Guessing at UPS capabilities - please report this ID string:\n");
	printf ("%s\n", raw);
	strlcpy(mstr, &raw[18], sizeof(mstr));
	mstr[10] = '\0';	/* 10 chars max, per the protocol */
	/* trim whitespace */
	rtrim(mstr, ' ');
	/* use SMS model-chars to attempt UPS detection */ 

	addinfo (INFO_MODEL, "", 0, 0);
        setinfo(INFO_MODEL, "SMS Unknown  %s", mstr);
        cap_upstemp = 1;

}
void getbaseinfo (char *port)
{
	char	temp[256], model[32], *raw;
	int	modelnum, i;

	/* dummy read attempt to sync - throw it out */
	upssendchar ('I');
	upssendchar (13);
	upsrecv (temp, sizeof(temp), ENDCHAR, "");

	/* now retrieve information and parse */
	upssendchar ('I');
	upssendchar (13);
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	raw = xstrdup (temp);

/*      Print Ups information   
        printf("-%s-\n",raw); */
	if (temp[0] != '#') {
		printf ("Bad UPS info start character\n");
		printf ("Got: [%s]\n", temp);
		exit (1);
	}

	temp[11] = 0;
	temp[27] = 0;

	/* manufacturer */
	rtrim(&temp[1], ' ');
	addinfo (INFO_MFR, &temp[1], 0, 0);
	/*  SMS LTDA         1300 VA   VER 1.0   
            L660A = PowerPal (L) @ 660 VA, American (A) version (115V) */

	/* grab full model string */
	rtrim(&temp[18], ' '); 
	snprintf (model, sizeof(model), "%s", &temp[18]);
/*        printf("Model ->%s<-\n",model); */
	modelnum = -1;

	/* figure out official model name and voltage info from table */
	for (i = 0; modeltab[i].mtext != NULL; i++) {
		if (!strcmp(modeltab[i].mtext, model)) {
			modelnum = i;
			lowvolt = modeltab[i].lowvolt;
			voltrange = modeltab[i].voltrange;
			cap_upstemp = modeltab[i].has_temp;
			break;
		}
	}

	/* table lookup fails -> guess */
	if (modelnum == -1)
		guessmodel (raw);
	else {
		addinfo (INFO_MODEL, modeltab[modelnum].desc, 0, 0);
		snprintf (temp, sizeof(temp), "%i", modeltab[modelnum].lowxfer);
		addinfo (INFO_LOWXFER, temp, 0, 0);

		snprintf (temp, sizeof(temp), "%i", modeltab[modelnum].highxfer);
		addinfo (INFO_HIGHXFER, temp, 0, 0);

		lownorm = modeltab[modelnum].lownorm;
		highnorm = modeltab[modelnum].highnorm;
	}

	if (cap_upstemp == 1)
		addinfo (INFO_UPSTEMP, "", 0, 0);

	/* now add instant command support info */
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
        addinfo (INFO_INSTCMD, "", 0, CMD_SIMPWF);

	printf ("Detected %s on %s\n", getdata(INFO_MODEL), port);
	free(raw);
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256], utility[16], loadpct[16], acfreq[16], battvolt[16],
		upstemp[16], stat[16], outvolt[16];
	int	util;
	double	bvoltp;

	upssendchar ('Q');
	upssendchar ('1');
	upssendchar (13);

	upsrecv (temp, sizeof(temp), ENDCHAR, "");

	/* (MMM.M NNN.N PPP.P QQQ RR.R S.SS TT.T  b7b6b5b4b3b2b1b0<cr>
	 *
	 * MMM.M : input voltage (utility)
	 * NNN.N : fault voltage (ignored)
	 * PPP.P : output voltage
	 */

	sscanf (temp, "%*c%s %*s %s %s %s %s %s %s", utility, outvolt, loadpct,
	        acfreq, battvolt, upstemp, stat);

	setinfo(INFO_OUTVOLT, "%s", outvolt);
	setinfo(INFO_UTILITY, "%s", utility);
	setinfo(INFO_BATTVOLT, "%s", battvolt);

	bvoltp = ((atof(battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	setinfo(INFO_BATTPCT, "%02.1f", bvoltp);

	strcpy (temp, "");

	if (stat[0] == '0')
		strcat (temp, "OL ");		/* on line */
	else
		strcat (temp, "OB ");		/* on battery */

	if (stat[1] == '1')
		strcat (temp, "LB ");		/* low battery */

	util = atoi (utility);

	if (stat[2] == '1') {		/* boost or trim in effect */
		if (util < lownorm)
			strcat (temp, "BOOST ");

		if (util > highnorm)
			strcat (temp, "TRIM ");
	}

	/* lose trailing space if present */
	if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	setinfo(INFO_STATUS, "%s", temp);

	if (cap_upstemp == 1)
		setinfo(INFO_UPSTEMP, "%s", upstemp);

	setinfo(INFO_ACFREQ, "%s", acfreq);
	setinfo(INFO_LOADPCT, "%s", loadpct);

	writeinfo();
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	char	temp[256], stat[32];

	upslogx(LOG_INFO, "Initiating UPS shutdown");
	printf ("Initiating forced UPS shutdown!\n");

	open_serial (port, B2400);

	/* basic idea: find out line status and send appropriate command */

	upssendchar ('Q');
	upssendchar ('1');
	upssendchar (13);
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	sscanf (temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0003<cr>, ups will cycle and return soon */

	upssendchar ('S');
	upssendchar ('0');
	upssendchar ('1');

	if (stat[0] == '0') {			/* on line */
		printf ("On line, sending shutdown+return command...\n");
		upssendchar ('R');
		upssendchar ('0');
		upssendchar ('0');
		upssendchar ('0');
		upssendchar ('3');
	}
	else
		printf ("On battery, sending normal shutdown command...\n");

	upssendchar (13);	/* end sequence */

	if (sddelay > 0) {
		printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit(0);
}

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			upssendchar ('C');
			upssendchar ('T');
			upssendchar (13);
			break;
		case CMD_BTEST1:	/* start battery test 10 Seconds*/
			upssendchar ('T');
			upssendchar (13);
			break;
                case CMD_SIMPWF: /* start battery test until Bat Low */
                        upssendchar ('T');
                        upssendchar ('L');
                        upssendchar (13);
                        break;


                default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x",
		        	auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.instcmd = instcmd;
	/* TODO: future */
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i;

	printf ("Network UPS Tools - SMS UPS driver 0.1 (%s)\n", UPS_VERSION);
	openlog ("sms", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	while ((i = getopt(argc, argv, "+d:hk:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/sms-%s", STATEPATH,
	          portname);

	open_serial (argv[0], B2400);

	initinfo();

	createmsgq();	/* try to create IPC message queue */

	getbaseinfo(argv[0]);

	setuphandlers();

	background();

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
