/* toshiba1500.c - model specific routines for Toshiba 1500 series units

$Id: toshiba1500.c,v 1.2 2000/11/24 10:41:30 cvs Exp $

Revision 1.2  2000/09/08 05:55:45  rpm
Rewrite receive path to make it deterministic, not timeout-driven.
Reduce getupsmsg wait to 1 second for faster updates.
Reduce stack-based buffer sizes to something reasonable.


Copyright (C) 2000  Kenneth Porter <shiva@well.com>
Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

Based on fentonups.c by Russell Kroll.

Thanks go to Dennis Kruep and Greg Mack of Toshiba for making the
protocol documentation available.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX 16

extern int upsc_debug; /* set to debug protocol */

int	shmok = 1;
int	sddelay = 90;	/* wait 90 seconds for shutdown */

/* character codes */
#define STX 0x02
#define ETX 0x03
#define EOT 0x04
#define ENQ 0x05
#define ACK 0x06
#define NAK 0x15

void initinfo (void)
{
	create_info(INFOMAX, shmok);

	/* setup variables that should always exist */
	addinfo (INFO_MFR, "Toshiba", FLAG_STRING, 0);
	addinfo (INFO_MODEL, "1500", FLAG_STRING, 0);
	addinfo (INFO_STATUS, "", FLAG_STRING, 0);
	addinfo (INFO_OUTVOLT, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	/* addinfo (INFO_OUTFREQ, "", 0, 0); */
}

/* retry sending message 3 times until ACK received or timeout
 * return 0 on ACK, -1 on timeout
 */

int waitforack (const char* message)
{
	int try_count;
	for (try_count = 0; try_count < 3; ++try_count)
	{
		char temp[4];
		upssend("%s", message);
		if (upsrecv (temp, 3, ACK, "") > 0)
			return 0; /* success */
	}
	return -1; /* retries exhausted */
}

void updatestatus (unsigned status)
{
	static unsigned last_status = 0;
	if (upsc_debug)
		printf ("updatestatus(%X)\n", status);
	if (status != last_status) {
		char status_string[256] = "";
		last_status = status;
		if (0x10 & status) /* input undervoltage or blackout */
			strcat (status_string,"OB ");
		else
			strcat (status_string,"OL "); /* online */
		if (0x08 & status) /* low battery */
			strcat (status_string,"LB ");
		if (0x04 & status) /* bypass operation */
			strcat (status_string,"BYP ");
		if (0x02 & status) /* inverter operation */
			strcat (status_string,"INV ");
		if (0x01 & status) /* synchronous operation */
			strcat (status_string,"SYNC ");
		/* todo: check 0x20 bit (fault) and read fault word
		 * trim trailing blank
		 */
		if (status_string[0])
			/* something was appended */
			status_string[strlen (status_string) - 1] = '\0';
		setinfo(INFO_STATUS, "%s", status_string);
	}
}

/* return 0 on success */

int select_toshiba (void)
{
	/* send 1-1-ENQ ("select") */
	if (-1 == waitforack ("11\005")) {
		if (upsc_debug)
			printf ("Bad reply to select\n");
		return -1;
	}
	return 0;
}

/* create value selection command followed by BCC */

void compose_toshiba_command (char* buf, int buflen,
							  char command_type,
							  const char* address)
{
	char        bcc;
	char*       p;

	snprintf (buf, buflen - 1, "\002%c%s\003", command_type, address);
	/* STX does not participate in BCC computation */
	bcc = 0;
	for (p = buf + 1; *p; ++p)
		bcc ^= *p;
	/* append BCC */
	*p++ = bcc;
	*p = '\0';
}

/* get one char (may be high ascii), with timeout */

int upsgetch (void)
{
	char	in;
	int		ret;
	struct 	sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);

	ret = read (upsfd, &in, 1);

	alarm (0);
	signal (SIGALRM, SIG_IGN);

	if (ret > 0) {
		nolongertimeout (); /* toggle flag */
		return in;
	}
	else
		return (-1);
}

/* return -1 on failure, 0 on success
 * stores null-terminated result string in buf,
 * stores status byte in *status
 */

int get_toshiba_result (char buf[5],
						char* status,
						char command_type,
						const char* address)
{
	char        temp[10];
	char*       statusp;
	int         bcc;
	char*       p;

	/* wait for up to 10 chars of response */
	if (upsrecv (temp, sizeof temp, ETX, "") < 1)
		return -1;

	/* attempt to parse value */
	if (STX != temp[0] || command_type != temp[1] ||
		  address[0] != temp[2] || address[1] != temp[3]) {
		if (upsc_debug)
			printf ("prefix wrong\n");
		return -1;
	}
	/* find the ETX (preceded by UPS status, followed by BCC) */
	/* check BCC */
	bcc = upsgetch ();
	if (bcc < 0)
		return -1;
	for (p = temp + 1; *p; ++p)
		bcc ^= *p;
	bcc ^= ETX; /* account for ETX eaten by upsrecv */
	if (bcc) {
		if (upsc_debug)
			printf ("bad BCC, result = %02X\n", bcc);
		return -1;
	}
	/* looks good, store it */
	statusp = temp + strlen (temp) - 1; /* status at end of message */
	*status = *statusp;
	*statusp = '\0'; /* terminate result string */
	strcpy (buf, temp + 4);
	return 0;
}

/* query a value from the UPS and update status bits
 * returns value (0-999) on success, -1 on failure
 */

int pollvalue (const char* address)
{
	char        temp[10];
	unsigned    try_count;
	unsigned    value;
	char	status;
	if (upsc_debug)
		printf ("pollvalue(%s)\n", address);

	memset (temp, '\0', sizeof temp);

	select_toshiba ();

	/* send command */

	compose_toshiba_command (temp, sizeof temp, 'M', address);

	/* now send command and wait for ACK */
	if (-1 == waitforack (temp)) {
		if (upsc_debug)
			printf ("Bad reply to command\n");
		return -1;
	}

	upssendchar (EOT); /* done selecting */

	/* now poll for value */

	upssend("%s", "10\005"); /* 1-0-ENQ */

	for (try_count = 0; try_count < 3; ++try_count) {
		if (-1 == get_toshiba_result (temp, &status, 'M', address)) {
			upssend("%s", "10\025"); /* 1-0-NAK */
			continue;
		}
		break; /* result ok */
	}
	upssendchar (EOT); /* signal UPS we're quitting */
	if (3 == try_count) {
		if (upsc_debug)
			printf ("giving up on parameter %s\n", address);
		return -1;
	}
	updatestatus (status); /* status at end of message */
	sscanf (temp,"%u",&value);
	if (upsc_debug)
		printf ("parameter %s has value '%s'\n", address, temp);
	return value;
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	unsigned value;         /* parsed result */
	double  dvalue;         /* same, but as a double */

#if 0 /* no poll result on 1500 */
	/* fault bits */
	pollvalue ("05");
#endif

	/* output voltage in percent */
	dvalue = 120 * pollvalue ("10");
	if (dvalue >= 0) {
		/* convert % to volts of rated voltage */
		setinfo(INFO_OUTVOLT, "%4.1f", dvalue / 100.0);
		writeinfo();
	}

#if 0 /* can't select on 1500 */
	/* input voltage in percent */
	dvalue = pollvalue ("20");
	if (dvalue >= 0) {
		/* convert % to volts of rated voltage */
		setinfo(INFO_UTILITY, "%4.1f", dvalue / 100.0);
		writeinfo();
	}
#endif

	/* output current in percent (not used) */
	value = pollvalue ("30");
	if (value >= 0) {
		setinfo(INFO_LOADPCT, "%u", value);
		writeinfo();
	}

	/* battery voltage in percent */
	value = pollvalue ("40");
	if (value >= 0) {
		setinfo(INFO_BATTPCT, "%u", value);
		writeinfo();
	}

	/* input frequency in Hz * 10 */
	dvalue = pollvalue ("50") / 10.0;
	if (dvalue >= 0) {
		setinfo(INFO_ACFREQ, "%3.1f", dvalue);
		writeinfo();
	}

#if 0
	/* output frequency in Hz * 10 (not used) */
	pollvalue ("51");
#endif
	
}

/* open port at 1200 bps, 7E1 framing */

void open_toshiba (char* port)
{
	struct	termios	tio;

	open_serial (port, B1200);

	/* switch to 7E1 */
	tcgetattr (upsfd, &tio);
	tio.c_cflag = CS7 | CLOCAL | CREAD | PARENB;
#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B1200);
	cfsetospeed (&tio, B1200);
#endif
	tcsetattr (upsfd, TCSANOW, &tio);

	/* timeouts are normal for this model,
	 * so don't log them
	 */
	flag_timeoutfailure = -1;
}

/* power down the attached load immediately */
void forceshutdown (char *port)
{
	unsigned    try_count;
	char	command[10], reply[10], status;

	/* address is 90 plus delay in 30 second intervals
	 * valid delay range is 30 seconds to 8 minutes 
	 */
	static const char address[] = "91";

	upslogx(LOG_INFO, "Initiating UPS shutdown");
	printf ("Initiating forced UPS shutdown!\n");

	open_toshiba (port);

	for (try_count = 0; try_count < 3; ++try_count) {

		if (-1 == select_toshiba ()) {
			upssendchar (EOT);
			continue;
		}

		compose_toshiba_command (command, sizeof command, 'W', address);
		upssend("%s", command);

		if (-1 == get_toshiba_result (reply, &status, 'C', address))
			continue;

		/* send confirmation */
		compose_toshiba_command (command, sizeof command, 'C', address);
		upssend("%s", command);

		/* check for acknowledge of confirmation */
		if (-1 == get_toshiba_result (reply, &status, 'W', address))
			continue;

		break; /* result ok */
	}
	upssendchar (EOT); /* signal UPS we're quitting */
	if (3 == try_count) {
		printf ("giving up on shutdown attempt\n");
		exit (1);
	}

	if (sddelay > 0) {
		printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit (0);
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i;

	printf ("Network UPS Tools - Toshiba 1500 series UPS driver 0.1 (%s)\n", UPS_VERSION);
	openlog ("toshiba1500", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	while ((i = getopt (argc, argv, "+d:ghk:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi (optarg);
				break;
			case 'k':
				forceshutdown (optarg);
				break;
			case 'h':
				help (prog);
				break;
			case 'g':
				upsc_debug = 1; /* turn on port debugging */
				break;
			default:
				usage (prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot ();

	portname = NULL;

	for (i = strlen (argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof statefn, "%s/toshiba1500-%s", STATEPATH,
			  portname);

	open_toshiba (argv[0]);

	initinfo ();

	createmsgq ();	/* try to create IPC message queue */

	if (!upsc_debug)
		background ();

	for (;;) {
		updateinfo ();

		/* wait up to 1 second for a message from the upsd */
		if (getupsmsg (1))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
