/* NVTV Gui -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See GPL.txt for details.
 *
 * The GTK graphical user interface.
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>

#include <gdk/gdkprivate.h>
#include <X11/Xmu/WinUtil.h>

#undef FALSE
#undef TRUE

#include "xfree.h"
#include "debug.h"
#include "nv_tvchip.h"
#include "main.h"
#include "data.h"
#include "data_bt.h"
#include "data_ch.h"
#include "gui.h"
#include "gui_bt.h"
#include "gui_ch.h"
#include "actions.h"
#include "backend.h"

/* -------- Global variables -------- */

GtkWidget *gui_main_window;

/* ---- GUI state ---- */

NVCrtRegs gui_crt;
NVTvRegs gui_tv;

NVSettings gui_set;

NVCrtRegs gui_orig_crt;
Bool gui_orig_present = FALSE;

int gui_flags = NV_PRIV_TVMODE;

NVTvChip gui_tv_chip = NV_NO_CHIP;

NVSystem gui_system = TV_SYSTEM_NONE;
NVMode *gui_list_mode = NULL;
NVMode *gui_act_mode = NULL;

Window gui_window = None;
int gui_window_width, gui_window_height;

Bool gui_auto = FALSE;
Bool gui_bypass = FALSE;
Bool gui_tvstate = FALSE; /* FALSE=off, TRUE=on */

/* ---- other variables ---- */

static CardPtr gui_card_list;

static CardPtr gui_card = NULL;

/* ---- "Changed" signal chains ---- */

static guint signal_notify = 0;

GuiNotify *update_crt;
GuiNotify *update_chip;
GuiNotify *update_mode;
GuiNotify *changed_all;
GuiNotify *show_bt;
GuiNotify *show_ch;
GuiNotify *hide_bt;
GuiNotify *hide_ch;

/* -------- Accelerators -------- */

GuiAccel gui_accel[ACCEL_LAST] = {
  {"Switch TV on:",    "F1"},
  {"Switch TV off:",   "F2"},
  {"Center X window:", "F5"},
  {"Reset values:",    "F9"},
  {"Print values:",    "F10"},
};

GtkAccelGroup *gui_main_accel_group;

/* -------- GUI Masks -------- */

struct mask_mode_page {
  GtkOptionMenu *system;
  GtkCList *modes;
  GtkLabel *res;
  GtkLabel *size;
  GtkLabel *overscan;
  GtkWidget *xcenter;
  GtkWidget *resize;
};

struct mask_mode_page gui_mode_mask;

struct mask_config_page {
  GtkLabel *io_addr, *probe;
  GtkOptionMenu *chip;
};

struct mask_config_page gui_config_mask;

struct mask_offset {
  GtkAdjustment *hadj, *vadj;
  gfloat hcenter, vcenter;
};

struct mask_offset gui_tv_offset, gui_mon_offset, gui_resize;

struct mask_settings {
  GtkAdjustment *brightness, *color, *contrast, *saturation;
  GtkAdjustment *flicker, *sharpness, *bandwidth;
  GtkToggleButton *dualview, *macrovis, *monochrome;
  GSList *output;
  int connect;
};

struct mask_settings gui_settings;

DataFunc *gui_func = NULL;

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_CRT(m) FIELD(gui_crt,m)

static struct reg_mask crt_mask [] = {
  {label:"HorizDisplay:",   bits:12, tick:8, FIELD_CRT(HDisplay)}, 
  {label:"HorizSyncStart:", bits:12, tick:8, FIELD_CRT(HSyncStart)}, 
  {label:"HorizSyncEnd:",   bits:12, tick:8, FIELD_CRT(HSyncEnd)}, 
  {label:"HorizTotal:",     bits:12, tick:8, FIELD_CRT(HTotal)}, 
  {label:"VertDisplay:",    bits:12, tick:1, FIELD_CRT(VDisplay)}, 
  {label:"VertSyncStart:",  bits:12, tick:1, FIELD_CRT(VSyncStart)},  
  {label:"VertSyncEnd:",    bits:12, tick:1, FIELD_CRT(VSyncEnd)},    
  {label:"VertTotal:",      bits:12, tick:1, FIELD_CRT(VTotal)},      
  {label:NULL }
};

static struct flag_mask crt_mask_flags [] = {
  {label:"Doublescan",     mask:V_DBLSCAN,   FIELD_CRT(Flags)}, 
  {label:"Interlace",      mask:V_INTERLACE, FIELD_CRT(Flags)}, 
  {label:NULL }
};


/* -------- GTK Helper Routines -------- */

/*
 *  Create an option menu with labels. 
 */

GtkWidget* create_option_menu (GtkSignalFunc func, ...)
{
  va_list ap;
  GtkWidget *opt, *menu, *item;
  char *label;
  int i;

  va_start (ap, func);
  opt = gtk_option_menu_new();
  menu = gtk_menu_new();
  for (i = 0; TRUE; i++) {
    label = va_arg (ap, char *);
    if (!label) break;
    item = gtk_menu_item_new_with_label (label);
    if (func) 
      gtk_signal_connect (GTK_OBJECT (item), "activate", func, (gpointer) i);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  va_end (ap);
  return opt;
}

/*
 *  Attempt to control the size of an entry 
 */

void entry_size_request_cb (GtkWidget *widget, GtkRequisition *req)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_ENTRY (widget));
  g_return_if_fail (req != NULL);

  /* No info about max or avg xsize, so guess */
  if (((GtkEntry *) widget)->text_max_length > 0) {
    req->width = ((GtkEntry *) widget)->text_max_length 
      * (widget->style->font->ascent + widget->style->font->descent) 
      + (widget->style->klass->xthickness + 2) * 2;
  }
}

GtkWidget *create_entry_with_size (gint max_length)
{
  GtkWidget *entry;

  entry = gtk_entry_new_with_max_length (max_length);
  gtk_signal_connect_after (GTK_OBJECT(entry), "size_request",  
			    GTK_SIGNAL_FUNC(entry_size_request_cb), NULL);
  return entry;
}

/*
 *  Attempt to control the size of a spinbutton
 */

void spinbutton_size_request_cb (GtkWidget *widget, GtkRequisition *req)
{
  GtkAdjustment *adj;
  register int s;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SPIN_BUTTON (widget));
  g_return_if_fail (req != NULL);

  /* WARNING: This is a very crude estimation of average font character width
     and arrow size using ascent and descent, and probably will not work with 
     all fonts. */
  adj = ((GtkSpinButton *) widget)->adjustment;

  if (adj) {
    s = ((int) log10 (adj->upper - adj->lower)) + 1 + 
        ((GtkSpinButton *) widget)->digits;
    s = s * (widget->style->font->ascent - widget->style->font->descent) +
        4 * widget->style->klass->xthickness + 
        (widget->style->font->ascent + widget->style->font->descent) / 2;
    /* ARROW_SIZE == 11 */
    req->width = s;
  }
}

GtkWidget *create_spinbutton_with_size (GtkAdjustment *adjustment,
  gfloat climb_rate, guint digits)
{
  GtkWidget *spin;

  spin = gtk_spin_button_new (adjustment, climb_rate, digits);
  gtk_signal_connect_after (GTK_OBJECT(spin), "size_request",  
			    GTK_SIGNAL_FUNC(spinbutton_size_request_cb), NULL);
  return spin;
}

/*
 *  Create an arrow button and attach to table.
 */

GtkWidget *create_arrow_button (GtkTable *table, GtkArrowType arrow_type,
  int x0, int x1, int y0, int y1)
{
  GtkWidget *button;
  GtkWidget *arrow;

  button = gtk_button_new();
  if (arrow_type >= GTK_ARROW_UP && arrow_type <= GTK_ARROW_RIGHT) {
    arrow = gtk_arrow_new (arrow_type, GTK_SHADOW_OUT);
    gtk_widget_set_usize (arrow, 20, 20); 
    gtk_container_add (GTK_CONTAINER (button), arrow);
  }
  gtk_table_attach (table, button, x0, x1, y0, y1, 
    GTK_FILL, GTK_FILL, 0,0);
  return button;
}

/*
 *  Create a vertical or horizontal radio box. Store radio group if
 *  non-null.
 */

GtkWidget *create_radio_box (int active, gboolean hor_vert, 
  GtkSignalFunc func, GSList **group, ...)
{
  GtkWidget *bbox;
  GtkWidget *button;
  char *label;
  GSList *slist;
  va_list ap;
  int i;

  va_start (ap, group);
  if (hor_vert) {
    bbox = gtk_hbox_new (FALSE, 0);
  } else {
    bbox = gtk_vbox_new (FALSE, 0);
  }
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  button = NULL; slist = NULL;
  for (i = 0; TRUE; i++) {
    label = va_arg (ap, char *);
    if (!label) break;
    if (button) slist = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
    button = gtk_radio_button_new_with_label (slist, label);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button),
      (active == i) ? TRUE : FALSE);
    if (func) 
      gtk_signal_connect (GTK_OBJECT (button), "toggled", func, (gpointer) i);
    gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, FALSE, 0);
  }
  va_end (ap);
  if (group) {
    *group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
  }
  return bbox;
}

/*
 * Set active index of radio group
 */

void set_radio_group_active (GSList *group, int active)
{
  group = g_slist_nth (group, g_slist_length (group) - active - 1);
  if (group) 
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (group->data), TRUE);
  }
}

/* 
 * Create notify widget (misuse adjustment for now) 
 */

GuiNotify *create_notify (void)
{
  return (GuiNotify *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
}

void change_adjustment (GtkAdjustment *adj, int value)
{
  if (adj->value != value)
    gtk_adjustment_set_value (adj, value);
}

/* -------- State methods -------- */

/* ---- register/flag (on signal chain) ---- */

void reg_changed_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");

  if (m->size == sizeof(int)) {
    *((int *) m->addr) = (int) adj->value; 
  } else if (m->size == sizeof(long)) {
    *((long *) m->addr) = (long) adj->value;
  }
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void reg_low_changed_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  long l;

  if (m->size != sizeof(long)) {
    fprintf (stderr, "reg_low_changed: not a long value\n");
    return;
  }
  l = *((long *) m->addr);
  l = (l / 100000) * 100000 + (long) adj->value;
  *((long *) m->addr) = l;
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void reg_high_changed_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  long l;

  if (m->size != sizeof(long)) {
    fprintf (stderr, "reg_high_changed: not a long value\n");
    return;
  }
  l = *((long *) m->addr);
  l = (l % 100000) + ((long) adj->value * 100000);
  *((long *) m->addr) = l;
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void reg_update_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size == sizeof(int)) {
    val = (gfloat) *((int *) m->addr);
  } else if (m->size == sizeof(long)) {
    val = (gfloat) *((long *) m->addr);
  }
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
}

void reg_low_update_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size != sizeof(long)) {
    fprintf (stderr, "reg_low_update: not a long value\n");
    return;
  }
  val = (gfloat) (*((long *) m->addr) % 100000);
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
}

void reg_high_update_cb (GtkAdjustment *adj, gpointer data)
{
  struct reg_mask *m = (struct reg_mask *) 
    gtk_object_get_data (GTK_OBJECT(adj), "adj");
  gfloat val;

  val = 0.0;
  if (m->size != sizeof(long)) {
    fprintf (stderr, "reg_high_update: not a long value\n");
    return;
  }
  val = (gfloat) (*((long *) m->addr) / 100000);
  gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), val);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
    GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
}

void flag_changed_cb (GtkWidget *button, gpointer data)
{
  struct flag_mask *m = (struct flag_mask *) 
    gtk_object_get_data (GTK_OBJECT(button), "mask");

  if (GTK_TOGGLE_BUTTON (button)->active) {
    *((int *) m->addr) |= m->mask;
  } else {
    *((int *) m->addr) &= ~m->mask;
  }
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

void flag_update_cb (GtkWidget *button, gpointer data)
{
  struct flag_mask *m = (struct flag_mask *) 
    gtk_object_get_data (GTK_OBJECT(button), "mask");

  gtk_signal_handler_block_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button),
    (*((int *) m->addr) & m->mask) ? TRUE : FALSE);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (button),
    GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
}

/* ---- crt registers ---- */

void gui_crt_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_crt), "changed");
}

void gui_crt_set (NVCrtRegs *crt)
{
  gui_crt = *crt;
  /* setup "crt" fields from "modeline" fields */
  DPRINTF ("gui_set_crt %p %i\n", crt, crt->HDisplay);
  gui_crt_changed ();
}

/* ---- chip registers ---- */

void gui_tv_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_chip), "changed");
}

void gui_tv_set (void)
{
  if (gui_func) {
    gui_func->init (gui_system, &gui_tv);
  }
  gui_tv_changed ();
}

/* ---- both registers ---- */

void gui_crt_tv_set (NVCrtRegs *crt, NVTvRegs *tv, Bool setup)
{
  if (crt) gui_crt = *crt;
  if (tv)  gui_tv = *tv;
  if (gui_func) {
    gui_func->init (gui_system, &gui_tv);
    if (setup) gui_func->setup (&gui_set, &gui_crt, &gui_tv);
  }
  gui_crt_changed ();
  gui_tv_changed ();
}

/* ---- actual mode ---- */

void gui_act_mode_changed (void)
{
  if (gui_act_mode) {
    gui_crt_tv_set (gui_act_mode->crt, gui_act_mode->tv, gui_bypass);
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
    /* FIXME -- all flags */
    /* FIXME: Store old value, and set to forced value */
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.dualview), 
      (gui_crt.PrivFlags & NV_MODE_DUALVIEW) ? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_settings.macrovis), 
      (gui_crt.PrivFlags & NV_MODE_MACROVIS) ? TRUE : FALSE);
  }
}

void gui_act_mode_set (NVMode *mode)
{
  char s[20];

  gui_act_mode = mode;
  if (gui_act_mode) {
    snprintf (s, 20, "%3i x %3i", gui_act_mode->res_x, gui_act_mode->res_y);
    gtk_label_set_text (gui_mode_mask.res, s);
    gtk_label_set_text (gui_mode_mask.size, gui_act_mode->size);
    snprintf (s, 20, "%06.3f x %06.3f", gui_act_mode->hoc, gui_act_mode->voc);
    gtk_label_set_text (gui_mode_mask.overscan, s);
  } else {
    gtk_label_set_text (gui_mode_mask.res, "<None>");
    gtk_label_set_text (gui_mode_mask.size, "<None>");
    gtk_label_set_text (gui_mode_mask.overscan, "<None>");
  }
  gui_act_mode_changed ();
}

/* ---- list mode (current selection) ---- */

void gui_list_mode_changed (void)
{
  gtk_signal_emit_by_name (GTK_OBJECT (update_mode), "changed");
}

void gui_list_mode_cb (GtkCList *clist, gint row, gint column,
  GdkEventButton *event, gpointer data)
{
  gui_list_mode = gtk_clist_get_row_data (gui_mode_mask.modes, row);
  gui_list_mode_changed ();
  gui_act_mode_set (gui_list_mode);
}

/* ---- list of all modes ---- */

/*
 * Set list of modes selected by gui_system.
 */

void gui_modes_init (void)
{
  NVMode *m;
  char sr[20], ss[20], so[20], sa[10];
  char *text [4];
  int row;
  int select;

  switch (gui_tv_chip) 
  {
    case NV_BROOKTREE: 
    case NV_CONEXANT: 
      m = bt_modes; break;
    case NV_CHRONTEL:  
      m = ch_modes; break;
    default: 
      m = NULL; break;
  }
  gtk_clist_freeze (gui_mode_mask.modes);
  gtk_clist_clear (gui_mode_mask.modes);
  if (m) {
    row = 0; select = 0;
    while (m->system != TV_SYSTEM_NONE) {
      if (m->system == gui_system) {
	if (opt_size) {
	  if (m->res_x == opt_res_x && m->res_y == opt_res_y && 
	      strcasecmp (m->size, opt_size) == 0) 
	  {
	    select = row;
	  }
	}
	snprintf (sr, 20, "%3i x %3i", m->res_x, m->res_y);
	snprintf (ss, 20, "%s", m->size);
	snprintf (so, 20, "%05.2f x %05.2f", m->hoc, m->voc);
	/* snprintf (sa, 10, "%7.5f", aspect); */
	text [0] = sr;
	text [1] = ss;
	text [2] = so;
	text [3] = sa;
	gtk_clist_append (gui_mode_mask.modes, text);
	gtk_clist_set_row_data (gui_mode_mask.modes, row, m);
	row++;
      }
      m++;
    }
    gtk_clist_select_row (gui_mode_mask.modes, select, 0); 
  }
  gtk_clist_thaw (gui_mode_mask.modes);
  /* FIXME: Create custom mode if opt_hoc and opt_voc are set */
}

void gui_modes_column_cb (GtkCList *clist, gint column, gpointer data)
{
  g_return_if_fail (GTK_IS_CLIST (clist));
  gtk_clist_set_sort_column (clist, column); 
  gtk_clist_sort (clist);
}

/* ---- tv system ---- */

void gui_system_changed (void)
{
  gui_modes_init ();
}

void gui_system_cb (GtkWidget *widget, gpointer data)
{
  gui_system = (NVSystem) data;
  gui_system_changed ();
}

void gui_system_set (NVSystem system)
{
  gui_system = system;
  if (gui_system > TV_SYSTEM_NONE)
    gtk_option_menu_set_history (gui_mode_mask.system, gui_system);
  gui_system_changed ();
}

void gui_system_init (void)
{
  gui_system = TV_SYSTEM_PAL; /* Must have some default. */
  if (opt_system != TV_SYSTEM_NONE) {
    gui_system = opt_system;
  }
  if (gui_system > TV_SYSTEM_NONE) {
    gtk_option_menu_set_history (gui_mode_mask.system, gui_system);
  }
  /* mode will be init by caller */
}

/* -------- Settings -------- */

void gui_get_settings (void)
{
  gui_set.tv_hoffset  = gui_tv_offset.hadj->value;
  gui_set.tv_voffset  = gui_tv_offset.vadj->value;
  gui_set.mon_hoffset = gui_mon_offset.hadj->value;
  gui_set.mon_voffset = gui_mon_offset.vadj->value;
  gui_set.brightness  = gui_settings.brightness->value;
  gui_set.color       = gui_settings.color->value;
  gui_set.saturation  = gui_settings.saturation->value;
  gui_set.contrast    = gui_settings.contrast->value;
  gui_set.flicker     = gui_settings.flicker->value;
  gui_set.sharpness   = gui_settings.sharpness->value;
  gui_set.bandwidth   = gui_settings.bandwidth->value;
  gui_set.macrovision = gui_settings.macrovis->active;
  gui_set.monochrome  = gui_settings.monochrome->active;
  gui_set.connector   = gui_settings.connect;
  if (gui_settings.dualview->active) {
    gui_flags = NV_PRIV_TVMODE | NV_PRIV_DUALVIEW;
  } else {
    gui_flags = NV_PRIV_TVMODE; 
  }
  if (gui_bypass) {
    gui_flags |= NV_PRIV_BYPASS;
  }
}

void gui_set_settings (void)
{
  change_adjustment (gui_tv_offset.hadj,      gui_set.tv_hoffset);
  change_adjustment (gui_tv_offset.vadj,      gui_set.tv_voffset);
  change_adjustment (gui_mon_offset.hadj,     gui_set.mon_hoffset); 
  change_adjustment (gui_mon_offset.vadj,     gui_set.mon_voffset); 
  change_adjustment (gui_settings.brightness, gui_set.brightness);  
  change_adjustment (gui_settings.color,      gui_set.color); 
  change_adjustment (gui_settings.saturation, gui_set.saturation);  
  change_adjustment (gui_settings.contrast,   gui_set.contrast);
  change_adjustment (gui_settings.flicker,    gui_set.flicker);  
  change_adjustment (gui_settings.sharpness,  gui_set.sharpness);   
  change_adjustment (gui_settings.bandwidth,  gui_set.bandwidth);   
}

void settings_cb (GtkObject *obj, gpointer data)
{
  static int lock = 0;

  DPRINTF ("settings %i\n", lock);
  if (lock > 0) return;
  lock++;
  if (gui_func) {
    gui_get_settings ();
    gui_func->clamp (&gui_set);
    gui_set_settings ();
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
  lock--;
}

void setup_cb (GtkWidget *button, gpointer data)
{
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data), TRUE);
  gui_crt_tv_set (&gui_crt, &gui_tv, TRUE);
}

void bypass_cb (GtkWidget *button, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (button)->active) {
    gui_bypass = TRUE; 
  } else {
    gui_bypass = FALSE; 
  }
}

void color_bar_cb (GtkButton *button, gpointer data)
{
  /* FIXME: What about settings? Do they influence the bars? */
  backend->setTestImage (NULL, NULL);
}

/* ---- connector ---- */

void gui_connector_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (widget)->active)
  {
    gui_settings.connect = (NVConnect) data;
  }
  settings_cb (NULL, 0);
}

/*
 *  Check which TV connections are available, and set radio group.
 */

void gui_connector_set (NVConnect connect)
{
  DPRINTF ("set connect %i %i\n", connect, opt_connect);
  if (connect == CONNECT_AUTO)
  {
    DPRINTF ("auto connect\n");
    connect = backend->getConnection ();
  }
  gui_settings.connect = connect;
  set_radio_group_active (gui_settings.output, (int) connect);
}

void gui_connector_init (void)
{
  if (opt_connect == CONNECT_NONE) {
    gui_connector_set (CONNECT_AUTO);
  } else {
    gui_connector_set (opt_connect);
  }
}

void check_monitor_cb (GtkWidget *widget, gpointer data)
{
  gui_connector_set (CONNECT_AUTO);
}

/* ---- chip (I2C bus device) ---- */

void gui_device_changed (NVTvChip last_type)
{
  DPRINTF ("gui_device_changed %i %i\n", last_type, gui_tv_chip);
  if (last_type != gui_tv_chip) 
  {
    switch (last_type) 
    {
      case NV_BROOKTREE:
      case NV_CONEXANT:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_bt), "changed");
	break;
      case NV_CHRONTEL:
	gtk_signal_emit_by_name (GTK_OBJECT (hide_ch), "changed");
	break;
      default: 
	break;
    }
    gui_func = data_func (gui_tv_chip);
    switch (gui_tv_chip) 
    {
      case NV_BROOKTREE:
      case NV_CONEXANT:
	gtk_signal_emit_by_name (GTK_OBJECT (show_bt), "changed");
	break;
      case NV_CHRONTEL:
	gtk_signal_emit_by_name (GTK_OBJECT (show_ch), "changed");
	break;
      default: 
	break;
    }
    if (gui_func) gui_func->defaults (&gui_set);
    if (opt_flicker >= 0) {
      gui_set.flicker = opt_flicker;
    }
    gui_func->clamp (&gui_set);
    gui_set_settings ();
  }
  gui_connector_init ();
  gui_modes_init (); 
  DPRINTF ("gui_device_changed end\n");
}

/*
 * Set selected I2C device (TV chip).
 */

void gui_device_cb (GtkWidget *widget, ChipPtr chip)
{
  NVTvChip last_type;

  last_type = gui_tv_chip;
  backend->setChip (chip, TRUE);
  gui_tv_chip = chip->chip;
  gui_device_changed (last_type);
}

void gui_device_set (ChipPtr chip)
{
  NVTvChip last_type;
  int i;
  ChipPtr c;

  last_type = gui_tv_chip;
  if (!chip) return;
  backend->setChip (chip, TRUE);
  gui_tv_chip = chip->chip;
  for (i = 0, c = gui_card->chips; c; c = c->next, i++)
  {
    if (c == chip) 
      gtk_option_menu_set_history (gui_config_mask.chip, i);
  }
  gui_device_changed (last_type);
  DPRINTF ("gui_device_set end\n");
}

void gui_device_init (Bool probe)
{
  GtkWidget *menu;
  GtkWidget *item;
  ChipPtr c;

  DPRINTF ("gui_device_init\n");
  menu = gtk_menu_new ();
  if (probe) backend->probeChips ();
  for (c = gui_card->chips; c; c = c->next)
  {
    item = gtk_menu_item_new_with_label (c->name);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC(gui_device_cb), (gpointer) c);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  if (!gui_card->chips) {
    item = gtk_menu_item_new_with_label ("<No Device>");
    gtk_menu_append (GTK_MENU (menu), item);
  }
  /* unrefs and frees the old menu ... hopefully. */
  gtk_option_menu_set_menu (gui_config_mask.chip, menu);
  /* make labels visible */
  gtk_widget_show_all (menu); 
  /* FIXME: respect opt_tv_chip, and look for chiptype,
     instead of using first one. */
  /* Menu has already first entry active ... */
  gui_device_set (gui_card->chips);
  DPRINTF ("gui_device_init end\n");
}

void check_chip_cb (GtkWidget *widget, gpointer data)
{
  gui_device_init (TRUE);
}

/* ---- pci card ---- */

void gui_card_changed (void)
{
  char s [20];

  if (gui_card) {
    snprintf (s, 20, "(I/O 0x%08lX)", gui_card->reg_base);
  } else {
    strncpy (s, "(No card)", 20);
  }
  gtk_label_set_text (gui_config_mask.io_addr, s);
  gui_device_init (FALSE);
}

void gui_card_set (CardPtr card)
{
  if (gui_card) {
    backend->closeCard ();
  }
  gui_card = card;
  backend->openCard (gui_card);
  gui_card_changed ();
}

void gui_card_cb (GtkWidget *widget, gpointer data)
{
  gui_card_set ((CardPtr) data);
}

/* -------- X Video Mode -------- */

/*
 * Switch to smallest fitting video mode.
 */

void xswitch_mode_cb (GtkWidget *widget, gpointer data)
{
  if (!gui_act_mode) return;
  switch_vidmode (gdk_display, gdk_screen, 
		  gui_act_mode->res_x, gui_act_mode->res_y);
}

/* -------- X window center & resize -------- */

void gui_xresize_cb (GtkWidget *widget, gpointer data);

/*
 *  Check if window size is current, and get new size & hints if not.
 */

void gui_xwindow_check (void)
{
  Window root;
  int x, y;
  unsigned width, height, borderw, depth;
  static XSizeHints *hints = NULL;
  long user;

  if (!gui_window) return;
  if (!hints) hints = XAllocSizeHints();
  /* FIXME: error handler. Call gui_xwindow_changed. */
  XGetGeometry (gdk_display, gui_window, &root, &x, &y, &width, &height, 
		&borderw, &depth);
  if (width != gui_window_width || height != gui_window_height)
  {
    gui_window_width = width;
    gui_window_height = height;
    XGetWMNormalHints (gdk_display, gui_window, hints, &user);
    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_resize.hadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    gtk_signal_handler_block_by_func (GTK_OBJECT (gui_resize.vadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    if (hints->flags & PResizeInc) {
      gui_resize.hadj->step_increment = hints->width_inc;
      gui_resize.hadj->page_increment = 10 * hints->width_inc;
      gui_resize.vadj->step_increment = hints->height_inc;
      gui_resize.vadj->page_increment = 10 * hints->height_inc;
    }
    gtk_adjustment_set_value (gui_resize.hadj, width);
    gtk_adjustment_set_value (gui_resize.vadj, height);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_resize.hadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
    gtk_signal_handler_unblock_by_func (GTK_OBJECT (gui_resize.vadj),
      GTK_SIGNAL_FUNC (gui_xresize_cb), NULL);
  }
}

/*
 *  Resize the X window.
 */

void gui_xresize_cb (GtkWidget *widget, gpointer data)
{
  gui_xwindow_check ();
  gui_window_width = (int) gui_resize.hadj->value;
  gui_window_height = (int) gui_resize.vadj->value;
  if (gui_window)
  {
    /* FIXME: error handler */
    XResizeWindow (gdk_display, gui_window, 
		   gui_window_width, gui_window_height);
    XSync (gdk_display, False); /* give the WM a chance */
    gui_xwindow_check ();
  }
}

/*
 *  The X Window selection has changed.
 */

void gui_xwindow_changed ()
{
  gui_window_width = -1;
  gui_window_height = -1;
  if (gui_window) 
  {
    gui_xwindow_check (); 
    /* get width and height */
    gui_resize.hcenter = (int) gui_window_width;
    gui_resize.vcenter = (int) gui_window_height;
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xcenter), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.resize), TRUE);
  } else {
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.xcenter), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (gui_mode_mask.resize), FALSE);
  }
}

/*
 *  Select an X window, and store original size.
 */

void xselect_window_cb (GtkWidget *widget, gpointer data)
{
  gui_window = Select_Window (gdk_display, gdk_screen);
  if (gui_window != None) 
    gui_window = XmuClientWindow (gdk_display, gui_window);
  gui_xwindow_changed ();
}

/*
 *  Center the X window in the X virtual viewport.
 */

void xcenter_window_cb (GtkWidget *widget, gpointer data)
{
  if (gui_window == None) return;
  /* FIXME: Should respect current mode, if tv-on */
  center_window (gdk_display, gdk_screen, gui_window, -1, -1);
}

/* -------- -------- */

void offset_center_cb (GtkWidget *widget, struct mask_offset *offset)
{
  /* FIXME: freeze */
  gtk_adjustment_set_value (offset->hadj, offset->hcenter);
  gtk_adjustment_set_value (offset->vadj, offset->vcenter);
  /* FIXME: thaw, signal changed */
}

void spin_dec_cb (GtkWidget *widget, gpointer data)
{
  gtk_spin_button_spin (GTK_SPIN_BUTTON(data), GTK_SPIN_STEP_BACKWARD, 0.0);
}

void spin_inc_cb (GtkWidget *widget, gpointer data)
{
  gtk_spin_button_spin (GTK_SPIN_BUTTON(data), GTK_SPIN_STEP_FORWARD, 0.0);
}

void changed_all_cb (GtkObject *obj, gpointer data)
{
  gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
}

/* -------- -------- */

void gui_tvstate_update (NVCrtRegs *crt)

{
  NVCrtRegs crt_temp;

  if (!crt) crt = &crt_temp;
  backend->getMode (crt, NULL);
  gui_tvstate = (crt->PrivFlags & NV_PRIV_TVMODE) ? TRUE : FALSE;
}

void tvmonitor_on_cb (GtkButton *button, gpointer data)
{
  NVCrtRegs new_crt;

  gui_get_settings ();
  gui_tvstate_update (&new_crt);
  if (!gui_tvstate) {
    gui_orig_crt = new_crt;
    gui_orig_present = TRUE;
  }
  backend->setModeSettings (gui_flags, &gui_crt, &gui_tv, &gui_set);
  gui_tvstate = TRUE;
}

void tvmonitor_off_cb (GtkButton *button, gpointer data)
{
  if (!gui_orig_present) {
    /* We don't have valid crt values, so try the X vidmode 
       current modeline. */
    if (get_vidmode (gdk_display, gdk_screen, &gui_orig_crt)) {
      gui_orig_present = TRUE;
    }
  }
  if (gui_orig_present) {
    backend->setModeSettings (0, &gui_orig_crt, NULL, NULL);
  } else {
    fprintf (stderr, "No monitor crt modeline found.\n");
  }
  gui_tvstate = FALSE;
}

void apply_cb (GtkButton *button, gpointer data)
{
  if (gui_tvstate) {
    gui_get_settings ();
    backend->setModeSettings (gui_flags, &gui_crt, &gui_tv, &gui_set);
  }
}

void auto_apply_cb (GtkWidget *widget, gpointer data)
{
  if (gui_tvstate && gui_auto) {
    gui_get_settings ();
    backend->setModeSettings (gui_flags, &gui_crt, &gui_tv, &gui_set);
  }
}

void auto_cb (GtkWidget *widget, gpointer data)
{
  if (GTK_TOGGLE_BUTTON (widget)->active) {
    gui_auto = TRUE;
  } else {
    gui_auto = FALSE;
  }
}

void gui_print_cb (GtkButton *button, gpointer data)
{
  NVTvRegs temp_tv;
  NVCrtRegs temp_crt;

  temp_tv = gui_tv;
  temp_crt = gui_crt;
  gui_get_settings ();
  if (gui_func && !gui_bypass) 
    gui_func->setup (&gui_set, &temp_crt, &temp_tv);
  if (gui_act_mode) {
    printf ("*** Resolution %03i x %03i  Overscan %06.3f x %06.3f\n",
	    gui_act_mode->res_x, gui_act_mode->res_y,
	    gui_act_mode->hoc, gui_act_mode->voc);
  }
  if ((int) data & PRINT_CRT_REGS) 
    print_crt_regs (&temp_crt);
  if ((int) data & PRINT_CHIP_REGS) 
    print_tv_regs (&temp_tv, gui_tv_chip);
};			      

void reset_crt_cb (GtkWidget *widget, gpointer data)
{
  if (gui_act_mode) {
    gui_crt_set (gui_act_mode->crt);
  }
}

void reset_setting_cb (GtkWidget *widget, gpointer data)
{
  if (gui_func) {
    gui_func->defaults (&gui_set);
    gui_set_settings ();
    auto_apply_cb (NULL, NULL);
  }
}

void quit_cb (GtkWidget *widget, gpointer data)
{
  gtk_main_quit ();
}

void gui_map_cb (GtkWidget *widget, GtkAccelGroup *group)
{
  gtk_window_add_accel_group (GTK_WINDOW (gui_main_window), group);
}

void gui_unmap_cb (GtkWidget *widget, GtkAccelGroup *group)
{
  gtk_window_remove_accel_group (GTK_WINDOW (gui_main_window), group);
}

/* -------- -------- */

gfloat lower_bound (int bits) 
{
  if (bits >= 0) return 0;
  bits = -bits - 1;
  return (gfloat) - (((long long)1 << bits) - 1);
}

gfloat upper_bound (int bits) 
{
  if (bits < 0) bits = -bits - 1;
  return (gfloat) (((long long)1 << bits) - 1);
}


/* -------- GUI Pages -------- */

GtkWidget *create_adjust_frame (char *title, int min, int max, 
  gboolean hor_vert, struct mask_offset *offset, GtkSignalFunc signal)
{
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *spin;   
  GtkWidget *box;
  GtkAdjustment *adj;

  frame = gtk_frame_new (title);

  table = gtk_table_new (3, 3, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  if (hor_vert) {
    box = gtk_hbox_new (FALSE, 0);
  } else {
    box = gtk_vbox_new (FALSE, 0);
  }
  gtk_table_attach (GTK_TABLE(table), box, 0,3, 3,4,
    GTK_FILL, GTK_FILL,  0,0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            (double) min, (double) max, 1.0, 10.0, 0.0);
  offset->hadj = adj;
  offset->hcenter = 0.0;

  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_box_pack_start(GTK_BOX (box), spin, TRUE, TRUE, 0);

  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_LEFT,  0,1, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_dec_cb), spin);
  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_RIGHT, 2,3, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_inc_cb), spin);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            (double) min, (double) max, 1.0, 10.0, 0.0);
  offset->vadj = adj;
  offset->vcenter = 0.0;

  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_box_pack_start(GTK_BOX (box), spin, TRUE, TRUE, 0);

  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_UP,    1,2, 0,1);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_dec_cb), spin);
  button = create_arrow_button (GTK_TABLE(table), GTK_ARROW_DOWN,  1,2, 2,3);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (spin_inc_cb), spin);

  button = create_arrow_button (GTK_TABLE(table), (GtkArrowType) -1, 1,2, 1,2);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (offset_center_cb), offset);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  if (signal) {
    gtk_signal_connect (GTK_OBJECT (offset->hadj), "value-changed", 
      signal, NULL);
    gtk_signal_connect (GTK_OBJECT (offset->vadj), "value-changed", 
      signal, NULL);
  }

  return frame;
}

void gui_mask_checkbutton (GtkWidget *table, GtkObject *changed, 
		     struct flag_mask *m, int x1, int x2, int y)
{
  GtkWidget *button;   

  button = gtk_check_button_new_with_label (m->label);
  gtk_table_attach (GTK_TABLE(table), button, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_object_set_data (GTK_OBJECT (button), "mask", m);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      GTK_SIGNAL_FUNC (flag_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (flag_update_cb), GTK_OBJECT (button));
}

/*
 * Create label with spinbutton, adjustment and signals. 
 */

void gui_mask_entry (GtkWidget *table, GtkObject *changed, 
		     struct reg_mask *m, int x1, int x2, int x3, int y)
{
  GtkWidget *spin;   
  GtkWidget *label;
  GtkAdjustment *adj;

  label = gtk_label_new (m->label);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
	  lower_bound (m->bits), upper_bound (m->bits),
	  m->tick, m->tick * 10, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin),
				 GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x2, x3, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_update_cb), GTK_OBJECT (adj));
}

/*
 * Create label with two spinbuttons, adjustments and signals. 
 * This is a workaround, since the stupid spinbuttons can only hold
 * float values, which is not enough for 32 bits.
 */

void gui_mask_twin_entry (GtkWidget *table, GtkObject *changed, 
  struct reg_mask *m, int x1, int x2, int x3, int x4, int y)
{
  GtkWidget *spin;   
  GtkWidget *label;
  GtkAdjustment *adj;

  label = gtk_label_new (m->label);
  gtk_table_attach (GTK_TABLE(table), label, x1, x2, y, y+1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 99999.0,
	  m->tick, m->tick * 100, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin), GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x2, x3, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_high_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_high_update_cb), GTK_OBJECT (adj));

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 99999.0,
	  m->tick, m->tick * 100, 0.0);
  spin = create_spinbutton_with_size (adj, 0.0, 0);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spin), GTK_SHADOW_OUT);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, x3, x4, y, y+1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  /* Connect functionality to adjustment, not to spinbutton */
  gtk_object_set_data (GTK_OBJECT (adj), "adj", m);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (reg_low_changed_cb), NULL);
  gtk_signal_connect_object (changed, "changed",
		      GTK_SIGNAL_FUNC (reg_low_update_cb), GTK_OBJECT (adj));
}

GtkWidget *gui_config_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *box;
  GtkWidget *button;
  int i;
  CardPtr card;
  GtkWidget *opt, *menu, *item;

  page = gtk_table_new (1,1,FALSE);

  /* Hardware, Card Detected */

  frame = gtk_frame_new ("Hardware");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,4,0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  table = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("PCI Card:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1,0,1, 
    GTK_FILL, GTK_FILL, 0, 0);

  opt = gtk_option_menu_new();
  menu = gtk_menu_new();
  for (card = gui_card_list; card; card = card->next) 
  {
    item = gtk_menu_item_new_with_label (card->name);  
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC(gui_card_cb), (gpointer) card);
    gtk_menu_append (GTK_MENU (menu), item);
  }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (opt), menu);
  gtk_table_attach (GTK_TABLE(table), opt, 1,2,0,1, 
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  label = gtk_label_new ("(I/O:   --------)");
  gui_config_mask.io_addr = GTK_LABEL (label);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 2,3, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new ("TV Chip:");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  opt = create_option_menu (NULL, "<No Device>", NULL);
  gui_config_mask.chip = (GtkOptionMenu *) (opt);
  gtk_table_attach (GTK_TABLE(table), opt, 1,2, 2,3, 
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Probe");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (check_chip_cb), NULL); 
  gtk_table_attach (GTK_TABLE(table), button, 2,3, 2,3, 
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_table_set_row_spacings (GTK_TABLE(table), 3);
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);

  /* Connector */  

  frame = gtk_frame_new ("Connector");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,4, 1,2, 
    GTK_FILL, GTK_FILL, 0,0);

  box = create_radio_box (2, TRUE, GTK_SIGNAL_FUNC (gui_connector_cb), 
    &gui_settings.output, "FBAS", "S-VHS", "Both", "Convert", NULL);
  gtk_container_set_border_width (GTK_CONTAINER (box), 5);
  gtk_container_add (GTK_CONTAINER (frame), box);

  button = gtk_button_new_with_label ("Probe");
  gtk_misc_set_padding (GTK_MISC (GTK_BIN(button)->child), 31, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (check_monitor_cb), NULL); 
  gtk_box_pack_end (GTK_BOX (box), button, FALSE, FALSE, 0);

  /* Accelerators */

  frame = gtk_frame_new ("Keyboard accelerators");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,4, 2,3,
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  table = gtk_table_new (1,1,FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_table_set_row_spacings (GTK_TABLE(table), 3);
  gtk_table_set_col_spacings (GTK_TABLE(table), 20);

  for (i = 0; i < ACCEL_LAST; i++) {
    label = gtk_label_new (gui_accel[i].label);
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
    gtk_table_attach (GTK_TABLE(table), label, 0,1, i,i+1, 
      GTK_FILL, GTK_FILL, 0, 0);

    label = gtk_label_new (gui_accel[i].accel);
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
    gtk_table_attach (GTK_TABLE(table), label, 1,2, i,i+1, 
      GTK_FILL, GTK_FILL, 0, 0);
  }

  return page;
}

GtkWidget *gui_crtc_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkAccelGroup *gui_crtc_accel_group;
  int i;
  int lines;

  gui_crtc_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_crtc_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_crtc_accel_group);

  frame = gtk_frame_new ("CRT Controller Register Values");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  lines = 4;
  for (i = 0; TRUE; i++) {
    struct reg_mask *m = crt_mask + i;
    int y =  i % lines;
    int x = (i / lines) * 2;

    if (!(m->label)) break;
    gui_mask_entry (table, GTK_OBJECT (update_crt), m, x, x+1, x+2, y);
  }
  for (i = 0; TRUE; i++) {
    struct flag_mask *m = crt_mask_flags + i;
    int y =  i % lines;
    int x = (i / lines) * 2 + 4;

    if (!(m->label)) break;
    gui_mask_checkbutton (table, GTK_OBJECT (update_crt), m, x, x+1, y);
  }

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

#if 0
  /* FIXME: Todo */
  button = gtk_check_button_new_with_label ("Interlace");
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("Doublescan");
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_container_add (GTK_CONTAINER(bbox), button);
#endif

#if 0
  button = gtk_button_new_with_label ("Check");
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  /* check if registers are "sane", correct if necessary */
  gtk_table_attach(GTK_TABLE(page), button, 2,3,1,2,
    GTK_FILL, GTK_FILL, 0, 0);
#endif

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), (gpointer) PRINT_CRT_REGS); 
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_crt_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_crtc_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}
  
GtkWidget *gui_mode_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *bbox;
  GtkWidget *label;
  GtkWidget *opt;
  GtkWidget *button;
  GtkWidget *scroller;
  GtkWidget *clist;

  page = gtk_table_new (1,1,FALSE);

  /* TV System */ 

  frame = gtk_frame_new ("TV System");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,1, 0,1, 
    GTK_FILL, GTK_FILL, 0,0);

  opt = create_option_menu (GTK_SIGNAL_FUNC (gui_system_cb), 
    "NTSC", "NTSC-J", "PAL", "PAL-60", "PAL-N", "PAL-Nc", "PAL-M", 
    "PAL-M60", "PAL-X", NULL);
  gui_mode_mask.system = (GtkOptionMenu *) opt;
  gtk_container_set_border_width (GTK_CONTAINER (opt), 3); 
  /* gtk_option_menu_set_history (GTK_OPTION_MENU (opt), mode_system); */
  gtk_container_add (GTK_CONTAINER (frame), opt);
  
  /* Resolution */

  frame = gtk_frame_new ("Resolution");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 1,2, 0,1, 
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new (NULL);
  gui_mode_mask.res = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);
  
  /* Size */ 

  frame = gtk_frame_new ("Size");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 2,3, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new (NULL);
  gui_mode_mask.size = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);
  
  /* Overscan */ 

  frame = gtk_frame_new ("Overscan");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 3,4, 0,1,
    GTK_FILL, GTK_FILL, 0,0);

  label = gtk_label_new (NULL);
  gui_mode_mask.overscan = (GtkLabel *) label;
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_misc_set_padding (GTK_MISC (label), 5, 3);
  gtk_container_add (GTK_CONTAINER (frame), label);

  /* Mode list */

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_set_border_width (GTK_CONTAINER (scroller), 5);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_table_attach (GTK_TABLE(page), scroller, 0,3, 1,3,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  clist = gtk_clist_new (3);
  gui_mode_mask.modes = (GtkCList *) clist;
  gtk_clist_set_column_title (GTK_CLIST (clist), 0, "Resolution");
  gtk_clist_set_column_title (GTK_CLIST (clist), 1, "Size");
  gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Overscan");
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  gtk_clist_column_titles_active (GTK_CLIST (clist)); /* must be after show */
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_IN); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 70); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 50); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 2, 90); 
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
    GTK_SIGNAL_FUNC (gui_list_mode_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
    GTK_SIGNAL_FUNC (gui_modes_column_cb), NULL);
  gtk_container_add (GTK_CONTAINER(scroller), clist);

  /* Buttons */

  bbox = gtk_vbutton_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_EDGE);
  gtk_table_attach (GTK_TABLE(page), bbox, 3,4, 1,2,
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  button = gtk_button_new_with_label ("X Video Mode");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xswitch_mode_cb), NULL); 
  gtk_container_add (GTK_CONTAINER (bbox), button);

  button = gtk_button_new_with_label ("X Window select");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xselect_window_cb), NULL); 
  gtk_container_add (GTK_CONTAINER (bbox), button);

  button = gtk_button_new_with_label ("X Window center");
  gui_mode_mask.xcenter = button;
  gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (xcenter_window_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group, 
    gui_accel[ACCEL_CENTER].key, gui_accel[ACCEL_CENTER].mods,
    GTK_ACCEL_VISIBLE);
  gtk_container_add (GTK_CONTAINER (bbox), button);

  /* Resize */

  frame = create_adjust_frame ("Resize Window", 0, 10000, FALSE, 
    &gui_resize, GTK_SIGNAL_FUNC (gui_xresize_cb));
  gui_mode_mask.resize = frame;
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
  gtk_table_attach (GTK_TABLE(page), frame, 3,4, 2,3, 
    GTK_FILL, GTK_FILL | GTK_EXPAND, 0,0);

  return page;
}
  
GtkWidget *gui_settings_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *bbox;
  GtkWidget *scale;
  GtkWidget *label;
  GtkWidget *widget;
  GtkAdjustment *adj;
  GtkAccelGroup *gui_setting_accel_group;

  gui_setting_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_setting_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_setting_accel_group);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);
  gtk_table_attach_defaults(GTK_TABLE(page), table, 0,5,0,1);

  /* TV Position */

  frame = create_adjust_frame ("TV Position", -100, 100, TRUE,
    &gui_tv_offset, GTK_SIGNAL_FUNC (settings_cb));
  gtk_table_attach (GTK_TABLE(table), frame, 0,1, 0,2, 
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0,0);

  /* Monitor Position */

  frame = create_adjust_frame ("Monitor Position", -100, 100, TRUE,
    &gui_mon_offset, GTK_SIGNAL_FUNC (settings_cb));
  gtk_table_attach (GTK_TABLE(table), frame, 1,2, 0,2, 
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0,0);

  /* Options */

  frame = gtk_frame_new ("Options");
  gtk_table_attach (GTK_TABLE(table), frame, 2,3, 0,2, 
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0,0);

  bbox = gtk_vbutton_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), bbox);

  button = gtk_check_button_new_with_label ("Dualview");
  gui_settings.dualview = GTK_TOGGLE_BUTTON (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("Macrovision");
  gui_settings.macrovis = GTK_TOGGLE_BUTTON (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("Monochrome");
  gui_settings.monochrome = GTK_TOGGLE_BUTTON (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_container_add (GTK_CONTAINER(bbox), button);

  /* TV Settings */

  frame = gtk_frame_new ("TV Settings");
  gtk_table_attach (GTK_TABLE(table), frame, 0,3, 2,3,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0,0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Brightness:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,0,1, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 200.0, 1.0, 10.0, 0.0);
  gui_settings.brightness = adj;
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_adjustment_set_value (adj, 100.0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,3,0,1, 
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Color:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 200.0, 1.0, 10.0, 0.0);
  gui_settings.color = adj;
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_adjustment_set_value (adj, 100.0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,3, 1,2, 
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Contrast:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 200.0, 1.0, 10.0, 0.0);
  gui_settings.contrast = adj;
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_adjustment_set_value (adj, 100.0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,3, 2,3,
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Saturation:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 3,4, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 200.0, 1.0, 10.0, 0.0);
  gui_settings.saturation = adj;
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_adjustment_set_value (adj, 100.0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,3, 3,4,
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Bandwidth:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 4,5, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 200.0, 1.0, 10.0, 0.0);
  gui_settings.bandwidth = adj;
  gtk_adjustment_set_value (adj, 100);
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,3, 4,5,
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Flicker:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 5,6, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 100.0, 1.0, 10.0, 0.0);
  gui_settings.flicker = adj;
  gtk_adjustment_set_value (adj, 100);
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,2, 5,6,
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  label = gtk_label_new ("Sharpness:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 6,7, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 
            0.0, 100.0, 1.0, 10.0, 0.0);
  gui_settings.sharpness = adj;
  gtk_adjustment_set_value (adj, 100);
  scale = gtk_hscale_new (adj);
  gtk_scale_set_digits (GTK_SCALE (scale), 0);
  gtk_scale_set_value_pos (GTK_SCALE (scale), GTK_POS_RIGHT);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
			GTK_SIGNAL_FUNC (settings_cb), NULL);
  gtk_table_attach (GTK_TABLE(table), scale, 1,2, 6,7,
    GTK_FILL | GTK_EXPAND,GTK_FILL, 0,0);

  gtk_table_set_col_spacing (GTK_TABLE(table), 1, 150);

  /* Carrier adjust, maybe other filters ... */

  widget = gtk_toggle_button_new_with_label ("Bypass");
  button = gtk_button_new_with_label ("Setup");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (setup_cb), (gpointer) widget); 
  gtk_table_attach(GTK_TABLE(page), button, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = widget;
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (bypass_cb), NULL); 
  gtk_table_attach(GTK_TABLE(page), button, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Color bars");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (color_bar_cb), NULL); 
  gtk_table_attach(GTK_TABLE(page), button, 2,3,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), 
    (gpointer)(PRINT_CRT_REGS | PRINT_CHIP_REGS));
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group, 
    gui_accel[ACCEL_PRINT].key, gui_accel[ACCEL_PRINT].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_setting_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_setting_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}
  
GtkWidget *gui_about_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *scroller;
  GtkWidget *label;

  page = gtk_table_new (1,1,FALSE);

  frame = gtk_frame_new ("About");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach (GTK_TABLE(page), frame, 0,1, 0,1,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_set_border_width (GTK_CONTAINER (scroller), 5);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_add (GTK_CONTAINER (frame), scroller);

  label = gtk_label_new (
  "TV-OUT for NVidia Cards (Version " VERSION ")\n\n"
  "Written by Dirk Thierbach\n\n"
  "Visit the Sourceforge project at\n"
  "http://sourceforge.net/projects/nv-tv-out/\n\n"
  "With many thanks to the members of the Sourceforge Rivatv project. "
  "Without their help this program would not have been possible.\n\n"
  "This program is protected by the Gnu Public License (GPL). You should "
  "have received the source code of this program together with a copy of "
  "the GPL. You may freely use, distribute, or modify the program, as long "
  "as anything derived in this way from the source code is covered by the "
  "GPL as well. See the text of the GPL for details.");
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_pattern (GTK_LABEL (label), ""); /* does not work... */
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_FILL);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroller), 
    label);

  return page;
}

void gui_main_buttons (GtkWidget *window, GtkWidget *table)
{
  GtkWidget *button;

  button = gtk_button_new_with_label ("TV on");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 0,1,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (tvmonitor_on_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group,
    gui_accel[ACCEL_TV_ON].key, gui_accel[ACCEL_TV_ON].mods,
    GTK_ACCEL_VISIBLE);

  button = gtk_button_new_with_label ("TV off");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 1,2,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (tvmonitor_off_cb), NULL);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_main_accel_group,
    gui_accel[ACCEL_TV_OFF].key, gui_accel[ACCEL_TV_OFF].mods,
    GTK_ACCEL_VISIBLE);

  button = gtk_button_new_with_label ("Apply");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 2,3,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (apply_cb), NULL);

  button = gtk_toggle_button_new_with_label ("AutoApply");
  gtk_widget_show (button);
  gtk_table_attach(GTK_TABLE(table), button, 3,4,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (auto_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
		      GTK_SIGNAL_FUNC (auto_apply_cb), NULL);

  button = gtk_button_new_with_label ("Quit");
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_table_attach(GTK_TABLE(table), button, 4,5,1,2,
    GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
}

void gui_accel_parse (void)
{
  GuiAccelIndex i;

  for (i = ACCEL_FIRST; i < ACCEL_LAST; i++) {
    gtk_accelerator_parse (gui_accel[i].accel, 
    &gui_accel[i].key, &gui_accel[i].mods);
  }
}

void gui_accel_root (GtkWidget *main_window)
{
/* TODO:
register gdk_root_window as gdk window, if not already registered
XSelectInput on root window
gdk_window_set_user_data (widget->window, main_window);
... and hope for the best...
 */
}

/* -------- Main -------- */
 
void gui_main (int argc, char *argv[], CardPtr card_list)
{
  GtkWidget *table;
  GtkWidget *notebook;
  GtkWidget *page;

  DPRINTF ("gui_main\n");
  gtk_init (&argc, &argv);

  gui_card_list = card_list;

  /* New "notify" signal for update signal chains. FIXME: Make that work. */

  signal_notify =
  gtk_object_class_user_signal_new (gtk_type_class (GTK_TYPE_DATA),
                    "notify",
                    GTK_RUN_LAST | GTK_RUN_NO_RECURSE,
                    gtk_marshal_NONE__POINTER,
                    GTK_TYPE_NONE, 1,
                    GTK_TYPE_POINTER);

  /* In the meantime, misuse adjustments for update signal chains. */

  update_crt  = create_notify ();
  update_chip = create_notify ();
  update_mode = create_notify ();
  changed_all = create_notify ();
  show_bt = create_notify ();
  show_ch = create_notify ();
  hide_bt = create_notify ();
  hide_ch = create_notify ();

  gui_ch_init ();
  gui_bt_init ();

  gui_main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (gui_main_window), "NVidia TV-Out");
  gtk_signal_connect (GTK_OBJECT (gui_main_window), "destroy",
		      GTK_SIGNAL_FUNC (quit_cb), NULL);

  gui_main_accel_group = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (gui_main_window), 
    gui_main_accel_group);
  gui_accel_parse ();

  table = gtk_table_new (1,1,FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (gui_main_window), table);
    
  notebook = gtk_notebook_new ();
  gtk_widget_show (notebook);
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table), notebook, 0,5,0,1);
    
  page = gui_mode_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("Mode"));

  page = gui_settings_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("Settings"));

  page = gui_config_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("Config"));

  page = gui_crtc_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("CRT Regs"));

  page = gui_bt_calc_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("BT Calc"));
  gtk_signal_connect_object (GTK_OBJECT (show_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_bt_reg_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("BT Regs"));
  gtk_signal_connect_object (GTK_OBJECT (show_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_bt_flag_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("BT Flags"));
  gtk_signal_connect_object (GTK_OBJECT (show_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_bt_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("Status"));
  gtk_signal_connect_object (GTK_OBJECT (show_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_bt), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_ch_reg_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("CH Regs"));
  gtk_signal_connect_object (GTK_OBJECT (show_ch), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_ch), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_ch_status_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("Status"));
  gtk_signal_connect_object (GTK_OBJECT (show_ch), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_show_all), GTK_OBJECT (page));
  gtk_signal_connect_object (GTK_OBJECT (hide_ch), "changed",
    GTK_SIGNAL_FUNC (gtk_widget_hide_all), GTK_OBJECT (page));

  page = gui_about_page ();
  gtk_widget_show_all (page);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, 
    gtk_label_new ("About"));

  gui_main_buttons (gui_main_window, table);

  /* Update values */

  gui_system_init (); /* Init system only once. */
  gui_card_set (card_list);
  gui_tvstate_update (NULL);
  if (opt_window != None) {
    gui_window = opt_window;
    gui_xwindow_changed ();
  }

  color_bar_cb (NULL, NULL);

  gtk_widget_show (gui_main_window);

  gtk_main ();
}
