/* NVTV Gui -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See GPL.txt for details.
 *
 * The GTK graphical user interface. Chrontel part.
 */

#include <gtk/gtk.h>

#include "debug.h"
#include "gui.h"
#include "gui_ch.h"
#include "backend.h"
#include "data_ch.h"

static GtkAdjustment *update_ch;

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_CH(m) FIELD(gui_tv.ch,m)

/* -------- GUI Masks -------- */

struct mask_ch_freq {
  GtkLabel *clk, *hsyn, *vsyn, *hover, *vover, *aspect;
  GtkLabel *pixels, *lines;
};

struct mask_ch_freq gui_mask_ch_freq;

struct mask_civ {
  GtkLabel *low, *high;
};

struct mask_civ gui_mask_civ;

static struct reg_mask ch_mask [] = {
  {label:"DMR-IR:",    bits:3,  tick:1, FIELD_CH(dmr_ir)},
  {label:"DMR-VS:",    bits:2,  tick:1, FIELD_CH(dmr_vs)},
  {label:"DMR-SR:",    bits:3,  tick:1, FIELD_CH(dmr_sr)},
  {label:"PLL-M:",     bits:9,  tick:1, FIELD_CH(pll_m)},
  {label:"PLL-N:",     bits:10, tick:1, FIELD_CH(pll_n)},
  {label:"PLLCAP:",    bits:1,  tick:1, FIELD_CH(pllcap)},
  {label:"DACG:",      bits:1,  tick:1, FIELD_CH(dacg)},
  {label:"BLR:",       bits:8,  tick:1, FIELD_CH(blr)},
  {label:"CE:",        bits:3,  tick:1, FIELD_CH(ce)},
  {label:"FFR-FC:",    bits:2,  tick:1, FIELD_CH(ffr_fc)},
  {label:"FFR-FY:",    bits:2,  tick:1, FIELD_CH(ffr_fy)},
  {label:"FFR-FT:",    bits:2,  tick:1, FIELD_CH(ffr_ft)},
  {label:"VBW-FLFF:",  bits:1,  tick:1, FIELD_CH(vbw_flff)},  
  {label:"VBW-CVBW:",  bits:1,  tick:1, FIELD_CH(vbw_cvbw)},  
  {label:"VBW-CBW:",   bits:2,  tick:1, FIELD_CH(vbw_cbw)},   
  {label:"VBW-YPEAK:", bits:1,  tick:1, FIELD_CH(vbw_ypeak)}, 
  {label:"VBW-YSV:",   bits:2,  tick:1, FIELD_CH(vbw_ysv)},   
  {label:"VBW-YCV:",   bits:1,  tick:1, FIELD_CH(vbw_ycv)},
  {label:"SAV:",       bits:9,  tick:1, FIELD_CH(sav)},
  {label:"HPR:",       bits:9,  tick:1, FIELD_CH(hpr)},
  {label:"VPR:",       bits:9,  tick:1, FIELD_CH(vpr)},
  {label:"ACIV:",      bits:1,  tick:1, FIELD_CH(aciv)},
  {label:"CIVH:",      bits:2,  tick:1, FIELD_CH(civh)},
#if 0 /* Test register */
  {label:"YLM:",       bits:9,  tick:1, FIELD_CH(ylm)},
  {label:"CLM:",       bits:9,  tick:1, FIELD_CH(clm)},
#endif
  {label:"mode:",      bits:5,  tick:1, FIELD_CH(mode)},
  {label:"macro:",     bits:2,  tick:1, FIELD_CH(macro)},
  {label:NULL}
};				    

static struct reg_mask ch_mask_fsci =
  {label:"FSCI:",      bits:32, tick:1, FIELD_CH(fsci)};

void reset_ch_cb (GtkWidget *widget, gpointer data)
{
  if (gui_tv_chip == NV_CHRONTEL && gui_act_mode) { 
    gui_tv.ch = gui_act_mode->tv->ch; 
    gui_tv_set ();
    gtk_signal_emit_by_name (GTK_OBJECT (update_ch), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
}

void update_ch_cb (GtkWidget *widget, gpointer data)
{
  if (gui_tv_chip == NV_CHRONTEL) { 
    gtk_signal_emit_by_name (GTK_OBJECT (update_ch), "changed");
  }
}

void ch_freq_calc_cb (GtkObject *obj, struct mask_ch_freq *m)
{
  int alo, tlo;
  double ato, tto;
  int tpl, tlf; /* total pixels/line, total lines/field */
  double sc;
  double hoc, voc, aspect;
  double Fref, Fpixel;
  double Fhsyn;
  double Fvsyn;
  int res_x, res_y;
  char s [20];

  static int tlf_pal[8]  = {500,625, -1,750,836, -1,-1,-1};
  static int tlf_ntsc[8] = {420,525,600,630,700,750,-1,-1};
  static double scale[8] = {5/4.0, 1/1.0, 7/8.0, 5/6.0, 3/4.0, 7/10.0, -1, -1};

  if (gui_tv_chip != NV_CHRONTEL) return;
  Fref = 14.31818; /* MHz */
  switch (gui_tv.ch.dmr_vs) {
    case 1: /* TV_SYSTEM_NTSC */
    case 3: /* TV_SYSTEM_NTSC_J */
      alo = 480;
      tlo = 262 * 2 + 1;
      ato = 52.65556; /* us */
      tto = 63.55556; /* us */ /* 63.55556;  */
      tlf = tlf_ntsc[gui_tv.ch.dmr_sr];
      break;
    case 0: /* TV_SYSTEM_PAL */
    case 2: /* TV_SYSTEM_PAL_M */
      alo = 576; 
      tlo = 312 * 2 + 1;
      ato = 52.0; /* us */
      tto = 64.0; /* us */
      tlf = tlf_pal[gui_tv.ch.dmr_sr];
      break;
  }
  switch (gui_tv.ch.dmr_ir) {
    case 0: res_x = 512; res_y = 384; break;
    case 1: res_x = 720; res_y = 400; break;
    case 2: res_x = 640; res_y = 400; break;
    case 3: res_x = 640; res_y = 480; break;
    case 4: res_x = 800; res_y = 600; break;
    case 5: res_x = 720; res_y = alo; break;
    default: res_x = 1; res_y = 1;
  }
  Fpixel = Fref * (gui_tv.ch.pll_n + 2.0) / (gui_tv.ch.pll_m + 2.0);
  sprintf (s, "%3.2f MHz", Fpixel);
  gtk_label_set_text (m->clk, s);

  sc = scale[gui_tv.ch.dmr_sr];
  tpl = (int) ((tto / 2.0) / (1.0 / Fpixel) * sc + 0.5);
  hoc = 1.0 - ((double) res_x / tpl) / ((double) ato / tto);
  voc = 1.0 - ((double) res_y / tlf) / ((double) alo / tlo);
  aspect = 1.0; /* FIXME TODO */

  Fhsyn = 1e3 * Fpixel / tpl;
  snprintf (s, 20, "%3.2f kHz", Fhsyn);
  gtk_label_set_text (m->hsyn, s);

  Fvsyn = 1e3 * Fhsyn / tlf;
  snprintf (s, 20, "%3.2f  Hz", Fvsyn);
  gtk_label_set_text (m->vsyn, s);

  snprintf (s, 20, "%06.3f %%", hoc * 100.0);
  gtk_label_set_text (m->hover, s);
  snprintf (s, 20, "%06.3f %%", voc * 100.0);
  gtk_label_set_text (m->vover, s);
  snprintf (s, 20, "%07.5f", aspect);
  gtk_label_set_text (m->aspect, s);

  snprintf (s, 20, "%3i / %3i", res_x, tpl);
  gtk_label_set_text (m->pixels, s);
  snprintf (s, 20, "%3i / %3i", res_y, tlf);
  gtk_label_set_text (m->lines, s);
}

gint check_civ_cb (struct mask_civ *m)
{
  long status;
  char s[20];

  if (gui_tv_chip != NV_CHRONTEL) return TRUE;
  status = backend->getStatus (0);
  if (status >= 0) {
    snprintf (s, 20, "%9li", status << 6);
    gtk_label_set_text (m->low, s);
    snprintf (s, 20, "%9li", status << 6 | 0x3f); 
    gtk_label_set_text (m->high, s);
  } else {
    gtk_label_set_text (m->low, "<None>");
    gtk_label_set_text (m->high, "<None>");
  }
  return TRUE;
}

/* -------- GUI Pages -------- */

GtkWidget *gui_ch_reg_page (void)
{
  GtkWidget *frame;
  GtkWidget *page;
  GtkWidget *table;
  GtkWidget *button;
  GtkAccelGroup *gui_ch_reg_accel_group;
  int i;
  int lines = 9;

  gui_ch_reg_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_ch_reg_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_ch_reg_accel_group);

  frame = gtk_frame_new ("Chrontel Register Values");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  for (i = 0; TRUE; i++) {
    struct reg_mask *m = ch_mask + i;
    int y =  i % lines;
    int x = (i / lines) * 3;

    if (!(m->label)) break;
    gui_mask_entry (table, GTK_OBJECT (update_ch), m, x, x+2, x+3, y);
  }
  gui_mask_twin_entry (table, GTK_OBJECT (update_ch), &ch_mask_fsci, 
    0, 2, 3, 4, lines);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), (gpointer) PRINT_CHIP_REGS);
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_ch_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_ch_reg_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

GtkWidget *gui_ch_status_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;

  page = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (page), 5);

  /* Frequencies CRT / TV */

  frame = gtk_frame_new ("Frequencies");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (3, 4, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Monitor dot clock");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.-- MHz");
  gui_mask_ch_freq.clk = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Monitor horiz. sync");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.-- kHz");
  gui_mask_ch_freq.hsyn = (GtkLabel *) label; 
  gtk_table_attach (GTK_TABLE(table), label, 1,2,2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Monitor vert. sync");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,3,4, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.--  Hz");
  gui_mask_ch_freq.vsyn = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,3,4, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV horiz. overscan");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,4,5, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--  %");
  gui_mask_ch_freq.hover = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,4,5, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV vert. overscan");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,5,6, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--  %");
  gui_mask_ch_freq.vover = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,5,6, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV aspect ratio");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,6,7, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("-.---  ");
  gui_mask_ch_freq.aspect = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,6,7, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Pixels (horiz.)");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,7,8, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--- / ---");
  gui_mask_ch_freq.pixels = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,7,8, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Lines (vert.)");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,8,9, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--- / ---");
  gui_mask_ch_freq.lines = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,8,9, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  /* CH Status */

  frame = gtk_frame_new ("CH Status");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (3, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("CIV");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---------");
  gui_mask_civ.low = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---------");
  gui_mask_civ.high = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  return page;
}
  
void gui_ch_init (void)
{
  update_ch = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (update_chip), "changed", 
    GTK_SIGNAL_FUNC (update_ch_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
    GTK_SIGNAL_FUNC (ch_freq_calc_cb), &gui_mask_ch_freq);

#ifndef DISABLE_TIMEOUT
  /* FIXME: Only add timeout if status page is mapped */
  gtk_timeout_add( 500 /* ms */, (GtkFunction) check_civ_cb, 
		   &gui_mask_civ);
#endif
}
