/* NVTV main -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Main routine. Parse options, execute actions or call gui.
 *
 */

#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include <unistd.h>
#include "xfree.h"

#include <X11/Xlib.h>
#ifdef HAVE_GTK
#include <gdk/gdkprivate.h>
#include "gui.h"
#endif

#include "backend.h"
#include "back_direct.h"
#include "back_null.h"
#include "back_client.h"
#include "main.h"
#include "actions.h"

/* Debug: */

#include "nv_tv.h"

/* -------- Global options -------- */

Bool opt_no_root = FALSE;

Bool opt_nvdev = FALSE;

int opt_res_x = -1, opt_res_y = -1;
float opt_hoc = -1.0, opt_voc = -1.0;

char *opt_size = NULL;
char *opt_dpy = NULL;
char *opt_window_name = NULL;

BackFuncPtr backend; /* global backend */

NVTvChip opt_tv_chip = NV_NO_CHIP;
NVSystem opt_system = TV_SYSTEM_NONE;
NVConnect opt_connect = CONNECT_AUTO;
int opt_flicker = -1;
Window opt_window = None;

static const char *short_options = "?bcdf:ghmno:pr:s:tw:C:NPS:W:X";

static struct option long_options[] =
  {{"tv-bars",     no_argument,       NULL, 'b'},
   {"center",      no_argument,       NULL, 'c'},
   {"debug",       no_argument,       NULL, 'd'},
   {"flicker",     required_argument, NULL, 'f'},
   {"gui",         no_argument,       NULL, 'g'},
   {"help",        no_argument,       NULL, 'h'},
   {"tv-off",      no_argument,       NULL, 'm'},
   {"no-root",     no_argument,       NULL, 'n'},
   {"overscan",    required_argument, NULL, 'o'},
   {"print",       no_argument,       NULL, 'p'},
   {"resolution",  required_argument, NULL, 'r'},
   {"size",        required_argument, NULL, 's'},
   {"tv-on",       no_argument,       NULL, 't'},
   {"win-name",    required_argument, NULL, 'w'},
   {"display",     required_argument, NULL, 'D'},
   {"connector",   required_argument, NULL, 'C'},
   {"nvdev",       no_argument,       NULL, 'N'},
   {"probe",       no_argument,       NULL, 'P'},
   {"system",      required_argument, NULL, 'S'},
   {"chip",        required_argument, NULL, 'T'},
   {"win-id",      required_argument, NULL, 'W'},
   {"switch-mode", no_argument,       NULL, 'X'},
   {"sync",        no_argument,       NULL, '.'},
   {"no-xshm",     no_argument,       NULL, '.'},
   {"name",        required_argument, NULL, '.'},
   {"class",       required_argument, NULL, '.'},
   {NULL,          0,                 NULL, 0}
};

/* -------- Actions -------- */

typedef enum {
  ACTION_NONE    = 0,
  ACTION_PRINT   = 1,
  ACTION_GUI     = 2,
  ACTION_TVON    = 3,
  ACTION_TVOFF   = 4,
  ACTION_TVBARS  = 5,
  ACTION_DEBUG   = 6,
  ACTION_PROBE   = 7,
} ActionMain;

typedef enum {
  ACTION_CENTER  = (1 << 0),
  ACTION_XMODE   = (1 << 1),
} ActionAux;

/* -------- */

void usage (void)
{
  fprintf (stderr,
      "usage:  nvtv [-options ...]\n\n");
  fprintf (stderr,
      "where options include:\n");
  fprintf (stderr,
      "  -h --help           print this message\n");
  fprintf (stderr,
      "     --display hst:#  X server to contact\n");
  fprintf (stderr,
      "  -p --print           calculate and print register values\n");
  fprintf (stderr,
      "  -m --tv-off          TV off, switch to monitor\n");
  fprintf (stderr,
      "  -t --tv-on           TV on\n");
  fprintf (stderr,
      "  -b --tv-bars         Color bars\n");
  fprintf (stderr,
      "  -r --resolution x,y  Resolution\n");
  fprintf (stderr,
      "  -o --overscan x,y    Overscan percentage (float)\n");
  fprintf (stderr,
      "  -s --size #          Size (for predefined modes)\n");
  fprintf (stderr,
      "  -S --system          NTSC,NTSC-J,PAL,PAL-60,PAL-NC,PAL-M,PAL-M60\n");
  fprintf (stderr,
      "  -C --connector       AUTO,FBAS,SVHS,BOTH,CONVERT\n");
  fprintf (stderr,
      "  -f --flicker         flicker filter setting (0%%-100%%)\n");
  fprintf (stderr,
      "     --chip            BROOKTREE,CHRONTEL\n");
  fprintf (stderr,
      "  -W --win-id id       select window by numerical id\n");
  fprintf (stderr,
      "  -w --win-name name   select window by name\n");
  fprintf (stderr,
      "  -c --center          center selected window\n");
  fprintf (stderr,
      "  -X --switch-mode     switch X mode\n");
  fprintf (stderr,
      "  -P --probe           probe and print system information\n");
  fprintf (stderr,
      "  -N --nvdev           enable usage of /dev/nv* devices\n");
  fprintf (stderr,
      "  -g --gui             always use gui\n");
  fprintf (stderr,
      "  -n --no-root         (only for debugging)\n");
  fprintf (stderr,
      "  -d --debug           (only for debugging)\n");
  exit (1);
}

void printRegs (NVMode *mode)
{
  printf ("*** Resolution %03i x %03i  Overscan %06.3f x %06.3f\n",
	  mode->res_x, mode->res_y, mode->hoc, mode->voc);
  print_tv_regs (mode->tv, opt_tv_chip);
  print_crt_regs (mode->crt);
}

void doDebug (void)
{
}

int main (int argc, char *argv[])
{
  CardPtr main_card;
  int c = '?';
  ActionMain act_main = ACTION_NONE;
  ActionMain act_next;
  ActionAux act_aux = 0;
  Display *main_dpy;
  int main_screen;
  NVMode main_mode;
  NVCrtRegs main_crt;
  NVTvRegs main_tv;
  NVSettings main_set;

  opterr = 0;
  while ((c = getopt_long (argc, argv, short_options, 
                long_options, NULL)) != EOF) 
  {
    act_next = ACTION_NONE;
    switch(c) 
    {
      case 'h': /* Print usage */
      case '?':
        usage(); 
	break;
      case 'D': 
	opt_dpy = optarg; 
	break;
      case '.': /* Gtk/Gdk option, ignore */
        break;
      case 'n':
	opt_no_root = TRUE; 
	break;
      case 'r':
	if (!optarg || sscanf (optarg, "%i,%i", &opt_res_x, &opt_res_y) != 2) 
	  usage ();
	break;
      case 'o':
	if (!optarg || sscanf (optarg, "%f,%f", &opt_hoc, &opt_voc) != 2)
	  usage ();
	break;
      case 'd':
	act_next = ACTION_DEBUG;
	break;
      case 'm':
	act_next = ACTION_TVOFF;
	break;
      case 't':
	act_next = ACTION_TVON;
	break;
      case 'b':
	act_next = ACTION_TVBARS;
	break;
      case 'p':
	act_next = ACTION_PRINT;
	break;
      case 'P':
	act_next = ACTION_PROBE;
	break;
      case 'g':
	act_next = ACTION_GUI;
	break;
      case 's':
	opt_size = optarg;
	break;
      case 'S':
	if (optarg && strcasecmp (optarg, "ntsc") == 0) 
	  opt_system = TV_SYSTEM_NTSC; 
	else if (optarg && strcasecmp (optarg, "ntsc-j") == 0) 
	  opt_system = TV_SYSTEM_NTSC_J; 
	else if (optarg && strcasecmp (optarg, "pal") == 0) 
	  opt_system = TV_SYSTEM_PAL; 
	else if (optarg && strcasecmp (optarg, "pal-60") == 0) 
	  opt_system = TV_SYSTEM_PAL_60; 
	else if (optarg && strcasecmp (optarg, "pal-n") == 0) 
	  opt_system = TV_SYSTEM_PAL_N; 
	else if (optarg && strcasecmp (optarg, "pal-nc") == 0) 
	  opt_system = TV_SYSTEM_PAL_NC; 
	else if (optarg && strcasecmp (optarg, "pal-m") == 0) 
	  opt_system = TV_SYSTEM_PAL_M; 
	else if (optarg && strcasecmp (optarg, "pal-m60") == 0) 
	  opt_system = TV_SYSTEM_PAL_M60; 
	else
	  usage ();
	break;
      case 'C':
	if (optarg && strcasecmp (optarg, "auto") == 0) 
	  opt_connect = CONNECT_AUTO; 
	else if (optarg && strcasecmp (optarg, "fbas") == 0) 
	  opt_connect = CONNECT_FBAS; 
	else if (optarg && strcasecmp (optarg, "svhs") == 0) 
	  opt_connect = CONNECT_SVHS; 
	else if (optarg && strcasecmp (optarg, "both") == 0) 
	  opt_connect = CONNECT_BOTH; 
	else if (optarg && strcasecmp (optarg, "convert") == 0) 
	  opt_connect = CONNECT_CONVERT; 
	else
	  usage ();
	break;
      case 'T':
	if (optarg && strcasecmp (optarg, "brooktree") == 0) 
	  opt_tv_chip = NV_BROOKTREE; 
	else if (optarg && strcasecmp (optarg, "conexant") == 0) 
	  opt_tv_chip = NV_CONEXANT; 
	else if (optarg && strcasecmp (optarg, "chrontel") == 0) 
	  opt_tv_chip = NV_CHRONTEL; 
	else if (optarg && strcasecmp (optarg, "philips") == 0) 
	  opt_tv_chip = NV_PHILIPS; 
	else
	  usage ();
	break;
      case 'f':
	if (!optarg || sscanf (optarg, "%i", &opt_flicker) != 1) 
	  usage ();
	break;
      case 'w':
	opt_window_name = optarg;
	break;
      case 'W':
	if (sscanf (optarg, "0x%lx", &opt_window) != 1) opt_window = None;
	if (opt_window == None) {
	  if (sscanf (optarg, "%ld", &opt_window) != 1) opt_window = None;
	}
	if (opt_window == None)
	  fprintf (stderr, "Invalid window id format: %s\n.", optarg);
	break;
      case 'c':
	act_aux |= ACTION_CENTER;
	break;
      case 'X':
	act_aux |= ACTION_XMODE;
	break;
      case 'N':
	opt_nvdev = TRUE;
	break;
    }
    if (act_next != ACTION_NONE) 
    {
      if (act_main != ACTION_NONE) {
	fprintf (stderr, "nvtv: More than one action. Choosing last one.\n");
      }
      act_main = act_next;
    }
  }

  /* Do the probing right away, without processing any other options. */

  if (act_main == ACTION_PROBE) {
#if 0 /* FIXME */
    probeSystem (main_card);
#else
    fprintf (stderr, "Probing currently disabled\n");
#endif
    return 0;
  }

  /* Start to process the options. First, if -s/-o is specified, default
     -r and -S, and assume -g if no main action is present. */

  if (opt_size || opt_hoc >= 0.0 || opt_voc >= 0.0) {
    if (opt_system == TV_SYSTEM_NONE) {
      fprintf (stderr, "Defaulting to PAL TV system.\n");
      opt_system = TV_SYSTEM_PAL;
    }
    if (opt_res_x <= 0) {
      opt_res_x = 800;
      fprintf (stderr, "Defaulting to x resolution %i\n", opt_res_x);
    }
    if (opt_res_y <= 0) {
      opt_res_y = 600;
      fprintf (stderr, "Defaulting to y resolution %i\n", opt_res_y);
    }
    if (act_main == ACTION_NONE) {
      act_main = ACTION_GUI;
    }
  }

  if (act_main == ACTION_NONE && !act_aux) {
    act_main = ACTION_GUI;
  }

  /* Next, choose the backend. If a main action is present, always try to 
     get a backend. If -n is specified, force the null backend. For
     printing, we can get away with having the null backend, as well. */
  
  if (act_main != ACTION_NONE) {
    if (opt_no_root) {
      DPRINTF ("init null\n");
      main_card = back_null_init (); 
    } else {
      if (back_root_avail ()) {
	DPRINTF ("init root\n");
	main_card = back_root_init ();
      } else if (back_client_avail ()) {
	DPRINTF ("init client\n");
	main_card = back_client_init ();
      } else if (opt_nvdev && back_nvdev_avail (TRUE)) {
	DPRINTF ("init nvdev\n");
	main_card = back_nvdev_init ();
      } else if (act_main == ACTION_PRINT) {
	DPRINTF ("init null\n");
	main_card = back_null_init ();
      } else {
	fprintf (stderr, "Cannot access cards. Either you are not root, "
                 "or the NVidia devices are\nnot accessible.\n");
	exit (1);
      }
    }
    if (!main_card) {
      fprintf (stderr, "No NVidia card found.\n");
      exit (1);
    }
  }

  /* Next, open the first card, unless the main action is none or the gui.
     Find the right chip, if specified. Note: If printing, we can find all 
     tv chip types in the null backend. */

  if (act_main != ACTION_NONE && act_main != ACTION_GUI) {
    backend->openCard (main_card);
    if (opt_tv_chip != NV_NO_CHIP) {
      ChipPtr chip;

      for (chip = main_card->chips; chip; chip = chip->next) {
	if (opt_tv_chip == chip->chip) break;
      }
      if (chip) {
	backend->setChip (chip, TRUE);
      } else {
	fprintf (stderr, "Cannot find specified chip.\n");
	exit (1);
      }
    }
  }

  /* Next, find the mode for printing -p and tv-on -t. This is required,
     fail if no mode is found. */

  if (act_main == ACTION_TVON || act_main == ACTION_PRINT) {
    if (opt_size) {
      if (! backend->findBySize (opt_system, opt_res_x, opt_res_y, opt_size,
				 &main_mode, &main_crt, &main_tv)) 
      {
	fprintf (stderr, "Cannot find '%s' mode %i x %i \n", 
		 opt_size, opt_res_x, opt_res_y);
	fprintf (stderr, "Please specify as e.g. -r 800,600 -s Normal\n");
	exit (1);
      }
    } else if (opt_hoc >= 0.0 || opt_voc >= 0.0) {
      if (opt_hoc < 0.0) opt_hoc = 0.10;
      if (opt_voc < 0.0) opt_voc = 0.10;
      /* FIXME: Defaulting message */
      /* FIXME: findByOverscan */
      fprintf (stderr, "Specifying modes by overscan is not implemented yet\n");
      exit (1);
    } else {
      fprintf (stderr, "No mode specified.\n");
      fprintf (stderr, "Please specify as e.g. -r 800,600 -s Normal\n");
      exit (1);
    }
  }
      
  /* For tv-off, the auxiliary actions or the gui, we need a display */

  if (act_main == ACTION_GUI || act_main == ACTION_TVOFF || act_aux) {
#if HAVE_GTK
    /* For some reason, just XOpenDisplay is not enough if GDK is 
       linked in / to be used later. */
    gdk_init (&argc, &argv);
    main_dpy = gdk_display;
    main_screen = gdk_screen;
#else
    main_dpy = XOpenDisplay (opt_dpy);
    main_screen = DefaultScreen (main_dpy);
#endif
  }

  /* For tv-off, we need a monitor mode. Get it with the XVidMode extension,
     and default resolution, if necessary. */

  if (act_main == ACTION_TVOFF) {
    if (opt_res_x > 0 && opt_res_y > 0) {
      find_vidmode (main_dpy, main_screen, opt_res_x, opt_res_y, &main_crt);
    } else 
    if (get_vidmode (main_dpy, main_screen, &main_crt)) {
      if (opt_res_x <= 0 || opt_res_y <= 0) {
	opt_res_x = main_crt.HDisplay;
	opt_res_y = main_crt.VDisplay;
      }
    } else {
      fprintf (stderr, "No monitor mode to switch tv off. "
	               "You need the XVidMode extension.\n");
      exit (1);
    }
    /* FIXME: Without XVidMode extension, try to get the mode from 
       somewhere else, i.e. internal VESA database? */
    /* NOTE: There must be a valid mode in main_crt here */
  }

  /* If there is still no resolution (i.e., opt_res both -1), then we cannot
     switch the X mode, even if requested, so disable that */

  if ((act_aux & ACTION_XMODE) && ((opt_res_x < 0) || (opt_res_y < 0))) {
    act_aux &= ~ACTION_XMODE;
    fprintf (stderr, "No resolution found. Switching X mode disabled.\n");
  }

  /* Finally, modify the settings, unless the main action is none or gui. */

  if (act_main != ACTION_NONE && act_main != ACTION_GUI) {
    backend->getSettings (&main_set);
    if (opt_connect > CONNECT_NONE) {
      main_set.connector = opt_connect;
    } else {
      main_set.connector = CONNECT_BOTH;
    }
    if (opt_flicker >= 0) {
      main_set.flicker = opt_flicker;
    }
    /* FIXME: Should probe for CONNECT_AUTO */
  }

  /* Now we start executing the actions. Printing, color bars and 
     debugging ignore the auxiliary actions, and are handled first. */

  switch (act_main) 
  {
    case ACTION_TVBARS:
      backend->setTestImage (NULL, &main_set);
      return 0;
    case ACTION_PRINT:
      printRegs (&main_mode);
      return 0;
    case ACTION_DEBUG:
      doDebug ();
      return 0;
    default:
      break;
  }
  
  /* If we reached this point, we have to process the auxiliary options.
     If the action is tv-off, we should execute the action first, so
     switching the X mode comes afterwards. There must be valid monitor
     mode in main_crt at this point. */

  if (act_main == ACTION_TVOFF) {
    backend->setMode (0, &main_crt, NULL);
  }

  /* Now, switch the X mode */

  if (act_aux & ACTION_XMODE) {
    switch_vidmode (main_dpy, main_screen, opt_res_x, opt_res_y);
    XSync (main_dpy, False);
  }

  /* Next, look for the window. */

  if (opt_window_name) {
    opt_window = Window_With_Name (main_dpy, 
      RootWindow(main_dpy, main_screen), opt_window_name);
    if (opt_window == None) {
      fprintf (stderr, "No window with name '%s' exists.\n", opt_window_name);
    }
  }

  /* Next, center the window, with respect to the resolution, if given. */

  if (act_aux & ACTION_CENTER) {
    if (opt_window != None) {
      center_window (main_dpy, main_screen, opt_window, opt_res_x, opt_res_y);
    } else {
      fprintf (stderr, "No window specified.\n");
      exit (1);
    }
  }

  /* Next, switch tv mode on. The main_mode must be a valid tv mode at
     this point. */

  if (act_main == ACTION_TVON) {
    backend->setModeSettings (NV_PRIV_TVMODE | NV_PRIV_DUALVIEW, 
			      main_mode.crt, main_mode.tv, &main_set);
  }

  /* Finally, start the gui. */

  if (act_main == ACTION_GUI) {
#ifdef HAVE_GTK
    gui_main (argc, argv, main_card); 
#else
    fprintf (stderr, "Compiled without gui\n");
    return 1;
#endif
  }

  return 0;
}

/* -------------------------------------------------------------------

Table of options/action:

                                   mode  backend   card
none     -r -X          -c         no    no        ignore
gui  -g  -r -X -s/-o -S -c -C/-f   opt   yes       ignore
on   -t  -r -X -s/-o -S -c -C/-f   yes   yes       open
off  -m  -r -X          -c         no    yes       open
bars -b                    -C/-f   no    yes       open
prt  -p  -r    -s/-o -S 	   yes   yes/none  open
     -P
     -d


Rules:

If only -s/-o is given, -r and -S are defaulted, regardless of action.
If -s/-o is specified and no action given, assume -g.
For -g, only auxiliary options are executed. No mode search, no settings.
Mode search is only done for -t and -p, and aborts if none is found/specified.
If card is opened, --chip will be respected.

----------------------------------------------------------------------



xine --display-ratio 0.979 # for 768x576 Large

Not implemented yet:

-x   --x-options         standard x options follow

GDK Options:

+ --display
+ --sync
+ --no-xshm
+ --name
+ --class
  --gxid_host
  --gxid_port
  --xim-preedit
  --xim-status

GTK Options

  --gtk-module
  --g-fatal-warnings

*/
