/* NVTV Nv_tv -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See GPL.txt for details.
 *
 * Access the BT chip (and the CRTC). This part could eventually become
 * a part of the XFree NV-Driver.
 *
 * Saving and restoring the CRTC should be modified in XFree, because
 * more registers are stored.
 *
 */


#include <stdio.h>
#include <string.h>
#include "debug.h"
#include "xfree.h"
#include "xf86i2c.h"
#include "nv_tv.h"
#include "nv_type.h"

/* -------- Global variables -------- */

Bool nvBusOk = TRUE;

NVTvState nvTvState = NV_TV_UNKNOWN;

/* -------- */

/* Little macro to construct bitmask for contiguous ranges of bits */

#define BITMASK(t,b) (((unsigned)(1U << (((t)-(b)+1)))-1)  << (b))
#define MASKEXPAND(mask) BITMASK(1?mask,0?mask)

/* Macro to set specific bitfields (mask has to be a macro x:y) ! */

#define SetBF(mask,value) ((value) << (0?mask))
#define GetBF(var,mask) (((unsigned)((var) & MASKEXPAND(mask))) >> (0?mask) )

#define MaskAndSetBF(var,mask,value) (var)=(((var)&(~MASKEXPAND(mask)) \
                                             | SetBF(mask,value)))

/* SetBitField: Move bit-range in 'from' to bit-range in 'to' */

#define SetBitField(value,from,to) SetBF(to, GetBF(value,from))
#define SetBitFlag(value,mask,to) ((value & mask) ? (1 << to) : 0)
#define SetBit(n) (1<<(n))
#define Set8Bits(value) ((value)&0xff)

static void NVTvI2CGetBits(I2CBusPtr b, int *clock, int *data)
{
    NVPtr pNv = NVPTR(xf86Screens[b->scrnIndex]);
    unsigned char val;

    /* Get the result. */
    VGA_WR08(pNv->riva.PCIO, 0x3d4, b->DriverPrivate.val);
    val = VGA_RD08(pNv->riva.PCIO, 0x3d5);

    *clock = (val & DDC_SCL_READ_MASK) != 0;
    *data  = (val & DDC_SDA_READ_MASK) != 0;

    DEBUG(ErrorF("NVTvI2CGetBits(%p) val=0x%x, returns clock %d, data %d\n",
                 b, val, *clock, *data));
}

static void NVTvI2CPutBits(I2CBusPtr b, int clock, int data)
{
    NVPtr pNv = NVPTR(xf86Screens[b->scrnIndex]);
    unsigned char val;

    VGA_WR08(pNv->riva.PCIO, 0x3d4, b->DriverPrivate.val+1);
    val = VGA_RD08(pNv->riva.PCIO, 0x3d5) & 0xf0;
    if (clock)
        val |= DDC_SCL_WRITE_MASK;
    else
        val &= ~DDC_SCL_WRITE_MASK;

    if (data)
        val |= DDC_SDA_WRITE_MASK;
    else
        val &= ~DDC_SDA_WRITE_MASK;

    VGA_WR08(pNv->riva.PCIO, 0x3d4, b->DriverPrivate.val+1);
    VGA_WR08(pNv->riva.PCIO, 0x3d5, val | 0x1);
    
    DEBUG(ErrorF("NVTvI2CPutBits(%p, %d, %d) val=0x%x\n", 
		 b, clock, data, val));
}

I2CBusPtr NVTvI2CInit (ScrnInfoPtr pScrn, char *name, int private)
{
  I2CBusPtr I2CPtr;

  I2CPtr = xf86CreateI2CBusRec();
  if (I2CPtr) {
    I2CPtr->BusName    = name;
    I2CPtr->scrnIndex  = pScrn->scrnIndex;
    I2CPtr->I2CPutBits = NVTvI2CPutBits;
    I2CPtr->I2CGetBits = NVTvI2CGetBits;
    I2CPtr->AcknTimeout = 5;
    I2CPtr->DriverPrivate.val = private; /* CRTC index for I2C Bus */
  }
  return I2CPtr;
}

Bool NVTvBusInit (ScrnInfoPtr pScrn)
{
  Bool ok;
  NVPtr pNv = NVPTR(pScrn);
  int bus;

  pNv->TvChain = NULL;
  pNv->TvDev = NULL;
  /* Must be named TV0, TV1, etc.; otherwise I2C_ID won't work */
  pNv->TvBusses[0] = NVTvI2CInit (pScrn, "TV0", 0x3e);
  pNv->TvBusses[1] = NVTvI2CInit (pScrn, "TV1", 0x36);
  pNv->TvBusses[2] = NVTvI2CInit (pScrn, "TV2", 0x50);
  
  ok = TRUE;
  for (bus = 0; bus < NV_NBUS; bus++) {
    if (!xf86I2CBusInit(pNv->TvBusses[bus])) {
      ErrorF ("Cannot init Bus %i\n", bus);
      ok = FALSE;
    }
  }
  return ok;
}

/* -------- Crt access -------- */ 

U008 readCrt (NVPtr pNv, int reg)
{
  VGA_WR08(pNv->riva.PCIO, 0x3d4, reg);
  return VGA_RD08(pNv->riva.PCIO, 0x3d5);
}

void writeCrt (NVPtr pNv, int reg, U008 val)
{
#ifdef FAKE_CRTC
  if (reg == 0x1f) {
#endif
  VGA_WR08(pNv->riva.PCIO, 0x3d4, reg);
  VGA_WR08(pNv->riva.PCIO, 0x3d5, val);
#ifdef FAKE_CRTC
  }
  FPRINTF ("%02x=%02x ", reg, val);
#endif
}

void orCrt (NVPtr pNv, int reg, U008 val)
{
#ifndef FAKE_CRTC
  register U008 tmp;

  VGA_WR08(pNv->riva.PCIO, 0x3d4, reg);
  tmp = VGA_RD08(pNv->riva.PCIO, 0x3d5);
  VGA_WR08(pNv->riva.PCIO, 0x3d5, tmp | val);
#else
  FPRINTF ("%02x|%02x ", reg, val);
#endif
}

void andCrt (NVPtr pNv, int reg, U008 val)
{
#ifndef FAKE_CRTC
  register U008 tmp;

  VGA_WR08(pNv->riva.PCIO, 0x3d4, reg);
  tmp = VGA_RD08(pNv->riva.PCIO, 0x3d5);
  VGA_WR08(pNv->riva.PCIO, 0x3d5, tmp & val);
#else
  FPRINTF ("%02x&%02x ", reg, val);
#endif
}

inline void unlockCrt (NVPtr pNv)
{
  writeCrt (pNv, 0x1f, 0x57); /* unlock extended registers */
}

inline void lockCrt (NVPtr pNv)
{
#ifndef FAKE_CRTC
#if 0 /**/
  writeCrt (pNv, 0x1f, 0x99); /* lock extended registers */
#endif
#endif
}

/* -------- I2C bus access -------- */

inline void NVWriteSeqBus (I2CDevPtr d, I2CByte subaddr, I2CByte *buf,
  int len)
{
#ifndef FAKE_I2C
  if (!xf86I2CWriteBytes (d, subaddr, buf, len)) nvBusOk = FALSE;
#else
  FPRINTF (" %02x=[%i]", subaddr, len);
#endif
}

inline void NVWriteBus (I2CDevPtr d, I2CByte subaddr, I2CByte data)
{
#ifndef FAKE_I2C
  if (!xf86I2CWriteByte (d, subaddr, data)) nvBusOk = FALSE;
#else
  FPRINTF (" %02x=%02x", subaddr, data);
#endif
}

inline void NVReadBus (I2CDevPtr d, I2CByte subaddr, I2CByte *data)
{
#ifndef FAKE_I2C
  if (!xf86I2CReadByte (d, subaddr, data)) nvBusOk = FALSE;
#else
  FPRINTF (" %02x?", subaddr);
  data = 0x00;
#endif
}

inline void NVStatusBus (I2CDevPtr d, I2CByte *data)
{
#ifndef FAKE_I2C
  if (!xf86I2CReadStatus(d, data)) nvBusOk = FALSE;
#else
  FPRINTF (" ss?");
  data = 0x00;
#endif
}

/* -------- Null chip -------- */

/* This is just to make sure nothing crashes if no chip can be detected. */

void NVInitNullRegs (I2CDevPtr dev)
{
  ErrorF ("Null chip: Init\n");
}

void NVSetNullRegs (I2CDevPtr dev, NVTvRegs *r)
{
  ErrorF ("Null chip: SetRegs\n");
}

void NVSetNullState (I2CDevPtr dev, NVTvState state)
{
  ErrorF ("Null chip: SetState\n");
}

NVConnect NVGetNullConnect (I2CDevPtr dev)
{
  ErrorF ("Null chip: Connect\n");
  return CONNECT_NONE;
}

long NVGetNullStatus (I2CDevPtr dev, int index)
{
  ErrorF ("Null chip: Status\n");
  return 0;
}

/* -------- Brooktree -------- */

/* I have no idea what registers 0xda-0xfc do, but mode 0 is supposed
 * to disable Macrovision. However, on my card Macrovision does not
 * seem to work anyway, so it improves just image quality :-) */

static I2CByte nvBtNtscMacro0 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x67,0xb5,0x90,0xb2,0x7d,0x00,0x00};
static I2CByte nvBtNtscMacro1 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x67,0xb5,0x90,0xb2,0x7d,0x63,0x00};
static I2CByte nvBtNtscMacro2 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x6c,0x31,0x92,0x32,0xdd,0xe3,0x00};
static I2CByte nvBtNtscMacro3 [] = {
  0x0f,0xfc,0x20,0xd0,0x6f,0x0f,0x00,0x00,0x0c,0xf3,0x09,
  0xbd,0x66,0xb5,0x90,0xb2,0x7d,0xe3,0x00};

static I2CByte nvBtPalMacro0 [] = {
  0x05,0x57,0x20,0x40,0x6e,0x7e,0xf4,0x51,0x0f,0xf1,0x05,
  0xd3,0x78,0xa2,0x25,0x54,0xa5,0x00,0x00};
static I2CByte nvBtPalMacro1 [] = {
  0x05,0x57,0x20,0x40,0x6e,0x7e,0xf4,0x51,0x0f,0xf1,0x05,
  0xd3,0x78,0xa2,0x25,0x54,0xa5,0x63,0x00};

void NVMacroBtMode (I2CDevPtr dev, NVTvRegs *r)
{
  DPRINTF ("nv macro bt %i\n", r->bt.macro); 
  switch (r->bt.macro) {
    case 0: /* NTSC APS 0 */
      NVWriteSeqBus (dev, 0xda, nvBtNtscMacro0, sizeof (nvBtNtscMacro0));
      break;
    case 1: /* NTSC APS 1 */
      NVWriteSeqBus (dev, 0xda, nvBtNtscMacro1, sizeof (nvBtNtscMacro1));
      break;
    case 2: /* NTSC APS 2 */
      NVWriteSeqBus (dev, 0xda, nvBtNtscMacro2, sizeof (nvBtNtscMacro2));
      break;
    case 3: /* NTSC APS 3 */
      NVWriteSeqBus (dev, 0xda, nvBtNtscMacro3, sizeof (nvBtNtscMacro3));
      break;
    case 4: /* PAL APS 0 */
      NVWriteSeqBus (dev, 0xda, nvBtPalMacro0, sizeof (nvBtPalMacro0));
      break;
    case 5: /* PAL APS 1 */
    case 6:
    case 7:
      NVWriteSeqBus (dev, 0xda, nvBtPalMacro1, sizeof (nvBtPalMacro1));
      break;
  }
}

/* All DACs off, clock output disabled. */

void NVInitBtRegs (I2CDevPtr dev)
{
  DPRINTF ("nv init bt\n");
  NVWriteBus (dev, 0xb8, 0x03);
  NVWriteBus (dev, 0xba, 0x17);
  NVWriteBus (dev, 0xc4, 0x00);
  NVWriteBus (dev, 0xc4, 0x01);
  NVWriteBus (dev, 0xc6, 0x98); /* En_Blank0, Sync Polarity */
  NVWriteBus (dev, 0xce, 0x18); 
  NVWriteBus (dev, 0xd4, 0x00); /* Mode2x=0 */
  NVWriteBus (dev, 0xd6, 0x00); /* Out_Mode */
}

void NVSetBtRegs (I2CDevPtr dev, NVTvRegs *r)
{
  DPRINTF ("nv set bt\n");
  NVWriteBus (dev, 0x6e, Set8Bits(r->bt.hsynoffset));
  NVWriteBus (dev, 0x70, SetBitField(r->bt.hsynoffset,9:8,7:6)
                       | SetBitField(r->bt.hsynwidth,5:0,5:0));
  NVWriteBus (dev, 0x72, Set8Bits(r->bt.vsynoffset));
  NVWriteBus (dev, 0x74, SetBitField(r->bt.vsynoffset,10:8,5:3)
                       | SetBitField(r->bt.vsynwidth,2:0,2:0));
  NVWriteBus (dev, 0x76, Set8Bits(r->bt.h_clko));
  NVWriteBus (dev, 0x78, Set8Bits(r->bt.h_active));
  NVWriteBus (dev, 0x7a, Set8Bits(r->bt.hsync_width));
  NVWriteBus (dev, 0x7c, Set8Bits(r->bt.hburst_begin));
  NVWriteBus (dev, 0x7e, Set8Bits(r->bt.hburst_end));
  NVWriteBus (dev, 0x80, Set8Bits(r->bt.h_blanko));
  NVWriteBus (dev, 0x82, Set8Bits(r->bt.v_blanko));
  NVWriteBus (dev, 0x84, Set8Bits(r->bt.v_activeo));
  NVWriteBus (dev, 0x86, SetBitField(r->bt.v_activeo,8:8,7:7)
                       | SetBitField(r->bt.h_active,9:8,5:4)
                       | SetBitField(r->bt.h_clko,11:8,2:0));
  NVWriteBus (dev, 0x88, Set8Bits(r->bt.h_fract));
  NVWriteBus (dev, 0x8a, Set8Bits(r->bt.h_clki));
  NVWriteBus (dev, 0x8c, Set8Bits(r->bt.h_blanki));
  NVWriteBus (dev, 0x8e, SetBitField(r->bt.h_blanki,8:8,3:3)
                       | SetBitField(r->bt.h_clki,10:8,2:0));
  NVWriteBus (dev, 0x90, Set8Bits(r->bt.v_linesi));
  NVWriteBus (dev, 0x92, Set8Bits(r->bt.v_blanki));
  NVWriteBus (dev, 0x94, Set8Bits(r->bt.v_activei));
  NVWriteBus (dev, 0x96, SetBitField(r->bt.clpf,1:0,7:6)
                       | SetBitField(r->bt.ylpf,1:0,5:4)
                       | SetBitField(r->bt.v_activei,9:8,3:2)
                       | SetBitField(r->bt.v_linesi,9:8,1:0));
  NVWriteBus (dev, 0x98, Set8Bits(r->bt.v_scale));
  NVWriteBus (dev, 0x9a, SetBitField(r->bt.h_blanko,9:8,7:6)
                       | SetBitField(r->bt.v_scale,13:8,5:0));
  NVWriteBus (dev, 0x9c, Set8Bits(r->bt.pll_fract));
  NVWriteBus (dev, 0x9e, Set8Bits(r->bt.pll_fract >> 8));
  NVWriteBus (dev, 0xa0, 0 /* en_xclk, by_pll */ 
                       | SetBitField(r->bt.pll_int,5:0,5:0));
  NVWriteBus (dev, 0xa2, SetBitField(r->bt.flags1,5:0,5:0));
  NVWriteBus (dev, 0xa4, Set8Bits(r->bt.sync_amp));
  NVWriteBus (dev, 0xa6, Set8Bits(r->bt.bst_amp));
  NVWriteBus (dev, 0xa8, Set8Bits(r->bt.mcr));
  NVWriteBus (dev, 0xaa, Set8Bits(r->bt.mcb));
  NVWriteBus (dev, 0xac, Set8Bits(r->bt.my));
  NVWriteBus (dev, 0xae, Set8Bits(r->bt.msc));
  NVWriteBus (dev, 0xb0, Set8Bits(r->bt.msc >> 8));
  NVWriteBus (dev, 0xb2, Set8Bits(r->bt.msc >> 16));
  NVWriteBus (dev, 0xb4, Set8Bits(r->bt.msc >> 24));
  NVWriteBus (dev, 0xba, ((nvTvState == NV_TV_OFF) ? 0x17 : 0x00)
	               | (r->bt.flags4 & BT_FLAG4_DAC));
  NVWriteBus (dev, 0xc8, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_YFLPF,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_FFILT,6)
		       | SetBitField(r->bt.f_selc,2:0,5:3)
                       | SetBitField(r->bt.f_sely,2:0,2:0));
  NVWriteBus (dev, 0xca, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMUSHY,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMSHY,6)
		       | SetBitField(r->bt.ycoring,2:0,5:3)
                       | SetBitField(r->bt.yattenuate,2:0,2:0));
  NVWriteBus (dev, 0xcc, SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMUSHC,7)
		       | SetBitFlag(r->bt.flags2,BT_FLAG2_DIS_GMSHC,6)
		       | SetBitField(r->bt.ccoring,2:0,5:3)
                       | SetBitField(r->bt.cattenuate,2:0,2:0));
  NVWriteBus (dev, 0xce, SetBitField(r->bt.out_muxa,1:0,1:0)
		       | SetBitField(r->bt.out_muxb,1:0,3:2)
                       | SetBitField(r->bt.out_muxc,1:0,5:4));
  NVMacroBtMode (dev, r);
}

void NVSetBtState (I2CDevPtr dev, NVTvState state)
{
  DPRINTF ("nv state bt\n");
  nvTvState = state;
  switch (state)
  {
    case NV_TV_OFF: 
      NVWriteBus (dev, 0xba, 0x17);
      NVWriteBus (dev, 0xc4, 0x00); 
      break;
    case NV_TV_BARS: 
      NVWriteBus (dev, 0xba, 0x00); 
      NVWriteBus (dev, 0xc4, 0x05); 
      break;
    case NV_TV_ON: 
      NVWriteBus (dev, 0xba, 0x00); 
      NVWriteBus (dev, 0xc4, 0x01); 
      break;
    default:
      break;
  }
}

int NVReadBt (I2CDevPtr dev, int index)
{
  I2CByte status;

  nvBusOk = TRUE;
  switch (nvTvState)
  {
    case NV_TV_OFF: 
      NVWriteBus (dev, 0xc4, 0x00 | SetBitField(index,1:0,7:6));
      break;
    case NV_TV_BARS: 
      NVWriteBus (dev, 0xc4, 0x05 | SetBitField(index,1:0,7:6)); 
      break;
    case NV_TV_ON: 
      NVWriteBus (dev, 0xc4, 0x01 | SetBitField(index,1:0,7:6)); 
      break;
    default:
      return -1;
  }
  NVStatusBus (dev, &status);
  if (!nvBusOk) 
    return -1;
  else 
    return status;
}

long NVGetBtStatus (I2CDevPtr dev, int index)
{ 
  return NVReadBt (dev, 2) | ((nvTvState == NV_TV_ON) ? 0x100 : 0);
}

NVConnect NVGetBtConnect (I2CDevPtr dev)
{
  int status;

  /* Enable all DACS and set check_stat */
  NVWriteBus (dev, 0xba, 0x40); 
  xf86usleep(50000); /* BT871 doc says: 20ms - 50ms */
  status = NVReadBt (dev, 1);
  DPRINTF ("bt conn %02x (%s)\n", status, nvBusOk ? "ok" : "err");
  /* BT871 doc says we should reset check_stat */
  NVWriteBus (dev, 0xba, 0x00); 
  /* FIXME. (But next SetTVState will write correct values.) */
  if (status >= 0) {
    switch (status & 0xe0) {
      case 0x80 : return CONNECT_FBAS; break;
      case 0x60 : return CONNECT_SVHS; break;
      case 0x20 : return CONNECT_CONVERT; break;
      default: 
	ErrorF("Strange Brooktree connection status %02x\n", status & 0xe0); 
      case 0xe0 : return CONNECT_BOTH; break;
    }
  } else {
    return CONNECT_NONE;
  }
}

/*
 * Check for Brooktree chip on device dev. Return version string if found, 
 * NULL otherwise. Don't use ReadBt.
 */

char *NVCheckBrooktree (I2CDevPtr dev, NVPtr pNv)
{
  I2CByte status;
  char *chip;
  static char version [50];

  nvBusOk = TRUE;
  /* crt should already be unlocked */
  if (readCrt (pNv, 0x28) & 0x80) {
    NVWriteBus (dev, 0xc4, 0x01);
  } else {
    NVWriteBus (dev, 0xc4, 0x00);
  }
  NVStatusBus (dev, &status);
  DPRINTF ("bt check %02x (%s)\n", status, nvBusOk ? "ok" : "err");
  if (!nvBusOk) return NULL;
  switch (status & 0xe0) {
    case 0x00 : chip = "868"; break;
    case 0x20 : chip = "869"; break;
    case 0x40 : chip = "870"; break;
    case 0x60 : chip = "871"; break;
    default : chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Brooktree BT%s Rev %i (%1s:%02X)", 
	      chip, status & 0x1f, I2C_ID(dev));
    return version;
  } else {
    return NULL;
  }
}

/*
 * Check for Conexant chip on device dev. Return version string if found, 
 * NULL otherwise. Make sure it cannot be a Philips chip, otherwise
 * register 6c (HTRIG) might be changed to an illegal value.
 * FIXME: Should use normal register readback for Conexant.
 */

char *NVCheckConexant (I2CDevPtr dev)
{
  I2CByte result;
  char *chip;
  static char version [50];

  nvBusOk = TRUE;  
  NVWriteBus (dev, 0x6c, 0x46); /* enable reg rd */
  DPRINTF ("cx check (%s)\n", nvBusOk ? "ok" : "err");
  NVReadBus (dev, 0x00, &result);
  NVWriteBus (dev, 0x6c, 0x06); /* disable reg rd, for bt compatibility */
  DPRINTF ("cx check %02x (%s)\n", result, nvBusOk ? "ok" : "err");
  if (!nvBusOk) return NULL;
  switch (result & 0xe0) {
    case 0x40 : chip = "25870"; break;
    case 0x60 : chip = "25871"; break;
    default : chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Conexant CX%s Rev %i (%1s:%02X)", 
	      chip, result & 0x1f, I2C_ID(dev));
    return version;
  } else {
    return NULL;
  }
}

/* -------- Chrontel -------- */

static I2CByte nvChDefMacro [0x40] = {
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};

static I2CByte nvChPalMacro00APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x2E,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x40,0x10,0x9F,0x9F,0xBA,0x9B,
  0x64,0x0A,0xEE,0x26,0x2A,0x7E,0x9A,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro08APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x37,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4C,0x30,0xBD,0xBD,0xDE,0x9B,
  0x64,0x3C,0x1B,0x2E,0x32,0x96,0xB7,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro14APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x3A,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x50,0x30,0xC8,0xC8,0xEA,0x9B,
  0x64,0x4C,0x29,0x30,0x34,0x9D,0xC0,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro20APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x47,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x61,0x38,0xF3,0xF3,0x1C,0x9B,
  0x64,0x94,0x6A,0x3B,0x40,0xC0,0xEA,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro03APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x00,0x00,0x00,0x00,0x00,
  0x00,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro03APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x01,0x98,0x98,0xBB,0x9E,
  0x2D,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro03APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x01,0x98,0x98,0xBB,0x98,
  0x29,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro11APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF0,0x00,0x00,0x00,0x00,
  0x00,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro11APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF1,0x67,0x67,0xE2,0x9E,
  0x2D,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro11APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF1,0x67,0x67,0xE2,0x98,
  0x29,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro17APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x00,0x00,0x00,0x00,0x00,
  0x00,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro17APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x01,0xAE,0xAE,0xD6,0x9E,
  0x2D,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro17APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x01,0xAE,0xAE,0xD6,0x98,
  0x29,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro24APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF0,0x00,0x00,0x00,0x00,
  0x00,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro24APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF9,0xDD,0xDD,0x11,0x9E,
  0x2D,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro24APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF9,0xDD,0xDD,0x11,0x98,
  0x29,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

/* 00: ---, 7:6, 7:0, ---, 7+4, 7:0, ---, ---, */
/* 08: 7:3, ---, ---, ---, 7:0, 7:4, 7:5, 7:0, */
/* 10: 7:4, 7:3, 7:0, 7:5, ---, ---, 7:0, 7:6, */ 
/* 18: 7:4, 7:4, 7:4, ---, ---, 7:4, 7:4, 7:4, */
/* 20: 7:6, 7:5, ---, ---, ---, ---, ---, 7:6  */ 
/* 28: ---, ---, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0  */ 
/* 30: 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0  */ 
/* 38: 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, ---  */ 

#if 0 /* not needed yet; ignore to prevent warning */
static I2CByte nvChMaskMacro [0x40] = {
  0xFF,0x3F,0x00,0xFF,0x7F,0x00,0x7F,0xFF,
  0x07,0xFF,0xFF,0xFF,0x00,0x0F,0x3F,0xC0,
  0x0F,0x07,0x00,0x1F,0xFF,0xFF,0x00,0xFF,
  0x0F,0x0F,0x0F,0xF0,0xF0,0xF0,0xF0,0xF0,
  0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
  0xFF,0xFF,0x80,0xFF,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x80,0x80,0x00,0x00,0x80,
  0x00,0x80,0x00,0x00,0x00,0xF8,0xFF,0xFF
};
#endif

typedef struct {
  int mode;
  I2CByte *regs[4];
} NVChMacroTable;

static NVChMacroTable nvChMacroTable [] = {
  { 0, {nvChDefMacro, 
	nvChPalMacro00APS1, nvChPalMacro00APS1, nvChPalMacro00APS1}},
  { 3, {nvChDefMacro, 
	nvChNtscMacro03APS1, nvChNtscMacro03APS2, nvChNtscMacro03APS3}},
  { 8, {nvChDefMacro, 
	nvChPalMacro08APS1, nvChPalMacro08APS1, nvChPalMacro08APS1}},
  {11, {nvChDefMacro, 
	nvChNtscMacro11APS1, nvChNtscMacro11APS2, nvChNtscMacro11APS3}},
  {14, {nvChDefMacro, 
	nvChPalMacro14APS1, nvChPalMacro14APS1, nvChPalMacro14APS1}},
  {17, {nvChDefMacro, 
	nvChNtscMacro17APS1, nvChNtscMacro17APS2, nvChNtscMacro17APS3}},
  {20, {nvChDefMacro, 
	nvChPalMacro20APS1, nvChPalMacro20APS1, nvChPalMacro20APS1}},
  {24, {nvChDefMacro, 
	nvChNtscMacro24APS1, nvChNtscMacro24APS2, nvChNtscMacro24APS3}},
  {-1, {nvChDefMacro, nvChDefMacro, nvChDefMacro, nvChDefMacro}}
};

I2CByte *NVMacroChRegs (NVTvRegs *r)
{
  NVChMacroTable *t;

  if (!r || r->ch.macro < 0 || r->ch.macro > 3) 
    return nvChDefMacro;
  for (t = nvChMacroTable; t->mode != -1; t++) {
    if (r->ch.mode == t->mode) {
      return t->regs [r->ch.macro];
    }
  }
  return nvChDefMacro;
}

/* Init chrontel register values (all done in SetChRegs)
 *
 * cfrb  = 0  (free running clock in master mode)
 * m/s*  = 1  (master mode)
 * mcp   = 0  (latch on negative edge)
 * xcm   = 00 (XCLK 1x) 
 * pcm   = 00 (P-Out 1x)
 * aciv  = 0  (auto civ off, since it has bad quality)
 * des   = 0  (no embedded sync)
 * syo   = 1  (output horizontal and vertical sync)
 * vsp   = 1  (vertical sync active high)
 * hsp   = 1  (horizontal sync active high)
 * dvdd2 = 1  (dvdd2 at 3.3V)
 * poutp = 1  (P-OUT polarity)
 * dsen  = 0  (Use hsync to determine start of active video) 
 * dsm   = 0  (should be 0 if dsen=0)
 */

void NVInitChRegs (I2CDevPtr dev)
{
  /* To avoid overclocking (just in case, shouldn't happen anyway).
     Will be set in NVSetChRegs as well */
  NVWriteBus (dev, 0x80|0x06, 0x40); /* master mode clock */
}

void NVSetChRegs (I2CDevPtr dev, NVTvRegs *r)
{
  I2CByte *m;
  int i;

  DPRINTF ("nv regs ch\n");
#ifdef FAKE_I2C
  FPRINTF ("\ni2c [");
#endif
  m = NVMacroChRegs (r); /* FIXME TODO */
  NVWriteBus (dev, 0x80|0x00, SetBitField(r->ch.dmr_ir,2:0,7:5)
		            | SetBitField(r->ch.dmr_vs,1:0,4:3)
		            | SetBitField(r->ch.dmr_sr,2:0,2:0));
  NVWriteBus (dev, 0x80|0x01, m[0x01] 
	                    | SetBitField(r->ch.ffr_fc,1:0,5:4)
		            | SetBitField(r->ch.ffr_fy,1:0,3:2)
 	                    | SetBitField(r->ch.ffr_ft,1:0,1:0));
  NVWriteBus (dev, 0x80|0x02, m[0x02]);
  NVWriteBus (dev, 0x80|0x03, m[0x03] 
	                    | SetBitField(r->ch.vbw_flff, 0:0,7:7)
	                    | SetBitField(r->ch.vbw_cvbw, 0:0,6:6)
	                    | SetBitField(r->ch.vbw_cbw,  1:0,5:4)
	                    | SetBitField(r->ch.vbw_ypeak,0:0,3:3)
	                    | SetBitField(r->ch.vbw_ysv,  1:0,2:1)
			    | SetBitField(r->ch.vbw_ycv,  0:0,0:0));
  NVWriteBus (dev, 0x80|0x04, m[0x04] 
	                    | SetBitField(r->ch.dacg,0:0,6:6)
                            | 0x05 /* idf=5 */ );
  NVWriteBus (dev, 0x80|0x05, m[0x05]);
  NVWriteBus (dev, 0x80|0x06, m[0x06] | 0x40); /* clock, s.a. */
  NVWriteBus (dev, 0x80|0x07, Set8Bits(r->ch.sav));
  NVWriteBus (dev, 0x80|0x08, m[0x08] 
	                    | SetBitField(r->ch.sav,8:8,2:2)
		            | SetBitField(r->ch.hpr,8:8,1:1)
		            | SetBitField(r->ch.vpr,8:8,0:0));
  NVWriteBus (dev, 0x80|0x09, Set8Bits(r->ch.blr));
  NVWriteBus (dev, 0x80|0x0a, Set8Bits(r->ch.hpr));
  NVWriteBus (dev, 0x80|0x0b, Set8Bits(r->ch.vpr));
  NVWriteBus (dev, 0x80|0x0c, m[0x0c]);
  NVWriteBus (dev, 0x80|0x0d, 0x07 | m[0x0d]); /* sync, s.a. */
  NVWriteBus (dev, 0x80|0x0e, m[0x0e] | 0x08 |
   ((nvTvState != NV_TV_ON) ? 0x05 
			    : SetBitField(r->ch.flags,1:0,CH_FLAG_DAC)));
  NVWriteBus (dev, 0x80|0x0f, m[0x0f]);
  NVWriteBus (dev, 0x80|0x10, 0x00 | m[0x10]); /* sense off */
  NVWriteBus (dev, 0x80|0x11, m[0x011] 
	                    | SetBitField(r->ch.ce,2:0,2:0));
  NVWriteBus (dev, 0x80|0x12, m[0x12]);
  NVWriteBus (dev, 0x80|0x13, m[0x13] 
	                    | SetBitField(r->ch.pll_n,9:8,2:1)
		            | SetBitField(r->ch.pll_m,8:8,0:0));
  NVWriteBus (dev, 0x80|0x14, Set8Bits(r->ch.pll_m));
  NVWriteBus (dev, 0x80|0x15, Set8Bits(r->ch.pll_n));
  NVWriteBus (dev, 0x80|0x16, m[0x16]);
  NVWriteBus (dev, 0x80|0x17, 0x00 | m[0x17]); /* BCO= 14MHz crystal */
  NVWriteBus (dev, 0x80|0x18, m[0x18] 
	                    | SetBitField(r->ch.fsci,31:28,3:0));
  NVWriteBus (dev, 0x80|0x19, m[0x19] 
	                    | SetBitField(r->ch.fsci,27:24,3:0));
  NVWriteBus (dev, 0x80|0x1a, m[0x1a] 
	                    | SetBitField(r->ch.fsci,23:20,3:0));
  NVWriteBus (dev, 0x80|0x1b, SetBitField(r->ch.fsci,19:16,3:0)
		            | 0x30);
  NVWriteBus (dev, 0x80|0x1c, SetBitField(r->ch.fsci,15:12,3:0)
		            | 0xc0);
  NVWriteBus (dev, 0x80|0x1d, m[0x1d] 
	                    | SetBitField(r->ch.fsci,11:8,3:0));
  NVWriteBus (dev, 0x80|0x1e, m[0x1e] 
	                    | SetBitField(r->ch.fsci,7:4,3:0));
  NVWriteBus (dev, 0x80|0x1f, m[0x1f] 
	                    | SetBitField(r->ch.fsci,3:0,3:0));
  NVWriteBus (dev, 0x80|0x20, m[0x20] 
	                    | SetBitField(r->ch.pllcap,0:0,4:4)
		            | 0x0a);
  NVWriteBus (dev, 0x80|0x21, m[0x21]
	                    | SetBitField(r->ch.civh,1:0,2:1)
	                    | SetBitField(r->ch.aciv,0:0,0:0));
  for (i = 0x2a; i <= 0x3d; i++) {
    NVWriteBus (dev, 0x80|i, m[i]);
  }
#if 0 /* Register 0x26-0x29 reserved for test */
  /* mtd, ms, rsa, bst, nst, de, ts */
  NVWriteBus (dev, 0x80|0x27, m[0x27] 
	                    | 0x00 /* FIXME */
	                    | SetBitField(r->ch.ylm,8:8,1:1)
		            | SetBitField(r->ch.clm,8:8,0:0));
  NVWriteBus (dev, 0x80|0x28, Set8Bits(r->ch.ylm));
  NVWriteBus (dev, 0x80|0x29, Set8Bits(r->ch.clm));
#endif
#ifdef FAKE_I2C
  FPRINTF ("]\n");
#endif
}
 
void NVSetChState (I2CDevPtr dev, NVTvState state)
{
  DPRINTF ("nv state ch\n");
  nvTvState = state;
#ifndef FAKE_I2C
  switch (state)
  {
    case NV_TV_OFF: 
    case NV_TV_BARS: 
      NVWriteBus (dev, 0x80|0x0e, 0x0d);
      break;
    case NV_TV_ON: 
      NVWriteBus (dev, 0x80|0x0e, 0x0b); 
      /* FIXME (But following NVSetRegs will set connector & mavrovis) */
      break;
    default:
      break;
  }
#endif
}

NVConnect NVGetChConnect (I2CDevPtr dev)
{
  I2CByte power, connect;

  NVReadBus  (dev, 0x80|0x0e, &power);
  NVWriteBus (dev, 0x80|0x0e, 0x0b); 
  NVWriteBus (dev, 0x80|0x10, 0x01); 
  xf86usleep (10);  /* tested */
  NVWriteBus (dev, 0x80|0x10, 0x00); 
  xf86usleep (10);  /* tested */
  NVReadBus  (dev, 0x80|0x10, &connect);
  NVWriteBus (dev, 0x80|0x0e, power); 
  if (nvBusOk) {
    switch (connect & 0x0e) {
      case 0x0c: return CONNECT_FBAS; break;
      case 0x02: return CONNECT_SVHS; break;
      case 0x0a: return CONNECT_CONVERT; break;
      default: 
	ErrorF("Strange Chrontel connection status %02x\n", connect & 0x0e); 
      case 0x00: return CONNECT_BOTH; break;
    }
  } else {
    return CONNECT_NONE;
  }
}

long NVGetChStatus (I2CDevPtr dev, int index)
{ 
  long civ;

  I2CByte result;

  nvBusOk = TRUE;  
  NVReadBus (dev, 0x80|0x21, &result);
  civ = SetBitField (result,4:3,0:1);
  NVReadBus (dev, 0x80|0x22, &result);
  civ = (civ << 8) | result;
  NVReadBus (dev, 0x80|0x23, &result);
  civ = (civ << 8) | result;
  NVReadBus (dev, 0x80|0x24, &result);
  civ = (civ << 8) | result;
  if (!nvBusOk) 
    return -1;
  else 
    return civ;
}

/*
 * Check for Chrontel chip on device dev. Return version string if found, 
 * NULL otherwise.
 */

char *NVCheckChrontel (I2CDevPtr dev)
{
  I2CByte result;
  char *chip;
  static char version [50];

  nvBusOk = TRUE;  
  NVReadBus (dev, 0x80|0x25, &result);
  DPRINTF ("ch version %02x (%s)\n", result, nvBusOk ? "ok" : "err");
  if (!nvBusOk) return NULL;
  switch (result)
  {
    case 0x32: chip = "7004C"; break; /* 00110010 */
    case 0x3a: chip = "7005C"; break; /* 00111010 */
    case 0x2a: chip = "7006C"; break; /* 00101010 */
    case 0x50: chip = "7007A"; break; /* 01010000 */
    case 0x40: chip = "7008A"; break; /* 01000000 */
    default:   chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Chrontel %s (%1s:%02X)", chip, I2C_ID(dev));
  } else {
    snprintf (version, 50, "Chrontel (id = %02X) (%1s:%02X)", 
	      result, I2C_ID(dev));
  }
  return version;
}

/* -------- Philips -------- */

/*
 * Check for Philips chip on device dev. Return version string if found, 
 * NULL otherwise.
 */

char *NVCheckPhilips (I2CDevPtr dev)
{
  I2CByte cid, ver;
  char *chip;
  static char version [50];

  nvBusOk = TRUE;  
  /* Neither Brooktree nor Conexant have register 0x1C */
  NVReadBus (dev, 0x1c, &cid); 
  DPRINTF ("ph check %02x (%s)\n", cid, nvBusOk ? "ok" : "err");
  if (!nvBusOk) return NULL;
  /* FIXME TODO: SAA7108, SAA7108B, SAA7109 */
  switch (cid) {
    case 0x02: chip = "7102"; break;
    case 0x03: chip = "7103"; break;
    default:   chip = NULL; break;
  }
  if (chip) {
    NVReadBus (dev, 0x00, &ver); 
    if (!nvBusOk) return NULL;
    snprintf (version, 50, "Philips SAA%s Rev %i (%1s:%02X)", 
	      chip, ver >> 5, I2C_ID(dev));
    return version;
  } else {
    snprintf (version, 50, "Philips (id = %02X) (%1s:%02X)", 
	      cid, I2C_ID(dev));
    return version;
  }
}

/* -------- CRT -------- */

/* The crt register settings are verified to be the same as the NVidia 
   driver settings, at least for the GeForce2 MX. This includes +1 instead
   of +0 (as in the X 'nv' driver) for horizSync. I do not know if all this 
   works for older/newer cards, since the bitranges might be different.  */

void NVSetCrtRegs (NVPtr pNv, DisplayModePtr mode)
{
  int horizDisplay    = (mode->CrtcHDisplay/8)    - 1;
  int horizBlankStart = (mode->CrtcHBlankStart/8) - 1;
  int horizSyncStart  = (mode->CrtcHSyncStart/8)  + 1; /* verified */
  int horizSyncEnd    = (mode->CrtcHSyncEnd/8)    + 1; /* verified */
  int horizBlankEnd   = (mode->CrtcHBlankEnd/8)   - 1;
  int horizTotal      = (mode->CrtcHTotal/8)      - 5;
  int vertDisplay     =  mode->CrtcVDisplay       - 1;
  int vertBlankStart  =  mode->CrtcVBlankStart    - 1;
  int vertSyncStart   =  mode->CrtcVSyncStart     - 1;
  int vertSyncEnd     =  mode->CrtcVSyncEnd       - 1;
  int vertBlankEnd    =  mode->CrtcVBlankEnd      - 1;
  int vertTotal       =  mode->CrtcVTotal         - 2;

  unlockCrt (pNv);
#ifdef FAKE_CRTC
  FPRINTF ("\ncrt[");
#endif
  writeCrt (pNv, 0x00, Set8Bits(horizTotal));
  writeCrt (pNv, 0x01, Set8Bits(horizDisplay));
  writeCrt (pNv, 0x02, Set8Bits(horizBlankStart));
  writeCrt (pNv, 0x03, SetBitField(horizBlankEnd,4:0,4:0) 
                     | SetBit(7)); /* 6:5 = disp_end_skew ... */
  writeCrt (pNv, 0x04, Set8Bits(horizSyncStart));
  writeCrt (pNv, 0x05, SetBitField(horizBlankEnd,5:5,7:7)
	             | SetBitField(horizSyncEnd,4:0,4:0));
  writeCrt (pNv, 0x06, Set8Bits(vertTotal));
  writeCrt (pNv, 0x07, SetBitField(vertTotal,8:8,0:0)
		     | SetBitField(vertDisplay,8:8,1:1)
		     | SetBitField(vertSyncStart,8:8,2:2)
		     | SetBitField(vertDisplay,8:8,3:3)
		     | SetBit(4) /* line_compare_8 */
		     | SetBitField(vertTotal,9:9,5:5)
		     | SetBitField(vertDisplay,9:9,6:6)
		     | SetBitField(vertSyncStart,9:9,7:7));
#if 0
  writeCrt (pNv, 0x08, 0x00); 
#endif
  writeCrt (pNv, 0x09, SetBitField(vertBlankStart,9:9,5:5)
		     | SetBit(6)
	             | ((mode->Flags & V_DBLSCAN) ? 0x80 : 0x00));
  writeCrt (pNv, 0x10, Set8Bits(vertSyncStart));
  writeCrt (pNv, 0x11, SetBitField(vertSyncEnd,3:0,3:0)
		     | SetBit(5));
  /* !!! SetBit(5) always by driver!! */
  writeCrt (pNv, 0x12, Set8Bits(vertDisplay));
  /* crt offset (= crt line increment)? in 0x13,8bit 0x19,10:8,5:7 */
  /* writeCrt(pNv, 0x13, 0xc0);
        ((pLayout->displayWidth/8) * (pLayout->bitsPerPixel/8))); */
  writeCrt (pNv, 0x15, Set8Bits(vertBlankStart));
  writeCrt (pNv, 0x16, Set8Bits(vertBlankEnd));
  /* FIXME TODO: Arbitration 0 (0x1b), 1 (0x20) */
  writeCrt (pNv, 0x25, SetBitField(vertTotal,10:10,0:0)
		     | SetBitField(vertDisplay,10:10,1:1)
		     | SetBitField(vertSyncStart,10:10,2:2)
		     | SetBitField(vertBlankStart,10:10,3:3)
		     | SetBitField(horizBlankEnd,6:6,4:4)
		     | (readCrt (pNv, 0x25) & 0x10)); /* bit 5: offset_11 */
  writeCrt (pNv, 0x2D, SetBitField(horizTotal,8:8,0:0)
		     | SetBitField(horizDisplay,8:8,1:1)
		     | SetBitField(horizBlankStart,8:8,2:2)
		     | SetBitField(horizSyncStart,8:8,3:3)
		     | (readCrt (pNv, 0x2D) & 0x18)); /* bit 5-4 */
  writeCrt (pNv, 0x33, 0x00);   /* bit 7 : Interlace off */
#ifdef FAKE_CRTC
  FPRINTF ("]\n");
#endif
  lockCrt (pNv);
}

void NVGetCrtRegs (NVPtr pNv, DisplayModePtr mode)
{
  int horizDisplay;
  int horizBlankStart;
  int horizSyncStart;
  int horizSyncEnd;
  int horizBlankEnd;
  int horizTotal;
  int vertDisplay;
  int vertBlankStart;
  int vertSyncStart;
  int vertSyncEnd;
  int vertBlankEnd;
  int vertTotal;

  unlockCrt (pNv);
  horizTotal      = Set8Bits(readCrt(pNv, 0x00)) 
                  | SetBitField(readCrt(pNv, 0x2d),0:0,8:8);
  horizDisplay    = Set8Bits(readCrt(pNv, 0x01)) 
                  | SetBitField(readCrt(pNv, 0x2d),1:1,8:8);
  horizBlankStart = Set8Bits(readCrt(pNv, 0x02))
                  | SetBitField(readCrt(pNv, 0x2d),2:2,8:8);
  horizBlankEnd   = SetBitField(readCrt(pNv, 0x03),4:0,4:0)
                  | SetBitField(readCrt(pNv, 0x05),7:7,5:5)
                  | SetBitField(readCrt(pNv, 0x25),4:4,6:6);
  horizSyncStart  = Set8Bits(readCrt(pNv, 0x04)) 
                  | SetBitField(readCrt(pNv, 0x2d),3:3,8:8);
  horizSyncEnd    = SetBitField(readCrt(pNv, 0x05),4:0,4:0);

  vertTotal       = Set8Bits(readCrt(pNv, 0x06)) 
                  | SetBitField(readCrt(pNv, 0x07),0:0,8:8)
                  | SetBitField(readCrt(pNv, 0x07),5:5,9:9)
                  | SetBitField(readCrt(pNv, 0x25),0:0,10:10);
  vertDisplay     = Set8Bits(readCrt(pNv, 0x12)) 
	          | SetBitField(readCrt(pNv, 0x07),3:3,8:8)
	          | SetBitField(readCrt(pNv, 0x07),6:6,9:9)
	          | SetBitField(readCrt(pNv, 0x25),1:1,10:10);
  vertBlankStart  = Set8Bits(readCrt(pNv, 0x15))
	          | SetBitField(readCrt(pNv, 0x07),3:3,8:8)
	          | SetBitField(readCrt(pNv, 0x09),5:5,9:9)
	          | SetBitField(readCrt(pNv, 0x25),3:3,10:10);
  vertBlankEnd    = Set8Bits(readCrt(pNv, 0x16));
  vertSyncStart   = Set8Bits(readCrt(pNv, 0x10))
	          | SetBitField(readCrt(pNv, 0x07),2:2,8:8)
	          | SetBitField(readCrt(pNv, 0x07),7:7,9:9)
	          | SetBitField(readCrt(pNv, 0x25),2:2,10:10);
  vertSyncEnd     = SetBitField(readCrt(pNv, 0x11),3:0,3:0);

  /* It is not clear if blank end is relative to blank start or
     total. I have chosen total here, since blank end is equal to
     total normally, anyway. */

  horizBlankEnd  |= SetBitField (horizTotal,8:7,8:7);
  while (horizBlankEnd < horizTotal) horizBlankEnd += SetBit (7);

  horizSyncEnd  |= SetBitField (horizSyncStart,8:5,8:5);
  while (horizSyncEnd < horizSyncStart) horizSyncEnd += SetBit (5);

  vertBlankEnd  |= SetBitField (vertTotal,10:8,10:8);
  while (vertBlankEnd < vertTotal) vertBlankEnd += SetBit (8);

  vertSyncEnd  |= SetBitField (vertSyncStart,10:4,10:4);
  while (vertSyncEnd < vertSyncStart) vertSyncEnd += SetBit (4);

  mode->CrtcHDisplay    = mode->HDisplay    = (horizDisplay    + 1) * 8;   
  mode->CrtcHBlankStart =		      (horizBlankStart + 1) * 8;
  mode->CrtcHBlankEnd   =		      (horizBlankEnd   + 1) * 8;  
  mode->CrtcHSyncStart	= mode->HSyncStart  = (horizSyncStart  - 1) * 8; 
  mode->CrtcHSyncEnd  	= mode->HSyncEnd    = (horizSyncEnd    - 1) * 8;   
  mode->CrtcHTotal    	= mode->HTotal      = (horizTotal      + 5) * 8;     

  mode->CrtcVDisplay    = mode->VDisplay    = vertDisplay    + 1;
  mode->CrtcVBlankStart =		      vertBlankStart + 1;
  mode->CrtcVBlankEnd   =		      vertBlankEnd   + 1;
  mode->CrtcVSyncStart  = mode->VSyncStart  = vertSyncStart  + 1;
  mode->CrtcVSyncEnd    = mode->VSyncEnd    = vertSyncEnd    + 1;
  mode->CrtcVTotal      = mode->VTotal      = vertTotal      + 2;

  mode->Clock = 0;  
  mode->Flags = 0;
  mode->PrivSize = 0;
  mode->Private = NULL;
  mode->PrivFlags = 0;
  mode->HSkew = mode->VScan = 0;  

  if (readCrt (pNv, 0x28) & 0x80) {
    mode->PrivFlags |= NV_PRIV_TVMODE;
  }

  lockCrt (pNv);
}

/* -------- Device routines -------- */

static TvFuncRec nvNullFunc = {
  InitRegs: NVInitNullRegs, SetRegs: NVSetNullRegs, SetState: NVSetNullState,
  GetConnect: NVGetNullConnect, GetStatus: NVGetNullStatus};

static TvFuncRec nvBtFunc = {
  InitRegs: NVInitBtRegs, SetRegs: NVSetBtRegs, SetState: NVSetBtState,
  GetConnect: NVGetBtConnect, GetStatus: NVGetBtStatus};

static TvFuncRec nvChFunc = {
  InitRegs: NVInitChRegs, SetRegs: NVSetChRegs, SetState: NVSetChState,
  GetConnect: NVGetChConnect, GetStatus: NVGetChStatus};

/* FIXME: Should have nvNullFunc, or check TvFunc otherwise */

void NVSetTvDevice (NVPtr pNv, I2CChainPtr chain, Bool init)
{
  DPRINTF ("nv set dev %p %i\n", chain, init);
  pNv->TvDev = chain->dev;
  switch (chain->chip) {
    case NV_BROOKTREE:
    case NV_CONEXANT:
      pNv->TvFunc = &nvBtFunc;
      break;
    case NV_CHRONTEL:
      pNv->TvFunc = &nvChFunc;
      break;
    case NV_PHILIPS:
    default:
      pNv->TvFunc = &nvNullFunc;
      break;
  }
  if (pNv->TvFunc && init) {
    unlockCrt (pNv);
    if (readCrt (pNv, 0x28) & 0x80) {
      /* Already in TV mode, don't init */
      nvTvState = NV_TV_ON;
    } else {
      nvBusOk = TRUE;
      pNv->TvFunc->InitRegs (chain->dev);
    }
    lockCrt (pNv);
  }
}

I2CChainPtr NVFindTvDevice (NVPtr pNv, NVTvChip chip)
{
  I2CChainPtr chain;

  if (!pNv) return NULL;
  for (chain = pNv->TvChain; chain; chain = chain->next)
  {
    if (chain->chip == chip) return chain;
  }
  return NULL;
}

void NVCreateChain (NVPtr pNv)
{
  int bus;
  I2CDevPtr dev;
  I2CChainPtr chain;
  char version[50];
  char *s;

  if (pNv->TvChain) DPRINTF ("NVCreateChain: WARNING! Memory leak\n");
  pNv->TvChain = NULL;
  for (bus = NV_NBUS; bus > 0; ) {
    bus--;
    for (dev = pNv->TvBusses[bus]->FirstDev; dev; dev = dev->NextDev) {
      chain = xcalloc (1, sizeof (I2CChainRec));
      chain->dev = dev;
      chain->next = pNv->TvChain;
      chain->chip = NV_NO_CHIP;
      switch (dev->SlaveAddr) {
	case 0x88: 
	  if ((s = NVCheckPhilips (dev))) { 
	    chain->chip = NV_PHILIPS; break; 
	  }
	  /* no break */
	case 0x8a:
	  if ((s = NVCheckConexant (dev))) { 
	    chain->chip = NV_CONEXANT; break; 
	  }
	  if ((s = NVCheckBrooktree (dev, pNv))) { 
	    chain->chip = NV_BROOKTREE; break; 
	  }
	  break;
	case 0xea:
	case 0xec:
	  if ((s = NVCheckChrontel (dev))) { 
	    chain->chip = NV_CHRONTEL; break; 
	  }
	  break;
      }
      if (chain->chip == NV_NO_CHIP) {
	snprintf (version, 50, "Unknown chip (%1s:%02X)", I2C_ID(dev));
	s = version;
      }
      chain->name = xalloc (strlen(s)+1);
      strcpy (chain->name, s);
      pNv->TvChain = chain;
    }
  }
}

/*
 * Free the chain of tv chips
 */

void NVDestroyChain (I2CChainPtr chain)
{
  I2CChainPtr c;

  while (chain) {
    xfree (chain->name);
    c = chain->next;
    xfree (chain);
    chain = c;
  }
}

/*
 * Free chain and device records of all tv chips
 */

void NVDestroyDevices (NVPtr pNv)
{
  I2CDevPtr dev;
  int bus;

  if (!pNv) return;
  if (pNv->TvChain) NVDestroyChain (pNv->TvChain);
  pNv->TvChain = NULL;
  pNv->TvDev = NULL;
  for (bus = 0; bus < NV_NBUS; bus++) {
    if (pNv->TvBusses[bus]) {
      while ((dev = pNv->TvBusses[bus]->FirstDev) != NULL) {
	xf86DestroyI2CDevRec(dev, TRUE); 
      }
    }
  }
}

/*
 * Free chain, device records, and all busses
 */

void NVDestroyBusses (NVPtr pNv)
{
  int i;

  if (!pNv) return;
  if (pNv->TvChain) NVDestroyChain (pNv->TvChain);
  pNv->TvChain = NULL;
  pNv->TvDev = NULL;
  for (i = 0; i < NV_NBUS; i++) {
    if (pNv->TvBusses[i]) 
      xf86DestroyI2CBusRec (pNv->TvBusses[i], TRUE, TRUE);
    pNv->TvBusses[i] = NULL;
  }
}

void NVProbeAllDevices (NVPtr pNv)
{
  I2CDevPtr dev;
  I2CSlaveAddr addr;
  int bus;
  static char name[8][0x100][NV_NBUS]; /* ugly, but avoids memory leak */

  for (bus = 0; bus < NV_NBUS; bus++) {
    for (addr = 0x00; addr < 0x100; addr += 2) {
      if (xf86I2CProbeAddress(pNv->TvBusses[bus], addr)) {
	dev = xf86CreateI2CDevRec();
	dev->DevName = &name[0][addr][bus];
	dev->SlaveAddr = addr; 
	dev->pI2CBus = pNv->TvBusses[bus];
	snprintf (dev->DevName, 6, "%1i:%02x", bus, addr);
	if (!xf86I2CDevInit(dev)) {
	  xf86DestroyI2CDevRec(dev, TRUE);
	}
      }
    }
  }
}

void NVProbeDevice (I2CBusPtr bus, I2CSlaveAddr addr, char *name)
{
  I2CDevPtr dev;

#ifndef FAKE_PROBE_ALL
#ifndef FAKE_PROBE_ADDR
  if (xf86I2CProbeAddress(bus, addr))
#else
  if (addr == FAKE_PROBE_ADDR)
#endif
#endif
  {
    dev = xf86CreateI2CDevRec();
    dev->DevName = name; 
    dev->SlaveAddr = addr;
    dev->pI2CBus = bus;
    if (!xf86I2CDevInit(dev)) {
      xf86DestroyI2CDevRec(dev, TRUE);
    }
  }
}

void NVProbeTvDevices (NVPtr pNv)
{
  if (!pNv) return;
  writeCrt (pNv, 0x1f, 0x57); /* unlock */
  NVProbeDevice (pNv->TvBusses[0], 0x88, "0:88");
  NVProbeDevice (pNv->TvBusses[0], 0x8A, "0:8A");
  NVProbeDevice (pNv->TvBusses[0], 0xEA, "0:EA");
  NVProbeDevice (pNv->TvBusses[0], 0xEC, "0:EC");
  NVProbeDevice (pNv->TvBusses[1], 0x88, "1:88");
  NVProbeDevice (pNv->TvBusses[1], 0x8A, "1:8A");
  NVProbeDevice (pNv->TvBusses[1], 0xEA, "1:EA");
  NVProbeDevice (pNv->TvBusses[1], 0xEC, "1:EC");
  NVProbeDevice (pNv->TvBusses[2], 0x88, "2:88");
  NVProbeDevice (pNv->TvBusses[2], 0x8A, "2:8A");
  NVProbeDevice (pNv->TvBusses[2], 0xEA, "2:EA");
  NVProbeDevice (pNv->TvBusses[2], 0xEC, "2:EC");
  /* NVCreateChain also determines chip type and version */
  NVCreateChain (pNv);
#ifdef PROBE_ALL_UNKOWN
  /* if no dev's found, scan all addresses */
  if (!pNv->TvChain) {
    NVProbeAllDevices (pNv);
    NVCreateChain (pNv);
  }
#endif
  pNv->TvDev = pNv->TvChain ? pNv->TvChain->dev : NULL;
}

/* -------- -------- */

/* Intermediate interface to TV. This might not be needed for the
 * XFree driver, and should be moved to a different file in that case.
 */

static int nvCrtIntr;

inline void prepCrt (NVPtr pNv)
{
  // unlockCrt (pNv);
  nvCrtIntr = readCrt (pNv, 0x11) & 0x10;
  // andCrt (pNv, 0x11, 0xef);   /* disable vert intr */
  // orCrt (pNv, 0x21, 0x80);    /* "virtual" changes */
  writeCrt (pNv, 0x21, 0xfa);
}

inline void commitCrt (NVPtr pNv)
{
  // andCrt (pNv, 0x21, 0x7f);      /* commit changes */
  // orCrt (pNv, 0x11, nvCrtIntr);  /* enable vert intr again */
}

void NVInitTvState (NVPtr pNv)
{
  if (nvTvState == NV_TV_UNKNOWN) {
    nvTvState = (readCrt (pNv, 0x28) & 0x80) ? NV_TV_ON : NV_TV_OFF;
  }
}

void NVSetTestImage (NVPtr pNv)
{
  /* FIXME? */
  DPRINTF ("nv tv testimage\n");
  unlockCrt (pNv);
  if (! (readCrt (pNv, 0x28) & 0x80)) {
    /* Only do bars if not in tv-mode */
    pNv->TvFunc->SetState (pNv->TvDev, NV_TV_BARS);
  }
  lockCrt (pNv);
}

/* FIXME: Check that if not changing mode, only registers are set,
   i.e. crt/tv regs in tv-mode and crt regs in non-tv mode
*/

void NVSetTvMode (NVPtr pNv, DisplayModePtr mode)
{
  NVTvState state;

  /* FIXME: Just convert to old 'state' for now */
  NVInitTvState (pNv);
  state = (mode->PrivFlags & NV_PRIV_TVMODE) ? NV_TV_ON : NV_TV_OFF;
  DPRINTF ("nv tv state %i\n", state);
  unlockCrt (pNv);
#if 1 /* FIXME */
  /* Experimental: Wait for vert retrace... */
  pNv->riva.PCIO[0x3c4] = 0x01;
  pNv->riva.PCIO[0x3c5] = 0x01;
  /* Wait until in image */
  while (pNv->riva.PCIO[0x3da] & 0x08);
  /* Wait until retrace starts */
  while (! (pNv->riva.PCIO[0x3da] & 0x08));
#if 0
  /* Wait until again in image */
  while (pNv->riva.PCIO[0x3da] & 0x08);
#endif
#endif /* FIXME */
  nvBusOk = TRUE;
  if (nvTvState == NV_TV_ON && state != NV_TV_ON) 
  {
    int tmp;
    int i;
    U032 a, b;

    prepCrt (pNv);
    andCrt (pNv, 0x28, 0x7f); 
#ifndef FAKE_NV_REGS
#if 0
    pNv->riva.PRAMDAC[0x50c/4] = 0x10000700; /* PLL_COEFF_SELECT */
#endif
    pNv->riva.PRAMDAC[0x50c/4] &= ~0x011f0000; /* PLL_COEFF_SELECT */
    pNv->riva.PRAMDAC[0x700/4] = 0x00000001; /* TV_SETUP */
    pNv->riva.PRAMDAC[0x880/4] |= 0x30000000;
#else
    FPRINTF ("nv[50c,700,880=mon] \n");
#endif
    andCrt (pNv, 0x1A, 0x3f);    /* Enable vsync, hsync */
    commitCrt (pNv);

#if 0 /* FIXME */
    /* Disable slave CRTC mode */
    tmp = readCrt (pNv, 0x28) & 0x7b; 
    writeCrt (pNv, 0x28, tmp); 
    for (i = 0; i < 25; i++) {
      a = pNv->riva.PCRTC[0x808/4]; /* NV_PCRTC_RASTER */
      xf86usleep (2000); /* 2ms */
      b = pNv->riva.PCRTC[0x808/4];
      DPRINTF ("off %08x %08x\n", a, b);
      if (a == b) {
	writeCrt (pNv, 0x28, tmp | 0x80); 
	xf86usleep (1000); /* 2ms */
	writeCrt (pNv, 0x28, tmp); 
      } else {
	break;
      }
    }
#endif
  }
  if (nvTvState != state) {
    pNv->TvFunc->SetState (pNv->TvDev, state);
  }
  if (state != NV_TV_OFF) 
  {
    pNv->TvFunc->SetRegs (pNv->TvDev, &((NVPrivate *) (mode->Private))->tv);
  }
  prepCrt (pNv);
  NVSetCrtRegs (pNv, mode);
  if (state == NV_TV_ON) 
  {
    writeCrt (pNv, 0x28, (readCrt (pNv, 0x28) & 3) | 0x80);   
#ifndef FAKE_NV_REGS
    pNv->riva.PRAMDAC[0x50C/4] = 0x10030700;  /* PLL Coeff Sel */
    pNv->riva.PRAMDAC[0x700/4] = 0x01000001;  /* TV setup      */
    pNv->riva.PRAMDAC[0x880/4] |= 0x30000000;
#else
    FPRINTF ("nv[50c,700,880=tv] \n");
#endif
    if (! (mode->PrivFlags & NV_PRIV_DUALVIEW))
    {
      orCrt (pNv, 0x1A, 0x80);           /* disable hsync */
    }
  }
  commitCrt (pNv);
  nvTvState = state;
  lockCrt (pNv);
  DPRINTF ("nv tv state end\n");
}

long NVGetTvStatus (NVPtr pNv, int index)

{
  long result; 

  unlockCrt (pNv);
  NVInitTvState (pNv);
  nvBusOk = TRUE;
  result = pNv->TvFunc->GetStatus (pNv->TvDev, index);
  lockCrt (pNv);
  return result;
}

NVConnect NVGetTvConnect (NVPtr pNv)
{
  NVConnect result;

  unlockCrt (pNv);
  NVInitTvState (pNv);
  nvBusOk = TRUE;
  result = pNv->TvFunc->GetConnect (pNv->TvDev);
  lockCrt (pNv);
  return result;
}

/* -------- -------- */

#if 0 /* FIXME: Remove */

void NVSaveCrt (NVPtr pNv, NVCrtState *state)
{
  int i;

  DPRINTF ("crt save %p\n", state);
  unlockCrt (pNv);
  for (i = 0x00; i <= 0x16; i++)
    state->regs [i] = readCrt(pNv, i);
  state->regs [0x25] = readCrt(pNv, 0x25);
  state->regs [0x2d] = readCrt(pNv, 0x2d);
  state->regs [0x33] = readCrt(pNv, 0x33);
#if 1
  state->regs [0x19] = readCrt(pNv, 0x19);
  state->regs [0x1a] = readCrt(pNv, 0x1a);
  state->regs [0x1b] = readCrt(pNv, 0x1b);
  state->regs [0x20] = readCrt(pNv, 0x20);
  state->regs [0x28] = readCrt(pNv, 0x28);
  state->regs [0x30] = readCrt(pNv, 0x30);
  state->regs [0x31] = readCrt(pNv, 0x31);
#endif
  lockCrt (pNv);
#if 0
  /**/ for (i = 0x00; i <= 0x33; i++) printf ("%02x ", state->regs [i]);
  /**/ printf ("\n");
#endif
}

void NVRestoreCrt (NVPtr pNv, NVCrtState *state)
{
  int i;

  DPRINTF ("crt restore %p\n", state);
  unlockCrt (pNv);
  for (i = 0x00; i <= 0x16; i++)
    writeCrt (pNv, i, state->regs [i]);  
  writeCrt (pNv, 0x25, state->regs [0x25]);
  writeCrt (pNv, 0x2d, state->regs [0x2d]);
  writeCrt (pNv, 0x33, state->regs [0x33]);
#if 0
  writeCrt (pNv, 0x19, state->regs [0x19]);
  writeCrt (pNv, 0x1a, state->regs [0x1a]);
  writeCrt (pNv, 0x1b, state->regs [0x1b]);
  writeCrt (pNv, 0x20, state->regs [0x20]);
  writeCrt (pNv, 0x28, state->regs [0x28]);
  writeCrt (pNv, 0x30, state->regs [0x30]);
  writeCrt (pNv, 0x31, state->regs [0x31]);
#endif
  lockCrt (pNv);
}

#endif

/* ----------------------------------------------------------------------

mode "1280x1024": 157.5 MHz, 91.1 kHz, 85.0 Hz
mode "1024x768": 94.5 MHz, 68.7 kHz, 85.0 Hz
mode "800x600": 56.3 MHz, 53.7 kHz, 85.1 Hz
mode "640x480": 36.0 MHz, 43.3 kHz, 85.0 Hz

# 1280x960 @ 85Hz (VESA) hsync: 85.9kHz
ModeLine "1280x1024" 157.5 1280 1344 1504 1728   1024 1025 1028 1072
h/8: a0 a8 bc d8


# 1024x768 @ 85Hz (VESA) hsync: 68.7kHz
ModeLine "1024x768"   94.5 1024 1072 1168 1376    768  769  772  808

# 800x600 @ 85Hz (VESA) hsync: 53.7kHz
ModeLine "800x600"    56.3  800  832  896 1048    600  601  604  631

# 640x480 @ 85Hz (VESA) hsync: 43.3kHz
ModeLine "640x480"    36.0  640  696  752  832    480  481  484  509

h sync 4:0 blank 6:0
v sync 3:0 blank 7:0

hor (dbssbt) 09f 09f 0a9 01d 057 0d3
ver (dbssbt) 3ff 3ff 400 003 02f 42e

hor (dbssbt) 07f 07f 087 013 02b 0a7
ver (dbssbt) 2ff 2ff 300 003 027 326

hor (dbssbt) 063 063 069 011 002 07e
ver (dbssbt) 257 257 258 00b 076 275

hor (dbssbt) 04f 04f 058 01f 067 063
ver (dbssbt) 1df 1df 1e0 003 0fc 1fb

*/
