/* $Id: virtual.c,v 1.37 2005/05/18 22:14:58 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#ifdef HAVE_MATH_H
#	include <math.h>
#endif

#include "diagnostic.h"
#include "messages.h"
#include "register.h"
#include "osutil.h"
#include "strutil.h"
#include "hosts.h"
#include "host_protocol.h"
#include "nws_proxy.h"

#include "nws_api.h"

/* this modules contains functions to talk to the proxy and to build a
 * virtual view of multiple NWS installation. So if we have say s cliques
 * and a 3rd clique that does as super-clique, we want to substitute
 * measurement in the super-clique, whenever measurement from/to hosts
 * in the others cliques are requested. This is because taking
 * measurements from all to all won't scale. The functions here are the
 * logic to do that */ 

/* modules global variables: we holds the proxy we want to talk to */
static void *lock = NULL;
static struct host_cookie *proxies = NULL;
static int howManyProxy = 0;
static int relax = 0;

extern int NwsUseServer(const NWSAPI_HostSpec *which, 
			struct host_cookie **cookies,
			int *howMany);


int
NWSAPI_UseProxy(const NWSAPI_HostSpec *which) {
	int ret;

	GetNWSLock(&lock);
	ret = NwsUseServer(which, &proxies, &howManyProxy);
	ReleaseNWSLock(&lock);

	return ret;
}

/* just find if a proxy registered with the nameserver and use it */
static int
SetProxyFromNameserver(int timeout) {
	char *tmp, filter[31+1];
	ObjectSet objs;
	Object obj;
	NWSAPI_HostSpec proxy;
	int ret;

	ret = 0;

	/* we don't have defined proxy: let's see if the
	 * nameserver knows anything about them */
	strcpy(filter, "(hostType=proxy)");
	if (NWSAPI_GetObjectsTimeout(filter, &objs, timeout)) {
		NWSAPI_ForEachObject(objs, obj) {
			tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
			if (tmp == NULL) {
				continue;
			}

			proxy = *NWSAPI_MakeHostSpec(tmp, DefaultHostPort(PROXY_HOST));
			ret += NWSAPI_UseProxy(&proxy);
			FREE(tmp);
		}
	}
	FreeObject(&objs);

	return (ret > 0);
}

/* utility to build a string suitable to send to a proxy */
static int
BuildProxyRequest(	const char *resource,
			const char *opts, 
			const char *hosts,
			char **request) {
	int len;
	char *tmp, host[MAX_HOST_NAME];
	const char *ptr;

	/* let's create the request: first resource and options */
	len = strlen(resource) + 1;
	if (opts != NULL) {
		len += strlen(opts) + 1;
	} else {
		len += 1;
	}
	tmp = MALLOC(sizeof(char) * len, 0);
	if (tmp == NULL) {
		FAIL("BuildProxyRequest: out of memory\n");
	}
	memcpy(tmp, resource, strlen(resource) + 1);
	len = strlen(resource) + 1;
	if (opts != NULL) {
		memcpy(tmp + len, opts, strlen(opts) + 1);
		len += strlen(opts) + 1;
	} else {
		/* empty string */
		memcpy(tmp + len, "\0", 1);
		len += 1;
	}

	/* ... then hosts */
	for (ptr = hosts; GETTOK(host, ptr, ",", &ptr); ) {
		if (host[0] == '\0') {
			/* empty host */
			continue;
		}

		/* let's do some trivial checking */
		if (strchr(host, ':') && strchr(host, '.') == NULL) {
			WARN1("BuildProxyRequest: %s does specify a port without domain name!!\n", host);
		}

		tmp = REALLOC(tmp, len + strlen(host) + 3, 0);
		if (tmp == NULL) {
			FAIL("BuildProxyRequest: out of memory\n");
		}

		memcpy(tmp + len, host, strlen(host) + 1);
		len += strlen(host) + 1;
	}
	*request = tmp;

	return len;
}

static int
CallCollect(	const char *resource,
		const char *opts,
		const char *hosts,
		NWSAPI_ForecastCollection **col,
		int tout) {
	NWSAPI_ForecastCollection tmp, *c;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor howManyDes = SIMPLE_DATA(INT_TYPE, 1);
	int i, j, z, y, howMany, *list, interMachine;
	char *request, host[MAX_HOST_NAME];
	const char *s;

	if (howManyProxy <= 0) {
		return 0;
	}

	/* let's build the request */
	des.repetitions = BuildProxyRequest(resource, opts, hosts, &request);
	if (des.repetitions <= 0) {
		return 0;
	}

	/* let's count the hosts */
	s = hosts;
	for(howMany = 0; GETTOK(host, s, ",", &s); howMany++)
		;

	/* let's see if we have a matrix or a vector of data */
	interMachine = NWSAPI_IntermachineResource(resource);

	/* let's got thorugh the proxies: the proxies are a global
	 * variables, so to be sure no others will modify them, we'll
	 * need to lock the section. */
	c = NULL;
	GetNWSLock(&lock);
	for (i = 0; i < howManyProxy; i++) {
		/* let's connect to the proxy */
		if (!ConnectToHost(&proxies[i], NULL)) {
			WARN1("CallCollect: failed to connect to %s\n", HostCImage(&proxies[i]));
			continue;
		}

		/* let's send the request */
		if (!SendMessageAndData(proxies[i].sd, GET_FORECASTS, request, &des, 1, tout)) {
			DROP_SOCKET(&proxies[i].sd);
			ERROR1("CallCollect: failed to talk %s\n", HostCImage(&proxies[i]));
			continue;
		}

		/* allocate space */
		if (c == NULL) {
			if (interMachine) {
				y = howMany * howMany;
			} else {
				y = howMany;
			}
			c = (NWSAPI_ForecastCollection *)MALLOC(sizeof(NWSAPI_ForecastCollection) * y, 1);
			/* let's initialize the values */
			for (z = 0; z < y; z++) {
				c[z].measurement.timeStamp = 0;
			}
		}

		/* let's get the forecasts: first how many hosts the
		 * proxy knows anything about ... */
		howManyDes.repetitions = 1;
		if (!RecvMessageAndData(proxies[i].sd, GOT_FORECASTS, &j, &howManyDes, 1, tout)) {
			DROP_SOCKET(&proxies[i].sd);
			ERROR1("CallCollect: failed to receive from %s\n", HostCImage(&proxies[i]));
			continue;
		} 

		/* ... then the list itself if we have more then 0 hosts */
		if (j <= 0) {
			INFO("CallCollect: proxy doesn't know anything\n");
			continue;
		}
		list = MALLOC(sizeof(int) * j, 1);
		howManyDes.repetitions = j;
		if (!RecvData(proxies[i].sd, list, &howManyDes, 1, tout)) {
			FREE(list);
			DROP_SOCKET(&proxies[i].sd);
			ERROR1("CallCollect: failed to receive from %s\n", HostCImage(&proxies[i]));
			continue;
		}

		/* we receive the forecastCollection one at a time */
		for (y = 0; y < j; y++) {
			/* y is destination and on single host resource
			 * cannot be > 0 */
			if (!interMachine && y > 0) {
				break;
			}
			for (z = 0; z < j; z++) {
				int ciccio;

				/* let's compute the index of the
				 * receiving forecast */
				if (interMachine && z == y) {
					/* we don't send/receive the
					 * diagonal in a 2 hosts
					 * resource. */
					continue;
				} else if (!interMachine) {
					/* very simple for 1 host
					 * resources */
					ciccio = list[z];
				} else {
					ciccio = list[z] + list[y]*howMany;
				}

				if (!RecvData(proxies[i].sd, &tmp, forecastCollectionDescriptor, forecastCollectionDescriptorLength, tout)) {
					DROP_SOCKET(&proxies[i].sd);
					ERROR("CallCollect: failed to receive forecasts\n");
					break;
				} 

				/* we keep the most recent */
				if (c[ciccio].measurement.timeStamp < tmp.measurement.timeStamp) {
					c[ciccio] = tmp;
				}
			}
			if (z < j) {
				/* error */
				break;
			}
		}
		FREE(list);
	}
	ReleaseNWSLock(&lock);
	FREE(request);

	*col = c;

	/* let's compute the # of forecasts returned */
	if (interMachine) {
		howMany *= howMany;
	}

	return howMany;
}

/* GetVirtualForecast is quite a beast, so we use few smaller routine to
 * make it easier to read. */

/* damn CS dept! We can have hosts with or without domain name, so we try
 * to compare them here, without doing to bad (a will match alicante with
 * strstr so we need to try to be smarter, we also need to match
 * test:9786 with test.cs.ucsb.edh:9786 thanks to our damn department's
 * broken DNS).
 *
 * Returns 1 if the hosts are 'similar'. 0 otherwise. */
int
NWSAPI_HostCompare(	const char *src,
			const char *dst) {
	char h[MAX_HOST_NAME + 9], 
	     *ptr;
	const char *cmp;
	
	/* before doing all the work, let's see if they are equal: it
	 * will save us time on well behaved DNS */
	if (strcmp(src, dst) == 0) {
		return 1;
	}

	/* now if we are not told to do relax mathing, we have failed */
	if (!relax) {
		return 0;
	}

	/* now we want to compare modified src with dst and modified dst
	 * with src */
	strcpy(h, src);
	cmp = dst;
	do {
		ptr = strchr(h, ':');
		if (!ptr) {
			if (!strchr(h, '.')) {
				/* so if the host doesn't have . nor : we
				 * try to adjust to what the cmp has. If
				 * there is a . we put .* */ 
				if (strchr(cmp , '.')) {
					strcat(h, ".*");
				}
			} 
			/* and now if the destination as a : we just add
			 * the match for any port */
			if (strchr(cmp , ':')) {
				strcat(h, ":*");
			}
		} else if (!strchr(h, '.')) {
			/* the nasty one: we got the port specified, but
			 * no domain: if cmp has port and no domain, we
			 * leave as it is. If cmp has domain and port we
			 * add a .*. If it doesn't have port we'll skip
			 * the port check. */
			if (strchr(cmp , '.')) {
				memmove(ptr + 2, ptr, strlen(ptr) + 1);
				*ptr = '.';
				*(ptr + 1) = '*';
			}
			if (!strchr(cmp , ':')) {
				ptr = strchr(h, ':');
				*ptr = '\0';
			}
		}
		if (strmatch(cmp, h)) {
			/* they are 'similar' */
			return 1;
		}

		if (cmp == src) {
			break;
		}
		/* let's switch them over now */
		cmp = src;
		strcpy(h, dst);
	} while(1);

	return 0;
}
static int
FindForecast(	const char *src,
		const char *dst,
		const char *list,
		int howMany,
		const char *resource,
		NWSAPI_ForecastCollection **matrix,
		int goToMemory) {
	int i, j, z, len;
	char h[MAX_HOST_NAME];
	const char *s;
	static char *filter = NULL;
	char *name;
	NWSAPI_ObjectSet series;
#define THIS_FILTER_MAX (35 + (MAX_HOST_NAME + 7) * 2)

	j = z = -1;
	for (i = 0, s = list; GETTOK(h, s, ",", &s); i++) {
		if (NWSAPI_HostCompare(src, h)) {
			j = i;
			if (z != -1) 
				break;
		}
		if (NWSAPI_HostCompare(dst, h)) {
			z = i;
			if (j != -1) 
				break;
		}
	}
	if (j == -1 && z == -1) {
		ABORT("FindForecasts: hosts non in the host list??\n");
	}
	i = j + z*howMany;

	if ((*matrix)[i].measurement.timeStamp == 0 && goToMemory) {
		INFO("FindForecasts: getting forecast from the memory\n");
		/* we have to go for the slow path */
		GetNWSLock(&lock);
		if (filter == NULL) {
			filter = MALLOC(THIS_FILTER_MAX + strlen(resource), 1);
		}
		snprintf(filter, THIS_FILTER_MAX + strlen(resource), "&(host=%s)(target=%s)(resource=%s)", src, dst, resource);
		len =  NWSAPI_GetObjects(filter, &series);
		ReleaseNWSLock(&lock);
		if (len && strlen(series) > 5) {
			name = NwsAttributeValue_r(FindNwsAttribute(series, "name"));
			len = NWSAPI_GetForecast(name, &((*matrix)[i]));
			FREE(name);
			if (!len) {
				INFO("FindForecasts: failed to get forecast\n");
				i = -1;
			}
		} else {
			INFO1("FindForecasts: search failed for %s\n", filter);
			i = -1;
		}
		FreeObjectSet(&series);
	}

	return (i);
}
static void
AddHost(	char **h,
		NWSAPI_ObjectSet path) {
	NWSAPI_Object obj;
	char host[MAX_HOST_NAME], 
	     host1[MAX_HOST_NAME],
	     *tmp;
	const char *ptr, *ptr1;
	int done;

	NWSAPI_ForEachObject(path, obj) {
		tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "hosts"));
		for (ptr = tmp; GETTOK(host, ptr, ",", &ptr); ) {
			done = 0;
			for (ptr1 = *h; GETTOK(host1, ptr1, ",", &ptr1); ) {
				if (NWSAPI_HostCompare(host, host1)) {
					/* we already have it */
					done = 1;
					break;
				}
			}
			if (done) {
				continue;
			}

			/* let's add it */
			if (relax) {
				if (strchr(host, ':')) {
					*strchr(host, ':') = '\0';
				}
				if (strchr(host, '.')) {
					*strchr(host, '.') = '\0';
				}
			}

			*h = REALLOC(*h, strlen(*h) + strlen(host) + 2, 1);
			sprintf(*h, "%s,%s", *h, host);
		}
		FREE(tmp);
	}
}
static int
CliqueOfHost(	const char *host,
		NWSAPI_ObjectSet objs,
		char **cliques,
		int howMany) {
	NWSAPI_Object obj;
	int i, z;
	const char *s;
	char name[MAX_HOST_NAME], *tmp, *tmp1;

	i = -1;
	NWSAPI_ForEachObject(objs, obj) {
		tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "member"));
		for(s = tmp; GETTOK(name, s, ",", &s) && i == -1; ) {
			if (i == -1 && NWSAPI_HostCompare(name, host)) {
				tmp1 = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
				for (z = 0; z < howMany; z++) {
					if (strcmp(cliques[z], tmp1) == 0) {
						i = z;
						break;
					}
				}
				FREE(tmp1);
			}
		}
		FREE(tmp);
	}

	return i;
}

int
NWSAPI_GetVirtualForecasts(	const char *resource,
				const char *opts,
				const char *hosts,
				int mode,
				int level,
				unsigned long valid,
				NWSAPI_ForecastCollection **col,
				const int tout) {
	char *tmp, 
		host[MAX_HOST_NAME], 
		host1[MAX_HOST_NAME], 
		*hlist,
		**cliques;
	const char *ptr, *ptr1, *s;
	int i, j, l, m, y, howMany, subs, virt, *matrix;
	double now;
	NWSAPI_ObjectSet objs, *paths;
	NWSAPI_Object obj;

	/* sanity check */
	if (resource == NULL || hosts == NULL || col == NULL) {
		FAIL("GetVirtualForecasts: NULL parameter!\n");
	}
	if (howManyProxy <= 0) {
		/* we don't have defined proxy: let's see if the
		 * nameserver knows anything about them */
		if (!SetProxyFromNameserver(tout)) {
			FAIL("GetVirtualForecasts: no proxy defined or found!\n");
		}
	}

	cliques = NULL;
	paths = NULL;
	objs = NULL;
	
	/* we keep a copy of the hosts list */
	hlist = strdup(hosts);
	if (hlist == NULL) {
		ABORT("GetVirtualForecasts: out of memory\n");
	}

	/* now let's see if we need to go virtual */
	howMany = subs = virt = 0;
	now = CurrentTime();
	if (mode > MODE_NO_VIRTUAL && mode <= MODE_MIN && level > 0 && NWSAPI_IntermachineResource(resource)) {
		/* let's get the cliques */
		if (NWSAPI_GetObjects(NWSAPI_CLIQUES, &objs)) {
			virt = 1;

			/* now let's build the clique to clique
			 * substitutions matrix: first let's allocate
			 * space and the vector of cliques names (we'll
			 * use it quite a bit). */
			NWSAPI_ForEachObject(objs, obj) {
				tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
				cliques = (char **)REALLOC(cliques, sizeof(char *)*(subs+1), 1);
				cliques[subs] = tmp;
				subs++;
			}
	
			/* we store all the possible paths: let's make
			 * room for them */
			paths = (NWSAPI_ObjectSet *)MALLOC(sizeof(NWSAPI_ObjectSet) * subs * subs, 1);
			for (i = 0; i < subs; i++) {
				for (j = 0; j < subs; j++) {
					paths[i + j*subs] = NULL;
					if (i == j) {
						continue;
					}

					/* let's get the path */
					if (!NWSAPI_FindCliquesPath(cliques[i], cliques[j], objs, &(paths[i + j*subs]), level)) {
						continue;
					}

					/* let's add the extra host */
					AddHost(&hlist, paths[i + j*subs]);
				}
			}
		}
	}

	/* let's build the request */
	howMany = CallCollect(resource, opts, hlist, col, tout);

	if (howMany <= 0 || !virt) {
		/* let's clean the memory */
		FREE(hlist);
		if (howMany <= 0) {
			FAIL("GetVirtualForecasts: failed to collect data!\n");
		} else {
			/* let's go though the data and nullify the non
			 * valid ones */
			if (valid > 0) {
				for (i = 0; i < howMany; i++) {
					if ((*col)[i].measurement.timeStamp + valid < now) {
						/* invalid */
						(*col)[i].measurement.timeStamp = 0;
					}
				}
			}
			return howMany;
		}
	}

	/* let's count the # of hosts in hlist */
#ifdef HAVE_SQRT
	howMany = sqrt(howMany);
#else
	s = hlist;
	for(howMany = 0; GETTOK(host, s, ",", &s); howMany++) 
		;
#endif

	/* we got all the data available to the proxy, so now we need to
	 * build the substitution matrix */
	matrix = (int *)MALLOC(sizeof(int) * subs * subs, 1);
	for (i = 0; i < subs; i++) {
		for (j = 0; j < subs; j++) {
			matrix[i + j*subs] = -1;
			if (i == j) {
				continue;
			}

			/* let's go through the path and get the right
			 * substitution */
			NWSAPI_ForEachObject(paths[i + j*subs], obj) {
				int best, p;

				tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "hosts"));
				host1[0] = '\0';
				best = -1;
				for(s = tmp; GETTOK(host, s, ",", &s); ) {
					/* we need 2 hosts */
					if (host1[0] == '\0') {
						strcpy(host1, host);
						continue;
					}

					p = FindForecast(host1, host, hlist, howMany, resource, col, 1);
					if (p == -1 || (valid > 0 && (*col)[p].measurement.timeStamp + valid < now)) {
						WARN2("GetVirtualForecasts: missing link in path (%s->%s)\n", host1, host);
						break;
					}

					/* destination is the source next
					 * time around */
					strcpy(host1, host);


					if (best == -1) {
						best = p;
					}

					/* let's see if the measurement
					 * we got is better */
					switch (mode) {
					case MODE_MAX:
						if ((*col)[best].measurement.measurement < (*col)[p].measurement.measurement) {
							best = p;
						}
						break;
					case MODE_MIN:
						if ((*col)[best].measurement.measurement > (*col)[p].measurement.measurement) {
							best = p;
						}
						break;
					default:
						ERROR("GetVirtualForecasts: unknown mode!\n");
						break;
					}
				}
				/* I need all the links in the paths to
				 * be able to substitute */
				if (p != -1) {
					matrix[i + j*subs] = best;
				}
				FREE(tmp);
			}
		}
	}

	/* let's go through the matrix and fill the empty spot */
	for (l=0, ptr = hosts; objs && GETTOK(host, ptr, ",", &ptr); l++) {
		for (m=0, ptr1 = hosts; GETTOK(host1, ptr1, ",", &ptr1); m++) {
			/* y keeps the index of the forecast we are
			 * dealing with */
			y = m + l*howMany;

			/* let's see if we got the result or not */ 
			if ((*col)[y].measurement.timeStamp != 0 
					|| l == m) {
				continue;
			}

			/* we skip the same host */
			if (NWSAPI_HostCompare(host, host1)) {
				continue;
			}

			/* we need the cliques of the hosts */
			i = CliqueOfHost(host, objs, cliques, subs);
			j = CliqueOfHost(host1, objs, cliques, subs);
			if (i != -1 && j != -1 && matrix[i + j*subs] != -1) {
				int p;
				char host3[MAX_HOST_NAME],
				     host4[MAX_HOST_NAME];

				if (valid <= 0) {
					/* easy way out: no check needed */
					(*col)[y] = (*col)[matrix[i + j*subs]];
					continue;
				}

				/* we need to check if the first
				 * and last link are all right. NOTE: we
				 * don't need to ask to go to the memory
				 * anymore (FindForecast) becasue we
				 * already done that check. */
				NWSAPI_ForEachObject(paths[i + j*subs], obj) {
					tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "hosts"));
					s = tmp; 
					GETTOK(host3, s, ",", &s);
					/* let's check the first
					 * link */
					if (!NWSAPI_HostCompare(host, host3)) {
						p = FindForecast(host3, host, hlist, howMany, resource, col, 0);
						if (p < 0 || (*col)[p].measurement.timeStamp + valid < now) {
							/* there is no
							 * such link */
							break;
						}
					}

					/* and now let's check the last
					 * link */
					do {
						strcpy(host4, host3);
					} while(GETTOK(host3, s, ",", &s));
					if (!NWSAPI_HostCompare(host4, host1)) {
						p =  FindForecast(host4, host1, hlist, howMany, resource, col, 0);
						if (p < 0 || (*col)[p].measurement.timeStamp + valid < now) {
							/* there is no
							 * such link */
							break;
						}
					}
					/* TODO: I might need to check
					 * every possible path, not only
					 * the first one */
					(*col)[y] = (*col)[matrix[i + j*subs]];
					break;
				}
			}
		}
	}
	/* let's clean the memory */
	FREE(hlist);
	FreeObjectSet(&objs);
	FREE(matrix);
	for (i = (subs * subs) - 1; i >= 0; i--) {
		FreeObjectSet(&(paths[i]));
	}
	FREE(paths);
	for (subs--; subs >= 0; subs--) {
		FREE(cliques[subs]);
	}
	FREE(cliques);

	/* now last steps: we have a matrix with more hosts than what the
	 * user asked: let's trim it. Let's count first how many hosts we
	 * have and how many we need */
	s = hosts;
	for(subs = 0; GETTOK(host, s, ",", &s); subs++) 
		;

	/* the order of the indeces allow us to do the assignment without
	 * override values! */
	for (i = 0; i < subs; i++) {
		for (j = 0; j < subs; j++) {
			/* now we have to check that the data is valid */
			if (valid > 0 && ((*col)[j + i*howMany].measurement.timeStamp + valid < now)) {
				/* data is stale */
				(*col)[j + i*subs].measurement.timeStamp = 0;
			} else {
				/* data is good */
				(*col)[j + i*subs] = (*col)[j + i*howMany];
			}
		}
	}

	/* free the extra memory */
	*col = (NWSAPI_ForecastCollection *)REALLOC(*col, sizeof(NWSAPI_ForecastCollection) * y, 1);

	return (subs*subs);
}

int
NWSAPI_GetCliqueHosts(	const char *clique_name,
			char **h) {
	const char *member;
	char filter[MAX_HOST_NAME], *tmp;
	int howMany;
	NWSAPI_ObjectSet objs;
	NWSAPI_NwsAttribute attr;

	/* sanity check */
	if (clique_name == NULL || clique_name[0] == '\0' || h == NULL) {
		ERROR("GetCliqueHosts: NULL parameters!\n");
		return 0;
	}

	/* let's talk to the nameserver and get clique registration */
	snprintf(filter, sizeof(filter), "(name=%s)", clique_name);
	if (!NWSAPI_GetObjects(filter, &objs)) {
		ERROR("GetCliqueHosts: no clique or no nameserver set\n");
		return 0;
	}

	/* now let's get the members and create the host list */
	attr = NWSAPI_FindNwsAttribute(objs, "member");
	if (attr == NO_ATTRIBUTE) {
		ERROR("GetCliqueHosts: no clique found?\n");
		NWSAPI_FreeObjectSet(&objs);
		return 0;
	}
	tmp = NwsAttributeValue_r(attr);
	NWSAPI_FreeObjectSet(&objs);

	/* let's count the members */
	for (howMany=0, member=tmp; GETTOK(filter, member, ",", &member); ) {
		howMany++;
	}

	/* let's return the hosts list */
	if (howMany > 0) {
		*h = tmp;
	} else {
		FREE(tmp);
	}

	return howMany;
}

/* returns common hosts between hosts lists */
static int
CommonHosts(	const char *h1,
		const char *h2,
		char **hosts) {
	int i, len, len1;
	char host1[MAX_HOST_NAME], host2[MAX_HOST_NAME], *ret;
	const char *member1, *member2;

	/* now let's find the common hosts */
	i = len = 0;
	ret = NULL;
	for (member1 = h1; GETTOK(host1, member1, ",", &member1); ) {
		for (member2 = h2; GETTOK(host2, member2, ",", &member2); ) {
			if (NWSAPI_HostCompare(host1, host2)) {
				/* we got one */
				len1 = strlen(host1) + 1;
				ret = (char *)REALLOC(ret, len + 1 + len1, 1);
				if (len > 0) {
					ret[len] = ',';
					len++;
				}
				memcpy(ret + len, host1, len1);
				len += len1;
				i++;
			}
		}
	}
	*hosts = ret;

	return i;
}


static NWSAPI_Object *
AddHostClique(	NWSAPI_ObjectSet obj,
		const char *clique,
		const char *host,
		int head) {
	NWSAPI_Object *tmpObj;
	char *t, *tmp, h[MAX_HOST_NAME];
	const char *s;

	/* let's add the host */
	tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "hosts"));

	/* host can be a comma separated list of hosts */
	GETTOK(h, host, ",", &s);
	if (head) {
		/* compare with the head */
		t = tmp;
	} else {
		/* compare with the tail */
		t = tmp + strlen(tmp) - strlen(h);
	}
	if (strncmp(h, t, strlen(h)) == 0) {
		/* it's the same, let's get the next one */
		if (!GETTOK(h, s, ",", &s)) {
			WARN("AddHostClique: cannot get second host!\n");
			GETTOK(h, host, ",", &s);
		}
	}


	t = MALLOC(strlen(tmp) + strlen(h) + 2, 1);
	
	if (head) {
		sprintf(t, "%s,%s", h, tmp);
	} else {
		sprintf(t, "%s,%s", tmp, h);
	}
	tmpObj = MALLOC(sizeof(NWSAPI_Object), 1);
	*tmpObj = NewObject();
	AddNwsAttribute(tmpObj, "hosts", t);
	FREE(t);
	FREE(tmp);

	/* now let's add the clique */
	tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "cliques"));
	t = MALLOC(strlen(tmp) + strlen(clique) + 2, 1);
	if (head) {
		sprintf(t, "%s,%s", clique, tmp);
	} else {
		sprintf(t, "%s,%s", tmp, clique);
	}
	AddNwsAttribute(tmpObj, "cliques", t);
	FREE(tmp);
	FREE(t);

	return tmpObj;
}


static void
FindCliques(	const NWSAPI_ObjectSet cliques,
		const char *src,
		const char *dst,
		char **history,
		int level,
		NWSAPI_ObjectSet *path) {
	NWSAPI_Object tmp, obj;
	char *m1, *m2, *common, host[MAX_HOST_NAME], clique[MAX_HOST_NAME], *c;
	int dstPresent, srcPresent;

	/* the first clique in the history it's the one I'm starting
	 * from. I need its data */
	if (!GETTOK(clique, *history, ",", NULL)) {
		ABORT("FindCliques: empty history!\n");
	}

	/* I need the member of the starting clique */
	c = m1 = NULL;
	NWSAPI_ForEachObject(cliques, obj) {
		c = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		if (strcmp(c, clique) == 0) {
			m1 = NwsAttributeValue_r(FindNwsAttribute(obj, "member"));
		}
		FREE(c);
		if (m1) {
			break;
		}
	}
	/* some basic check */
	if (m1 == NULL) {
		ABORT("FindCliques: clique with no members!\n");
	}

	/* now let's do real job */
	NWSAPI_ForEachObject(cliques, tmp) {
		/* we need the name of the current clique */
		c = NwsAttributeValue_r(FindNwsAttribute(tmp, "name"));

		/* now let's check we are not getting into a loop */
		dstPresent = 0;
		for (m2 = *history; GETTOK(host, m2, ",", (const char **)&m2);){
			if (strcmp(host, c) == 0) {
				dstPresent = 1;
				break;
			}
		}
		if (dstPresent) {
			FREE(c);
			continue;
		}

		/* we need to know if dst and/or src are in the member
		 * list. We can't use strstr because we can pick more
		 * than what we'd like */
		dstPresent = srcPresent = 0;
		common = NwsAttributeValue_r(FindNwsAttribute(tmp,"member"));
		for (m2 = common; GETTOK(host, m2, ",", (const char**)&m2); ) {
			if (NWSAPI_HostCompare(host, dst)) {
				dstPresent = 1;
			}
			if (NWSAPI_HostCompare(host, src)) {
				srcPresent = 1;
			}
		}
		/* if src is present we don't deal with the clique: it
		 * will be dealt later. */
		if (srcPresent) {
			FREE(common);
			FREE(c);
			continue;
		}

		/* we'll need to know the common hosts */
		CommonHosts(m1, common, &m2);
		FREE(common);

		/* we pick now only one common host: if dst in common, we
		 * prefer it */
		host[0] = '\0';
		srcPresent = 0;
		if (dstPresent && m2) {
			for (common = m2; GETTOK(host, common, ",", (const char **)&common);) {
				if (NWSAPI_HostCompare(dst, host)) {
					srcPresent = 1;
					break;
				}
			}
		}
		if (!srcPresent) {
			GETTOK(host, m2, ",", NULL);
		}
		FREE(m2);

		/* If there are no common hosts, we can't do anything. */
		if (host[0] == '\0') {
			FREE(c);
			continue;
		}

		/* we got the common hosts: it's our destination in tmp? */
		if (!dstPresent) {
			ObjectSet toAdd;
			Object *o;

			/* nope we need to loop deeper */
			if (level <= 0) {
				/* well, I guess not this time */
				FREE(c);
				continue;
			}

			/* we need to call FindCliques: let's adjust the
			 * history now */
			common = MALLOC(strlen(*history) + strlen(c) + 2, 1);
			sprintf(common, "%s,%s", c, *history);
			FREE(*history);
			*history = common;

			toAdd = NewObjectSet();
			FindCliques(cliques, src, dst, history, level - 1, &toAdd);

			/* let's fix the returned path and add it to the
			 * real returned path */
			NWSAPI_ForEachObject(toAdd, obj) {
				o = AddHostClique(obj, c, host, 1);
				AddObject(path, *o);
				FreeObject(o);
			}
			FreeObjectSet(&toAdd);
		} else {
			/* we found a new path: let's create a new object
			 * with it. We add the destination host too */
			m2 = MALLOC(strlen(host) + strlen(dst) + 2, 1);
			sprintf(m2, "%s,%s", host, dst);
			obj = NewObject();
			AddNwsAttribute(&obj, "hosts", m2);
			FREE(m2);

			/* now let's add the target clique */
			common = NwsAttributeValue_r(FindNwsAttribute(tmp, "name"));
			AddNwsAttribute(&obj, "cliques", common);
			FREE(common);

			/* add the new object to the new
			 * object set */
			AddObject(path, obj);
			FreeObject(&obj);
		}
		FREE(c);
	}
	FREE(m1);

	return;
}

/* utility function for FindCliquesPath: find a host belonging to one
 * clique. We preferred hosts that belong *only* to that clique */
static int
UniqueHost(	const char *clique,
		const char *blacksheeps,
		NWSAPI_ObjectSet objs,
		char *where) {
	NWSAPI_Object obj;
	char *members, 
	     	h[MAX_HOST_NAME],
	     	h1[MAX_HOST_NAME],
		*tmp;
	const char *s;
	int done;

	/* let's find the list of members of this clique */
	members = NULL;
	NWSAPI_ForEachObject(objs, obj) {
		tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		if (tmp == NULL) {
			continue;
		}

		if (strcmp(clique, tmp) == 0) {
			members = NwsAttributeValue_r(FindNwsAttribute(obj, "member"));
		}
		FREE(tmp);
		if (members != NULL) {
			break;
		}
	}
	if (members == NULL) {
		ERROR("UniqueHost: couldn't find clique or clique members\n");
		return 0;
	}

	/* now let's see if we can find the unique host */
	where[0] = '\0';
	for (tmp = members; GETTOK(h, tmp, ",", (const char **)&tmp); ) {
		char *c;

		strcpy(where, h);
		NWSAPI_ForEachObject(objs, obj) {
			c = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
			if (strcmp(c, clique) == 0) {
				FREE(c);
				continue;
			}
			c = NwsAttributeValue_r(FindNwsAttribute(obj, "member"));

			/* we need again to use HostCompare because of
			 * our damn CS dept and the broken DNS */
			for (s = c; GETTOK(h1, s, ",", &s); ) {
				if (NWSAPI_HostCompare(h, h1)) {
					where[0] = '\0';
					break;
				}
			}
			FREE(c);
			if (where[0] == '\0') {
				break;
			}
		}
		if (where[0] != '\0') {
			/* ok we got the host */
			break;
		}
	}
	if (where[0] == '\0') {
		/* we didn't get the unique host: let's just pick
		 * whatever it's not in the blacksheeps list */
		for (tmp = members; GETTOK(h, tmp, ",", (const char **)&tmp);) {
			done = 1;
			for (s = blacksheeps; GETTOK(h1, s, ",", &s); ) {
				if (NWSAPI_HostCompare(h, h1)) {
					done = 0;
					break;
				}
			}
			if (done) {
				/* h is not in the blacksheeps, we are
				 * done */
				strcpy(where, h);
				break;
			}
		}
	}
	FREE(members);

	return (where[0] != '\0');
}

int
NWSAPI_FindCliquesPath(	const char *c1,
			const char *c2,
			NWSAPI_ObjectSet objs,
			NWSAPI_ObjectSet *path,
			int level) {
	char src[MAX_HOST_NAME],
		dst[MAX_HOST_NAME],
		*m, *m1, *c;
	NWSAPI_Object obj;
	NWSAPI_ObjectSet newSet;
	int ret;

	/* sanity check */
	if (c1 == NULL || c2 == NULL || path == NULL || c1[0] == '\0' 
			|| c2[0] == '\0' || objs == NULL) {
		ERROR("FindCliquesPath: NULL parameters\n");
		return 0;
	}
	if (strcmp(c1, c2) == 0) {
		ERROR("FindCliquesPath: I need 2 different cliques\n");
		return 0;
	}
	if (level <= 0 && level > 20) {
		ERROR("FindCliquesPath: level out of allowed limits\n");
		return 0;
	}

	/* let's get the unique hosts for both cliques. We have to use
	 * some blacksheeps, that is host we don't want to use: hosts in
	 * either cliques qualifies as such. */
	m = m1 = c = NULL;
	NWSAPI_ForEachObject(objs, obj) {
		c = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		if (strcmp(c, c1) == 0) {
			m = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(obj, "member"));
		}
		if (strcmp(c, c2) == 0) {
			m1 = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(obj, "member"));
		}
		FREE(c);
		if (m && m1) {
			break;
		}
	}
	if (UniqueHost(c1, m1, objs, src) && UniqueHost(c2, m, objs, dst)) {
		ret = NWSAPI_FindHostsPath(src, dst, objs, path, level);
		if (ret) {
			/* we need to be sure we got c1 and c2 in the
			 * list: this happens on pathological cases, that
			 * is with cliques with very few hosts and so no
			 * unique host can be found. We so add them to
			 * the list (this is extremely important for the
			 * Tree of dependencies we want to build */
			newSet = NewObjectSet();
			NWSAPI_ForEachObject(*path, obj) {
				const char *tmp;
				Object *o, *o1;

				c = NwsAttributeValue_r(FindNwsAttribute(obj, "cliques"));

				/* let's check the head first */
				o = NULL;
				GETTOK(src, c, ",", NULL);
				if (NWSAPI_HostCompare(src, c1)) {
					/* c1 isn't the first clique: we
					 * need to add it */
					o = AddHostClique(obj, c1, m, 1);
				}

				/* now let's check the tail */
				for (tmp = c; GETTOK(dst, tmp, ",", &tmp); ) {
					strcpy(src, dst);
				}
				if (NWSAPI_HostCompare(src, c2)) {
					/* c2 is not the end clique */
					if (o) {
						o1 = AddHostClique(*o,c2,m1,0);
						FreeObject(o);
						o = o1;
					} else {
						o = AddHostClique(obj,c2,m1,0);
					}
				}
				FREE(c);

				/* let's add the object */
				if (o) {
					AddObject(&newSet, *o);
					FreeObject(o);
				} else {
					AddObject(&newSet, obj);
				}
			}
			FreeObjectSet(path);
			*path = newSet;
		}
	} else {
		INFO("FindCliquesPath: problems looking for members!\n");
		ret = 0;
	}
	FREE(m);
	FREE(m1);

	return ret;
}

int
NWSAPI_FindHostsPath(	const char *h1,
			const char *h2,
			NWSAPI_ObjectSet objs,
			NWSAPI_ObjectSet *path,
			int level) {
	NWSAPI_ObjectSet result;
	NWSAPI_Object obj, tmpObj;
	char *c, *tmp, host[MAX_HOST_NAME];
	const char *member;
	int srcPresent;

	/* sanity check */
	if (h1 == NULL || h2 == NULL || path == NULL || h1[0] == '\0' 
			|| h2[0] == '\0' || objs == NULL) {
		ERROR("FindHostsPath: NULL parameters\n");
		return 0;
	}
	if (strcmp(h1, h2) == 0) {
		ERROR("FindHostsPath: I need 2 different hosts\n");
		return 0;
	}
	if (level <= 0 && level > 20) {
		ERROR("FindHostsPath: level out of allowed limits\n");
		return 0;
	}

	/* let's find the starting clique */
	*path = NULL;
	result = NewObjectSet();
	NWSAPI_ForEachObject(objs, obj) {
		ObjectSet tmpSet;

		tmp = NwsAttributeValue_r(NWSAPI_FindNwsAttribute(obj, "member"));
		if (tmp == NULL) {
			continue;
		}

		/* we need to see if src is in the member list */
		srcPresent = 0;
		for (member = tmp; GETTOK(host, member, ",", &member); ) {
			if (NWSAPI_HostCompare(host, h1)) {
				srcPresent = 1;
				break;
			}
		}
		FREE(tmp);

		/* if h1 (the source) is not present this is not a
		 * starting point */
		if (!srcPresent) {
			continue;
		}

		/* let's fix the history and find the path */
		tmpSet = NewObjectSet();
		tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		FindCliques(objs, h1, h2, &tmp, level - 1, &tmpSet);
		FREE(tmp);

		/* if we got something we need to add our
		 * host/clique to the found path and add it to the
		 * returned path */
		c = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
		NWSAPI_ForEachObject(tmpSet, tmpObj) {
			Object *o;

			o = AddHostClique(tmpObj, c, h1, 1);
			AddObject(&result, *o);
			FreeObject(o);
		}
		FREE(c);
		FreeObjectSet(&tmpSet);
	}

	if (result != NULL && strlen(result) > 2) {
		*path = result;
	}

	return (*path != NULL);
}

int
NWSAPI_FindCliquesTree(	char ***cliques,
			int **tree) {
	NWSAPI_ObjectSet objs, path;
	NWSAPI_Object obj;
	char *tmp, clique[MAX_HOST_NAME];
	int howMany, i, j, z, t;
	const char *ptr;

	*cliques = NULL;
	howMany = 0;

	/* let's save the clique names */
	if (NWSAPI_GetObjects(NWSAPI_CLIQUES, &objs)) {
		NWSAPI_ForEachObject(objs, obj) {
			tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
			*cliques = (char **)REALLOC(*cliques, sizeof(char *)*(howMany+1), 1);
			(*cliques)[howMany] = tmp;
			howMany++;
		}
	}
	
	/* allocate space for the matrix of 'child of' */
	*tree = MALLOC(sizeof(int) * howMany * howMany, 1);
	bzero((*tree), sizeof(int) * howMany * howMany);

	/* to find out the dependencies we use FindCliquesPath */
	for (i = 0; i < howMany; i++) {
		for (j = 1 + i; j < howMany; j++) {
			path = NULL;
			
			if (!NWSAPI_FindCliquesPath((*cliques)[i], (*cliques)[j], objs, &path, 2)) {
				continue;
			}

			NWSAPI_ForEachObject(path, obj) {
				tmp = NwsAttributeValue_r(FindNwsAttribute(obj, "cliques"));
				/* let's count the cliques */
				for(z = 0, ptr = tmp; GETTOK(clique, ptr, ",", &ptr);) {
					z++;
				}
				if (z != 3) {
					/* we need 3 cliques to
					 * have a superClique */
					continue;
				}

				/* let's go again, we got a super
				 * clique */
				for(z = 0, ptr = tmp; GETTOK(clique, ptr, ",", &ptr); z++) {
					if (z != 1) {
						continue;
					}

					/* we need the index of the super
					 * clique */
					for (t = 0; t < howMany; t++) {
						if (strcmp(clique, (*cliques)[t]) == 0) {
							break;
						}
					}
					if (t >= howMany) {
						ABORT("non-existent clique\n");
					}
					(*tree)[i + t*howMany] = 1;
					(*tree)[j + t*howMany] = 1;
				}
				FREE(tmp);
			}
			FreeObjectSet(&path);
		}
	}
	FreeObjectSet(&objs);

	return howMany;
}

void
NWSAPI_RelaxHostNameCheck(int r) {
	relax = r;
}

int
NWSAPI_GetCliquesForecasts(	const char *res,
				const char *clique_name,
				char **h,
				int mode,
				int level,
				unsigned long valid,
				NWSAPI_ForecastCollection **col,
				int tOut) {
	int ret, howMany, len;
	char *hosts, clique[MAX_HOST_NAME], *tmp;
	const char *ptr, *ptr1;

	/* only h needs to be checked here */
	if (h == NULL) {
		FAIL("GetCliqueForecasts: NULLL parameter!\n");
	}

	/* we might have more then one clique */
	*h = hosts = NULL;
	len = howMany = 0;
	for (ptr = clique_name; GETTOK(clique, ptr, ",", &ptr); ) {
		/* let's get the clique's member */
		ret = NWSAPI_GetCliqueHosts(clique, &hosts);
		if (ret == 0) {
			continue;
		}
		howMany += ret;
		for (ptr1 = hosts; GETTOK(clique, ptr1, ",", &ptr1); ) {
			if (relax) {
				/* we have problems with the DNS so we
				 * avoid using domain name */
				tmp = strchr(clique, '.');
				if (tmp) {
					*tmp = '\0';
				}
				tmp = strchr(clique, ':');
				if (tmp) {
					*tmp = '\0';
				}
			}
			*h = REALLOC(*h, len + strlen(clique) + 2, 1);
			if (len > 0) {
				len++;
				strcat(*h, ",");
			}
			memcpy(*h + len, clique, strlen(clique) + 1);
			len += strlen(clique);
		}
		FREE(hosts);

	}
	if (howMany == 0) {
		return 0;
	}

	/* now let's get the data */
	if (howMany < 2) {
		FREE(*h);
		ERROR("GetCliqueForecast: not enough member in the clique!\n");
		ret = 0;
	} else {
		/* now let's get the data */
		ret = NWSAPI_GetVirtualForecasts(res, NULL, *h, mode, level, valid, col, tOut);
	}

	return ret;
}

int
NWSAPI_StartCliqueFetching(	const NWSAPI_HostSpec *proxy,
				const char *clique_name,
				const int timeOut) {
	struct host_cookie tmp;
	int ret;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (proxy == NULL || clique_name == NULL || clique_name[0] == '\0') {
		ERROR("StartCliqueFetching: NULL parameters\n");
		return 0;
	}

	/* let's connect to the proxy */
	MakeHostCookie(proxy->machineName, proxy->machinePort, &tmp);
	if (!ConnectToHost(&tmp, NULL)) {
		FAIL1("StartCliqueFetching: failed to connect to %s\n", HostCImage(&tmp));
	}

	/* send the message */
	des.repetitions = strlen(clique_name);
	if (!SendMessageAndData(tmp.sd, START_ACTIVITY_FETCHING, clique_name, &des, 1, timeOut)) {
		DROP_SOCKET(&tmp.sd);
		FAIL1("StartCliqueFetching: failed to talk %s\n", HostCImage(&tmp));
	}

	if (!RecvMessage(tmp.sd, PROXY_ACK, (size_t *)(&ret), timeOut)) {
		DROP_SOCKET(&tmp.sd);
		FAIL("StartCliqueFetching: failed to receive ack\n");
	}
	DROP_SOCKET(&tmp.sd);

	return 1;
}

int
NWSAPI_StartFetching(	const NWSAPI_HostSpec *proxy,
			const char *resource,
			const char *opts,
			const char *hosts,
			const int timeOut) {
	struct host_cookie tmp;
	char *request;
	size_t i;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);

	/* sanity check */
	if (proxy == NULL || resource == NULL || hosts == NULL) {
		ERROR("StartFetching: NULL parameter!\n");
		return 0;
	}

	/* let's connect to the proxy */
	MakeHostCookie(proxy->machineName, proxy->machinePort, &tmp);
	if (!ConnectToHost(&tmp, NULL)) {
		FAIL1("StartFetching: failed to connect to %s\n", HostCImage(&tmp));
	}

	des.repetitions = BuildProxyRequest(resource, opts, hosts, &request);
	if (des.repetitions == 0) {
		DROP_SOCKET(&tmp.sd);
		FAIL("StartFetching: failed to build request\n");
	}

	/* let's send the request */
	i = SendMessageAndData(tmp.sd, START_FETCHING, request, &des, 1, timeOut);
	FREE(request);
	if (!i) {
		DROP_SOCKET(&tmp.sd);
		FAIL1("StartFetching: failed to talk %s\n", HostCImage(&tmp));
	}

	if (!RecvMessage(tmp.sd, PROXY_ACK, &i, timeOut)) {
		DROP_SOCKET(&tmp.sd);
		ERROR("StartFetching: failed to receive ack\n");
	}
	DROP_SOCKET(&tmp.sd);

	return 1;
}
