(*
 * Graph: generic graph library
 * Copyright (C) 2004
 * Sylvain Conchon, Jean-Christophe Filliatre and Julien Signoles
 * 
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2, as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * See the GNU Library General Public License version 2 for more details
 * (enclosed in the file LGPL).
 *)

(* $Id: classic.mli,v 1.11 2004/02/20 14:37:40 signoles Exp $ *)

(** Some classic graphs *)

module type S = sig

  type graph 

  val divisors : int -> graph
    (** [divisors n] builds the graph of divisors. 
      Vertices are integers from [2] to [n]. [i] is connected to [j] if
      and only if [i] divides [j]. Raises [Invalid_argument] is [n < 2]. *)

  val de_bruijn : int -> graph
    (** [de_bruijn n] builds the de Bruijn graph of order [n].
      Vertices are bit sequences of length [n] (encoded as their
      interpretation as binary integers). The sequence [xw] is connected
      to the sequence [wy] for any bits [x] and [y] and any bit sequence 
      [w] of length [n-1]. 
      Raises [Invalid_argument] is [n < 1] or [n > Sys.word_size-1]. *)

  val vertex_only : int -> graph
    (** [vertex_only n] builds a graph with [n] vertices and no edge. *)

  val full : ?self:bool -> int -> graph
    (** [full n] builds a graph with [n] vertices and all possible edges.
      The optional argument [self] indicates if loop edges should be added
      (default value is [true]). *)

end

module P (G : Sig.P with type V.label = int) : S with type graph = G.t
  (** Classic Persistent Graphs *)

module I (G : Sig.I with type V.label = int) : S with type graph = G.t
  (** Classic Imperative Graphs *)
