%%NAME
%%  eshadois - draw scaled shadow image of a matrix 
%%
%%SYNOPSIS
%%  [x colorMapNew]=eshadois(matrix[,colorMap[,legendOrientation[,legendScale]]])
%%
%%PARAMETER(S)
%%  matrix             matrix for image 
%%  colorMap           define own colormap  
%%  legendOrientation  side of the image where the legend appears 
%%                     character 's'(south),'n'(north),'w'(west) or 'e'(east) 
%%                     (default orientation is east)
%%  legendScale        scale vector of legend [start step end]
%%                     special cases of scale vector are:
%%                     if start=0 and end=0 then autorange=on 
%%                     if step=0 then autoscale=on
%%                     (default scale vector=[0 0 0])
%%
%%  if the next return parameters are used then no output
%%  x                  scaled shadow image matrix
%%  colorMapNew        colormap of x
%%
%%GLOBAL PARAMETER(S)
%%  eImageLegendScale
%%  eImageDefaultColorMap
%%  eImageLegendVisible 
%%  eImageLegendWidth
%%  eImageLegendHeight
%%  ePlotAreaPos
%%  ePlotAreaWidth
%%  ePlotAreaHeight
%%  eAxesTicLongLength
%%  eAxesValueSpace
%%  eAxesValueFontSize
%%  eAxesLabelFontSize
%%  eAxesLabelTextFont
%%  eAxesColor
%%  eAxesLineWidth
%%  eAxesTicShortLength
%%  eAxesTicLongLength
%%  eAxesTicLongMaxN
%%  eAxesValueSpace
%%  eImageLegendPos
%%  eImageLegendLabelDistance
%%  eImageLegendValueFormat
%%  eImageLegendVisible,...
%%  eImageLegendLabelText
%%  eYAxisWestScale
%%  eXAxisSouthScale
% written by Stefan Mueller stefan.mueller@fgan.de 1999

function [x,colorMapNew]=eshadois(matrix,colorMap,legendOrientation,legendScale)
  if nargin >4
    usage('[x colorMapNew]=eshadois(matrix[,coloMap[,legendOrientation[,legendScale]]])');
  end
  eglobpar;
  if nargin<4
    legendScale=eImageLegendScale;
  end
  if nargin<3
    legendOrientation='e';
  end
  if nargin<2
    colorMap=ecolors(eImageDefaultColorMap);
  end
  if nargin==0
    x=-3*pi:0.1:3*pi;
    [a b]=meshgrid(x,x);
    R=sqrt(a.^2+b.^2) + eps;
    matrix=sin(R)./R;
    colorMap=[1 1 0;1 0 0;0 1 0;0 0 1;1 0 1];
  end

  if legendScale(1)==legendScale(3)
    minval=min (min (matrix));
    maxval=max (max (matrix));
  else
    minval=legendScale(1);
    maxval=legendScale(3);
  end
  legendScale(1)=minval;
  legendScale(3)=maxval;
  [x colorMapNew]=eshadoi(matrix,colorMap);	
  
  if nargout==0
    
    % color legend
    if eImageLegendVisible & ...
       sum(2*colorMap(:,1)-colorMap(:,2)-colorMap(:,3))
      if eImageLegendWidth==0
        if legendOrientation=='s' | ...
           legendOrientation=='n'
          xl=ePlotAreaWidth;
          yl=eImageLegendHeight;
          scaleLength=xl;
        else
          yl=ePlotAreaHeight;
          xl=eImageLegendHeight;
          scaleLength=yl;
        end
      else 
        if legendOrientation=='s' | ...
           legendOrientation=='n'
          xl=eImageLegendWidth;
          yl=eImageLegendHeight;
          scaleLength=xl;
        else
          xl=eImageLegendHeight;
          yl=eImageLegendWidth;
          scaleLength=yl;
        end
      end
      scaleSpace=eAxesTicLongLength+eAxesValueSpace+eAxesValueFontSize;
      if legendOrientation=='s'
       colorImage=(1:length(colorMap));
       legendPosX=ePlotAreaPos(1)+eImageLegendPos(1);
       legendPosY=ePlotAreaPos(2)+eImageLegendPos(2);
       scalePosX=legendPosX;
       scalePosY=legendPosY;
       textPosX=scalePosX+scaleLength/2;
       textPosY=scalePosY-scaleSpace-eAxesLabelFontSize*0.72-...
                eImageLegendLabelDistance;
       textAngle=0;
      elseif legendOrientation=='n'
       colorImage=(1:length(colorMap));
       legendPosX=(ePlotAreaPos(1)+eImageLegendPos(1));
       legendPosY=(ePlotAreaPos(2)+ePlotAreaHeight-eImageLegendPos(2));
       scalePosX=legendPosX;
       scalePosY=legendPosY+yl;
       textPosX=scalePosX+scaleLength/2;
       textPosY=scalePosY+scaleSpace+yl+eAxesLabelFontSize*0.72+...
                eImageLegendLabelDistance;
       textAngle=0;
      elseif legendOrientation=='e'
       colorImage=(length(colorMap):-1:1)';
       legendPosX=(ePlotAreaPos(1)+ePlotAreaWidth-eImageLegendPos(2));
       legendPosY=(ePlotAreaPos(2)+eImageLegendPos(1));
       scalePosX=legendPosX+xl;
       scalePosY=legendPosY;
       textPosX=scalePosX+scaleSpace+xl+eAxesLabelFontSize*0.72+...
                eImageLegendLabelDistance;
       textPosY=scalePosY+scaleLength/2;
       textAngle=90;
      elseif legendOrientation=='w'
       colorImage=(length(colorMap):-1:1)';
       legendPosX=(ePlotAreaPos(1)+eImageLegendPos(2)-xl);
       legendPosY=(ePlotAreaPos(2)-eImageLegendPos(1));
       scalePosX=legendPosX;
       scalePosY=legendPosY;
       textPosX=scalePosX-scaleSpace-eAxesLabelFontSize*0.72-...
                eImageLegendLabelDistance;
       textPosY=scalePosY+scaleLength/2;
       textAngle=90;
      end
      eimagexy(eFile,colorImage,colorMap,...
               legendPosX*eFac,legendPosY*eFac,xl*eFac,yl*eFac);
      erect(eFile,legendPosX*eFac,legendPosY*eFac,...
            xl*eFac,yl*eFac,eAxesLineWidth*eFac,[0 0 0],0);
      escalexy(eFile,legendOrientation,...
               scalePosX*eFac,scalePosY*eFac,0,0,...
               scaleLength*eFac,...
               legendScale(1),...
               legendScale(2),...
               legendScale(3),...
               eImageLegendValueFormat,...
               eImageLegendValueVisible,...
               eAxesValueFontSize*eFac,...
               eAxesLineWidth*eFac,...
               eAxesTicShortLength*eFac,...
               eAxesTicLongLength*eFac,...
               eAxesTicLongMaxN,...
               eAxesValueSpace*eFac,...
               eAxesColor);
      if strcmp(eImageLegendLabelText,'')~=1
        etext(eImageLegendLabelText,...
               textPosX,...
               textPosY,...
               eAxesLabelFontSize,0,eAxesLabelTextFont,textAngle);  
      end
    end    
  
    %image
    eimage(x,colorMapNew);	
  
    % image axes
    if eYAxisWestScale(1)==eYAxisWestScale(3)
      eYAxisWestScale=[0 0 size(matrix,1)];
    end
    if eXAxisSouthScale(1)==eXAxisSouthScale(3)
      eXAxisSouthScale=[0 0 size(matrix,2)];
    end
    egrid;
    eaxes;
  end
