%% Copyright (C) 2009 Carlo de Falco
%% 
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%% 
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%% 
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

%% -*- texinfo -*-
%% @deftypefn {Function File} {[@var{dRdx}, @var{dRdy}]=} nrbbasisfungradient (@{@var{dzdu}, @var{dzdv}@}, @{@var{dxdu}, @var{dydu}, @var{dxdv}, @var{dydv}@})
%% @deftypefnx {Function File} {[@var{dRdx}]=} nrbbasisfungradient (@var{dzdu}, @var{dxdu}, @var{nrb})
%% NRBBASISFUNGRADIENT Compute the gradient of the basis functions of a NURBS surface at the
%% specified parametric points.
%%
%% INPUT:
%% @itemize @minus
%% @item @var{dzdu}, @var{dzdv}: basis function derivatives with respect
%% to parameters @code{u} and @code{v}
%% @item @var{dxdu}, @var{dydu}, @var{dxdv},
%% @var{dydv}: NURBS geometry map derivatives
%% @var{nrb}:  NURBS structure describing the surface
%% @end itemize
%%
%% OUTPUT:
%% @itemize @minus
%% @item @var{dRdx} derivative of the basis functions with respect to
%% the @code{x} direction
%% @item @var{dRdy} derivative of the basis functions with respect to
%% the @code{y} direction
%% @end itemize
%%
%% @seealso{nrbbasisfunder,nrbbasisfun,nrbderiv}
%% @end deftypefn

%% Author: Carlo de Falco <carlo@guglielmo.local>
%% Created: 2009-04-29

function [varargout]  = nrbbasisfungradient (dz, N, nrb)

  if (iscell(dz))

    dzdu=dz{1}; 
    dzdv=dz{2}; 

    X = squeeze(nrb.coefs(1,:,:)./nrb.coefs(4,:,:));
    Y = squeeze(nrb.coefs(2,:,:)./nrb.coefs(4,:,:));

    nfun = size(N,2);
    tmp = sum(X(N).*dzdu, 2);
    dxdu= tmp(:, ones(nfun,1)); 
    tmp = sum(Y(N).*dzdu, 2);
    dydu= tmp(:, ones(nfun,1));
    tmp = sum(X(N).*dzdv, 2);
    dxdv= tmp(:, ones(nfun,1));
    tmp = sum(Y(N).*dzdv, 2);
    dydv= tmp(:, ones(nfun,1));
    
    detjac = dxdu.*dydv - dydu.*dxdv;
    
    varargout{1} = ( dydv .* dzdu - dydu .*dzdv)./detjac;
    varargout{2} = (-dxdv .* dzdu + dxdu .*dzdv)./detjac;
    %%keyboard
  elseif (~iscell(dz) && (nargout==1))

    varargout{1} = dzdu ./ dxdu;

  else
    
    print_usage();
    
  end

end

%!test
%! p = 2;   q = 3;   m = 4; n = 5;
%! Lx  = 1; Ly  = 1; 
%! nrb = nrb4surf   ([0 0], [Lx 0], [0 Ly], [Lx Ly]);
%! nrb = nrbdegelev (nrb, [p-1, q-1]);
%! nrb = nrbkntins  (nrb, {linspace(0,1,m)(2:end-1), linspace(0,1,n)(2:end-1)});
%! nrb.coefs (4,:,:) += rand (size (nrb.coefs (4,:,:)));
%! 
%! [u(2,:,:), u(1,:,:)] = meshgrid(rand (1, 20), rand (1, 20));
%! 
%! 
%! uv (1,:) = u(1,:,:)(:)';
%! uv (2,:) = u(2,:,:)(:)';
%! 
%! [dzdu, dzdv, connect] =  nrbbasisfunder (uv, nrb);
%! nd        = nrbderiv(nrb);
%! [ndp, dp] = nrbdeval(nrb, nd, uv);
%! 
%! [dzdx, dzdy]= nrbbasisfungradient ({dzdu, dzdv}, connect, nrb);
%! assert(norm(sum(dzdx, 2), inf), 0, 1e-10)
%! assert(norm(sum(dzdy, 2), inf), 0, 1e-10)