## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_contour(@var{x},@var{y},@var{z},[@var{prop},@var{val}])
## Displays a contour plot defined by @var{x},@var{y},@var{z} matrices,
## where @var{x} and @var{y} are typically formed by meshgrid.
##
## Optional @var{prop},@var{val} arguments can be used to change
## properties of the plot.  Currently, valid properties are
## ";NConts;".
##
## @end deftypefn
## @seealso{vtk_contourf, vtk_surf}

## Author: Dragan Tubic


function vtk_contour( varargin )
  
  valid_props = ";NConts;";
  
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, all_va_args);
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
    ncts = properties.NConts;  # number of contours;
  else
    properties.empty = 1;
    ncts=10;
  end

  if ( no_numerical_params < 1 | no_numerical_params > 3 )
    error("Syntax is vtk_contour( x, (y), (z))");
  end
  
  if ( no_numerical_params == 1 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = z;
  elseif ( no_numerical_params == 2 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = nth (varargin,1);
  elseif ( no_numerical_params == 3 )
    x = nth (varargin,1);
    y = nth (varargin,2);
    z = nth (varargin,3);
    c = z;
  end
  
  ## form the array that specifies connecting points
  [nr nc] = size(x);
  m=nc; n=nr;
  i=0:m-2;
  j=0:n-2;
  [i,j]=meshgrid(i,j);
  i=i(:); j=j(:);
  t=[i*n+j,(i+1)*n+j,(i+1)*n+(j+1) ; (i+1)*n+(j+1),i*n+(j+1),i*n+j];

  x = x(:);
  y = y(:);
  z = z(:);
  c = c(:);
  
  
  vtk_init;	
  
  if ( length(x) != length(y) | length(x) != length(z) | length(x) != length(c) )
    error("Lengths of all three coordinates have to be the same.");
  end
  
  %% We'll create the building blocks of polydata including data attributes.
  surface = vtkPolyData();
  points   = vtkPoints();
  polys    = vtkCellArray();
  scalars  = vtkFloatArray();
  
  %% Load the point, cell, and data attributes.
  %% for (i=0; i<8; i++) points->InsertPoint(i,x[i]);
  coords = vtkFloatArray; 
  coords.SetNumberOfTuples( length(x) );
  coords.SetNumberOfComponents(3);
  pts = [x y z]';
  pts(3,:) = min(z) - max( [max(x)-min(x) max(y)-min(y) max(z)-min(z) ] )/4;  #puts the contour lines in one plane
  coords.SetArray( pts(:), 3*length(x), 0 );
  points.SetData(coords);
  
  no_tris = length(t);
  t = [ones(no_tris,1)*3 t]';
  ptids = vtkIdTypeArray;
  ptids.SetArray( t(:), no_tris*4, 0 );
  polys.SetCells( no_tris, ptids );

  ## manually create color map table instead; a must if later add a scalar bar
  lut = vtkLookupTable();
  lut.SetHueRange(0.66667, 0.0);
  lut.SetNumberOfColors(256);
  lut.SetRampToLinear();
  lut.Build();
  scalars.SetArray( c, length(c) , 0 );
  
  %% We now assign the pieces to the vtkPolyData.
  surface.SetPoints(points);
  surface.SetPolys(polys);
  surface.GetPointData().SetScalars(scalars);
  
  contours = vtkContourFilter();
  contours.SetInput(surface)
  contours.GenerateValues(ncts, min(c), max(c))
  
  %% The contour lines are mapped to the graphics library.
  contMapper = vtkPolyDataMapper();
  contMapper.SetInput(contours.GetOutput())
  contMapper.SetLookupTable(lut);            % use manually created color map table
  contMapper.SetScalarRange(min(c),max(c));  % relative scaling
  
  contActor = vtkActor();
  contActor.SetMapper(contMapper)

  f = vtk_figure(0);
  f.renderer.AddActor(contActor);
  %f.axes.YAxisVisibilityOff();  # turn off the "Z" axes  (but use YAxis command!?) - must be a bug, JJS 7/23/05
  vtk_update(f);
  
endfunction
