## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_print(@var{filename},@var{options})
## "Prints" to a file.  Default output is
## postscript if not specified.
##
## @var{options}:
## @table @code
## @item -d@var{device}
##   Output device, where @var{device} is one of:
##   @table @code
##   @itemx ps
##   @itemx eps
##   @itemx pdf
##     Postscript or pdf type output
##   @itemx tiff
##   @itemx jpeg
##   @itemx png
##     Image output
##   @itemx rib
##     Renderman output
##   @end table
## @item -vect
## Use only vector graphic output for postscript/pdf.  Default is to
## write 3D props as an image.  Warning:  incorrect polygon placement
## and/or huge filesize is probable with this option.
## @item -tex
##   Write text to a tex file and exclude text from postscript/pdf.
## @item -nocompress
##   Do not compress images in pdf and tiff output.
## @item -qualityXX
##   Set jpeg quality.  Default is 100.
## @end table
##
## The vtk window must be the top window on your display (i.e.
## unobstructed) for most of the output devices to work properly.  This
## function now attempts to raise your vtk window to the top, but your
## windows manager may intercept that call (e.g. KDE).   To prevent
## this, you should turn off so-called ``Focus stealing prevention''
## options in your windows manager.
## @end deftypefn
## @seealso{mark_as_command}

## Author: Jonathan Stickel

## Important note:  all GL2PS output results in incorrectly placed text
## when used on my desktop, but the results are OK on my laptop.  My
## guess is a buggy compile of the GL2PS libraries.

## TBD:  guess DEVICE based on file extension?

function vtk_print(varargin)

  ## process the input
  filename = 0;
  vect = 0;
  tex = 0;
  compress = 1;
  quality = 100;
  DEVICE = "ps";
  for i=1:nargin
    arg = varargin{i};
    if (!isstr(arg))
      error("Only string arguments may be given")
    elseif (arg(1) != "-")
      if (filename)
	error("Only one argument (the filename) may be given without a leading '-'")
      endif
      filename = arg;
    elseif ( length(arg)>1 && arg(1:2) == "-d" )
      DEVICE = arg(3:end);
    elseif ( strcmp(arg,"-vect") )
      vect = 1;
    elseif ( strcmp(arg,"-tex") )
      tex = 1;
    elseif ( strcmp(arg,"-nocompress") )
      compress = 0;
    elseif ( length(arg)>8 && arg(1:8)=="-quality" )
      quality = str2num(arg(9:end));
    else
      warning(["Unknown option '",arg,"' ignored"])
    endif
  endfor
  if (!filename)
    filename = "vtkoutput";
  endif

  
  f = vtk_figure(0);
  w = f.window;
  vtkRaiseWindow (w);  ## must bring the window to the top to export images correctly

  ## GL2PS export types
  if ( strcmp(DEVICE,"ps") | strcmp(DEVICE,"eps") | strcmp(DEVICE,"pdf") )
    e = vtkGL2PSExporter;
    e.SetRenderWindow(w);
    e.CompressOff();    ## compression should be off for postscript; is an option for pdf

    ## strip the extension for GL2PS output
    idx = find(filename==".");
    if ( isscalar(idx) )  # test that there is one and only one period in the filename
      filename = filename(1:idx-1);
    endif
    e.SetFilePrefix(filename)
    
    %%e.DrawBackgroundOff()  % make this an option ??
    if (vect)
      shdw = f.axes.GetAxisTitleTextProperty().GetShadow();
      f.axes.GetAxisTitleTextProperty().ShadowOff()  % otherwise "sort simple" flips the text shadow
      ##e.SetSortToSimple()  % Simple is default
      e.SetGlobalPointSizeFactor(1/3);
      e.SetGlobalLineWidthFactor(1/3);
    else
      e.Write3DPropsAsRasterImageOn()
      e.SetSortToOff()   # otherwise text shadow flipped
    endif
    if (tex)
      e.SetFileFormatToTeX()
      e.Write()
      e.TextOff()
    endif
    
    if (strcmp(DEVICE,"ps"))
      e.SetFileFormatToPS()
    elseif (strcmp(DEVICE,"eps"))
      e.SetFileFormatToEPS()
    elseif (strcmp(DEVICE,"pdf"))
      if (compress)
	e.CompressOn()
      endif
      e.SetFileFormatToPDF()
    endif
    e.Write();
    if (vect)
      f.axes.GetAxisTitleTextProperty().SetShadow(shdw);  % reset for normal viewing
    endif
    
  ## image types
  elseif ( strcmp(DEVICE,"jpeg") | strcmp(DEVICE,"tiff") | strcmp(DEVICE,"png") )
    w2if = vtkWindowToImageFilter();
    if (strcmp(DEVICE,"jpeg"))
      writer = vtkJPEGWriter();
      writer.SetQuality(quality);
    elseif (strcmp(DEVICE,"png"))
      writer = vtkPNGWriter();
    elseif (strcmp(DEVICE,"tiff"))
      writer = vtkTIFFWriter();
      if (compress)
	writer.SetCompressionToDeflate();
      else
	writer.SetCompressionToNoCompression()
      endif
    endif
    w2if.SetInput(w);
    writer.SetInput( w2if.GetOutput() );
    writer.SetFileName( filename );
    writer.Write();

  ## renderman export
  elseif ( strcmp(DEVICE,"rib") )
    rib_exporter = vtkRIBExporter();
    rib_exporter.SetInput (w);
    rib_exporter.SetPixelSamples(4,4);
    %% Apparently buggy but not sure, it segfaults ocasionaly if ExportArrays is On
    %%rib_exporter.ExportArraysOn();
    rib_exporter.SetSize(1024,768);
    rib_exporter.SetBackground(1);
    if (strcmp(filename(end-3:end),".rib"))
      filename = filename(1:end-4);
    endif
    rib_exporter.SetFilePrefix(filename);
    rib_exporter.Write ();      
  else
    warning("DEVICE specified is not supported; no output");
  endif

endfunction
