#include <odindata/fileio.h>
#include <odindata/utils.h>
#include <odindata/filter.h>

/**
  * \page miconv Conversion utility for medical images
  * \verbinclude cmdline-utils/miconv.usage
  *
  * \section miconv_examples Some examples how to use miconv:
  *
  * Read a bunch of DICOM files (format 'dcm'), produced by Siemens, from directory 'mosaic' and convert it to NIFTI file 'mosaic.nii'
  * with a dialect suitable for FSL:
  *   \verbatim
       miconv -rf dcm -rdialect siemens -wdialect fsl mosaic mosaic.nii
      \endverbatim
  *
  * Converting a short (16 bit) raw data file '2dseq' of size 256x256 to a DICOM file 'image.dcm':
  *   \verbatim
       miconv -nx 256 -ny 256 -rf short 2dseq image.dcm
      \endverbatim
  *
  *
  * Extracting the data from the 2nd slice (index 1) of Vista file 'image.v' and store it in
  * new Vista file 'slice1.v':
  *   \verbatim
       miconv -srange 1 image.v slice1.v
      \endverbatim
  */

struct MiConvOpts : JcampDxBlock {

  JDXstring protfile;
  JDXstring parlist;
  JDXbool non_verbose;

  MiConvOpts() {

    protfile.set_cmdline_option("p").set_description("Load this protcol for defaults");
    append_member(protfile,"protfile");

    parlist.set_cmdline_option("l").set_description("Lists space-separated set of protocol parameters and exits. Use '-l help' to get a list of possible parameters and '-l all' to list all parameters. The parameter 'output-file' must be omitted when using this option.");
    append_member(parlist,"parlist");

    non_verbose.set_cmdline_option("nv").set_description("be less verbose, show progress in percent only");
    append_member(non_verbose,"non_verbose");
  }

};

////////////////////////////////////////

void usage(const Protocol& prot) {
  FileReadOpts ropts;
  FileWriteOpts wopts;
  MiConvOpts mopts;
  FilterFactory filter;
  STD_cout << "miconv:  Converts medical image data between formats." << STD_endl;
  STD_cout << "         File formats are automatically identified by their file extension." << STD_endl;
  STD_cout << "Usage: miconv [options] <input-file> [<output-file>]" << STD_endl;
  STD_cout << "General options:" << STD_endl;
  STD_cout << mopts.get_cmdline_usage("\t");
  STD_cout << "Protocol options:" << STD_endl;
  STD_cout << prot.get_cmdline_usage("\t");
  STD_cout << "File read options:" << STD_endl;
  STD_cout << ropts.get_cmdline_usage("\t");
  STD_cout << "File write options:" << STD_endl;
  STD_cout << wopts.get_cmdline_usage("\t");
  STD_cout << "Filters:" << STD_endl;
  STD_cout << filter.get_cmdline_usage("\t");
  STD_cout << "Other options:" << STD_endl;
  STD_cout << "\t" << LogBase::get_usage() << STD_endl;
  STD_cout << "\t" << helpUsage() << STD_endl;
  STD_cout << "Supported file extensions(formats):" << STD_endl;
  STD_cout << FileIO::autoformats_str("\t") << STD_endl;
}


////////////////////////////////////////

int main(int argc, char* argv[]) {
  if(LogBase::set_log_levels(argc,argv)) return 0;

  Log<OdinData> odinlog("miconv","main");

  char optval[ODIN_MAXCHAR];

  Protocol prot_template;
  FileReadOpts  ropts;
  FileWriteOpts wopts;
  MiConvOpts mopts;

  if(hasHelpOption(argc,argv)) {usage(prot_template); return 0;}
  if(argc<3) {usage(prot_template); return 0;}

  // set defaults
  prot_template.seqpars.set_MatrixSize(readDirection,1);
  prot_template.seqpars.set_MatrixSize(phaseDirection,1);
  prot_template.seqpars.set_MatrixSize(sliceDirection,1);

  mopts.parse_cmdline_options(argc,argv);

  STD_string infile;
  STD_string outfile;


  if(mopts.parlist!="") {
    infile=argv[argc-1];
  } else {
    infile=argv[argc-2];
    outfile=argv[argc-1];
  }

  if(mopts.parlist=="help") {
    STD_cout << "Possible protocol parameters:" << STD_endl;
    for(unsigned int i=0; i<prot_template.numof_pars(); i++) {
      JcampDxClass& ldr=prot_template[i];
      if(ldr.get_filemode()!=exclude) {
	STD_cout <<  ldr.get_label() << STD_endl;
      }
    }
    return 0;
  }

  svector pars;
  if(mopts.parlist=="all") {
    for(unsigned int i=0; i<prot_template.numof_pars(); i++) {
      JcampDxClass& ldr=prot_template[i];
      if(ldr.get_filemode()!=exclude) pars.push_back(ldr.get_label());
    }
  } else {
    pars=tokens(mopts.parlist);
  }

  int npars=pars.size();
  if(npars) FileIO::set_trace_status(false);

  if(mopts.protfile!="") {
    ODINLOG(odinlog,infoLog) << "Loading default protocol " << optval << STD_endl;
    prot_template.load(mopts.protfile);
  }

  // override parameters of protocol file
  prot_template.parse_cmdline_options(argc,argv,false); // do not modify argc/argv on 1st pass

  FileIO::ProtocolDataMap pdmap_in;

  // Read data
  ropts.parse_cmdline_options(argc,argv);

  int readresult;
  if(mopts.non_verbose){
    ProgressDisplayConsole display;
    ProgressMeter mProgress(display);
    readresult=FileIO::autoread(pdmap_in, infile, ropts, prot_template,&mProgress);
  }
  else readresult=FileIO::autoread(pdmap_in, infile, ropts, prot_template);

  if(readresult<0) {
    ODINLOG(odinlog,errorLog) << "autoread failed" << STD_endl;
    return readresult;
  }

  // If requested, print formatted protocol properties and exit
  if(npars) {
    sarray table(npars+1,pdmap_in.size()+1);
    for(int ipar=0; ipar<npars; ipar++) {
      table(ipar,0)=pars[ipar];
    }
    table(npars,0)="Dataset";
    int imap=1;
    for(FileIO::ProtocolDataMap::const_iterator pdit=pdmap_in.begin(); pdit!=pdmap_in.end(); ++pdit) {
      for(int ipar=0; ipar<npars; ipar++) {
        table(ipar,imap)=rmblock(pdit->first.printval(pars[ipar]), "\n", ""); // truncate after first newline
      }
      STD_ostringstream oss;
      oss << pdit->second.shape();
      table(npars,imap)=oss.str();
      imap++;
    }

    STD_cout << print_table(table);

    return 0;
  }

  ODINLOG(odinlog,normalDebug) << "mem(in ): " << Profiler::get_memory_usage() << STD_endl;

  // Create copy of protocol-data map, thereby modify protocol
  FileIO::ProtocolDataMap pdmap_out;
  for(FileIO::ProtocolDataMap::const_iterator pdit=pdmap_in.begin(); pdit!=pdmap_in.end(); ++pdit) {
    Protocol prot(pdit->first);
    prot.parse_cmdline_options(argc,argv); // override parameters of infile
    pdmap_out[prot].reference(pdit->second);
  }

  ODINLOG(odinlog,normalDebug) << "mem(out): " << Profiler::get_memory_usage() << STD_endl;

  // Parse before filters to remove options
  wopts.parse_cmdline_options(argc,argv);

  FilterChain filterchain(argc,argv);

  //apply filters on pdmap_out
  if(!filterchain.apply(pdmap_out)) return -1;


  // Write data
  int result=FileIO::autowrite(pdmap_out, outfile, wopts);

  if(result>0) result=0;
  return result;
}
