/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PatchModel.h"
#include "NodeModel.h"
#include "ConnectionModel.h"
#include <cassert>
#include <iostream>

using std::cerr; using std::cout; using std::endl;

namespace LibOmClient {


void
PatchModel::path(const string& new_path)
{
	m_path = new_path;
	for (NodeModelMap::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
		(*i).second->path(m_path +"/"+ (*i).second->name());
}


NodeModel*
PatchModel::get_node(const string& name)
{
	NodeModelMap::iterator i = m_nodes.find(name);

	return ((i != m_nodes.end()) ? (*i).second : NULL);
}


void
PatchModel::remove_node(const string& name)
{
	NodeModelMap::iterator i = m_nodes.find(name);
	if (i != m_nodes.end()) {
		delete i->second;
		m_nodes.erase(i);
		return;
	}
	
	cerr << "[PatchModel::remove_node] " << m_path << ": failed to find node " << name << endl;
}


// FIXME: switch to names
void
PatchModel::rename_node(const string& old_path, const string& new_path)
{
	NodeModelMap::iterator i = m_nodes.find(OmPath::name(old_path));
	NodeModel* nm = NULL;
	
	if (i != m_nodes.end()) {
		nm = (*i).second;
		for (list<ConnectionModel*>::iterator j = m_connections.begin(); j != m_connections.end(); ++j) {
			if (OmPath::parent((*j)->src_port_path()) == old_path)
				(*j)->src_port_path(new_path +"/"+ OmPath::name((*j)->src_port_path()));
			if (OmPath::parent((*j)->dst_port_path()) == old_path)
				(*j)->dst_port_path(new_path +"/"+ OmPath::name((*j)->dst_port_path()));
		}
		m_nodes.erase(i);
		nm->path(new_path);
		m_nodes[OmPath::name(new_path)] = nm;
		return;
	}
	
	cerr << "[PatchModel::rename_node] " << m_path << ": failed to find node " << old_path << endl;

	cerr << "NODE DUMP: " << endl;
	for (NodeModelMap::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i) {
		cerr << "\t" << ((*i).second)->path() << endl;
	}
}


ConnectionModel*
PatchModel::get_connection(const string& src_port_path, const string& dst_port_path)
{
	for (list<ConnectionModel*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i)
		if ((*i)->src_port_path() == src_port_path && (*i)->dst_port_path() == dst_port_path)
			return (*i);
	return NULL;
}


void
PatchModel::add_connection(ConnectionModel* cm)
{
	ConnectionModel* existing = get_connection(cm->src_port_path(), cm->dst_port_path());

	if (existing == NULL)
		m_connections.push_back(cm);
}


void
PatchModel::remove_connection(const string& src_port_path, const string& dst_port_path)
{
	ConnectionModel* cm = NULL;
	for (list<ConnectionModel*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		cm = (*i);
		if (cm->src_port_path() == src_port_path && cm->dst_port_path() == dst_port_path) {
			delete cm;
			m_connections.erase(i);
			assert(get_connection(src_port_path, dst_port_path) == NULL); // no duplicates
			return;
		}
	}
	cerr << "[PatchModel::remove_connection] WARNING: Failed to find connection " <<
		src_port_path << " -> " << dst_port_path << endl;
	return;
}


} // namespace LibOmClient
