;;;; Emacs interface for logging hours to onShore TimeSheet. ;;;;
;;
;; timesheet.el
;;
;; Copyright 1999 -- 2001, onShore Development Inc. <URL:http://www.onshore-devel.com/>
;;
;;
;; This program is free software under the terms of the GNU General Public
;; License (GPL). A copy of the GPL, "COPYING", should have been made
;; available with this software.  If not, a copy may be obtained at 
;; http://www.fsf.org/copyleft/gpl.html
;;
;; 
;;
;; Questions to kfogel@onshore.com.
;;


;; For `comint-read-noecho':
(require 'comint)

;; Compatibility code for Emacsen which don't have base64-encode-string
(if (not (fboundp 'base64-encode-string))
    (defun base64-encode-string (str)
      (base64-encode str)))


(defvar timesheet-dir (expand-file-name "~/.timesheet")
  "*Where local timesheet files live.")

(defvar timesheet-unlogged-file (expand-file-name "hours" timesheet-dir)
  "*File containing not-yet-uploaded timesheet data.")

(defvar timesheet-error-file (expand-file-name "errors" timesheet-dir)
  "*File containing upload error details.")

(defvar timesheet-logme-file (expand-file-name "LOGGING" timesheet-dir)
  "*File containing timesheet data in the process of uploading.")

(defvar timesheet-archive-prefix (expand-file-name "logged" timesheet-dir)
  "*Prefix for saved hour logs (suffixes are dates).")

(defvar timesheet-yawn-timeout 10
  "*Number of seconds to wait for data from server before giving up.")



;; todo: use defcustom?

(defvar timesheet-host 
  nil
  "The host running timesheet services.")

(defvar timesheet-default-host
  "localhost" 
  "Default host for timesheet services.")

(defvar timesheet-list-root-url
  nil
  "Where you'd point a browser to get the client and job lists.")

(defvar timesheet-client-list-url
  nil
  "Where to find a pre-compiled list of client numbers and names.")

(defvar timesheet-job-list-url
  nil
  "Where to find a pre-compiled list of job numbers and names.")


(defun timesheet-set-dependant-vars () 
  (or timesheet-host
      (setq timesheet-host 
            (read-string "onShore TimeSheet host: " timesheet-default-host)))
  (or timesheet-list-root-url
      (setq timesheet-list-root-url 
	    (concat timesheet-host
                    (read-string "onShore TimeSheet application root: " 
                                 "/onshore-timesheet/"))))
  (or timesheet-client-list-url
      (setq timesheet-client-list-url
	    (concat "http://" 
                    timesheet-list-root-url 
                    "dump-clients-jobs.cgi?dump=client")))
  (or timesheet-job-list-url
      (setq timesheet-job-list-url 
	    (concat "http://" 
                    timesheet-list-root-url
                    "dump-clients-jobs.cgi?dump=job")))
  (timesheet-set-username-password))

      
(defvar timesheet-logger-program "timesheet-log-from.pl"
  "*Point this at the logging program.")

(defvar timesheet-pulldata-program "timesheet-get-data.pl"
  "*Point this at the data getting program")




(defun timesheet-decompose-url (url)
  ;; returns (serverhostname path), path appropriate for HTTP GET.
  (save-match-data
    ;; strip off leading trash
    (if (string-match "^https*://" url)
        (setq url (substring url (match-end 0))))
    ;; divide server from path, no path means use "/" as path
    (let* ((match (string-match "/" url))
           (path-idx (if match (1- (match-end 0))))
           (server (if path-idx (substring url 0 path-idx) url))
           (path (if path-idx (substring url path-idx) "/")))
      (list server path))))


(defun timesheet-server-from-url (url)
  (car (timesheet-decompose-url url)))

(defun timesheet-path-from-url (url)  ; todo: is this useful anymore?
  (car (cdr (timesheet-decompose-url url))))


(defun timesheet-cache-file-from-url (url)
  "Takes timesheet-{client,job}-list-url, returns the name of the
corresponding local cache file."
  (save-match-data
    (let ((significant-part (string-match "=\\([a-zA-Z]*\\)$" url)))
      (concat timesheet-dir "/" (substring url (match-beginning 1)) "s.txt"))))



(defvar timesheet-debug nil "*Set to t to record elisp->perl traffic.")

(defvar timesheet-debug-log (concat timesheet-dir "/debuglog"))

(defun timesheet-proc-sendstr (process string)
  (if timesheet-debug
      (save-excursion
        (set-buffer (find-file-noselect timesheet-debug-log))
        (goto-char (point-max))
        (insert string)
        (save-buffer)))
  (process-send-string process string))



(defun timesheet-url-or-cache-into-buffer 
  (url &optional use-local buffer)
  "Read the source text of URL (or CACHED version) into BUFFER at point.
Return BUFFER.
BUFFER omitted or nil means create a new buffer."
  (message "Reading %s..." url)
  (let* ((buf (if (buffer-live-p buffer)
                  buffer   ; just use the buffer given
                (generate-new-buffer " *timesheet*")))
         (server (timesheet-server-from-url url))
         (cache-file (timesheet-cache-file-from-url url)))
    (if use-local
        (save-excursion
          (set-buffer buf)
          (buffer-disable-undo)
          (erase-buffer)
          (insert-file cache-file)
          buf)
      (let ((proc (timesheet-start-pulldata-process 
                   timesheet-username timesheet-password url)))
	(save-excursion
	  (set-buffer (process-buffer proc))
	  (buffer-disable-undo (process-buffer proc))
	  (erase-buffer)
	  (message "Reading %s..." url)
          (while (accept-process-output proc timesheet-yawn-timeout))
          (write-file cache-file)
          (set-buffer-modified-p nil))
        (message "Reading %s...done" url)
        (process-buffer proc)))))


(defun timesheet-url-or-cache-to-list (url use-local behavior)
  ;; Parse jobs or clients into a list.
  ;; BEHAVIOR is `job' or `client'
  (let ((buf (timesheet-url-or-cache-into-buffer url use-local))
        lst)
    (save-excursion
      (set-buffer buf)
      (goto-char (point-min))
      (while (re-search-forward "[0-9]+::" nil t)
        (beginning-of-line)
        (let ((num1 (string-to-number
                     (buffer-substring
                      (point) (progn (forward-word 1) (point)))))
              (name (buffer-substring 
                     (progn (forward-char 2) (point))
                     (if (eq behavior 'client)
                         (progn (end-of-line) (point))
                       (end-of-line)
                       (search-backward "::")
                       (point))))
              (num2 (if (eq behavior 'client)
                        nil
                      (forward-char 2)
                      (string-to-number
                       (buffer-substring
                        (point) (progn (end-of-line) (point)))))))
          (if num2
              (setq lst (cons (list num1 name num2) lst))
            (setq lst (cons (list num1 name) lst)))))
      (kill-buffer buf)
      lst)))


(defun timesheet-client-list (&optional use-local)
  ;; Return list of the form ((ClientNumber ClientName) ...)
  (timesheet-url-or-cache-to-list timesheet-client-list-url use-local 'client))


(defun timesheet-job-list (&optional use-local)
  ;; Return list of the form ((JobNumber JobName ClientNumber) ...)
  (timesheet-url-or-cache-to-list timesheet-job-list-url use-local 'job))



(defvar timesheet-jobarray-cache nil
  "Obarray of clients and jobs, for `completing-read'.
Each symbol's name is the name of a client.  Each value is a
list: (JobName[string] JobNumber[num]).  Each function value is the
client number.")


;; Sets and returns global variable of same name
;; But always use this function, not the variable
(defun timesheet-jobarray (&optional use-local)
  (if timesheet-jobarray-cache
      timesheet-jobarray-cache
    ;; Else rebuild from server or from local-copy
    (let ((new-obarray (make-vector 79 nil))
          (client-list (timesheet-client-list use-local))
          (job-list (timesheet-job-list use-local)))
      (while job-list
        (let* ((this-job (car job-list))
               (jnum (car this-job))
               (jnam (car (cdr this-job)))
               (cnum (car (cdr (cdr this-job))))
               (cnam (or (car (cdr (assoc cnum client-list)))
                         (format "Unknown Client: %d" cnum)))
               (csym (intern cnam new-obarray))
               (jobs (if (boundp csym) (symbol-value csym))))
          (fset csym cnum)
          (set csym (cons (list jnam jnum) jobs)))
        (setq job-list (cdr job-list)))
      (setq timesheet-jobarray-cache new-obarray))))



(defun timesheet-unjust-completion-filter (elt)
  "Why does obarray completion even offer nil entries?  I don't know,
but this filter makes up for the deficiency."
  elt)


(defun timesheet-read-or-check-client (&optional client use-local)
  ;; If CLIENT is nil, prompts; else checks that CLIENT is a valid client
  ;; USE-LOCAL means use local cache of the lists.
  ;; Returns a timesheet-jobarray symbol, or errors
  (if client
      (let ((csym (intern-soft client (timesheet-jobarray use-local))))
        (or csym (error "No such client \"%s\"!" client)))
    (let ((completion-ignore-case t))
      (intern
       (completing-read "Client: " (timesheet-jobarray use-local)
                        'timesheet-unjust-completion-filter t)
       (timesheet-jobarray)))))


(defun timesheet-read-job (&optional use-local)
  "Return `(job-name job-number client-name client-number)' from user.
Optional argument USE-LOCAL non-nil means use cached copies of the
client and job lists (because no network, for example)."
  (interactive)
  (let* ((csym (timesheet-read-or-check-client nil use-local))
         (cnum (symbol-function csym))
         (jlst (symbol-value csym))
         (job  (assoc (let ((completion-ignore-case t))
                        (completing-read "Job: " jlst)) jlst)))
    (append job (list (symbol-name csym) cnum))))



(defun timesheet-date-string (&optional time)
  "Convert Emacs `current-time-string' result to Timesheet date format.
Optional argument TIME is either like the result of
`current-time-string' or `current-time'."
  (or time (setq time (current-time-string)))
  (if (consp time) (setq time (current-time-string time)))
  (let ((months '(("Jan" "01") ("Feb" "02") ("Mar" "03") ("Apr" "04")
                  ("May" "05") ("Jun" "06") ("Jul" "07") ("Aug" "08")
                  ("Sep" "09") ("Oct" "10") ("Nov" "11") ("Dec" "12")))
        (month  (substring time 4 7))
        (day    (substring time 8 10))
        (year   (substring time 22 24)))
    (setq month (car (cdr (assoc month months))))
    (if (string-match "^\\( \\|\t\\)" day)
        (aset day 0 ?0))
    (format "%s/%s/%s" month day year)))



(defun timesheet-insure-dir ()
  "Make sure the timesheet-dir exists and is writeable."
  (if (and (file-exists-p timesheet-dir)
           (file-readable-p timesheet-dir)
           (file-writable-p timesheet-dir))
      nil
    ;; Else create it
    (make-directory timesheet-dir)))



(defvar timesheet-mode-map
  (let ((map (make-sparse-keymap "Timesheet")))
    (define-key map "\C-c\C-c" 'timesheet-log-accumulated-hours)
    (define-key map "\C-c\C-b" 'timesheet-toggle-billable)
    (define-key map "\C-c\C-u" 'timesheet-duplicate-previous-entry)
    (define-key map "\C-c\C-f" 'timesheet-flush-client/job-cache)
    (define-key map "\C-c\C-n" 'timesheet-new-entry)
    (define-key map "\C-c\C-t" 'timesheet-total-hours)
    (define-key map "\C-c\C-q" 'timesheet-bury-buffer)
    (define-key map "\C-c\C-w" 'widen)
    map)
  "Commands for Timesheet Mode.")


(defalias 'timesheet-new-entry 'timesheet)


(defun timesheet-bury-buffer ()
  "Make it go away, but save to disk first."
  (interactive)
  (save-buffer)
  (bury-buffer))


(defun timesheet-total-hours ()
  "Display the total number of hours in the current log."
  (interactive)
  (save-excursion
    (save-restriction
      (widen)
      (goto-char (point-min))
      (let ((total (float 0)))
        (while (re-search-forward "^Hours:\\s-+\\([0-9]*\\.*[0-9]*\\)" nil t)
          (let ((num (float (string-to-number (match-string 1)))))
            (setq total (+ total num))
            (forward-line 1)))
        (let* ((str (format "%s" total))
               (signifindex (string-match "\\." str))
               (finalstr (substring str 0 (if signifindex (+ signifindex 2)))))
          (message "Total hours: %s" finalstr))))))


(defun timesheet-flush-client/job-cache ()
  "Empty the client and job list, forcing refetch from server next time."
  (interactive)
  (setq timesheet-jobarray-cache nil)
  (message
   "Cache flushed; client and job lists will be refetched next time."))


(defun timesheet-toggle-billable ()
  "Change Billable from y to n, or n to y."
  (interactive)
  (save-excursion
    (or (search-backward timesheet-entry-separator nil t)
        (re-search-backward "^Client: " nil t)
        (re-search-backward "^Serial: " nil t))
    (let ((search-start (progn (re-search-forward "^Billable:") (point)))
          (limit (save-excursion (end-of-line) (point))))
      (if (re-search-forward "\\(y\\|\\yes\\|Y\\|Yes\\)" limit t)
          (replace-match "n")
        (goto-char search-start)
        (if (re-search-forward "\\(n\\|\\no\\|N\\|No\\)" limit t)
            (replace-match "y")
          (error "Can't find Billable field"))))))
          

(defun timesheet-duplicate-previous-entry ()
  "Start a new entry, with fields the same as previous entry."
  (interactive)
  (find-file timesheet-unlogged-file)
  (widen)
  (goto-char (point-max))
  (let ((new-start (point)))
    (if (not (search-backward timesheet-entry-separator nil t))
        (error "No previous entry to duplicate")
      (beginning-of-line)
      (let ((str (buffer-substring (point) new-start)))
        (goto-char new-start)
        (insert "\n\n")
        (setq new-start (point))
        (insert str)
        (goto-char new-start)
        (forward-line 1)
        (setq new-start (point))
        (narrow-to-region new-start (point-max))
        (goto-char (point-min))
        (timesheet-mode)
        (message "Duplicated previous entry.")))))


(defvar timesheet-mode-hook nil
  "Hook for Timesheet mode, run at the end of mode setup.")


(defun timesheet-mode ()
  "Major mode for writing up hours.
The following special keybinding are in effect:

\\{timesheet-mode-map}
"
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'timesheet-mode
        mode-name  "Timesheet Mode")
  (use-local-map timesheet-mode-map)
  (put 'timesheet-mode 'mode-class 'special)
  (run-hooks 'timesheet-mode-hook))


(defvar timesheet-entry-separator (concat (make-string 72 ?=) "\n")
  "Separates entries in the human-edited log file.")
         

(defvar timesheet-header-separator "----------------\n"
  "Separates headers from descriptions/comments in an entry.")



(defalias 'timesheet-log 'timesheet-log-accumulated-hours)


(defun timesheet-handle-simple-field (log-proc prep-str &optional default)
  ;; A helper for the common cases of timesheet-handle-field
  ;; When you call this, point should be on the field, when you're
  ;; done, it will be on the next line.  Must return t, because
  ;; timesheet-handle-field does.
  (timesheet-proc-sendstr log-proc prep-str)
  (search-forward ":")
  (let ((limit (save-excursion (end-of-line) (point))))
    (if (re-search-forward "\\(\\S-+\\)\\s-*$" limit t)
        (let ((str (match-string 1)))
          (if str
              (progn (timesheet-proc-sendstr log-proc str)
                     (timesheet-proc-sendstr log-proc "\n"))))
      (timesheet-proc-sendstr log-proc (or default "\n"))
      (timesheet-proc-sendstr log-proc "\n")))
  (forward-line 1)
  t)


(defun timesheet-handle-field (log-proc)
  ;; Call this in the hours buffer just *below* an entry separator
  (cond
   ((looking-at "Client:")
    (timesheet-proc-sendstr log-proc "\\CLIENT_ID\n")
    (re-search-forward "[0-9]+")
    (forward-word -1)
    (let ((client-id
           (buffer-substring (point)
                             (progn (forward-word 1) (point)))))
      (timesheet-proc-sendstr log-proc client-id)
      (timesheet-proc-sendstr log-proc "\n"))
    (timesheet-proc-sendstr log-proc "\\CLIENT_NAME\n")
    (search-forward "(")
    (let ((client-name
           (buffer-substring (point)
                             (progn (end-of-line)
                                    (search-backward ")")
                                    (point)))))
      (timesheet-proc-sendstr log-proc client-name)
      (timesheet-proc-sendstr log-proc "\n"))
    (forward-line 1)
    t
    )
   ((looking-at "Job:")
    (timesheet-proc-sendstr log-proc "\\JOB_ID\n")
    (re-search-forward "[0-9]+")
    (forward-word -1)
    (let ((job-id
           (buffer-substring (point)
                             (progn (forward-word 1) (point)))))
      (timesheet-proc-sendstr log-proc job-id)
      (timesheet-proc-sendstr log-proc "\n"))
    (forward-line 1)
    t)
   ((looking-at "Date:")
    (timesheet-handle-simple-field log-proc "\\DATE_ENTERED\n"))
   ((looking-at "Hours:")
    (timesheet-handle-simple-field log-proc "\\TOTAL_HOURS\n"))
   ((looking-at "Time In:")
    (timesheet-handle-simple-field log-proc "\\TIME_IN\n"))
   ((looking-at "Time Out:")
    (timesheet-handle-simple-field log-proc "\\TIME_OUT\n"))
   ((looking-at "Parking:")
    (timesheet-handle-simple-field log-proc "\\PARKING\n"))
   ((looking-at "Travel:")
    (timesheet-handle-simple-field log-proc "\\TRAVEL\n"))
   ((looking-at "Billable:")
    (timesheet-handle-simple-field log-proc "\\BILLABLE\n"))
   ((looking-at timesheet-header-separator)
    (re-search-forward "^Hours Description:")
    (beginning-of-line)
    t)
   ((looking-at "Hours Description:")
    (timesheet-proc-sendstr log-proc "\\HOURS_DESCRIPTION\n")
    (search-forward ":")
    (if (looking-at "\\s-*$")
        (forward-line 1))
    (let ((str (buffer-substring (point) (progn 
                                           (re-search-forward "^Comment:")
                                           (forward-line -1)
                                           (end-of-line)
                                           (point)))))
      (timesheet-proc-sendstr log-proc str)
      (timesheet-proc-sendstr log-proc "\n"))
    t)
   ((looking-at "Comment:")
    (timesheet-proc-sendstr log-proc "\\COMMENT\n")
    (search-forward ":")
    (if (looking-at "\\s-*$")
        (forward-line 1))
    (let ((str (buffer-substring
                (point)
                (progn 
                  (if (search-forward timesheet-entry-separator nil 0)
                      (forward-line -2))
                  (end-of-line)
                  (if (eobp) ; force final newline
                      (insert "\n"))
                  (point)))))
      (timesheet-proc-sendstr log-proc str)
      (timesheet-proc-sendstr log-proc "\n"))
    (forward-line 1)
    t)
   ((or (eobp) (looking-at timesheet-entry-separator))
    ;; nothing left in this entry, end it
    nil)
   (t 
    (forward-line 1) 
    t)
   ))


(defun timesheet-start-logger-process ()
  (let ((process-connection-type "not-nil"))
    (start-process "timesheet-logger" "*timesheet logger*" 
                   timesheet-logger-program
                   (concat "-approot=" "http://" 
                           timesheet-list-root-url))))


(defun timesheet-start-pulldata-process (user password url)
  (let ((process-connection-type nil))
    (let ((proc (start-process
                 "timesheet-pulldata" "*timesheet pull data*" 
                 timesheet-pulldata-program 
                 url)))
      (process-send-string proc (format "%s\n" user))
      (process-send-string proc (format "%s\n" password))
      (process-send-eof proc)
      proc)))
  

(defvar timesheet-username nil
  "*Username for logging into the server.")


(defvar timesheet-password nil
  "*Password for logging into the server.")


(defun timesheet-set-username-password ()
  (or timesheet-username
      (setq timesheet-username (read-string "onShore TimeSheet username: ")))
  (or timesheet-password
      (setq timesheet-password (comint-read-noecho "onShore TimeSheet password: " t))))  

(defun timesheet-send-username-password (proc)
  (timesheet-set-username-password) 
  (timesheet-proc-sendstr proc (format "\\USERNAME\n%s\n\\PASSWORD\n%s\n" 
                                    timesheet-username timesheet-password)))


(defun timesheet-this-entry-date ()
  ;; helper for below, gets this entry's date:
  (save-excursion
    (save-match-data
      (re-search-forward "^Date:\\s-*\\([0-9]+/[0-9]+/[0-9]+\\)")
      (match-string 1))))


(defun timesheet-handle-entry (log-proc serialnum)
  (let ((entry-start (point))
        (hourbuf (current-buffer))
        (errfile timesheet-error-file)
        (serialstr
         (format "%s-item-%d" (timesheet-this-entry-date) serialnum)))
    (message "Handling entry %s..." serialstr)
    (timesheet-proc-sendstr log-proc "\\BEGIN\n")
    (timesheet-proc-sendstr log-proc (format "\\SERIAL\n%s\n" serialstr))
    (forward-line 1)                   ; hop past separator
    (insert "Serial:   " serialstr "\n") ; tag entry with an id number
    (while (timesheet-handle-field log-proc))
    (timesheet-proc-sendstr log-proc "\\END\n\n")

    ;; Check for results:
    (save-excursion
      (set-buffer (process-buffer log-proc))
      (display-buffer (process-buffer log-proc))
      (let ((no-results-yet t))
      (while no-results-yet
        (message "Waiting for results on %s..." serialstr)
        
        (cond
         ((progn (goto-char (point-min)) (search-forward "\\SUCCESS" nil t))
          (message "Waiting for results on %s...done" serialstr)
          (sit-for 1)
          (erase-buffer)
          ;; Archive successful entry:
          (save-excursion
            (set-buffer hourbuf)
            (append-to-file entry-start (point) (timesheet-archive-file))
            (delete-region entry-start (point)))
          (setq no-results-yet nil))
         ((progn (goto-char (point-min)) (search-forward "\\FAILURE" nil t))
          (save-excursion
            (set-buffer hourbuf)
            (append-to-file entry-start (point) timesheet-error-file)
            (delete-region entry-start (point)))
          (append-to-file (point-min) (point-max) timesheet-error-file)
          (message "Failure details for %s archived in %s"
                   serialstr timesheet-error-file)
          (sit-for 1)
          (erase-buffer)
          (setq no-results-yet nil))
         (t
          ;; (while (accept-process-output log-proc timesheet-yawn-timeout))
          (accept-process-output nil timesheet-yawn-timeout)
          )))))
    (message "Handling entry %s...done" serialstr)))


(defun timesheet-archive-file ()
  (let* ((datestr (current-time-string))
         (month-word (substring datestr 4 7))
         (month (cdr (assoc month-word '(("Jan" . "01") ("Feb" . "02")
                                         ("Mar" . "03") ("Apr" . "04")
                                         ("May" . "05") ("Jun" . "06")
                                         ("Jul" . "07") ("Aug" . "08")
                                         ("Sep" . "09") ("Oct" . "10")
                                         ("Nov" . "11") ("Dec" . "12")))))
         (day        (substring datestr 8 10))
         (year       (substring datestr 20 24))
         (newstr     (concat year month day)))
    (if (string-match "[ \t]" (char-to-string (aref newstr 6)))
        (aset newstr 6 ?0))
    (concat timesheet-archive-prefix "." newstr)))


(defun timesheet-log-accumulated-hours ()
  "Upload the local timesheet log file to the server."
  (interactive)
  ;; The server log_from.pl script wants this in a certain format.
  ;; Our approach is to translate the log file to that format, save
  ;; that in a buffer, then send the buffer to log_from.pl.
  (message "Logging hours...")
  (save-excursion
    (set-buffer (find-file-noselect timesheet-unlogged-file))
    (widen)
    (save-buffer)
    (sit-for 0.1)

    ;; And they're off!
    (goto-char (point-min))
    (let ((proc (timesheet-start-logger-process))
          (serialnum 1))
      (save-excursion 
        (set-buffer (process-buffer proc))
        (buffer-disable-undo (process-buffer proc))
        (erase-buffer))
      (timesheet-send-username-password proc)
      ;; Loop over the data, uploading as you go:
      (while (search-forward timesheet-entry-separator nil t)
        (forward-line -1)
        (timesheet-handle-entry proc serialnum)
        (setq serialnum (1+ serialnum)))
      (timesheet-proc-sendstr proc "\\QUIT\n")
      (if (not (eq (process-status proc) 'exit))
          (process-send-eof proc))
      (save-buffer)
      (kill-buffer (current-buffer))
      (delete-other-windows))

    ;; That's all, folks.
    (if (> (or (nth 7 (file-attributes timesheet-error-file)) 0) 0)
        (message "See %s for error information." timesheet-error-file)
      (message "Hours logged."))
    ))



(defun timesheet (use-local)
  "Log hours for the onShore TimeSheet database.  Prefix argument
means load client and job lists from locally-cached copies, instead of
getting them from the server.

Hours are not actually uploaded to the server until
`timesheet-log-accumulated-hours' is run.
Logs are stored in \"~/.timesheet/hours\" (or in whatever directory is
specified by the `timesheet-dir' variable).

Variables you may want/need to customize:

   timesheet-dir
   timesheet-host
   timesheet-list-root-url
"
  (interactive "P")
  (timesheet-set-dependant-vars)
  (timesheet-insure-dir)
  (let* ((job  (timesheet-read-job use-local))
         (jnam (nth 0 job))
         (jnum (nth 1 job))
         (cnam (nth 2 job))
         (cnum (nth 3 job))
         (date  (read-string "Date: " (timesheet-date-string (current-time))))
         (hours (read-string "Hours: "))
         (opoint nil)
         )
    (find-file timesheet-unlogged-file)
    (timesheet-mode)
    (goto-char (point-max))

    ;; Make sure we have a separator line before every entry
    (if (not (save-excursion
               (forward-line -1)
               (looking-at (regexp-quote timesheet-entry-separator))))
        (insert (if (bobp) "" "\n\n") timesheet-entry-separator))
    
    (setq opoint (point))
    ;; Insert the "headers"
    (insert (format "Client:   %-5d  (%s)\n" cnum cnam))
    (insert (format "Job:      %-5d  (%s)\n" jnum jnam))
    (insert (format "Date:     %s\n" date))
    (insert (format "Hours:    %s\n" (if (> (length hours) 0) hours "?")))
    (insert (format "Time In:  \n"))
    (insert (format "Time Out: \n"))
    (insert (format "Parking:  \n"))
    (insert (format "Travel:   0\n"))
    (insert (format "Billable: y\n"))
    (insert timesheet-header-separator)
    (insert "\n")
    (insert "Hours Description:\n")
    (insert "\n\n")
    (insert "Comment: \n")
    ;; Position point in description, the most common case:
    (search-backward "Hours Description:")
    (save-buffer)
    (forward-line 1)
    (narrow-to-region opoint (point-max))
    ))
;;;; End timesheet.el ;;;;

