<?php
/* 	OpenDb - Open Lending Database Project
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

include_once("./functions/function.php");
include_once("./functions/utils.php");
include_once("./functions/borrowed_item.php");
include_once("./functions/item.php");
include_once("./functions/http.php");
include_once("./functions/fileutils.php");
include_once("./functions/user.php");
include_once("./functions/review.php");
include_once("./functions/item_attribute.php");
include_once("./functions/item_type.php");
include_once("./functions/widgets.php");
include_once("./functions/parseutils.php");
include_once("./functions/status_type.php");

/*
* @parma $item_r - The item the update is being performed against.  This allows us to check the
* 					borrow status of the item, as well as the owner_id, which allows us to
* 					validate updates.  If an administrator is performing the
* 					update, the validations should still be performed in context of
* 					the owner of the item.

* @param $old_status_type_r - Originating s_status_type record as returned from fetch_status_type_r
* @param $new_status_type_r - Proposed update s_status_type record as returned from fetch_status_type_r
* 
* This function assumes the basic checks, such as the old and new s_status_type are different,
* This function does not do 'update' specific logic, but merely tests that the change
* change from one s_status_type to another is valid.
* 
* Refer to docs/notes/s_status_type.txt for more information
*/
function is_update_status_type_valid($item_id, $instance_no, $owner_id, $old_status_type_r, $new_status_type_r, &$errors)
{
	global $LANG_VARS;

	// New status cannot be closed.
	if($new_status_type_r['closed_ind'] != 'Y')
	{
		if($new_status_type_r['update_ind'] == 'Y')
		{
			// Either no existing borrowed_item records, or new s_status_type borrow_ind must allow borrowing (==Y), or
			// only temporarily disable it (==N).
			if( (($new_status_type_r['borrow_ind'] != 'Y' && $new_status_type_r['borrow_ind'] != 'N' && $new_status_type_r['borrow_ind'] != 'B' && is_exists_item_instance_borrowed_item($item_id, $instance_no)) || 
								($new_status_type_r['borrow_ind'] != 'Y' && $new_status_type_r['borrow_ind'] != 'N' && is_item_borrowed($item_id, $instance_no))) )
			{
				$errors = replace_lang_vars(array('usertype'=>get_usertype_description($owner_user_type),'s_status_type_desc'=>$new_status_type_r['description']),$LANG_VARS['operation_not_avail_s_status_type']);
				return FALSE;
			}
			else
			{
				$owner_user_type = fetch_user_type($owner_id);
								
				// Owner must have enough permission to create items of this type.
				if(strlen($new_status_type_r['min_create_user_type'])==0 || in_array($new_status_type_r['min_create_user_type'], get_min_user_type_r($owner_user_type)))
				{
					// Now check that there is no restrictions on how many instances of this item and s_status_type, can be created by the owner.
					if($new_status_type_r['new_owner_instance_ind'] == 'Y' || !is_exists_item_instance_with_owner_and_status($item_id, $new_status_type_r['s_status_type'], $owner_id))
					{
						return TRUE;
					}
					else//if($new_status_type_r['new_owner_instance_ind'] == 'Y' || !is_exists_item_instance_with_owner_and_status($item_id, $new_status_type_r['s_status_type'], $owner_id))
					{
						$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
						return FALSE;
					}
				}//if(strlen($new_status_type_r['min_create_user_type'])==0 || in_array($new_status_type_r['min_create_user_type'], get_min_user_type_r($owner_user_type)))
				else
				{
					$errors = replace_lang_vars(array('usertype'=>get_usertype_description($owner_user_type),'s_status_type_desc'=>$new_status_type_r['description']),$LANG_VARS['s_status_type_create_access_disabled_for_usertype']);
					return FALSE;
				}
			}
		}//if($old_status_type_r['update_ind'] == 'Y')
		else
		{
			$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
			return FALSE;
		}
	}
	else//if($new_status_type_r['closed_ind'] != 'Y')
	{
		$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['s_status_type_not_supported']),'detail'=>'');
		return FALSE;
	}
}

/*
* Handles complete new items, as well as new item_instances.  Checks whether item_id
* is defined, before performing new item_instance specific checks.
*/
function is_newitem_status_type_valid($owner_id, $new_status_type_r, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;

	// New status cannot be closed.
	if($new_status_type_r['closed_ind'] != 'Y')
	{
		if($new_status_type_r['insert_ind'] == 'Y')
		{
			$owner_user_type = fetch_user_type($owner_id);
									
			// Owner must have enough permission to create items of this type.
			if(strlen($new_status_type_r['min_create_user_type'])==0 || in_array($new_status_type_r['min_create_user_type'], get_min_user_type_r($owner_user_type)))
			{
				return TRUE;
			}
			else
			{
				$errors = replace_lang_vars(array('usertype'=>get_usertype_description($owner_user_type),'s_status_type_desc'=>$new_status_type_r['description']),$LANG_VARS['s_status_type_create_access_disabled_for_usertype']);
				return FALSE;
			}
		}//if($new_status_type_r['insert_ind'] == 'Y')
		else
		{
			$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
			return FALSE;
		}
	}
	else//if($new_status_type_r['closed_ind'] != 'Y')
	{
		$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['s_status_type_not_supported']),'detail'=>'');
		return FALSE;
	}
}

function is_newinstance_status_type_valid($item_id, $owner_id, $new_status_type_r, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	
	if(is_newitem_status_type_valid($owner_id, $new_status_type_r, $errors))
	{
		// Overriding config variable.
		if($CONFIG_VARS['item_input.new_instance_owner_only']!==TRUE || is_user_owner_of_item($item_id, NULL, $owner_id))
		{
			// If both are set to 'Y', we do not care who already has a copy of what!
			if($new_status_type_r['new_owner_instance_ind'] == 'Y' && $new_status_type_r['new_not_owner_instance_ind'] == 'Y')
			{
				return TRUE;
			}
			else
			{
				if(is_exists_item_instance_with_owner_and_status($item_id, $new_status_type_r['s_status_type'], $owner_id))
				{
					if($new_status_type_r['new_owner_instance_ind'] == 'Y')
						return TRUE;
					else
					{
						$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
						return FALSE;
					}
				}
				else
				{
					if($new_status_type_r['new_not_owner_instance_ind'] == 'Y')
						return TRUE;
					else
					{
						$errors = array('error'=>replace_lang_var('s_status_type_desc', $new_status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
						return FALSE;
					}
				}
			}
		}
		else//if($CONFIG_VARS['item_input.new_instance_owner_only']!==TRUE || is_user_owner_of_item($item_r['item_id'], NULL, $HTTP_SESSION_VARS['user_id']))
		{
			$errors = array('error'=>$LANG_VARS['operation_not_avail_new_instance'],'detail'=>'');
			return FALSE;
		}
	}
	else
	{
		return FALSE;
	}
}

/**
	Returns a formatted filename to be used where a item_attribute is defined as a upload
	or saveurl widget type.
*/
function get_item_attr_filename($fieldname, $item_id, $extension)
{
	global $CONFIG_VARS;

	return str_replace("{ext}", 
					$extension, 
					str_replace(
						"{item_id}", 
						$item_id, 
						str_replace("{fieldname}", 
								$fieldname, 
								$CONFIG_VARS['item_input.item_attr_file_mask']
								)
							)
						);
}

/**
	Special functionality to handle file upload widgets
*/
function handle_item_upload($prompt, $fieldname, $item_id, $http_post_file, &$errors)
{
	global $CONFIG_VARS;
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$value = save_upload_file($http_post_file['tmp_name'], 
				$CONFIG_VARS['item_input.item_attr_save_dir'], 
				get_item_attr_filename($fieldname, $item_id, get_file_ext($http_post_file['name'])), 
				TRUE);
			
	if($value !== FALSE)
	{
		opendb_log("Uploaded file. (item_id=$item_id, file=".$http_post_file['name'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
		return $value;
	}
	else
	{
		$errors[] = replace_lang_var("prompt", $prompt, $LANG_VARS['file_upload_error']);
		return NULL;
	}
}

/**
	Special functionality to handle saveurl widget.
	
	Note: Does not check extension, as this is done in the validate_item_attributes
	function.

	If $url cannot be downloaded, the original $url will be returned.
*/
function handle_item_saveurl($prompt, $fieldname, $item_id, $url, &$errors)
{
	global $CONFIG_VARS;
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;
	
	// Do a double wammy check to ensure we are not trying to resave file.
	if(!file_exists($url) && is_url_absolute($url))
	{
		$value = save_url_file($url, 
							$CONFIG_VARS['item_input.item_attr_save_dir'], 
							get_item_attr_filename($fieldname, $item_id, get_file_ext($url)),
							TRUE);

		if($value !== FALSE)
		{
			opendb_log("Saved URL. (item_id=$item_id, url=$url, update_who=".$HTTP_SESSION_VARS['user_id'].")");
			return $value;
		}
		else
		{
			$errors[] = replace_lang_var("prompt", $prompt, $LANG_VARS['url_save_error']);
			return NULL;
		}
	}
	else // Invalid url - could be local image, or no image at all.
	{   
		// The str_replace removes the single quotes as they look a bit ugly anywhere but in a javascript alert!
		if(file_exists($url))
			$errors[] = replace_lang_var("prompt", $prompt, $LANG_VARS['url_already_saved_local']);
		else
			$errors[] = replace_lang_var("prompt", $prompt, $LANG_VARS['invalid_url_error']);
			
		// But return original $url anyway, so it can be saved to the database.
		return $url;
	}
}

/*
* If a valid file is found, we should delete it.
*/
function delete_item_attribute_file($filename, $input_type, $fieldname, $item_id)
{
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;
	
	if(is_file_widget_input_type($input_type))
	{
		$basename = trim(basename($filename));
		
		// This proves that the file is associated with the current attribute.
		if($basename == trim(get_item_attr_filename($fieldname, $item_id, get_file_ext($basename))))
		{
			// Now check if the basename'd file exists in the item_input upload directory, if so it is
			// safe to delete it!
			if(file_exists($CONFIG_VARS['item_input.item_attr_save_dir'].(substr($CONFIG_VARS['item_input.item_attr_save_dir'],-1) == "/"?"/":"").$basename))
			{
				if(@unlink($filename))
					opendb_log("Deleted file. (item_id=$item_id, file=".$filename.", update_who=".$HTTP_SESSION_VARS['user_id'].")");
				else
				{
					opendb_log("Failed to delete file. (item_id=$item_id, file=".$filename.", update_who=".$HTTP_SESSION_VARS['user_id'].")");

					// Failed to delete existing file.				
					return FALSE;
				}
			}
		}
	}
	//else
	return TRUE;
}

/*
* 	This assumes a certain amount of input validation has been performed before calling this
*	function.
* 
* 	NOTE: Assumes the validate_item_attributes(...) has been called before this, to put the
* 		actual values into the $HTTP_VARS[$fieldname] value.
*/
function handle_item_attributes($op, $item_r, $attr_results, $HTTP_VARS, $HTTP_POST_FILES, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	$attributes_updated = 0;
	while($item_attribute_type_r = mysql_fetch_array($attr_results, MYSQL_ASSOC))
	{
		$input_widget_type = get_function_type(trim($item_attribute_type_r['input_type']));

		// For all operations the {CATEGORY,DURATION,TITLE,STATUSTYPE,STATUSCMNT,ITEM_ID} cannot be 
		// updated because they exist at item/item_instance level.
		if(	$item_attribute_type_r['s_field_type']!='CATEGORY' && $item_attribute_type_r['s_field_type']!='DURATION' && 
					$item_attribute_type_r['s_field_type']!='TITLE' && $item_attribute_type_r['s_field_type']!='STATUSTYPE' &&
					$item_attribute_type_r['s_field_type']!='STATUSCMNT' && $item_attribute_type_r['s_field_type']!='ITEM_ID')
		{
			$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']);

			// A refresh field.
			if(isset($HTTP_VARS[$fieldname.'_new']) && ($HTTP_VARS[$fieldname] == 'old' || $HTTP_VARS[$fieldname] == 'new'))
			{
				if($HTTP_VARS[$fieldname] == 'new')
					$HTTP_VARS[$fieldname] = $HTTP_VARS[$fieldname.'_new'];
				else
					continue; // Do not perform any update if old attribute value selected.
			}

			// Is it an upload operation
			if(is_array($HTTP_POST_FILES) && is_array($HTTP_POST_FILES[$fieldname]) && is_uploaded_file($HTTP_POST_FILES[$fieldname]['tmp_name'])) 
				$value = ifempty(handle_item_upload($item_attribute_type_r['prompt'], $fieldname, $item_r['item_id'], $HTTP_POST_FILES[$fieldname], $errors), $item_attribute_type_r['attribute_val']);// if no upload file, restore the original attribute value
			else if(is_array($HTTP_POST_FILES) && is_array($HTTP_POST_FILES[$fieldname."_upload"]) && is_uploaded_file($HTTP_POST_FILES[$fieldname."_upload"]['tmp_name'])) 
				$value = ifempty(handle_item_upload($item_attribute_type_r['prompt'], $fieldname, $item_r['item_id'], $HTTP_POST_FILES[$fieldname."_upload"], $errors), $item_attribute_type_r['attribute_val']);// if no upload file, restore the original attribute value
			else if($HTTP_VARS[$fieldname."_saveurl"] == "y") // saveurl request - generate error for local image url's (whether exists or not!)
				$value = ifempty(handle_item_saveurl($item_attribute_type_r['prompt'], $fieldname, $item_r['item_id'], $HTTP_VARS[$fieldname], $errors), $HTTP_VARS[$fieldname]);
			else if(is_array($HTTP_VARS[$fieldname]))//field of array
				$value = get_item_attr_col_value($item_attribute_type_r['s_attribute_type'], $HTTP_VARS[$fieldname]);
			else // normal field
				$value = $HTTP_VARS[$fieldname];
				
			// If attribute value found - an existing attribute, so do an update.
			if($item_attribute_type_r['attribute_val'] !== NULL)
			{
				// The attribute should actually be set
				if(strlen($value)>0)
				{
					if(update_item_attribute($item_r['item_id'], $item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no'], $value))
					{
						// If previous attribute_val represents a local file, and the new value is NOT pointing at the same file (having overwritten it!)
						// we should delete the old file.
						if(strlen($item_attribute_type_r['attribute_val'])>0 && $value != $item_attribute_type_r['attribute_val'])
						{
							// We have logged whether the file was deleted or not, but do not bother reporting
							// failure the user, because what could they do.
							delete_item_attribute_file($item_attribute_type_r['attribute_val'], $item_attribute_type_r['input_type'], $fieldname, $item_r['item_id']);
						}
						$attributes_updated++;
					}
				}
				else
				{
					//attribute is empty, so delete it.
					if(delete_item_attribute($item_r['item_id'], $item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']))
					{
						// We have logged whether the file was deleted or not, but do not bother reporting
						// failure the user, because what could they do.
						delete_item_attribute_file($item_attribute_type_r['attribute_val'], $item_attribute_type_r['input_type'], $fieldname, $item_r['item_id']);

						$attributes_updated++;
					}
				}
			}
			else if(strlen($value)>0)
			{
				if(insert_item_attribute($item_r['item_id'], $item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no'], $value))
   		   		 	$attributes_updated++;
			}
		}
	}
	mysql_free_result($attr_results);
	
	// Indicate how many attributes updated.
	return $attributes_updated;
}

/*
* Validate item_attributes will actually update the $HTTP_VARS variable
* with the final filtered value
*/
function validate_item_attributes($op, $s_item_type, &$HTTP_VARS, $HTTP_POST_FILES, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;
	
	$errors = NULL;
	$all_fields_validated=TRUE;
	
	$attr_results = fetch_item_attribute_type_rs($s_item_type, TRUE, TRUE);
	if($attr_results)
	{
		while($item_attribute_type_r = mysql_fetch_array($attr_results, MYSQL_ASSOC))
		{
			// Item_ID is purely a read-only attribute.
			if($item_attribute_type_r['s_field_type']!='ITEM_ID')
			{
				// Force compulsory_ind for several s_field_type attributes, in case of bad data.
				if( $item_attribute_type_r['s_field_type'] == 'TITLE')
					$item_attribute_type_r['compulsory_ind'] = 'Y';
				else if( $item_attribute_type_r['s_field_type'] == 'STATUSTYPE' || $item_attribute_type_r['s_field_type'] == 'STATUSCMNT')
					$item_attribute_type_r['compulsory_ind'] = 'N';
				else if($item_attribute_type_r['s_field_type'] == 'DURATION')//Duration can never be compulsory.
					$item_attribute_type_r['compulsory_ind'] = 'N';
					
				$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']);

				// A refresh field.
				if(isset($HTTP_VARS[$fieldname.'_new']) && ($HTTP_VARS[$fieldname] == 'old' || $HTTP_VARS[$fieldname] == 'new'))
				{
					if($HTTP_VARS[$fieldname] == 'new')
						$HTTP_VARS[$fieldname] = $HTTP_VARS[$fieldname.'_new'];
					else
						continue; // Do not perform any update if old attribute value selected.
				}
				
				// Is it an upload operation
				if(is_array($HTTP_POST_FILES) && is_array($HTTP_POST_FILES[$fieldname]))
					$value = $HTTP_POST_FILES[$fieldname]['name'];
				else if(is_array($HTTP_POST_FILES) && is_array($HTTP_POST_FILES[$fieldname."_upload"])) 
					$value = $HTTP_POST_FILES[$fieldname."_upload"]['name'];
				else if($HTTP_VARS[$fieldname."_saveurl"] == "y") // saveurl request - generate error for local image url's (whether exists or not!)
					$value = ($HTTP_VARS[$fieldname] = filter_input_field($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['input_type'], $HTTP_VARS[$fieldname]));
				else // normal field
					$value = ($HTTP_VARS[$fieldname] = filter_input_field($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['input_type'], $HTTP_VARS[$fieldname]));
				
				// Indicate at least one field failed validation.
				if(!validate_input_field($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['prompt'], $item_attribute_type_r['input_type'], $item_attribute_type_r['compulsory_ind'], $value, $errors))
				{
					$all_fields_validated=FALSE;
				}
				else
				{
					// So we have the filtered version for the handle_update / handle_insert functions.
					if(isset($HTTP_VARS[$fieldname.'_new']) && $HTTP_VARS[$fieldname] == 'new')
						$HTTP_VARS[$fieldname.'_new'] = $HTTP_VARS[$fieldname];
				}
			}				
		}
		mysql_free_result($attr_results);
		
		if(!$all_fields_validated)
			return FALSE;
		else
			return TRUE;
	}
	else
	{
		//else - what else can I do here?
		$errors[] = array('error'=>$LANG_VARS['undefined_error'],'detail'=>'');
		return FALSE;
	}
}

/*
 * Returns:
 * 	TRUE  				- Successful execution
 *  FALSE 	 			- Failed execution
 *  "__CONFIRM__" 		- Operation requires confirmation
 *  "__ABORTED__"		- Operation was aborted
 * "__INVALID_DATA__" 	- indicates that the data entered was not validated
 * 
 * NOTE:
 * 	Assumes that $item_r['owner_id'] has been validated before being assigned, will do no more
 * 	checking against this value, but will use it as the owner for which this new item is created.
 */
function handle_item_insert($parent_item_r, &$item_r, $status_type_r, $HTTP_VARS, $HTTP_POST_FILES, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;

	// Either a normal (parent) item insert (is_empty_array($parent_item_r)) OR a child item insert, and we are checking
	// the permissions of the user, to ensure they are either the owner, or an administrator, who can edit the users item, by adding a new child.
	if(is_empty_array($parent_item_r) || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $parent_item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
	{
		if(is_empty_array($parent_item_r) || $CONFIG_VARS['item_input.linked_item_support']!==FALSE)
		{
			// No parent (is_empty_array($parent_item_r)), or parent and child are same item type, or allowed to be different.
			if(is_empty_array($parent_item_r) || $parent_item_r['s_item_type'] == $item_r['s_item_type'] || $CONFIG_VARS['item_input.link_same_type_only']!==TRUE)
			{
				// Before trying to insert items into this structure, first ensure it is valid.
				if(is_valid_item_type_structure($item_r['s_item_type']))
				{
					// We need to get the title	
					if(validate_item_attributes("insert", $item_r['s_item_type'], $HTTP_VARS, $HTTP_POST_FILES, $errors))
					{
						// Either adding a child (linked) item in which case no s_status_type restrictions apply, or we validate
						// the s_status_type restrictions for the creation of a new normal (parent) item.  The $item_r['owner_id']
						// is artificially populated back in the main item_input.php script, but this allows us to introduce functionality
						// later on for administrators to create items for other users.
						if(is_not_empty_array($parent_item_r) || is_newitem_status_type_valid($item_r['owner_id'], $status_type_r, $errors))
						{
							$title_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'TITLE');
							$item_r['title'] = $HTTP_VARS[get_field_name($title_attr_type_r['s_attribute_type'], $title_attr_type_r['order_no'])];

							$is_exists_owner_title = FALSE;
							$is_exists_linked_title = FALSE;
							$is_exists_title = FALSE;
							
							// A child item or do not check for non-owner duplicates
							if(is_not_empty_array($parent_item_r))
							{
								if($CONFIG_VARS['item_input.duplicate_title_support']!==TRUE || ($HTTP_VARS['confirmed'] != 'true' && $CONFIG_VARS['item_input.confirm_duplicate_linked_item_insert']!==FALSE))
									$is_exists_linked_title = is_exists_title($item_r['title'], $item_r['s_item_type'], NULL, $item_r['parent_id']);
							}
							else
							{
								// Check owner context
								if($CONFIG_VARS['item_input.duplicate_title_support']!==TRUE || ($HTTP_VARS['confirmed'] != 'true' && $CONFIG_VARS['item_input.confirm_duplicate_owner_insert']!==FALSE))
									$is_exists_owner_title = is_exists_title($item_r['title'], $item_r['s_item_type'], $item_r['owner_id']);
									
								// Only check title/s_item_type context								
								if($CONFIG_VARS['item_input.duplicate_title_support']!==TRUE || ($HTTP_VARS['confirmed'] != 'true' && $CONFIG_VARS['item_input.confirm_duplicate_insert']!==FALSE))
									$is_exists_title = is_exists_title($item_r['title'], $item_r['s_item_type']);
							}
							
							// Unless duplicate title support is allowed we cannot continue.
							if((!$is_exists_title && !$is_exists_owner_title && !$is_exists_linked_title) || $CONFIG_VARS['item_input.duplicate_title_support']!==FALSE)
							{
								if(!$is_exists_title && !$is_exists_owner_title && !$is_exists_linked_title)
								{
									// The category input field name corresponds to the s_item_attribute_type record where s_field_type = 'CATEGORY'
									$category_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'CATEGORY');
									$category_val = get_item_attr_col_value(
													$category_attr_type_r['s_attribute_type'], 
													$HTTP_VARS[get_field_name($category_attr_type_r['s_attribute_type'], $category_attr_type_r['order_no'])]);
							
									$new_item_id = insert_item($item_r['parent_id'], $item_r['s_item_type'], $item_r['title'], $category_val, $item_r['owner_id']);
									if($new_item_id !== FALSE)
									{
										$item_r['item_id'] = $new_item_id;
											
										// Who cares if the attribute insert fails, if everything else worked.			
										$attr_results = fetch_item_attribute_type_rs($item_r['s_item_type'], FALSE, TRUE);
										if($attr_results)
											handle_item_attributes($HTTP_VARS['op'], $item_r, $attr_results, $HTTP_VARS, $HTTP_POST_FILES, $errors);
					
										// If no parent relationship, insert a instance record.
										if(is_empty_array($parent_item_r))
										{
											$duration_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'DURATION');
											$borrow_duration = get_item_attr_col_value(
														$duration_attr_type_r['s_attribute_type'], 
														$HTTP_VARS[get_field_name($duration_attr_type_r['s_attribute_type'], $duration_attr_type_r['order_no'])]);
																
											if(strlen($borrow_duration)>0)
											{
												if($status_type_r['borrow_ind'] != 'Y' && $status_type_r['borrow_ind'] != 'N' && $status_type_r['borrow_ind'] != 'B')
												{
													// Actually this is a warning!
													$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_borrow_duration_not_supported']) ,'detail'=>'');
													$borrow_duration = NULL;
												}
											}
											
											$statuscmnt_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'STATUSCMNT');
											$status_comment = $HTTP_VARS[get_field_name($statuscmnt_attr_type_r['s_attribute_type'], $statuscmnt_attr_type_r['order_no'])];
													
											if(strlen($status_comment)>0 && $status_type_r['status_comment_ind'] != 'Y' && $status_type_r['status_comment_ind'] != 'H')
											{
												// Actually this is a warning!
												$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_status_comments_not_supported']) ,'detail'=>'');
												$status_comment = NULL;
											}
											
											// Remember the $item_r['owner_id'] is artificially populated back in the main item_input script.
											$instance_no = insert_newitem_instance($new_item_id, $item_r['s_status_type'], $status_comment, $borrow_duration, $item_r['owner_id']);
					
											// Get the mysql_error before it is reset by next SQL call.
											if($instance_no===FALSE)
											{
												$errors = array('error'=>$LANG_VARS['item_instance_not_added'],'detail'=>mysql_error());
												return FALSE;
											}
											$item_r['instance_no'] = $instance_no;
										}
	
										return TRUE;
									}
									else
									{
										$errors = array('error'=>$LANG_VARS['item_not_added'],'detail'=>mysql_error());
										return FALSE;
									}
								}
								else if($HTTP_VARS['confirmed'] != 'false')// if explicitly false, then we are aborting insert.
								{
									 if($is_exists_owner_title)
										return "__CONFIRM_EXISTS_OWNER_TITLE__";
									else if($is_exists_title)
										return "__CONFIRM_EXISTS_TITLE__";
									else if($is_exists_linked_title)
										return "__CONFIRM_EXISTS_LINKED_TITLE__";
								}
								else //insert aborted.
								{
									return "__ABORTED__";
								}
							}
							else // cannot insert duplicate.
							{
								if($is_exists_owner_title)
									$errors = array('error'=>replace_lang_vars(array('title'=>$item_r['title'],'s_item_type'=>$item_r['s_item_type']), $LANG_VARS['title_same_type_and_owner_exists']),'detail'=>'');
								else if($is_exists_title)
									$errors = array('error'=>replace_lang_vars(array('title'=>$item_r['title'],'s_item_type'=>$item_r['s_item_type']), $LANG_VARS['title_same_type_exists']),'detail'=>'');
								else if($is_exists_linked_title)
									$errors = array('error'=>replace_lang_vars(array('title'=>$item_r['title'],'s_item_type'=>$item_r['s_item_type']), $LANG_VARS['title_linked_item_exists']),'detail'=>'');
								
								return FALSE;
							}
						}
						else//is_status_type_insert_valid(NULL, $item_r['owner_id'], $status_type_r, $errors)
						{
							return FALSE;
						}
					}
					else //if(validate_item_attributes("insert", $item_r['s_item_type'], $errors))
					{
						return "__INVALID_DATA__";
					}
				}
				else // if(is_valid_item_type_structure($item_r['s_item_type']))
				{
					$errors = array('error'=>replace_lang_var("s_item_type", $item_r['s_item_type'], $LANG_VARS['invalid_item_type_structure']),'detail'=>'');
					
					// An error like this is a big problem, and should be dealt with quickly, but there is no sense in alarming the
					// user by sending back an error.
					opendb_log("Experienced a problem with a system item type.  Either it does not exist, or there is an error in its structure. (s_item_type=".$item_r['s_item_type'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
					return FALSE;
				}
			}
			else // NOT $CONFIG_VARS['item_input.link_same_type_only']
			{
				$errors = array('error'=>replace_lang_var("s_item_type", $parent_item_r['s_item_type'], $LANG_VARS['linked_item_must_be_type']),'detail'=>'');
				return FALSE;
			}
		}
		else//if(is_empty_array($parent_item_r) || $CONFIG_VARS['item_input.linked_item_support']!==FALSE)
		{
			$errors = array('error'=>$LANG_VARS['linked_items_not_supported'],'detail'=>mysql_error());
			return FALSE;
		}
	}
	else // not owner of parent item.
	{
		$errors = array('error'=>$LANG_VARS['cannot_update_item_not_owned']);
		opendb_log("Attempted to add a linked item to parent item they do not own (parent_id=".$parent_item_r['item_id'].", parent_instance_no=".$parent_item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
		return FALSE;
	}
}

/*
* @param $item_r - Is the item_instance we are going to copy.
* 
* Note: The $item_r['status_comment'] is not transferred to the new instance.
*/
function handle_item_instance_insert($parent_item_r, &$item_r, $status_type_r, $HTTP_VARS, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;

	if($CONFIG_VARS['item_input.item_instance_support']!==FALSE)
	{
		// Ensure valid owner_id specified
		if(strlen($HTTP_VARS['owner_id'])>0 && ($HTTP_VARS['owner_id'] == $HTTP_SESSION_VARS['user_id'] || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type'])))
			$owner_id = $HTTP_VARS['owner_id'];
		else
			$owner_id = $HTTP_SESSION_VARS['user_id'];
		
		if(is_empty_array($parent_item_r))
		{
			if(is_newinstance_status_type_valid($item_r['item_id'], $owner_id, $status_type_r, $errors))
			{
				$new_instance_no = insert_item_instance($item_r['item_id'], $item_r['s_status_type'], NULL, $item_r['borrow_duration'], $owner_id);
				if($new_instance_no !== FALSE)
				{
					// Now $item_r represents new instancge of item
					$item_r['instance_no'] = $new_instance_no;
					return TRUE;
				}
				else//if($new_instance_no !== FALSE)
				{
					$errors = array('error'=>$LANG_VARS['item_instance_not_added'],'detail'=>mysql_error());
					return FALSE;
				}
			}
			else//if(is_status_type_insert_valid($item_id, $owner_id, $new_status_type_r, $errors))
			{
				return FALSE;
			}
		}
		else//if(is_empty_array($parent_item_r))
		{
			$errors = array('error'=>$LANG_VARS['operation_not_available'],'detail'=>'');
			return FALSE;
		}
	}
	else//if($CONFIG_VARS['item_input.item_instance_support']!==FALSE)
	{
		$errors = array('error'=>$LANG_VARS['operation_not_avail_new_instance'],'detail'=>'');
		return FALSE;
	}
}

/*
 No assumptions are made about whether the op is an update or delete.  In fact we
 assume 'refresh' functionality even if a normal update, which simplifies our
 task considerably.
 * 
 * Return "__INVALID_DATA__" - indicates that the data entered was not validated
 */
function handle_item_update($parent_item_r, &$item_r, $status_type_r, $HTTP_VARS, $HTTP_POST_FILES, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	// If no parent, then it must be owned by the current user.
	if(!is_array($parent_item_r) || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $parent_item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
	{
		// If $parent_item_r defined, then the test for parent ownership is sufficient!
		if(is_not_empty_array($parent_item_r) || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
		{
			if(validate_item_attributes("update", $item_r['s_item_type'], $HTTP_VARS, $HTTP_POST_FILES, $errors))
			{
				/*
				* Handles simple HTTP variables, does not handle $HTTP_POST_FILE, which are supported by
				* more complex attributes.
				*/
				function get_http_field_value($item_attribute_type_r, $HTTP_VARS)
				{
					$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']);
		
					// A refresh field.
					if(isset($HTTP_VARS[$fieldname.'_new']) && ($HTTP_VARS[$fieldname] == 'old' || $HTTP_VARS[$fieldname] == 'new'))
					{
						if($HTTP_VARS[$fieldname] == 'new')
							$HTTP_VARS[$fieldname] = $HTTP_VARS[$fieldname.'_new'];
						else // Get original field.
							return NULL;
					}
	
					// Either not a refresh, or chosen new field.
					if(isset($HTTP_VARS[$fieldname]))
					{
						if(is_array($HTTP_VARS[$fieldname]))//field of array
							return get_item_attr_col_value($item_attribute_type_r['s_attribute_type'], $HTTP_VARS[$fieldname]);
						else // normal field.
							return $HTTP_VARS[$fieldname];
					}
					else // no field found.
						return NULL;		
				}
				
				$item_r['title'] = ifempty(
								get_http_field_value(fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'TITLE'), $HTTP_VARS),
								$item_r['title']);
				
				$item_r['category'] = ifempty(
								get_http_field_value(fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'CATEGORY'), $HTTP_VARS),
								$item_r['category']);
		
				if(update_item($item_r['item_id'], $item_r['title'], $item_r['category']))
				{
					// Who cares if the attribute insert fails, if everything else worked.			
					$attr_results = fetch_item_attribute_type_with_val_rs($item_r['s_item_type'], $item_r['item_id'], TRUE, TRUE, TRUE);
					if($attr_results)
						handle_item_attributes($HTTP_VARS['op'], $item_r, $attr_results, $HTTP_VARS, $HTTP_POST_FILES, $errors);

					if(is_not_empty_array($parent_item_r))
					{
						// No need to update the instance, as we are updating a child item!
						return TRUE;
					}
					else// If no parent relationship, update instance record.
					{
						$update_status_type = get_http_field_value(fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'STATUSTYPE'), $HTTP_VARS);
						
						// If the s_status_type is being updated, perform validations for the update type.
						// Otherwise, reset to original s_status_type and ignore request to update.
						if(strlen($update_status_type)>0 && $item_r['s_status_type'] != $update_status_type)
						{
							$update_status_type_r = fetch_status_type_r($update_status_type);
							if(is_not_empty_array($update_status_type_r))
							{
								if(is_update_status_type_valid($item_r['item_id'], $item_r['instance_no'], $item_r['owner_id'], $status_type_r, $update_status_type_r, $errors))
								{
									$item_r['s_status_type'] = $update_status_type;
									$status_type_r = $update_status_type_r;
								}
							}
						}//if(strlen($update_s_status_type)>0 && $item_r['s_status_type'] != $update_s_status_type)
						
						// *****************************
						// If s_status_type validations were successful, then $status_type_r array has reference to the
						// s_status_type to update to, or else on failure the old s_status_type.
						// *****************************
						
						// Do not bother with these checks, unless item actually allows updating of these values.
						$borrow_duration = get_http_field_value(fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'DURATION'), $HTTP_VARS);
						if($borrow_duration !== NULL)
						{
							// If borrow_duration is equal to 'N', it means borrowing is not available, but the borrow
							// duration can still be updated.
							if(strlen($borrow_duration)>0 && $status_type_r['borrow_ind'] != 'Y' && $status_type_r['borrow_ind'] != 'N' && $status_type_r['borrow_ind'] != 'B')
							{
								// Actually this is a warning!
								$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_borrow_duration_not_supported']) ,'detail'=>'');
								$borrow_duration = NULL;
							}
						}
						else
							$borrow_duration = FALSE; // Not defined, so do not update
								
						$status_comment = get_http_field_value(fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'STATUSCMNT'), $HTTP_VARS);
						if($status_comment !== NULL)
						{
							if(strlen($status_comment)>0 && $status_type_r['status_comment_ind'] != 'Y' && $status_type_r['status_comment_ind'] != 'H')
							{
								// Actually this is a warning!
								$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_status_comments_not_supported']) ,'detail'=>'');
								$status_comment = NULL;
							}
						}
						else
							$status_comment = FALSE; // Not defined, so do not update
						
						// Now update item_instance!
						if(!update_item_instance($item_r['item_id'], $item_r['instance_no'], $item_r['s_status_type'], $status_comment, $borrow_duration))
						{
							$errors = array('error'=>$LANG_VARS['item_instance_not_updated'],'detail'=>mysql_error());
							return FALSE;
						}
					
						return TRUE;
					}
				}
				else //if(update_item($item_r['item_id'], ifempty($title_val, $item_r['title']), $category_val))
				{
					$errors = array('error'=>$LANG_VARS['item_not_updated'],'detail'=>mysql_error());
					return FALSE;
				}
			}
			else
			{
				return "__INVALID_DATA__";
			}
		}
		else//if(is_not_empty_array($parent_item_r) || is_user_owner_of_item($item_r['item_id'], $item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
		{ 
			$errors = array('error'=>$LANG_VARS['cannot_update_item_not_owned'],'detail'=>'');
			opendb_log("Attempted to update an item they do not own (item_id=".$item_r['item_id'].", instance_no=".$item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
			return FALSE;
		}
	}
	else //if(is_empty_array($parent_item_r) || is_user_owner_of_item($parent_item_r['item_id'], $parent_item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
	{
		$errors = array('error'=>$LANG_VARS['cannot_update_item_not_owned'],'detail'=>'');
		opendb_log("Attempted to update linked item for parent item they do not own (item_id=".$item_r['item_id'].", parent_id=".$parent_item_r['item_id'].", parent_instance_no=".$parent_item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
		return FALSE;		
	}
}

/*
* Update s_status_type value only.  Used to toggle between status'.  However
* the calling process will be passing the new s_status_type value, so there
* is no need for clever logic in this function.
* 
* Return values:
* 
* 	__ABORTED__
* 	__CONFIRM__
*/
function handle_item_instance_update($parent_item_r, $item_r, $status_type_r, $HTTP_VARS, &$errors)
{
	global $LANG_VARS;
	global $HTTP_SESSION_VARS;

	if(is_empty_array($parent_item_r))
	{
		if(is_not_empty_array($item_r))
		{
			if(is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
			{
				if($HTTP_VARS['confirmed'] == 'true')
				{
					$status_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'STATUSTYPE');
					$update_status_type = $HTTP_VARS[get_field_name($status_attr_type_r['s_attribute_type'], $status_attr_type_r['order_no'])];
						
					// Do not allow use of default s_status_type, it needs to be explicitly specified.
					if(strlen($update_status_type)>0 && $item_r['s_status_type'] != $update_status_type)
						$update_status_type_r = fetch_status_type_r($update_status_type);
					else // Otherwise s_status_type is not being udated
					{
						$update_status_type = $status_type_r['s_status_type'];
						$update_status_type_r = $status_type_r; // current s_status_type
					}
							
					// If $update_status_type_r not defined, then we are not updating the s_status_type, only the status_comment
					if($item_r['s_status_type'] == $update_status_type || is_update_status_type_valid($item_r['item_id'], $item_r['instance_no'], $item_r['owner_id'], $status_type_r, $update_status_type_r, $errors))
					{
						$duration_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'DURATION');
						$borrow_duration = get_item_attr_col_value(
										$duration_attr_type_r['s_attribute_type'], 
										$HTTP_VARS[get_field_name($duration_attr_type_r['s_attribute_type'], $duration_attr_type_r['order_no'])]);
																
						if($borrow_duration !== NULL)
						{
							// If borrow_duration is equal to 'N', it means borrowing is not available, but the borrow
							// duration can still be updated.
							if(strlen($borrow_duration)>0 && $update_status_type_r['borrow_ind'] != 'Y' && $update_status_type_r['borrow_ind'] != 'N' && $update_status_type_r['borrow_ind'] != 'B')
							{
								// Actually this is a warning!
								$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $update_status_type_r['description'], $LANG_VARS['s_status_type_borrow_duration_not_supported']) ,'detail'=>'');
								$borrow_duration = NULL;
							}
						}
						else
							$borrow_duration = FALSE; // Not defined, so do not update
						
						$statuscmnt_attr_type_r = fetch_sfieldtype_item_attribute_type_r($item_r['s_item_type'], 'STATUSCMNT');
						$status_comment = $HTTP_VARS[get_field_name($statuscmnt_attr_type_r['s_attribute_type'], $statuscmnt_attr_type_r['order_no'])];
							
						if($status_comment !== NULL)
						{
							if(strlen($status_comment)>0 && $update_status_type_r['status_comment_ind'] != 'Y' && $update_status_type_r['status_comment_ind'] != 'H')
							{
								// Actually this is a warning!
								$errors = array('error'=>replace_lang_var('s_status_type_desc', $update_status_type_r['description'], $LANG_VARS['s_status_type_status_comments_not_supported']) ,'detail'=>'');
							}
						}
						else
							$status_comment = FALSE; // Not defined, so do not update

						// Borrow Duration is not updated, when updating instance
						if(update_item_instance($item_r['item_id'], $item_r['instance_no'], $update_status_type, $status_comment, $borrow_duration))
						{
							return TRUE;
						}
						else//if(update_item_instance($item_r['item_id'], $item_r['instance_no'], $update_status_type, $HTTP_VARS['status_comment'], FALSE))
						{
							$errors = array('error'=>$LANG_VARS['item_instance_not_updated'],'detail'=>mysql_error());
							return FALSE;
						}
					}
					else//if(is_update_status_type_valid($item_r['item_id'], $item_r['instance_no'], $item_r['owner_id'], $status_type_r, $update_status_type_r, $errors))
					{
						return FALSE;
					}
				}
				else if($HTTP_VARS['confirmed'] != 'false')
				{
					return "__CONFIRM__";
				}
				else // confirmation required.
				{
					return "__ABORTED__";				
				}
			}
			else//if(is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || is_user_owner_of_item($item_r['item_id'], $item_r['instance_no'], $HTTP_SESSION_VARS['user_id']))
			{
				$errors = array('error'=>$LANG_VARS['cannot_update_item_not_owned'],detail=>'');
				opendb_log("Attempted to update item instance they do not own (item_id=".$item_r['item_id'].", instance_no=".$item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
				return FALSE;
			}
		}
		else//if(is_not_empty_array($item_r))
		{
			$errors = array('error'=>$LANG_VARS['item_not_found'],'detail'=>'');
			return FALSE;
		}
	}
	else//if(is_empty_array($parent_item_r))
	{
		$errors = array('error'=>$LANG_VARS['operation_not_available'],'detail'=>'');
		return FALSE;
	}
}

function fetch_attribute_type_rs_into_array($results)
{
	$attribute_type_array = array();
	
	if($results)
	{
		while($attribute_val_r = mysql_fetch_array($results, MYSQL_ASSOC))
		{
			$attribute_type_array[] = $attribute_val_r;
		}
		mysql_free_result($results);
	}
	return $attribute_type_array;
}

function search_attribute_val(&$attribute_val_rs_array, $attribute_type)
{
	if(is_array($attribute_val_rs_array))
	{
		$attribute_val = NULL;
		for($i = 0; $i < count($attribute_val_rs_array); $i++)
		{
			if(strcmp($attribute_val_rs_array[$i]['s_attribute_type'],$attribute_type) === 0)
			{
				$attribute_val = $attribute_val_rs_array[$i]['attribute_val'];
				
				// Remove matched attribute value array element.
				array_splice($attribute_val_rs_array, $i, 1);
				
				break; // break of of for loop to return attribute_val
			}
		}
		return $attribute_val;

	}
	else
		return FALSE;
}

function copy_item_attributes($old_item_type, $old_item_id, $new_item_type, $new_item_id)
{
	$results = fetch_item_attribute_type_with_val_rs($old_item_type, $old_item_id, FALSE, TRUE);
	if($results)
	{
		$attribute_val_rs_array = fetch_attribute_type_rs_into_array($results);
		if(is_array($attribute_val_rs_array))
		{
			$results = fetch_item_attribute_type_rs($new_item_type, TRUE, TRUE);
			if($results)
			{
				while($item_attribute_type_r = mysql_fetch_array($results))
				{
					$attribute_val = search_attribute_val($attribute_val_rs_array, $item_attribute_type_r['s_attribute_type']);
					if(strlen($attribute_val)>0)
					{
						insert_item_attribute(
							$new_item_id, 
							$item_attribute_type_r['s_attribute_type'], 
							$item_attribute_type_r['order_no'], 
							$attribute_val);
					}
				}
				mysql_free_result($results);
			}
		}
	}
}

/*
* @param $item_r - Is the item_instance we are going to copy.
* 
*/
function handle_clone_item(&$item_r, $HTTP_VARS, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;
	
	if($CONFIG_VARS['item_input.clone_item_support']!==FALSE)
	{
		if(is_not_empty_array($item_r))
		{
			if(strlen($item_r['parent_id']) == 0)
			{
				if($HTTP_VARS['confirmed'] == 'true')
				{
					// Ensure valid owner_id specified
					if(strlen($HTTP_VARS['owner_id'])>0 && ($HTTP_VARS['owner_id'] == $HTTP_SESSION_VARS['user_id'] || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type'])))
						$owner_id = $HTTP_VARS['owner_id'];
					else
						$owner_id = $HTTP_SESSION_VARS['user_id'];
	
					if(strlen($HTTP_VARS['s_status_type'])>0)
						$status_type_r = fetch_status_type_r($HTTP_VARS['s_status_type']);
					else
						$status_type_r = fetch_status_type_r($item_r['s_status_type']);
							
					if(is_newitem_status_type_valid($owner_id, $status_type_r, $errors))
					{
						if(strlen($HTTP_VARS['s_item_type'])>0)
							$new_item_type = $HTTP_VARS['s_item_type'];
						else
							$new_item_type = $item_r['s_item_type'];
						
						$new_item_id = insert_item(NULL, $new_item_type, $item_r['title'], $item_r['category'], $owner_id);
						if($new_item_id!==FALSE)
						{
							$old_item_id = $item_r['item_id'];
							$old_item_type = $item_r['s_item_type'];
							
							if(strlen($HTTP_VARS['borrow_duration'])==0)
								$HTTP_VARS['borrow_duration'] = $item_r['borrow_duration'];
								
							if(strlen($HTTP_VARS['borrow_duration'])>0)
							{
								if($status_type_r['borrow_ind'] != 'Y' && $status_type_r['borrow_ind'] != 'N' && $status_type_r['borrow_ind'] != 'B')
								{
									// Actually this is a warning!
									$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_borrow_duration_not_supported']) ,'detail'=>'');
									$HTTP_VARS['borrow_duration'] = NULL;
								}
							}
							
							if(strlen($HTTP_VARS['status_comment'])>0)
							{
								if($status_type_r['status_comment_ind'] != 'Y' && $status_type_r['status_comment_ind'] != 'H')
								{
									// Actually this is a warning!
									$errors[] = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['s_status_type_status_comments_not_supported']) ,'detail'=>'');
									$HTTP_VARS['status_comment'] = NULL;
								}
							}

							$instance_no = insert_newitem_instance($new_item_id, $status_type_r['s_status_type'], $HTTP_VARS['status_comment'], $HTTP_VARS['borrow_duration'], $owner_id);
							if($instance_no!==FALSE)
							{
								// Now update item_r for new cloned item.
								$item_r['item_id'] = $new_item_id;
								$item_r['instance_no'] = $instance_no;
								$item_r['s_item_type'] = $new_item_type;
								
								copy_item_attributes($old_item_type, $old_item_id, $new_item_type, $new_item_id);
									
								// Copy child items
								$results = fetch_child_item_rs($old_item_id);
								if($results)
								{
									while($child_item_r = mysql_fetch_array($results, MYSQL_ASSOC))
									{
										// Either we are coercing all children, or leave as is!
										if($HTTP_VARS['coerce_child_item_type'] == 'Y')
											$new_child_item_type = $new_item_type;
										else
											$new_child_item_type = $child_item_r['s_item_type'];
											
										$new_child_item_id = insert_item($new_item_id, $new_child_item_type, $child_item_r['title'], $child_item_r['category'], $owner_id);
										copy_item_attributes($child_item_r['s_item_type'], $child_item_r['item_id'], $new_child_item_type, $new_child_item_id);
									}
									mysql_free_result($results);
								}
								
								// As soon as we get here, we have correctly cloned the item.									
								return TRUE;
							}
							else//if($instance_no!==FALSE)
							{
								$errors = array('error'=>$LANG_VARS['failed_to_clone_item'],detail=>'');
								opendb_log("Failed to create item_instance record for cloned item (item_id=".$item_r['item_id'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
								return FALSE;
							}
						}
						else//if($new_item_id!==FALSE)
						{
							$errors = array('error'=>$LANG_VARS['failed_to_clone_item'],detail=>'');
							opendb_log("Failed to clone item (item_id=".$item_r['item_id'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
							return FALSE;
						}
					}
					else//if(is_newitem_status_type_valid($owner_id, $status_type_r, $errors))
					{
						return FALSE;
					}
				}
				else if($HTTP_VARS['confirmed'] != 'false')
				{
					return "__CONFIRM__";
				}
				else // confirmation required.
				{
					return "__ABORTED__";				
				}
			}
			else//if(strlen($item_r['parent_id']) == 0)
			{
				$errors = array('error'=>$LANG_VARS['cannot_clone_linked_item'],detail=>'');
				opendb_log("User attempted to clone linked item (item_id=".$item_r['item_id'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
				return FALSE;
			}
		}
		else//if(is_not_empty_array($item_r))
		{
			$errors = array('error'=>$LANG_VARS['item_not_found'],'detail'=>'');
			return FALSE;
		}
	}
	else
	{
		$errors = array('error'=>$LANG_VARS['clone_item_not_supported'],detail=>'');
		return FALSE;
	}
}

/*
 * This function assumes, that the delete confirm functionality has already
 * been processed, before being called.  It does not confirm delete checking
 * as a result.
 * 
 * Returns:
 * 	TRUE  			-	Successful execution
 *  FALSE 	 		-	Failed execution
 *  "__CONFIRM__" 	-	Operation requires confirmation
 *  "__ABORTED__"	-	Operation was aborted
 */
function handle_item_delete($parent_item_r, $item_r, $status_type_r, $HTTP_VARS, &$errors)
{
	global $LANG_VARS;
	global $CONFIG_VARS;
	global $HTTP_SESSION_VARS;
	
	if(is_empty_array($parent_item_r) || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $parent_item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
	{
		// If $parent_item_r defined, then the test for parent ownership is sufficient!
		if(is_not_empty_array($parent_item_r) || is_user_admin($HTTP_SESSION_VARS['user_id'], $HTTP_SESSION_VARS['user_type']) || $item_r['owner_id'] == $HTTP_SESSION_VARS['user_id'])
		{
			if(is_not_empty_array($parent_item_r) || $status_type_r['delete_ind'] == 'Y')
			{
				// Child item cannot have any borrowed items attached.
				if(is_not_empty_array($parent_item_r) || !is_exists_item_instance_borrowed_item($item_r['item_id'], $item_r['instance_no']))
				{
					if($HTTP_VARS['confirmed'] == 'true' || ((is_not_empty_array($parent_item_r) && $CONFIG_VARS['item_input.confirm_linked_item_delete']!==TRUE) || (is_empty_array($parent_item_r) && $CONFIG_VARS['item_input.confirm_item_delete']===FALSE)))
					{
						// If normal item, (Not a child!)
						if(is_empty_array($parent_item_r))
						{
							if(!delete_item_instance($item_r['item_id'], $item_r['instance_no']))
							{
								$errors = array('error'=>$LANG_VARS['item_not_deleted'],'detail'=>mysql_error());
								return FALSE;
							}
						}
	
						// If child or no more instance left, 
						// proceed with item and item_attribute delete.
						if(is_not_empty_array($parent_item_r) || !is_exists_item_instance($item_r['item_id'], NULL))
						{
							if(delete_item($item_r['item_id']))
							{
								$results = fetch_item_attribute_type_with_val_rs($item_r['s_item_type'], $item_r['item_id'], FALSE, FALSE, TRUE);
								if($results)
								{
									while($item_attribute_type_r = mysql_fetch_array($results, MYSQL_ASSOC))
									{
										if(delete_item_attribute($item_r['item_id'], $item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no'], $HTTP_SESSION_VARS['user_id']))
										{
											$fieldname = get_field_name($item_attribute_type_r['s_attribute_type'], $item_attribute_type_r['order_no']);
						
											// We have logged whether the file was deleted or not, but do not bother reporting
											// failure the user, because what could they do.
											delete_item_attribute_file($item_attribute_type_r['attribute_val'], $item_attribute_type_r['input_type'], $fieldname, $item_r['item_id']);	
										}
									}
								}
							}
							
							// Get rid of all reviews as well.
							if(is_item_reviewed($item_r['item_id']))
								delete_reviews($item_r['item_id']);
							
							// As long as delete_item has worked, we do not care about anything else.
							return TRUE;
						}
					}
					else if($HTTP_VARS['confirmed'] != 'false')
					{
						return "__CONFIRM__";
					}
					else // confirmation required.
					{
						return "__ABORTED__";				
					}
				}
				else
				{
					if(is_item_reserved_or_borrowed($item_r['item_id'], $item_r['instance_no']))
						$errors = array('error'=>$LANG_VARS['item_not_deleted'],'detail'=>$LANG_VARS['item_reserved_or_borrowed']);
					else
						$errors = array('error'=>$LANG_VARS['item_not_deleted'],'detail'=>$LANG_VARS['item_has_inactive_borrowed_item']);
						
					opendb_log("Attempted to delete item with attached reservation record(s) (item_id=".$item_r['item_id'].", instance_no=".$item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
					return FALSE;
				}
			}
			else//if($status_type_r['delete_ind'] == 'Y')
			{
				$errors = array('error'=>replace_lang_var('s_status_type_desc', $status_type_r['description'], $LANG_VARS['operation_not_avail_s_status_type']),'detail'=>'');
				return FALSE;
			}
		}
		else
		{
			$errors = array('error'=>$LANG_VARS['cannot_delete_item_not_owned'],'detail'=>'');
			opendb_log("Attempted to delete item instance they do not own (item_id=".$item_r['item_id'].", instance_no=".$item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
			return FALSE;
		}
	}
	else // not owner of parent item.
	{
		$errors = array('error'=>$LANG_VARS['cannot_update_item_not_owned'],'detail'=>'');
		opendb_log("Attempted to delete linked item from parent they do not own (item_id=".$item_r['item_id'].", parent_id=".$parent_item_r['item_id'].", parent_instance_no=".$parent_item_r['instance_no'].", update_who=".$HTTP_SESSION_VARS['user_id'].")");
		return FALSE;
	}
}
?>
