/*
 * archive-file.cc --
 *
 *      Archive file datatype functions
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "archive/archive-file.h"
#include "archive/rtp-archive.h"
#include "misc/nethost.h"
#include "misc/mtrace.h"
#ifndef WIN32
#   include <sys/param.h>
#endif
#include "archive/timeval.h"


DEFINE_OTCL_CLASS(ArchiveFile, "ArchiveFile") {
	INSTPROC(open);
	INSTPROC(close);
	INSTPROC(header);
	INSTPROC(private_header);

	PROC(ts2string);
}


DEFINE_OTCL_CLASS(DataFile, "ArchiveFile/Data") {
}


DEFINE_OTCL_CLASS(IndexFile, "ArchiveFile/Index") {
}


int
ArchiveFile::Write(const FileHeader *hdr, const Byte *pb /*=NULL*/)
{
	Tcl &tcl = Tcl::instance();
	int current;
	u_int32_t privateLen;

	if (IsOpen()==FALSE) {
		tcl.resultf("file not open");
		goto error;
	}

	current     = Tell();
	privateLen  = net2host(hdr->privateLen);
	headerSize_ = sizeof(FileHeader) + privateLen;

	if (Seek(0, SEEK_SET) < 0) {
		tcl.resultf("cannot seek to beginning of file: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	if (Write((Byte*)hdr, sizeof(FileHeader)) != sizeof(FileHeader)) {
		tcl.resultf("cannot write file header: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	if (pb!=NULL && privateLen > 0) {
		if (Write((Byte*)pb, privateLen) < 0) {
			tcl.resultf("cannot write private header: %s",
				    strerror(Tcl_GetErrno()));
			goto error;
		}
	}

	if (Seek(current, SEEK_SET) < 0) {
		tcl.resultf("cannot seek to original position: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	tcl.resultf("");
	return sizeof(FileHeader);

error:
	tcl.add_errorf("\ninvoked from within "
		       "ArchiveFile::Write(const FileHeader *, ...)");
	return -1;
}


/*
 * header might be NULL as well!
 * this allows us to simply read the header and figure out the start of the
 * actual contents
 */
int
ArchiveFile::Read(FileHeader *hdr /*=NULL*/, Byte *pb /*=NULL*/,
		  u_int32_t pbLen /*=0*/)
{
	Tcl &tcl = Tcl::instance();
	FileHeader tmp, *header=&tmp;
	int current;
	u_int32_t privateLen;
	int headerRead;

	if (hdr!=NULL) header = hdr;

	if (IsOpen()==FALSE) {
		tcl.resultf("file not open");
		goto error;
	}

	current = Tell();
	if (Seek(0, SEEK_SET) < 0) {
		tcl.resultf("cannot seek to beginning of file: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}
	if (Read((Byte*) header, sizeof(FileHeader)) != sizeof(FileHeader)) {
		tcl.resultf("cannot read header from file: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}
	headerRead = sizeof(FileHeader);
	privateLen = net2host(header->privateLen);
	if (privateLen==0 || pb==NULL) {
		goto end;
	}

	if (pbLen < privateLen) {
		tcl.resultf("private header contains %lu bytes, buffer"
			    " has space for only %lu bytes",
			    (unsigned long) privateLen,
			    (unsigned long) pbLen);
		goto error;
	}

	if (Read((Byte *)pb, privateLen) != int(privateLen)) {
		tcl.resultf("cannot read private header from file: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}
	headerRead +=privateLen;
end:
	if (Seek(current, SEEK_SET) < 0) {
		tcl.resultf("cannot seek to original position: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	headerSize_ = sizeof(FileHeader) + privateLen;
	tcl.resultf("");
	return headerRead;

error:
	tcl.add_errorf("\ninvoked from within "
		       "ArchiveFile::Read(FileHeader *, ...)");
	return -1;
}


int
ArchiveFile::close(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	Close();
	return TCL_OK;
}


int
ArchiveFile::open(int argc, const char * const *argv)
{
        const char *filename, *mode;
        int permissions;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(filename);
	ARG_DEFAULT(mode, "r");
	ARG_DEFAULT(permissions, 0644);
	END_PARSE_ARGS;

	strncpy(filename_, filename, 99);
	return Open(filename, mode, permissions);
}


int
ArchiveFile::Open(const char *filename, const char *mode, int permissions)
{
	Tcl &tcl = Tcl::instance();

	channel_ = Tcl_OpenFileChannel(tcl.interp(), (char*)filename,
				       (char*)mode, permissions);
	if (channel_==NULL) {
		tcl.resultf("cannot open file: '%s'", filename);
		tcl.add_errorf("\ninvoked from within ArchiveFile::Open()");
		return TCL_ERROR;
	}

	return Tcl_SetChannelOption(tcl.interp(), channel_,
				    "-translation", "binary");
}


int
ArchiveFile::header(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	int returnValue;
	FileHeader hdrHost, hdrNet;
	const char *hdrArray;
	timeval tv;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(hdrArray);
	END_PARSE_ARGS;

	if (IsOpen()==FALSE) {
		tcl.resultf("file not open");
		goto error;
	}

	returnValue = Read(&hdrNet);
	if (returnValue!=sizeof(FileHeader)) {
		tcl.resultf("cannot read file header: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	net2host(hdrNet, hdrHost);

	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "version",
			hdrHost.version, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "protocol",
			hdrHost.protocol, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "media",
			hdrHost.media, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "cname",
			hdrHost.cname, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "name",
			hdrHost.name, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}

	tv.tv_sec = hdrHost.start_sec;
	tv.tv_usec= hdrHost.start_usec;
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "start",
			tvtoa(tv), TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}

	tv.tv_sec = hdrHost.end_sec;
	tv.tv_usec= hdrHost.end_usec;
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "end",
			tvtoa(tv), TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}

	tcl.resultf("");
	return TCL_OK;

error:
	tcl.add_errorf("\ninvoked from within ArchiveFile::Header()");
	return TCL_ERROR;
}

int
ArchiveFile::private_header(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	int returnValue;
	FileHeader hdrNet;
	RTPprivatehdr phdrNet, phdrHost;
	const char *hdrArray;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(hdrArray);
	END_PARSE_ARGS;

	if (IsOpen()==FALSE) {
		tcl.resultf("file not open");
		goto error;
	}

	returnValue = Read(&hdrNet, (Byte *)&phdrNet, sizeof(RTPprivatehdr));
	if (returnValue!=sizeof(FileHeader)+sizeof(RTPprivatehdr)) {
		tcl.resultf("cannot read private file header: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	net2host(phdrNet, phdrHost);

	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "email",
			phdrHost.email, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "phone",
			phdrHost.phone, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "loc",
			phdrHost.loc, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "tool",
			phdrHost.tool, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}
	char c_ssrc[16];
	sprintf(c_ssrc, "%u", phdrHost.ssrc);
	if (Tcl_SetVar2(tcl.interp(), (char*)hdrArray, "ssrc",
			c_ssrc, TCL_LEAVE_ERR_MSG)==NULL) {
		tcl.resultf("\nwhile writing to array '%s'", hdrArray);
		goto error;
	}


	tcl.resultf("");
	return TCL_OK;

error:
	tcl.add_errorf("\ninvoked from within ArchiveFile::Header()");
	return TCL_ERROR;
}


int
ArchiveFile::ts2string(int argc, const char * const * argv)
{
	double ts;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(ts);
	END_PARSE_ARGS;

	time_t time_t_ts = (time_t) ts;

	/* FIXME: not thread safe (both the static string and the call to ctime*/
	static char time_string[256];
	strcpy(time_string, ctime(&time_t_ts));
	// strip the trailing "\n"
	int len = strlen(time_string);
	while (len > 0 && \
	       (time_string[len-1]=='\n' || time_string[len-1]=='\r')) len--;
	time_string[len] = '\0';

	Tcl::instance().result(time_string);
	return TCL_OK;
}


void
host2net(const IndexRecord &idxHost, IndexRecord &idxNet)
{
	idxNet.sentTS_sec  = host2net(idxHost.sentTS_sec);
	idxNet.sentTS_usec = host2net(idxHost.sentTS_usec);
	idxNet.recvTS_sec  = host2net(idxHost.recvTS_sec);
	idxNet.recvTS_usec = host2net(idxHost.recvTS_usec);
	idxNet.seqno       = host2net(idxHost.seqno);
	idxNet.filePointer = host2net(idxHost.filePointer);
}


void
net2host(const IndexRecord &idxNet, IndexRecord &idxHost)
{
	idxHost.sentTS_sec  = net2host(idxNet.sentTS_sec);
	idxHost.sentTS_usec = net2host(idxNet.sentTS_usec);
	idxHost.recvTS_sec  = net2host(idxNet.recvTS_sec);
	idxHost.recvTS_usec = net2host(idxNet.recvTS_usec);
	idxHost.seqno       = net2host(idxNet.seqno);
	idxHost.filePointer = net2host(idxNet.filePointer);
}


void host2net(const FileHeader &hdrHost, FileHeader &hdrNet)
{
	hdrNet = hdrHost;
	hdrNet.start_sec  = host2net(hdrHost.start_sec);
	hdrNet.start_usec = host2net(hdrHost.start_usec);
	hdrNet.end_sec    = host2net(hdrHost.end_sec);
	hdrNet.end_usec   = host2net(hdrHost.end_usec);
	hdrNet.privateLen = host2net(hdrHost.privateLen);
}


void net2host(const FileHeader &hdrNet, FileHeader &hdrHost)
{
	hdrHost = hdrNet;
	hdrHost.start_sec  = net2host(hdrNet.start_sec);
	hdrHost.start_usec = net2host(hdrNet.start_usec);
	hdrHost.end_sec    = net2host(hdrNet.end_sec);
	hdrHost.end_usec   = net2host(hdrNet.end_usec);
	hdrHost.privateLen = net2host(hdrNet.privateLen);
}

