/*
 * lts.cc --
 *
 *      Logical Time System, used for time stamping
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
/*static char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/lts.cc,v 1.17 2002/02/03 03:09:26 lim Exp $";*/
#endif

#include "archive/lts.h"
#include "misc/mtrace.h"
#include "archive/timeval.h"


timeval operator - (timeval tv1, timeval tv2)
{
	timeval ret;

	if (tv2.tv_usec > tv1.tv_usec) {
		tv1.tv_usec += 1000000;
		tv2.tv_sec += 1;
	}
	ret.tv_sec  = tv1.tv_sec  - tv2.tv_sec;
	ret.tv_usec = tv1.tv_usec - tv2.tv_usec;
	return ret;
}


timeval operator + (timeval tv1, timeval tv2)
{
	timeval ret;

	ret.tv_sec = tv1.tv_sec + tv2.tv_sec;
	ret.tv_usec = tv1.tv_usec + tv2.tv_usec;
	if (ret.tv_usec > 1000000L) {
		ret.tv_sec += 1;
		ret.tv_usec -= 1000000L;
	}
	return ret;
}


char *
tvtoa(const timeval &tv)
{
	/* FIXME: not thread safe */
	static char tvStr[256];

	sprintf(tvStr, "%ld.%06ld", (long int) tv.tv_sec,
		(long int) tv.tv_usec);
	return tvStr;
}


DEFINE_OTCL_CLASS(LTS, "LTS") {
	INSTPROC(speed);
	INSTPROC(now_logical);
	INSTPROC(now_system);
	INSTPROC(set_reference);
}


LTS::LTS()
	: speed_(0.0)
{
	refLogical_.tv_sec = refLogical_.tv_usec = 0;
	refSystem_ .tv_sec = refSystem_ .tv_usec = 0;
}


int
LTS::speed(int argc, const char * const *argv)
{
	if (argc==2) {
		// we are just querying the current speed
		char speedStr[TCL_DOUBLE_SPACE+1];
		Tcl &tcl = Tcl::instance();

		Tcl_PrintDouble(tcl.interp(), Speed(), speedStr);
		tcl.resultf("%s", speedStr);
		return TCL_OK;
	}
	else {
		double speedVal;
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(speedVal);
		END_PARSE_ARGS;

		MTrace(trcLTS, ("Changing LTS speed to %lf", speedVal));
		Speed(speedVal);
		ActivateTriggers(LTS_SPEED);
		return TCL_OK;
	}
}


int
LTS::now_logical(int argc, const char *const *argv)
{
	if (argc==2) {
		// we're just querying
		timeval logical = NowLogical();
		Tcl::instance().resultf("%s", tvtoa(logical));
		return TCL_OK;
	}
	else {
		// we're changing the mapping of logical to system time
		double logical;
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(logical);
		END_PARSE_ARGS;

		MTrace(trcLTS, ("Changing the mapping of current system time "
				"to %lf", logical));
		NowLogical(ftotv(logical));
		ActivateTriggers(LTS_REFERENCE);
		return TCL_OK;
	}
}


int
LTS::now_system(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	timeval system = NowSystem();
	Tcl::instance().resultf("%s", tvtoa(system));
	return TCL_OK;
}


int
LTS::set_reference(int argc, const char * const *argv)
{
	double logical, system;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(logical);
	ARG(system);
	END_PARSE_ARGS;

	MTrace(trcLTS, ("Changing reference mapping to %lf (logical) -> "
			"%lf (system)", logical, system));
	SetReference(ftotv(logical), ftotv(system));
	ActivateTriggers(LTS_REFERENCE);
	return TCL_OK;
}


timeval
LTS::LogicalTimeOf(timeval system)
{
	// logical = (system - refSystem_)*speed_ + refLogical_

	timeval logical = ftotv((tvtof(system) - tvtof(refSystem_)) * speed_ +
				tvtof(refLogical_));
	return logical;
}
void
LTS::AdjustLogicalReference(double delta) {
  refLogical_ = ftotv(tvtof(refLogical_)+delta);
}

timeval
LTS::SystemTimeOf(timeval logical)
{
	// system = (logical - refLogical_)/speed_ + refSystem_
	timeval system;

	if (speed_==0) {
		system.tv_sec = system.tv_usec = -1;
	}
	else {
		system = ftotv((tvtof(logical) - tvtof(refLogical_)) / speed_ +
				tvtof(refSystem_));
	}
	return system;
}

// This needs to be fixed, breaks if no triggers are set?
void
LTS::ActivateTriggers(u_int32_t triggerType)
{
	ListIndex idx;
	LTSTrigger *trigger;
	MTrace(trcLTS, ("Trying to activate triggers for trigger-type %u",
			triggerType));
	for (idx = triggers_.getFirst(); triggers_.IsDone(idx)==FALSE;
	     idx = triggers_.getNext(idx)) {
		trigger = triggers_.getData(idx);
		MTrace(trcLTS|trcVerbose, ("Comparing with trigger %p",
					   trigger));
		if ( trigger->type() & triggerType ) {
			MTrace(trcLTS, ("Activating trigger %p", trigger));
			trigger->callback(trigger->type() & triggerType);
		}
	}
}



void
LTSTimer::sched(timeval logical)
{
	timeval now;

	if (lts_->Speed()==0.0) {
		// the speed has been set to zero
		now = lts_->NowLogical();
		if (logical <= now) {
			// the current time is past the timer we're trying to
			// schedule => cause a timeout right away!

			MTrace(trcLTS,("Speed is zero; logical time %s has "
				       "passed, scheduling timeout right away",
					tvtoa(logical)));
			timeout();
			return;
		}

		// the timeout has to happen in the future and logical speed
		// is zero i.e. the timeout won't happen at all!

		MTrace(trcLTS, ("Speed is zero; time %s is in the future, "
				"can't schedule timeout", tvtoa(logical)));
		return;
	}

	timeval system, diff;

	system = lts_->SystemTimeOf(logical);
	now = lts_->NowSystem();

	if (system <= now) {
		// the current time is past the timer we're trying to
		// schedule => cause a timeout right away!

		MTrace(trcLTS, ("Speed is non-zero; system time %s has passed,"
				" scheduling timeout right away",
				tvtoa(system)));
		timeout();
		return;
	}

	diff = system - now;
	MTrace(trcLTS, ("Speed is non-zero; system time %s is %ld.%06ld "
			"seconds in the future,scheduling a timeout for later",
			tvtoa(system), (long)diff.tv_sec,(long)diff.tv_usec));
	msched(diff.tv_sec*1000 + diff.tv_usec/1000);
}
