/*
 * renderer-dali.cc --
 *
 *      A Dali-object renderer: it gets a frame, transforms it into a 
 *      Dali-like structure, and calls the effect implementation. 
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/renderer-dali.cc,v 1.1 2002/02/07 04:17:01 chema Exp $ */


//
//	TBD: it's not clear how to pass the ssrc_ and ts_ values 
//	from a decoder to a renderer (most don't need it). Ketan solved this by 
//	duplicating the decoders (which is bad because it implies maintaining the 
//	same code in two different places) and passing at the target_->recv()
//	a new VidRep structure. It should be easier and more elegant to 
//	permit the Renderer object querying the decoder about ts_ and ssrc_. 
//
//	decimation is passed from tcl when the renderer is instantiated. All 
//	decoders return the value of the variable decimation_ when 
//	XXXDecoder::command("decimation") is executed. This is always 411 in 
//	H.261 (when it's really 420) and 422 in MJPEG (even it could be 411 or 
//	420).
//

#include <stdlib.h>
#include "renderer.h"
#include "vw.h"
#include "rv.h"

#include "tclcl.h"
#include "module.h"

#include "renderer-dali.h"

DaliRenderer::DaliRenderer(int decimation) 
	: Renderer(), input_id_(0), target_(0), effect_(0), frame_handler_(0),
		decimation_(decimation)
{
}

int DaliRenderer::command (int argc, const char*const* argv) {
	Tcl& tcl = Tcl::instance();

	if (argc == 2) {
		if (strcmp(argv[1], "effect") == 0) {
			if (effect_ != NULL) {
				printf ("%s\n", effect_);
			} else {
				printf ("effect_ is empty\n");
			}
			return (TCL_OK);
		}
	}

	if (argc == 3) {
		// effect_ is used for tcl-defined effects (DaliSubprograms)
		if (strcmp(argv[1], "effect") == 0) {
			effect_ = (char *) malloc(strlen(argv[2])+1);
			strcpy(effect_, argv[2]);
			return (TCL_OK);
		}

		if (strcmp(argv[1], "set-frame-buffer") == 0) {
			// get a pointer to the frame
			VidRep *fb;
			fb = (VidRep *)TclObject::lookup(argv[2]);
			set_frame_buffer(fb);

			// keep the frame handler name to be able to do tcl calls
			frame_handler_ = (char *) malloc(strlen(argv[2])+1);
			strcpy(frame_handler_, argv[2]);
			return TCL_OK;
		}
	}

	if ((argc == 3) || (argc == 4)) {
		// target_ is used for C++ defined effects (EffectModule inheriting)
		if (strcmp(argv[1], "attach") == 0) {
			// get the input id
			if (argc == 3) {
				input_id_ = 0;
			} else {
				input_id_ = atoi(argv[3]);
			}
			//target_ = (Module *) TclObject::lookup(argv[2]);
			target_ = (EffectModule *) TclObject::lookup(argv[2]);
			if (target_ == 0) {
				tcl.resultf("%s attach: no such effect: %s",
						argv[0], argv[2]);
				return (TCL_ERROR);
			}
			return (TCL_OK);
		}

	}

	return (Renderer::command(argc, argv));
}


void DaliRenderer::recv (Buffer* bp, int codec, int quality)
{
	recv (bp);
}


void DaliRenderer::recv (Buffer* bp)
{
	printf ("DaliRenderer::recv -> SHOULDN'T BE HERE!!\n");
	abort();
	return;
}

void DaliRenderer::resize (int w, int h) {
	printf ("DaliRenderer::resize -> SHOULDN'T BE HERE!!\n");
	abort();
	return;
}


//
// UncompressedDaliRenderer
//
UncompressedDaliRenderer::UncompressedDaliRenderer(int decimation)
	: DaliRenderer(decimation)
{
	// create the new Uncompressed image
	frame_ = new Uncompressed;

	return;
}


//
//	UncompressedDaliRenderer::init_frame
//
//	Inputs:
//		vf: the input VideoFrame object
//
//	Description:
//	Initializes the Uncompressed frame to match the input 
//	VideoFrame data (dimensions, color subsampling scheme, buffers, ...)
//
void UncompressedDaliRenderer::init_frame(VideoFrame* vf)
{
	// initialize the underlying FrameModule
	FrameModule::size(vf->width_, vf->height_);

	// get the frame horizontal and vertical subsampling ratios
	if (decimation_ == 411) {
		// TBD: this is incoherent with the definition of 411 but all the code 
		//	is wrong!
		decimation_ = 420;
	}
	int css_h, css_v;
	frame_->convert_csss(decimation_, css_h, css_v);

	// use the provided buffer instead of allocating more memory
	u_char *ldata = vf->bp_;
	u_char *crdata = vf->bp_ + (vf->width_ * vf->height_);
	u_char *cbdata = vf->bp_ + (vf->width_ * vf->height_) + 
			((vf->width_/css_h) * (vf->height_/css_v));

	frame_->init (vf->width_, vf->height_, css_h, css_v, css_h, css_v, 
			vf->width_, vf->height_, 0, 0, 
			ldata, crdata, cbdata, VIDREP_OVERLOAD_MEMORY);
	return;
}



int UncompressedDaliRenderer::command (int argc, const char*const* argv)
{
	return (DaliRenderer::command(argc, argv));
}


void UncompressedDaliRenderer::recv (Buffer* bp)
{
	VideoFrame* vf = (VideoFrame*)bp;

	// point our Uncompressed object to the received VideoFrame
	if (!sameframe(vf)) {
		init_frame(vf);
	}

	// keep the timestamp
	now_ = frame_->ts_ = vf->ts_;

//
// TBD: we should think on passing to the effect the rvts_ vector. In 
//	that way a lot of copying could be avoided when frames are static. 
//	Currently the subsampling-scheme adapters are to copy the chroma 
//	contents always!!
//


#if 0
	YuvFrame* yf = (YuvFrame*)vf;
	// count the number of blocks
	u_int now = now_;
	const u_int8_t* ts = yf->crvec_;
	u_int bcnt = 0;
	int w = width_ >> 3;
	int y;
	for (y = height_ >> 3; --y >= 0; ) {
		for (int x = 0; x < w; ++x) {
			if (now == ts[x]) {
				++bcnt;
			}
		}
		ts += w;
	}
	printf ("%i: %i blocks\n", now, bcnt);

	// save the frame to disk
	frame_->save_disk("udl");
	return;
#endif


	// if an EffectModule object is waiting for this frame, send it
	if (target_ != 0) {
		//target_->EffectModule::recv(frame_, input_id_);
		target_->recv(frame_, input_id_);
	}

	// if a tcl module is waiting for this frame, send it
	if ((effect_ != 0) && (frame_ != 0)) {
		Tcl& tcl = Tcl::instance();
		char callback[1024];
		sprintf (callback, "%s recv %s", effect_, frame_handler_);
		tcl.eval(callback);
	}
	return;
}



//
// create the tcl versions of the classes
//

// DaliRenderer is an abstract class, so you cannot create an object
//static class DaliRendererClass : public TclClass {
//public:
//	DaliRendererClass() : TclClass("Renderer/Dali") {}
//  TclObject* create(int argc, const char*const* argv) {
//		if (argc != 4)
//				abort();
//		return (new DaliRenderer());
//	}
//} dalirenderer_class;


static class UncompressedDaliRendererClass : public TclClass {
public:
	UncompressedDaliRendererClass() : TclClass("Renderer/Dali/Uncompressed") {}
  TclObject* create(int argc, const char*const* argv) {
		if (argc != 5)
				abort();
		int decimation = atoi(argv[4]);
		return (new UncompressedDaliRenderer(decimation));
	}
} uncompresseddalirenderer_class;

