#include <string.h>   // for strncpy()

#include "tclcl.h"
#include "tcpchan.h"

#include "srm/srm.h"
#include "srm/appmgr-srm.h"
#include "srm/session-srm.h"

/*
 * Packet header format for an srm cache packet
 * XXX we should leverage the SRM ADU name, but we don't.
 * 32-bit offset
 * 32-bit length
 * 16-bit key length
 * key
 * data
 */
struct cachehdr {
	/* notused */
};


class CacheBuf {
public:
	CacheBuf* next_;
	CacheBuf(const char* key) {
		len_ = 0;
		data_ = 0;
		off_ = 0;
		key_ = new char[strlen(key) + 1];
		strcpy(key_, key);
	}
	~CacheBuf() {
		delete[] data_;
		delete[] key_;
	}
	inline void set(const char* data, int len) {
		delete[] data_;
		len_ = len;
		data_ = new char[len];
		memcpy(data_, data, len);
	}
	inline void copyin(int off, const char* p, int len) {
		memcpy(&data_[off], p, len);
	}
	inline void length(int len) {
		delete [] data_;
		data_ = new char[len];
		len_ = len;
	}
	inline void start() { off_ = 0; }
	inline void adjust(int cc) { off_ += cc; }
	inline int offset() const { return (off_); }
	inline int length() const { return (len_); }
	inline char* data() const { return (data_); }
	inline char* key() const { return (key_); }
private:
	char* data_;
	char* key_;
	int len_;
	int off_;
};

/*
 * The MCManager is MASHCast's subclass from SRM_AppMgr. It
 * defines a number of virtual methods called by the SRM objects. 
 */
class MCManager : public SRM_AppMgr {
public:
	MCManager();
	~MCManager();

        /*
	 * Fill in the next message ("data unit") to be sent out. 
	 * Called by the SRM sending code. 
	 */
	virtual int next_ADU(u_char *db, int len, srm_src &id, 
			     int &pkt_type, int &next);
        /*
	 * Called when a new source is identified. We create a new
	 * MCReceiver to handle any data coming from that source. In
	 * this case, the MCReceivers will be forwarding the data
	 * via TCP to the browser so we pass the MCReceiver's 
	 * constructor the TCP Server Handler so it knows how to send. 
	 */
	virtual SRM_PacketHandler *new_source(const srm_src &sid, 
					      int islocal);

	/*
	 * Application fills in the "last sequence number" that it has 
	 * sent out so far. 
	 */
	virtual int periodic_update(u_char */*buf*/) {
		return 0; 
	}
	
	virtual void handle_request(const srm_src &/*sid*/, u_char */*pb*/, 
				    int /*len*/) {}
	virtual void handle_reply(const srm_src &/*sid*/, u_char */*pb*/, 
				  int /*len*/) {}
	virtual void handle_SA(const srm_src &/*sid*/, u_char */*pb*/, 
			       int /*len*/) {}
	int command(int argc, const char*const* argv);

	/*
	 * Called when we receive a packet from the net
	 * (from ANY source)
	 */
	virtual void recv(u_char *buf, int len);

private:
	SRM_Session *session_;
	/*
	 * hash table routines for actual cache of distributed data
	 */
	void enter(const char* key, const char* buffer, int len);
	CacheBuf* create(const char* key);
	CacheBuf* lookup_by_key(const char* key);
	void start(const char* key);
	Tcl_HashTable cache_;
	CacheBuf* active_;
	CacheBuf* tail_;
};

static class MCMgrTclClass : public TclClass {
public:
	MCMgrTclClass() : TclClass("MCManager") {}
	
	TclObject* create(int argc, const char*const* /*argv*/) {
		if (argc==4) {
			return (new MCManager());
		} else {
			return (TclObject*)NULL;
		}
	}
} mcmgr_class;

/* 
 * Each SRM participant has a MCReceiver for each sender in
 * the group. The job of the MCReceiver is to handle the message
 * when it comes in. In this case, that means caching the document
 * and posting it to the browser.
 */
class MCReceiver : public SRM_PacketHandler {
public:
	MCReceiver(MCManager* p) : manager_(p) {}
	~MCReceiver(void) {}
	virtual void recv(u_char *buf, int len) {
		manager_->recv(buf, len);
	}
protected:
	MCManager* manager_;
};

//virtual
void MCManager::recv(u_char *pkt, int plen)
{
	char key[300];

	u_int32_t off = ntohl(*(u_int32_t*)&pkt[0]);
	u_int32_t len = ntohl(*(u_int32_t*)&pkt[4]);
	u_int32_t keylen = ntohs(*(u_int16_t*)&pkt[8]);
        u_int32_t hlen = keylen + 10;
	u_int32_t last = hlen + off;
	if (last > (u_int32_t)plen || last > len) {
		/*XXX bump an error counter */
		return;
	}
	if (keylen > sizeof(key))
		/*XXX*/
		return;
	memcpy(key, &pkt[10], keylen);
	key[keylen] = 0;

	CacheBuf* cb = lookup_by_key(key);
	if (cb == 0) {
		cb = create(key);
		cb->length(len);
	}
	cb->copyin(off, (char*)&pkt[keylen + 10], plen - hlen);

	Tcl::instance().evalf("%s receive %s %d %d %d",
			      name(), key, off, plen - hlen, len);
}

MCManager::MCManager() :
	SRM_AppMgr(0), session_(0), active_(0), tail_(0)
{
	Tcl_InitHashTable(&cache_, TCL_STRING_KEYS);
	
}

MCManager::~MCManager()
{
/* XXX need to delete hash table and all its contents */
/* And MCReceivers though we don't have refs to them! */
}

//virtual
SRM_PacketHandler *
MCManager::new_source(const srm_src &/*sid*/, int /*islocal*/)
{
	return (new MCReceiver(this));
}

/*
 * Fill in the next message ("data unit") to be sent out. 
 * Called by the SRM sending code. 
 */
//virtual
int
MCManager::next_ADU(u_char *pkt, int pktlen, srm_src &/*id*/, 
			     int &pkt_type, int & next)
{
again:
	int off = active_->offset();
	int len = active_->length();
	int cc = len - off;
	if (cc <= 0) {
		active_ = active_->next_;
		if (active_ != 0)
			goto again;
		return (0);
	}
	int keylen = strlen(active_->key());
	int hlen = keylen + 10;
	if (cc + hlen > pktlen)
		cc = pktlen - hlen;
	*(u_int32_t*)&pkt[0] = htonl(off);
	*(u_int32_t*)&pkt[4] = htonl(len);
	*(u_int16_t*)&pkt[8] = htons(keylen);
	memcpy((char*)&pkt[10], active_->key(), keylen);
	memcpy((char *)&pkt[hlen], active_->data() + off, cc);
	active_->adjust(cc);
	pkt_type = APP_DATA;

	int leftover = len - (off + cc);
	if (leftover > 0) {
		leftover += keylen + 10;
		/*XXX*/
		if (leftover > 1000)
			leftover = 1000;
		next = leftover;
	} else if (active_->next_) {
		/*XXX packet size rather than ADU length */
		int len = active_->next_->length();
		if (len > 1000)
			len = 1000;/*XXX*/
		next = len;
	} else {
		active_ = 0;
	}

        return (cc + keylen + 10);
}

void MCManager::enter(const char* key, const char* buffer, int len)
{
	CacheBuf* cb = create(key);
	cb->set(buffer, len);
}

CacheBuf* MCManager::create(const char* key)
{
	int newEnt;
	Tcl_HashEntry* he = Tcl_CreateHashEntry(&cache_, (char*)key,
						(int*)&newEnt);
	if (!newEnt) {
		/* XXX - shouldn't happen.  delete the old entry */
		CacheBuf* cb = (CacheBuf*)Tcl_GetHashValue(he);
		delete cb;
	}
	CacheBuf* cb = new CacheBuf(key);
	Tcl_SetHashValue(he, (char*)cb);
	return (cb);
}

CacheBuf* MCManager::lookup_by_key(const char* key)
{
	Tcl_HashEntry* he = Tcl_FindHashEntry(&cache_, (char*)key);
	if (he != 0)
		return ((CacheBuf*)Tcl_GetHashValue(he));
	return (0);
}

void MCManager::start(const char* key)
{
	Tcl_HashEntry* he = Tcl_FindHashEntry(&cache_, (char*)key);
	if (he == 0)
		/*XXX shouldn't happen */
		return;

	CacheBuf* cb = (CacheBuf*)Tcl_GetHashValue(he);

	/*
	 * If we're already streaming packets, add this buffer
	 * to the tail of the queue.  Otherwise, activate
	 * the streaming process.
	 */
	cb->next_ = 0;
	if (active_) {
		tail_->next_ = cb;
		tail_ = cb;
	} else {
		active_ = cb;
		tail_ = cb;
		active_->start();
		/*
		 * XXX SRM question - why do we have to give
		 * the byte count?  XXX also, this is wrong
		 * since it wants the next packet size
		 * not the next chunk size...
		 */
		int len = active_->length();
		if (len  > 1000)
			len = 1000;/*XXX*/
		session_->begin_xmit(len);
	}
}

int 
MCManager::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (!strcmp(argv[1],"attach_session")) {
		SRM_Session* p = (SRM_Session*)TclObject::lookup(argv[2]);
		if (p == 0) {
			tcl.result("Couldn't find session");
			return TCL_ERROR;
		}
		session_ = p;
		return (TCL_OK);
	}
	if (!strcmp(argv[1], "lookup")) {
		CacheBuf* cb = lookup_by_key(argv[2]);
		if (cb == 0)
			return (TCL_OK);

		char* prefix = cb->data();
		int len = cb->length();
		int plen = strlen(prefix) + 1;
		char* data = prefix + plen;
		len -= plen;

		/*
		 * XXX hack - the data is passed in a buffer of binary
		 * data stored in the global variable buffer$self.
		 */
		char wrk[20];
		sprintf(wrk, "buffer%s", name());
		Tcl_Obj* id = Tcl_NewStringObj(wrk, strlen(wrk));
		/*XXX maybe we should save the Tcl_Obj in CacheBuf */
		Tcl_Obj* p = Tcl_NewStringObj(data, len);

		(void)Tcl_ObjSetVar2(tcl.interp(), id, (Tcl_Obj *)0, p,
				     TCL_GLOBAL_ONLY);

		tcl.result(prefix);
		return (TCL_OK);
	}
			
	/*
	 * <otcl> MultiCache insert { key data len }
	 * Insert the data with the corresponding key
	 * into the underlying reliable multicast session.
	 * Causes the data to be sent over the network.
	 * XXX data/len should perhaps be replaced with a bufferID.
	 * XXX newer hack - buffer pass in global var
	 */
	if (!strcmp(argv[1], "insert")) {
		if (argc != 5) {
			tcl.result("wrong number of args to send");
			return (TCL_ERROR);
		}
		const char* key = argv[2];
		int len = atoi(argv[3]);
		/*XXX hack - a hook to put some prefixed text */
		const char* prefix = argv[4];
		/*
		 * XXX hack - the data is passed in a buffer of binary
		 * data stored in the global variable buffer$self.
		 */
		char wrk[20];
		sprintf(wrk, "buffer%s", name());
		Tcl_Obj* id = Tcl_NewStringObj(wrk, strlen(wrk));
		Tcl_Obj* p = Tcl_ObjGetVar2(tcl.interp(), id, (Tcl_Obj *)0,
					    TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY);
		if (p == 0)
			/*XXX tcl should set up this var */
			abort();

		int slen;
		char* data = Tcl_GetStringFromObj(p, &slen);
		if (len != slen)
			abort();
		/*
		 * Enter the page into the cache, then
		 * kick it out to the session.
		 */

		CacheBuf* cb = create(key);
		int plen = strlen(prefix);
		cb->length(len + plen + 1);
		cb->copyin(0, prefix, plen + 1);
		cb->copyin(plen + 1, data, len);
		start(key);

		/*
		 * Free objects.
		 */
		Tcl_DecrRefCount(id);
		Tcl_DecrRefCount(p);

		return (TCL_OK);
	}
	return (TclObject::command(argc, argv));
}






