/*
 * np-stubs.cc --
 *
 *      Wrapper implementations of plugin API functions which invoke the
 *      corresponding C++ object methods
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "np-plugin.h"
#include <stdio.h>


#ifdef XP_UNIX
char*
NPP_GetMIMEDescription(void)
{
	return PluginInstance::mimeDescription;
}


NPError
NPP_GetValue(void */*future*/, NPPVariable variable, void *value)
{
	NPError err = NPERR_NO_ERROR;

	switch (variable) {
	case NPPVpluginNameString:
		*((char **)value) = PluginInstance::pluginName;
		break;

	case NPPVpluginDescriptionString:
		*((char **)value) = PluginInstance::pluginDescription;
		break;

	default:
		err = NPERR_GENERIC_ERROR;
	}
	return err;
}
#endif


NPError
NPP_Initialize(void)
{
	/*
	 * Plugin developers must define PluginInstance::Initialize() in
	 * their code
	 */
	NPError error = PluginInstance::Initialize();
	return error;
}


jref
NPP_GetJavaClass()
{
	return NULL;
}


void
NPP_Shutdown(void)
{
	/*
	 * Plugin developers must define PluginInstance::Shutdown() in
	 * their code
	 */
	PluginInstance::Shutdown();
}


NPError
NPP_New(NPMIMEType mimeType,
	NPP instance,
	uint16 mode,
	int16 argc,
	char* argn[],
	char* argv[],
	NPSavedData* saved)
{
	if (instance==NULL) return NPERR_INVALID_INSTANCE_ERROR;

	PluginInstance *plugin =
		PluginInstance::CreateInstance(mimeType, instance, mode,
					       argc, argn, argv, saved);
	instance->pdata = (void*) plugin;
	if (plugin==NULL)
		return NPERR_OUT_OF_MEMORY_ERROR;

	return NPERR_NO_ERROR;
}


NPError
NPP_Destroy(NPP instance, NPSavedData** save)
{
	if (instance==NULL) return NPERR_INVALID_INSTANCE_ERROR;

	PluginInstance* plugin = (PluginInstance*) instance->pdata;
	if (plugin==NULL) return NPERR_INVALID_PLUGIN_ERROR;

	NPError result = plugin->SaveInstanceData(save);
	delete plugin;

	return result;
}


NPError
NPP_SetWindow(NPP instance, NPWindow* window)
{
	if (instance==NULL) return NPERR_INVALID_INSTANCE_ERROR;

	PluginInstance* plugin = (PluginInstance*) instance->pdata;
	if (plugin==NULL) return NPERR_INVALID_PLUGIN_ERROR;

	return plugin->SetWindow(window);
}


NPError
NPP_NewStream(NPP instance,
	      NPMIMEType type,
	      NPStream *stream,
	      NPBool seekable,
	      uint16 *stype)
{
	if (instance==NULL)
		return NPERR_INVALID_INSTANCE_ERROR;

	PluginInstance *plugin = (PluginInstance*) instance->pdata;
	if (plugin==NULL) return NPERR_INVALID_PLUGIN_ERROR;

	/*
	 * initialize to the default stream type
	 * CreateStream can override it, if it wants
	 */
	*stype = NP_NORMAL;
	PluginStream *theStream = plugin->CreateStream(type, stream, seekable,
						       stype);
	stream->pdata = (void*) theStream;

	if (theStream==NULL) return NPERR_OUT_OF_MEMORY_ERROR;
	return NPERR_NO_ERROR;
}


/* PLUGIN DEVELOPERS:
 *	These next 2 functions are directly relevant in a plug-in which
 *	handles the data in a streaming manner. If you want zero bytes
 *	because no buffer space is YET available, return 0. As long as
 *	the stream has not been written to the plugin, Navigator will
 *	continue trying to send bytes.  If the plugin doesn't want them,
 *	just return some large number from NPP_WriteReady(), and
 *	ignore them in NPP_Write().  For a NP_ASFILE stream, they are
 *	still called but can safely be ignored using this strategy.
 */

int32
NPP_WriteReady(NPP instance, NPStream *stream)
{
	if (instance==NULL) return -1;

	PluginStream *theStream = (PluginStream*) stream->pdata;
	if (theStream==NULL) return -1;

	return theStream->WriteReady();
}


int32
NPP_Write(NPP instance, NPStream *stream, int32 offset, int32 len,void *buffer)
{
	if (instance==NULL) return -1;

	PluginStream* theStream = (PluginStream*) stream->pdata;
	if (theStream==NULL) return -1;

	return theStream->Write(offset, len, buffer);
	/* return the number of bytes accepted */
}


NPError
NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
	if (instance==NULL) return NPERR_INVALID_INSTANCE_ERROR;

	PluginStream* theStream = (PluginStream*) stream->pdata;
	if (theStream==NULL) return NPERR_GENERIC_ERROR;

	NPError retErr = theStream->Destroy(reason);
	delete theStream;

	return retErr;
}


void
NPP_StreamAsFile(NPP instance, NPStream *stream, const char* fname)
{
	if (instance==NULL) return;

	PluginStream* theStream = (PluginStream*) stream->pdata;
	if (theStream==NULL) return;

	theStream->StreamAsFile(fname);
}


void
NPP_Print(NPP instance, NPPrint* printInfo)
{
	if (printInfo==NULL) return;   // trap invalid parm

	if (instance != NULL) {
		PluginInstance* plugin = (PluginInstance*) instance->pdata;
		plugin->Print(printInfo);
	}
}


int16
NPP_HandleEvent(NPP instance, void* event)
{
	NPBool eventHandled = FALSE;
	if (instance==NULL) return eventHandled;

	PluginInstance *plugin = (PluginInstance*) instance->pdata;
	if (plugin!=NULL) eventHandled = plugin->HandleEvent(event);

	return eventHandled;
}


void NPP_URLNotify(NPP /*instance*/,
		   const char*  /*url*/,
		   NPReason     /*reason*/,
		   void*        /*notifyData*/)
{
}

