/*
 * synch-buffer.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1999-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "tclcl.h"
#include "ntp-time.h"
#include "synch-buffer.h"
#include "source.h"

static class SynchBufferClass : public TclClass {
public:
  SynchBufferClass() : TclClass("SynchBuffer") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new SynchBuffer());
  }
} synch_buffer_class_;

DEFINE_OTCL_CLASS(SynchLTS, "SynchLTS") {
  INSTPROC(speed);
  INSTPROC(now_logical);
  INSTPROC(now_system);
  INSTPROC(set_reference);
}

void
SynchBuffer::recv(pktbuf *pb) {
  if (lts_ == 0) {
    /* Can't do anything without this. Release packet */
    pb->release();
  }

  rtphdr *rh = (rtphdr *) pb->dp;

  /* Update frame arrival stats */
  if (rh->rh_flags & htons(RTP_M)) {
    num_frames_++;
  }

  /* If resetting reference do that now */
  if (reset_flag_) {
    lts_->set_reference_by_mts(ntohl(rh->rh_ts));
    reset_flag_ = 0;
    reschedule();
  }

  /* Queue the packet */

  if (q_.insert(pb)) {
    /* Packet ended up at the front, call reschedule to reset next
     *  transmission time and/or send packet now if late
     */
    reschedule();
  }
}

void
SynchBuffer::callback(u_int32_t triggerType) {
  reschedule();
}

void
SynchBuffer::timeout() {
  send();
}

void
SynchBuffer::reschedule() {
  if (q_.empty()) {
    return;
  }

  if (lts_ == 0) {
    q_.flush();
    return;
  }

  sched(ftotv(((double) ntohl(q_.head_ts()))/lts_->clock_speed()));
}

void
SynchBuffer::send() {
  if (q_.empty()) {
    return;
  }

  if (lts_ == 0) {
    q_.flush();
    return;
  }

  if (!q_.head_loss()) {
    u_int32_t ts;
    pktbuf *pb = q_.pop();
    rtphdr *rh = (rtphdr *) pb->dp;
    ts = rh->rh_ts;

    last_out_delta_ = (((double) ntohl(ts))/lts_->clock_speed()) - tvtof(lts_->NowLogical());

    if (target_ != 0) {
      target_->recv(pb);
    } else {
      pb->release();
    }

    while((!q_.empty()) && (q_.head_ts() == ts)) {
      pb = q_.pop();
      if (target_ != 0) {
	target_->recv(pb);
      } else {
	pb->release();
      }
    }
  }
  reschedule();
}

int
SynchBuffer::command(int argc, const char*const*argv) {
  Tcl &tcl = Tcl::instance();
  char* res = tcl.buffer();

  if (argc == 2) {
    if (strcmp(argv[1], "lts") == 0) {
      if (lts_ != 0) {
	strcpy(res, lts_->name());
      } else {
	strcpy(res, "");
      }
      tcl.result(res);
      return TCL_OK;
    }
    if (strcmp(argv[1], "target") == 0) {
      if (target_ != 0) {
	strcpy(res, target_->name());
      } else {
	strcpy(res, "");
      }
      tcl.result(res);
      return TCL_OK;
    }
    if (strcmp(argv[1], "reset_reference") == 0) {
      reset_flag_ = 1;
      return TCL_OK;
    }
    if (strcmp(argv[1], "buffer_latency") == 0) {
      if (q_.empty() || (lts_ == 0)) {
	sprintf(res, "%f", last_out_delta_);
      } else {
	double now = tvtof(lts_->NowLogical());
	double then = ((double) ntohl(q_.tail_ts()))/lts_->clock_speed();
	sprintf(res, "%f", (then - now));
      }
      tcl.result(res);
      return TCL_OK;
    }
    if (strcmp(argv[1], "num_frames") == 0) {
      sprintf(res, "%d", num_frames_);
      tcl.result(res);
      return TCL_OK;
    }
  }
  if (argc == 3) {
    if (strcmp(argv[1], "lts") == 0) {
      SynchLTS *new_lts;
      new_lts = (SynchLTS *)TclObject::lookup(argv[2]);

      if (lts_ != 0) {
	lts_->UnsetTrigger((LTSTrigger *)this);
      }

      lts_=new_lts;
      LTSTrigger::LTS_(new_lts);
      LTSTimer::LTS_(new_lts);

      if (lts_ != 0) {
	lts_->SetTrigger((LTSTrigger *) this);
      }

      return TCL_OK;
    }
    if (strcmp(argv[1], "target") == 0) {
      target_ = (Module *)TclObject::lookup(argv[2]);
      return TCL_OK;
    }
    if (strcmp(argv[1], "synch_to_source") == 0) {
      if (lts_ != 0) {
	Source *s = (Source *)TclObject::lookup(argv[2]);
	if (s != 0) {
	  Source::Layer &sl = s->layer(0);
	  timeval tv;
	  tv.tv_sec = (sl.ntp_ts_sec() - GETTIMEOFDAY_TO_NTP_OFFSET);
	  tv.tv_usec = ((u_int32_t) (((double) sl.ntp_ts_fsec()) * 0.00023283));
	  u_int32_t mts = sl.mts();
	  lts_->SetReference(ftotv(((double) mts)/lts_->clock_speed()), tv);
	  reschedule();
	}
      }
      return TCL_OK;
    }
  }
  return (PacketModule::command(argc, argv));
}



int
SynchLTS::command(int argc, const char*const*argv) {
  Tcl &tcl = Tcl::instance();
  char* res = tcl.buffer();

  if (argc == 2) {
    if (strcmp(argv[1], "lts") == 0) {
      LTS *lts = LTS_();
      if (lts != 0) {
	strcpy(res, lts->name());
      } else {
	strcpy(res, "");
      }
      tcl.result(res);
      return TCL_OK;
    }
  }
  if (argc == 3) {
    if (strcmp(argv[1], "lts") == 0) {
      LTS *new_lts;
      new_lts = (LTS *)TclObject::lookup(argv[2]);

      if (LTS_() != 0) {
	LTS_()->UnsetTrigger((LTSTrigger *)this);
      }
      LTS_(new_lts);

      if (new_lts != 0) {
	new_lts->SetTrigger((LTSTrigger *) this);
      }

      return TCL_OK;
    }
    if (strcmp(argv[1], "adjust") == 0) {
      double delta = atof(argv[2]);
      AdjustLogicalReference(delta);
      return TCL_OK;
    }
  }
  return (LTS::command(argc, argv));
}

