/*
 * session-srm.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <math.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#include "source-srm.h"
#include "tclcl.h"
#include "net.h"

#include "timer.h"
#include "ntp-time.h"
#include "session-srm.h"
#include "srm.h"
#include "appmgr-srm.h"
#include <misc/random.h>

class SRM_Request;
class SRM_Reply;

static class SRM_SessionClass : public TclClass {
public:
    SRM_SessionClass() : TclClass("Session/SRM") {
            Random::seed_heuristically();
    }
    TclObject* create(int /*argc*/, const char*const* /*argv*/) {
	return (new SRM_Session);
    }
} srm_session_class;

SRM_Session *SRM_Session::instance_=NULL;

srm_src::operator char *() const
{
  /* FIXME: static string is not thread safe!!! */
  static char string[256];
  sprintf(string, "%lx@%lx", (long unsigned) ss_uid, (long unsigned) ss_addr);
  return string;
}

SRM_Session::SRM_Session() :
    min_delay_(5), min_reply_(100), min_request_(30),
    c1_(1.0), c2_(2.0), d1_(1.0), d2_(2.0),     /* delay parameters */
    mtu_(MAX_BURST_SIZE),
    avbl_bw_(KBps),
    bkt_size_(MAX_BURST_SIZE/2),
    // slow start? We could have given a
    // larger bkt_size to start with.
    ps_inst_(0),
    badversion_(0),
    badoptions_(0),
    badpacket_(0),
    badext_(0),
    nrunt_(0),
    pool_(0),
    pktbuf_(NULL),
    appmgr_(NULL),
    sa_timer_(NULL)
{
    dh_.manager(this);
    ch_.manager(this);
    ps_inst_ = gettime();
    pktbuf_ = new u_char[2 * SRM_MTU];
    memset(pktbuf_, 0, sizeof(pktbuf_));

    phtDelays_ = new Tcl_HashTable;
    Tcl_InitHashTable(phtDelays_, sizeof(srm_src)/sizeof(int));

    instance_ = this;
    /*
     * Yatin: changed SRM_Request::sse_ to SRM_Session::instance_, so we don't
     * have to do an upcall into Tcl and assume the existence of a V(session)
     * variable in the Tcl code
     */
}


SRM_Session::~SRM_Session()
{
    cancel();

    Tcl_HashSearch hsearch;

    Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtDelays_, &hsearch);
    while (pEntry) {
        delete ((SRM_delayInfo *)Tcl_GetHashValue(pEntry));
        pEntry=Tcl_NextHashEntry(&hsearch);
    }
    Tcl_DeleteHashTable(phtDelays_);
    delete phtDelays_;

    delete[] pktbuf_;
}

/*
SRM_Session::alloc_srcid(u_int32_t addr) const
{
    timeval tv;
    ::gettimeofday(&tv, 0);
    u_int32_t srcid = u_int32_t(tv.tv_sec + tv.tv_usec);
    srcid += (u_int32_t)getuid();
    srcid += (u_int32_t)getpid();
    srcid += addr;
    return (srcid);
}*/



/*
 * needed because it is defined as an abstract virtual function inside
 * class SessionHandler
 */
void
SRM_Session::announce(CtrlHandler*)
{

}


int
SRM_Session::command(int argc, const char*const* argv)
{
    Tcl& tcl = Tcl::instance();
    char* cp = tcl.buffer();
    if (argc == 2) {
	if (strcmp(argv[1], "exit") == 0) {
	    exit(0);
	}
	if (strcmp(argv[1], "data-net") == 0) {
	    sprintf(cp, "%s", dh_.net()->name());
	    tcl.result(cp);
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "ctrl-net") == 0) {
	    sprintf(cp, "%s", ch_.DataHandler::net()->name());
	    tcl.result(cp);
	    return (TCL_OK);
	}
        if (strcmp(argv[1], "sa-timer") == 0) {
            if (!sa_timer_) tcl.result("");
            else tcl.result(sa_timer_->name());
	    return (TCL_OK);
	}
    } else if (argc == 3) {
	if (strcmp(argv[1], "source-manager") == 0) {
	    sm_ = (SRM_SourceManager*)TclObject::lookup(argv[2]);
	    if (sm_ == 0)
		printf("session:: Error in initialization of sm_");
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "buffer-pool") == 0) {
	    pool_ = (SRM_BufferPool*)TclObject::lookup(argv[2]);
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "app-mgr") == 0) {
	    appmgr_ = (SRM_AppMgr*)TclObject::lookup(argv[2]);
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "sa-timer") == 0) {
	    sa_timer_ = (SA_Timer*)TclObject::lookup(argv[2]);
	    if (sa_timer_) sa_timer_->session(this);
	    return (TCL_OK);
	}
	/*if (strcmp(argv[1], "random-srcid") == 0) {
	    sprintf(cp, "%u", alloc_srcid(inet_addr(argv[2])));
	    tcl.result(cp);
	    return (TCL_OK);
	}*/
	if (strcmp(argv[1], "mtu") == 0) {
	    mtu_ = atoi(argv[2]);
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "begin-xmit") == 0) {
	    SRM_Packet *sp = (SRM_Packet*)TclObject::lookup(argv[2]);
	    int status = begin_xmit(sp->length());
	    sprintf(cp, "%u", status);
	    tcl.result(cp);
	    return (TCL_OK);
	}
    } else if (argc == 4) {
	/* FIXME ignore channel argument */
	if (strcmp(argv[1], "data-net") == 0) {
	    dh_.net((Network*)TclObject::lookup(argv[2]));
	    return (TCL_OK);
	}
	if (strcmp(argv[1], "ctrl-net") == 0) {
	    ch_.net((Network*)TclObject::lookup(argv[2]));
	    return (TCL_OK);
	}
    } else if (argc == 6) {
        if (strcmp(argv[1], "delay-params")==0) {
            double c1, c2, d1, d2;
            if ( (TCL_OK!=Tcl_GetDouble(tcl.interp(), (char*)argv[2], &c1))
                 || (TCL_OK!=Tcl_GetDouble(tcl.interp(), (char*)argv[3], &c2))
                 || (TCL_OK!=Tcl_GetDouble(tcl.interp(), (char*)argv[4], &d1))
                 || (TCL_OK!=Tcl_GetDouble(tcl.interp(), (char*)argv[5], &d2)) ) {
                Tcl_AddErrorInfo(tcl.interp(), "cannot convert c1");
                return TCL_ERROR;
            } else {
                c1_ = (float)c1; c2_ = (float)c2; d1_ = (float)d1; d2_ = (float)d2;
                MTrace(trcSRM, ("set c1=%.2f c2=%.2f d1=%.2f d2=%.2f",
                                c1_, c2_, d1_, d2_));
                return TCL_OK;
            }
        }
    }
    return (SRM_Transmitter::command(argc, argv));
}


void
SRM_Session::transmit(pktbuf* pb)
{
    Network* n = 0;

    srmhdr *sh = (srmhdr *)pb->data;
    pb->dp = pb->data; // point back to data.
    u_int16_t type = (ntohs(sh->sh_flags) >> 8) & 0x000f;
    switch (type){
	/*
	 * Data and Repair replies tend to be 'bulky',
	 * so send them over the rate-controlled channel
	 * Control packets don't need to be rate controlled
	 */
    case APP_DATA :
    case SRM_REPLY :
	n = dh_.net();
	break;
    case SRM_REQUEST :
    case SRM_CNAME   :
    case SRM_BYE  :
    case SRM_SA   :
	n = ch_.DataHandler::net();
	break;
#ifdef SRM_DEBUG
    default:
        fprintf(stderr,"Err: attempting to send invalid pkt!\n");
#endif // SRM_DEBUG

    };

    /*
     * FIXME check for null network: can happen because sender/receiver
     * subscriptions aren't yet completely separated.
     * THIS NEEDS TO BE FIXED.
     */

    if (n != 0)
    {
	n->send(pb);
        release(pb);
    }
}

/* for now, release() of buffers are all done in SRM_Session, for recv()
 * and transmit() */
void
SRM_Session::recv(DataHandler* dh)
{
    pktbuf* pb = pool_->alloc();
    memset(pb->data, 0, PKTBUF_SIZE);
    u_int32_t addr;
    int port;
    int cc = dh->recv(pb->data, sizeof(pb->data), addr, port);
    MTrace(trcSRM|trcVerbose,("recv %d B data for %x", cc, addr));

    if (cc <= 0)
	return;
    u_int16_t flags = ntohs(*(u_int16_t*)pb->data);
    if ( !((SRM_PROTO << 14 | SRM_COMPAT_BIT << 12 | SRM_VERSION << 6)
	   & flags) || (0 != (flags & 0x3f)) ) {
	++badversion_;
	release(pb);
	return;
    }
    if (cc < int(sizeof(srmhdr))) {
	++nrunt_;
	release(pb);
	return;
    }
    pb->len = cc - sizeof(srmhdr);
    pb->dp = pb->data + sizeof(srmhdr);
    demux(pb, addr);
    release(pb);
}

/* In demux(), addr has the IP address of the sender,
   src_addr is the address of the __original__ source */
void
SRM_Session::demux(pktbuf* pb, u_int32_t /*addr*/)
{
    srmhdr* sh = (srmhdr *)pb->data;
    srm_src ss;
    ss.ss_uid  = ntohl(sh->sh_src.ss_uid);
    ss.ss_addr = sh->sh_src.ss_addr;
    u_int32_t lts  = ntohl(sh->sh_ts);
    u_int16_t seqno= ntohs(sh->sh_seqno);
    u_int16_t type = (ntohs(sh->sh_flags) >> 8) & 0x000f;


    if (type == SRM_REPLY) {
	/*
	 * If it is a repair reply, we don't care to demux it.
	 * The app is handed the ADU and it must cancel the
	 * repair request and update the 'data tree'.
	 */
	appmgr_->handle_reply(ss, (pb->data + sizeof(srmhdr)), pb->len);
	return;
    }
    if (sm_ == 0) {
	printf(" NULL sm_\n");
	return;
    }

    SRM_Source* s = sm_->demux(ss, seqno);
    // This source was never heard before, so create an entry for it.
    if (s == 0) {
      // for some reason the app didn't want to create this source
      // we'll ignore it
      return;
    }
    s->lts(lts);

    /* If I  hear myself, don't pass it up */
    if (sm_->is_local_source(s))
	return;

    // Pass the data buffer to the packet handfler after setting the length
    SRM_PacketHandler *hdlr = s->handler();
    if (hdlr == 0) return;
    hdlr->recv(pb->dp, pb->len);
    return;
}


/* note: we use a fixed buffer for receiving, so there is not release() */
void
SRM_Session::recv(CtrlHandler* ch)
{
    MTrace(trcSRM|trcVerbose, ("Received control information on the control channel"));

    u_int32_t src;
    int port;
    memset(pktbuf_, 0, 2 * SRM_MTU);
    int cc = ch->recv(pktbuf_, 2 * SRM_MTU, src, port);

    if (cc <= 0)
	return;

    u_int16_t flags = ntohs(*(u_int16_t*)pktbuf_);
    if ( !((SRM_PROTO << 14 | SRM_COMPAT_BIT << 12 | SRM_VERSION << 6)
	   & flags) || (0 != (flags & 0x3f)) ) {
	++badversion_;
	return;
    }
    if (cc < int(sizeof(srmhdr))) {
	++nrunt_;
	return;
    }
    parse_ctrl(pktbuf_, cc);
}


void
SRM_Session::parse_ctrl(u_char *pb, int len) {
    srmhdr* sh = (srmhdr *)pb;
    u_int16_t type = (ntohs(sh->sh_flags) >> 8) & 0x000f;
    srm_src src = sh->sh_src;
    src.ss_uid = ntohl(src.ss_uid);

    switch(type) {
    case SRM_REQUEST:
	appmgr_->handle_request(src, (pb + sizeof(srmhdr)),
				(len - sizeof(srmhdr)));
	break;
    case SRM_SA:
    {
        srm_sa* pSA = (srm_sa*)(pb + sizeof(srmhdr));
        u_int32_t nElt = ntohl(pSA->sa_nElt);
	if ((unsigned int)len < sizeof(srmhdr) + sizeof(srm_sa) +
	    nElt*sizeof(srm_dinfo)) {
		++badpacket_;
		break;
	}
        u_int32_t tsend = ntohl(sh->sh_ts);
        MTrace(trcSRM|trcVerbose, ("--- tsend=%f ms, now=%f ms",
                                ntp2msec(tsend), ntp2msec(ntptime()) ));
        update_delay(src, tsend, nElt, &pSA->sa_arDInfo[0]);
        int offset = sizeof(srmhdr) + sizeof(srm_sa) + nElt*sizeof(srm_dinfo);
	appmgr_->handle_SA(src, pb + offset, len - offset);
	break;
    }
    case SRM_CNAME:
    {
	srm_src sid;
	SRM_Source *src;
	char *name;
	int namelen;
	len -= sizeof(srmhdr);
	pb  += sizeof(srmhdr);

	while (len > int(sizeof(srm_src))) {
	    sid = *((srm_src*) pb);
	    name= (char*)pb + sizeof(srm_src);
	    /* FIXME: this might cause problems for corrupted packets!
	     * should send len of name in packet */
	    namelen = strlen(name) + 1;
	    if ( namelen + (int) sizeof(srm_src) > len) {
		    ++badpacket_;
		    break;
	    }
	    sid.ss_uid = ntohl(sid.ss_uid);
	    /* FIXME: create_source will create a new packet handler!
	     *      we should only have packet handlers for active
	     *      sources */
	    /* FIXME: this allows one security attack: create packets
	     *      with different source_id fields, the application
	     *      will be bloated with source data structures in
	     *      notime */
	    src = sm_->demux(sid, 0);

	    // Don't make an upcall per SRM_CNAME packet
	    // Upcall only if there's a change in the CNAME

	    if (!src->cname() || strcmp(name, src->cname())) {
		src->cname(name);
		MTrace(trcSRM, ("Registering new cname for %d@%s: %s",
				sid.ss_uid,InetNtoa(sid.ss_addr),
				src->cname()));
	    }

	    if ( (namelen % 4)!=0 ) {
		namelen = (namelen/4 + 1) * 4;
	    }
	    len -= sizeof(srm_src) + namelen;
	    pb  += sizeof(srm_src) + namelen;
	}
	break;
    }
    default:
	MTrace(trcSRM, ("Err: invalid pkt in ctrl channel! (%d B)", len));
	break;
    }
}


int
SRM_Session::begin_xmit(int len)
{
    double wait_time;
    double cs_inst;
    double earned_bytes = 0;

    if (busy_)
	return (ERR_BUSY);
    if (len > MAX_BURST_SIZE)
	return (ERR_TOO_BIG);

    cs_inst = gettime();
    earned_bytes = (cs_inst - ps_inst_) * avbl_bw_*1e3;
    MTrace(trcSRM | trcVerbose, ("bkt_size_=%0.4g earned_bytes=%0.4g\n",
            bkt_size_, earned_bytes));

    ps_inst_ = cs_inst;

    bkt_size_ +=  earned_bytes;
    if (bkt_size_ > mtu_)
	bkt_size_ = mtu_;

    /* avbl_bw_ is in 1e3 Bytes, wait_time is in ms */
    wait_time = (len - sizeof(srmhdr) - bkt_size_)/avbl_bw_;
    /* if we have enough schedule to execute immediately */
    /* FIXME: will this slow down network response? */
    if (wait_time < 0.0) wait_time = 0.0;
    MTrace(trcSRM | trcVerbose, ("wait_time = %.3g\n", wait_time));
    msched((int)ceil(wait_time));
    busy_ = 1;
    return (ERR_BUSY);
}


void
SRM_Session::timeout()
{
    int cc;
    int type;
    double e_len;
    double cs_inst;
    int next=0;
    struct srm_src id;
    cs_inst = gettime();
    e_len = (cs_inst - ps_inst_)*avbl_bw_*1e3;
    ps_inst_ = cs_inst;
    bkt_size_ += (e_len);
    if (bkt_size_ > mtu_)
	bkt_size_ = mtu_;

    pktbuf *pkt = pool_->alloc();
    id = sm_->first_local_source()->id();
    type = APP_DATA;
    next = 0;
    int maxlen = (bkt_size_ > SRM_MTU) ? SRM_MTU : int(bkt_size_);
    pkt->len = appmgr_->next_ADU(pkt->dp, maxlen - sizeof(srmhdr),
                                 id, type, next);
    MTrace(trcSRM|trcVerbose, ("in timeout, maxplen = %d, next = %d\n",
            maxlen - sizeof(srmhdr), next));

    if (pkt->len > 0) {
	pkt->len += sizeof(srmhdr);
	if (pkt->len > maxlen)
	    fprintf(stderr, "Too much data from application\n");

	cc = send_ADU(pkt, id, type);
	if (cc < 0)
	    perror("send");
	bkt_size_-= cc;
    }
    busy_ = 0;

    if (next != 0)
	begin_xmit(next);
}


int
SRM_Session::send_ADU(pktbuf *pkt, const srm_src &id, int type)
{
    int len = pkt->len;
    /* Set the type field in the header */
    //srmhdr *sh = (srmhdr *)pkt->data;
    pool_->sethdr(pkt, id, type);
    switch (type){
	/*
         * If packet is a repair request delay its tramsmission
         * by the estimated round-trip time.
	 */
    case APP_DATA :
    case SRM_REPLY :
        transmit(pkt);
	break;
    case SRM_BYE  :
    case SRM_SA   :
    case SRM_CNAME :
	break;
    };
    return len;
}


double
SRM_Session::gettime()
{
    timeval tv;
    ::gettimeofday(&tv, 0);
    return (tv.tv_sec + 1e-6 * tv.tv_usec);
}


void
SRM_Session::fill_cname(pktbuf *pkt, SRM_Source *local)
{
    char *lname = local->cname();
    int lname_len;

    MTrace(trcVerbose|trcSRM, ("[fill cname] : cname = %s (%d)", lname, strlen(lname)));

    srm_src *sid = (srm_src*) pkt->dp;
    *sid = local->id();
    sid->ss_uid = htonl(sid->ss_uid);
    strcpy((char*) pkt->dp + sizeof(srm_src), lname);
    lname_len = strlen(lname) + 1;
    if ( (lname_len % 4)!=0 ) {
	// round off the length to a multiple of 4 bytes
	lname_len = (lname_len/4 + 1) * 4;
    }
    pkt->dp += sizeof(srm_src) + lname_len;
    pkt->len+= sizeof(srm_src) + lname_len;
}

double
SRM_Session::request_time(const srm_src& rqt_tgt)
{
    double delay = delay_est(rqt_tgt);

    MTrace(trcSRM|trcVerbose, ("Delay estimate %x@%s is %f", rqt_tgt.ss_uid,
                    intoa(rqt_tgt.ss_addr), delay));
    if (delay==0.0) delay = min_request_;
    if (delay < min_reply_) delay = min_delay_;
    double rand = Random::uniform(c1_, c1_+c2_);
    double waitTime = delay*rand;

    MTrace(trcSRM|trcVerbose, ("Scheduling a request timer for %f ms", waitTime));
    return waitTime;
}

void
SRM_Session::sched_request(SRM_Request *sr, const srm_src& rqt_tgt)
{
    sr->set_source(sm_->first_local_source()->id());
    sr->set_tgt(rqt_tgt);
    sr->msched((int)request_time(rqt_tgt));
}

/* returns a random time to wait before send out the reply, note that
 * this does not include the backoff multiplier */
double
SRM_Session::reply_time(const srm_src& rpy_tgt)
{
    double delay = delay_est(rpy_tgt);

    MTrace(trcSRM, ("Delay estimate for %x@%s is %f", rpy_tgt.ss_uid,
                    intoa(rpy_tgt.ss_addr), delay));
    if (delay == 0.0)       delay = min_reply_;
    if (delay < min_delay_) delay = min_delay_;
    double rand = (double)random()/double(INT_MAX);
    double waitTime = delay*(d1_ + d2_*rand);
    MTrace(trcSRM, ("Scheduling a repair timer for %f", waitTime));
    return waitTime;

}

void
SRM_Session::sched_reply(SRM_Reply *sr, const srm_src& rpy_tgt)
{
    sr->set_source(sm_->first_local_source()->id());
    sr->set_tgt(rpy_tgt);
    sr->msched((int)reply_time(rpy_tgt));
}

/* tsend is in ntp format */
void
SRM_Session::update_delay(const srm_src& srcId, u_int32_t tsend,
                          u_int32_t nElt, srm_dinfo* arDInfo)
{
   /*
    * t1
    *   \
    *   t2
    *    |
    *   t3
    *   /
    * t4
    */
    srm_src local_sid = sm_->first_local_source()->id();
    if (local_sid == srcId) return; // don't care about own SA

    float now = (float) ntptime();
    int created;
    Tcl_HashEntry* pEntry = Tcl_CreateHashEntry(phtDelays_, (char*)&srcId,
                                                &created);
    SRM_delayInfo* pDI=NULL;
    if (created) {
        pDI = new SRM_delayInfo;
        pDI->delay = pDI->t2m1 = 0.0;
        Tcl_SetHashValue(pEntry, pDI);
    } else {
        pDI = (SRM_delayInfo*) Tcl_GetHashValue(pEntry);
    }
    /* relative to incoming packet, t2=now, t1=tsend, so t2m1 = now - tsend */

    /* t2m1 is in 2^-16 secs */
    pDI->t2m1 = now - (float)tsend;
    MTrace(trcSRM|trcVerbose, ("tsend=%f ms; now= %f ms; t2m1=%f ms",
            ntp2msec(tsend), ntp2msec(now), ntp2msec(pDI->t2m1)));

    /* for convenience, change the local source id to network byteorder
     * since addr is already in network byteorder, just byteswap uid */
    local_sid.ss_uid = ntohl(local_sid.ss_uid);
    /*
     * t4=now
     *  td=t1 + delta = t1 + t3 - t2 = t3 - (t2 - t1)
     *  delay_est = 0.5(t4 - t1 - delta) = 0.5(now - td)
     */
    for (u_int i=0; i<nElt; i++) {
        if (arDInfo[i].di_ss==local_sid) {
            /* convert to float to prevent overflow/underflow of unsigned */
            u_int32_t tmp = ntohl( *( (u_int32_t*)&(arDInfo[i].di_td) ) );
            float ftd = *((float*)&tmp);
            MTrace(trcSRM|trcVerbose, ("recv td=%f ms", ntp2msec(ftd)));
            /* REVIEW: decide on the right values for exp. averaging */
            float new_sample = ntp2msec((float)(0.5*(now - ftd)));
            if (new_sample > 0) {
                    if (pDI->delay!=0.0) {
                            pDI->delay = 0.7F*pDI->delay + 0.3F*new_sample;
                    } else {
                            pDI->delay = new_sample;
                    }
                    MTrace(trcSRM, ("DELAY for %s: sample=%.2f, new=%.2fms",
                                    InetNtoa(srcId.ss_addr),
                                    ntp2msec((float)(0.5*(now - ftd))),
                                    pDI->delay));
                    break;
            }
        }
    }
}

/* returns the delay estimate in millisec */
float
SRM_Session::delay_est(const srm_src& src)
{
    if (src.ss_uid==0 && src.ss_addr==0) return 0.0;
    Tcl_HashEntry* pEntry = Tcl_FindHashEntry(phtDelays_, (char*)&src);
    if (!pEntry) return 0.0;
    return ((SRM_delayInfo*)Tcl_GetHashValue(pEntry))->delay;
}
/*
 * Fill in the SRM session announcement information
 *  - updates pb->dp & pb-len
 * Review: should check length to prevent overflow
 */
void
SRM_Session::fill_SA(pktbuf* pb)
{
    srm_sa* pSA = (srm_sa*)(pb->dp);
    Tcl_HashSearch hs;
    int nElt = 0;
    float now = (float)ntptime();
    /* td = t1 + delta, where delta = t3 - t2
          = t1 + t3 - t2
          = t3 - t2m1, where t2m1 = t2 - t1 and t3=now
    */
    float ftmp = 0.0;
    u_int32_t itmp = 0;
    for (Tcl_HashEntry* pEntry=Tcl_FirstHashEntry(phtDelays_, &hs);
         pEntry; pEntry = Tcl_NextHashEntry(&hs) ) {
         srm_src sid = *(srm_src*)Tcl_GetHashKey(phtDelays_, pEntry);
         sid.ss_uid = htonl(sid.ss_uid);
         SRM_delayInfo *pDI = (SRM_delayInfo *)Tcl_GetHashValue(pEntry);
         if (pDI->t2m1==0.0) {
                 continue;      // don't send out machines with invalid ests.
         }
         pSA->sa_arDInfo[nElt].di_ss = sid;
         MTrace(trcSRM|trcVerbose, ("now=%f ms;  t2m1=%f ms", ntp2msec(now),
                         ntp2msec(pDI->t2m1)));
         ftmp = now - pDI->t2m1;
         MTrace(trcSRM|trcVerbose, ("ftmp=%f", ntp2msec(ftmp)));
         itmp = htonl( *((u_int32_t*)&ftmp) );
         pSA->sa_arDInfo[nElt++].di_td = *(float*)&itmp;
         MTrace(trcSRM|trcVerbose,
                ("pkt: (%x)",
                 *(u_int32_t*)&(pSA->sa_arDInfo[nElt-1].di_td)));
    }
    pSA->sa_nElt = htonl(nElt);
    int len = sizeof(srm_sa) + nElt*sizeof(srm_dinfo);
#ifdef SRM_DEBUG
    describeSA(stderr, pb);
#endif // SRM_DEBUG
    pb->len += len;
    pb->dp += len;
}

#ifdef SRM_DEBUG
void
SRM_Session::describeSA(FILE* /*of*/, pktbuf* pb)
{
    srm_sa* pSA = (srm_sa*)(pb->dp);
    int nElt = ntohl(pSA->sa_nElt);

    MTrace(trcSRM, ("nElt=%d  now=%f ms", nElt, ntp2msec(ntptime())));

    for(int i=0; i<nElt; i++) {
        MTrace(trcSRM, ("pkt_td: %x\n",
              *((u_int32_t*)&(pSA->sa_arDInfo[i].di_td))));
        int32_t tmp = ntohl( *( (u_int32_t*)&(pSA->sa_arDInfo[i].di_td) ) );
        float ftd = *((float*)&tmp);

        MTrace(trcSRM, ("%lx@%s td=%f ms\n",
                    ntohl(pSA->sa_arDInfo[i].di_ss.ss_uid),
                    InetNtoa(pSA->sa_arDInfo[i].di_ss.ss_addr),
                    ntp2msec(ftd)));
    }
}
#endif // SRM_DEBUG

SRM_Request::SRM_Request()
    : isDone_(0),  localSrc_(), tgt_(), pb_(0),
      multiplier_(1), ignore_backoff_(0.0)
{
    assert(SRM_Session::instance_ && "no Session/SRM object defined");
}

double
SRM_Request::newtime()
{
        double newtime =
                (SRM_Session::instance_->request_time(tgt_)) * multiplier_;
        MTrace(trcSRM, ("(%x) new rqt timer:%.2f m=%d",
                        this, newtime, multiplier_));
        return newtime;
}


SRM_Reply::SRM_Reply() : SRM_Request()
{
     ignore_ = 0;
}

/*
 * called when the reply is done
 * ignore request for 3*d[s, localSrc_]
 *            (s is either the original source of data or the requester)
 *
 * Caveat: multiple calls to reply_done() will cause the reply to stay for
 *         3*d[s, localSrc_] after the LAST call....
 */
double
SRM_Reply::reply_done()
{
    /* when ignore_ = 1, MBReply::fill_ADU() will cancel the reply on the next
     * timeout, and meanwhile the reply will stay MB's reply queue causing
     * new requests to be ignored
     * FIXME: needs to rethink the mechanism for SRM v2
     */
    ignore_ = 1;
    SRM_Session* ps = SRM_Session::instance_;
    /*
     * tgt_ is the target of reply/requester
     * localSrc_ is the local source
     * origSId() is the original source id
     */
    double d = (localSrc_ == origSId()) ?
            ps->delay_est(tgt_) : ps->delay_est(origSId());
    if (d == 0.0) d=100.0;      // default for unheard of source
    d*=3.0;
    msched((int)d);
    assert(!isDone_);
    return d;
}

void
SRM_Request::fill(const srm_src& ss)
{
    pb_ = SRM_Session::instance_->alloc();
    SRM_Session::instance_->sethdr(pb_, ss, SRM_REQUEST);
    int adu_len = fill_ADU();
    if (adu_len > 0) {
            doubleMult();
    }
    pb_->len = adu_len + sizeof(srmhdr);
};


/*virtual*/
double
SRM_Reply::newtime()
{
        double newtime =
                (SRM_Session::instance_->reply_time(tgt_))*multiplier_;
        MTrace(trcSRM, ("(%x) new repair time %.2f", this, newtime));
        return newtime;
}

void
SRM_Reply::fill(const srm_src& ss) {
    pb_ = SRM_Session::instance_->alloc();
    SRM_Session::instance_->sethdr(pb_, ss, SRM_REPLY);
    int adu_len = fill_ADU();
    pb_->len = adu_len + sizeof(srmhdr);
};

void
SRM_Request::doubleMult()
{
        /* FIXME: det max back off */
        if (multiplier_ < 1024)
                multiplier_*=2; // backoff at most 9 times
}

/*
 * backoff after doubling the multiplier, if we are not in the
 * ignore_backoff stage
 */
int SRM_Request::backoff()
{
    timeval tp;
    if (ignore_backoff_ != 0.0) {
        ::gettimeofday(&tp, 0);
        double now = tp.tv_sec + (tp.tv_usec*1e-6);
        if (now < ignore_backoff_) {
            MTrace(trcSRM, ("(%x) rqt ignore backoff", this));
            return 0;
        } else {
            MTrace(trcSRM, ("(%x) rqt ignore backoff over", this));
            ignore_backoff_ = 0.0; // ignore period exceeded
        }
    }

    // a random timer using original multiplier
    double di = newtime();

    doubleMult();
    double d = newtime();

    /* FIXME: this is slightly different from floyd-ton96, which schedules
     * half way to the expiry time */
    ::gettimeofday(&tp, 0);
    ignore_backoff_ = tp.tv_sec + (tp.tv_usec*1e-6) + (di*0.5e-3);
    MTrace(trcSRM, ("(%x) now=%.4f igbk til %.4f", this,
                    tp.tv_sec + (tp.tv_usec*1e-6), ignore_backoff_));
    msched((int)d);                // reschedule

    return 1;
}

#ifdef OLD
/*
 * Need to clean this up: make this a  common
 * function for all types of packets.
 */
void SRM_Request::fill_hdr(const srm_src& ss) {

    struct srmhdr *sh  = (srmhdr *)(pb_->data);
    /*
     * Version is already filled in by
     * the alloc() function
     */
    sh->sh_flags = htons ( ntohs(sh->sh_flags) | (SRM_REQUEST << 8) );
    sh->sh_ts          = htonl( ntptime() );
    sh->sh_src.ss_uid  = htonl( ss.ss_uid );
    sh->sh_src.ss_addr = ss.ss_addr;
}


void SRM_Reply::fill_hdr(const srm_src& ss) {

    struct srmhdr *sh  = (srmhdr *)(pb_->data);
    /*
     * Version is already filled in by
     * the alloc() function
     */
    sh->sh_flags = htons ( ntohs(sh->sh_flags) | (SRM_REPLY << 8) );
    sh->sh_ts          = htonl( ntptime() );
    sh->sh_src.ss_uid  = htonl( ss.ss_uid );
    sh->sh_src.ss_addr = ss.ss_addr;
}
#endif //old


void
SRM_Request::timeout()
{
	if (isDone_) {
		delete this;
		return;
	}
	if (SRM_Session::instance_ == NULL) {
		fprintf(stderr, "request: session object not found\n");
		return;
	}
	fill(localSrc_);

	if (pb_->len > int(sizeof(srmhdr))) {
		SRM_Session::instance_->transmit(pb_);
		pb_ = NULL;             // transmit() will release the buffer
	}
	/* check for termination in case cancel is called during
	 * the last fill operation */
        if (!isDone_) {
		msched((int)newtime());
	} else {
		delete this;
		/* noting should come after this line before return */
	}
	return;
}

/* same as request, except we have the ignore_ stuff for ignoring new requests */
void
SRM_Reply::timeout()
{
	if (isDone_) {
		delete this;
		return;
	}
	if (SRM_Session::instance_ == NULL) {
		fprintf(stderr, "repair: session object not found\n");
		return;
	}
	fill(localSrc_);
	if (pb_->len > int(sizeof(srmhdr))) {
		SRM_Session::instance_->transmit(pb_);
		pb_ = NULL;
	}
	if (!ignore_) {
		// reply_done have already scheduled otherwise
		msched((int) newtime());
	}
}


/*
 * Session Announcement Timer
 */

static class SA_Timer_Class : public TclClass {
public:
    SA_Timer_Class() : TclClass("TimerSA") {}
    TclObject* create(int /*argc*/, const char*const* /*argv*/) {
        /* REVIEW: decide on values for the timer */
	return (new SA_Timer());
    }
} sa_timer_class;


SA_Timer::~SA_Timer()
{
}

SA_Timer::SA_Timer(int period): Timer()
{
    /* period in milliseconds */
    period_ = period;
    // Make an upcall into tcl to get the
    // value of the current session pointer
    sn_ = 0;

    return;
}


/*
 * Commands to vary the period of messages
 */
int
SA_Timer::command(int argc, const char*const* argv)
{
    Tcl& tcl = Tcl::instance();
    char* cp = tcl.buffer();
    if (argc == 2) {
	if (strcmp(argv[1], "period") == 0) {
	    sprintf(cp, "%u", period_);
	    tcl.result(cp);
	    return (TCL_OK);
	}

    }

    if (argc == 3) {
	/*if (strcmp(argv[1], "session") == 0) {
	    ss_ = (SRM_Session*)TclObject::lookup(argv[2]);
#ifdef SRM_DEBUG
	    printf("\n\n\nSession was set to %s \n\n", argv[2]);
#endif
	    if (ss_ == 0)
		printf("session_announce:: Error in initialization of ss_\n");
	    return (TCL_OK);
	}*/

	if (strcmp(argv[1], "period") == 0) {
	    period_ = atoi(argv[2]);
	    cancel();
	    msched(period_);
	    return (TCL_OK);
	}
    }
    return TCL_ERROR;
}


/*
 * On a timeout, a session update is fetched form the
 * application and sent out on the control channel.
 * Along with this send out the Cname announcement packet
 */
const int cCNAME_FREQ = 5;      // ratio of # cname pkts : # SA's
void
SA_Timer::timeout()
{
    List<SRM_Source> *lsrc;
    srm_src first_sid;

    if (ss_==0) {
	fprintf(stderr, "no session object specified for Timer/SA object\n");
	return;
    }

    lsrc = (ss_->source_manager())->local_sources();
    if (lsrc->PeekAtHead()==NULL) {
	MTrace(trcSRM, ("no local source!\n"));
	msched((int)(period_+0.5*period_*(double)random()/double(INT_MAX)));
	return;
    }
    else {
	first_sid = lsrc->PeekAtHead()->id();
    }

    // Send a CNAME announcement
    // FIXME: for now the srm header in the announcement will contain
    // the srm_src id of the first (of multiple) local source
    /* Imp Note: send out cname with freq cCNAME_FREQ */
    if (sn_ == cCNAME_FREQ - 1) {
            pktbuf *cname_pkt = ss_->pool()->alloc();
            ss_->sethdr(cname_pkt, first_sid, SRM_CNAME);

            cname_pkt->dp = cname_pkt->data + sizeof(srmhdr);
            cname_pkt->len= sizeof(srmhdr);
            ListIndex idx;
            for (idx = lsrc->getFirst();
                 lsrc->IsDone(idx)==FALSE;
                 idx = lsrc->getNext(idx)) {
                    ss_->fill_cname(cname_pkt, lsrc->getData(idx));
            }
            /* A pointer to the SRM session is stored in ss_ */
            ss_->transmit(cname_pkt);
    } else {
            // FIXME: for now the srm header in the announcement will contain
            // the srm_src id of the first (of multiple) local source
            pktbuf *pkt = ss_->pool()->alloc();
            ss_->sethdr(pkt, first_sid, SRM_SA);
            pkt->dp = pkt->data + sizeof(srmhdr);
            pkt->len = sizeof(srmhdr);

            /* fill_SA will update pkt->dp and pkt->len */
            ss_->fill_SA(pkt);
            /* A pointer to the SRM session is stored in ss_ */
            int len  = (ss_->app_manager())->periodic_update(pkt->dp);
            pkt->len += len;
            assert(len < SRM_MTU && "session annoucement too big");

            /* REVIEW: the order is wrong, should set timers after app_mgr */
            srmhdr *sh = (srmhdr*)pkt->data;
            sh->sh_ts  = htonl(ntptime());

            ss_->transmit(pkt);
    }
    sn_ = (sn_ + 1) % cCNAME_FREQ;
    cancel();     // in case somewhere else someone scheduled a timer
    /* randomize timers a bit to prevent global sync */
    msched((int)(period_+0.5*period_*(double)random()/double(INT_MAX)));
}
