/*
 * pktbuf-srmv2.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <tclcl.h>

#include "pktbuf-srmv2.h"
#include "ntp-time.h"
#include "pktbuf.h"
#include "srmv2.h"


#include <math.h>

static class SRMv2_BufferPoolClass : public TclClass {
public:
	SRMv2_BufferPoolClass() : TclClass("BufferPool/SRMv2") {}
	virtual TclObject* create(int, const char*const* )
	{
		return (new SRMv2_BufferPool());
	}
} SRMv2_bp_class;



pktbuf *
SRMv2_BufferPool::alloc(unsigned char type)
{
	pktbuf* pb = BufferPool::alloc();
	build_cmnhdr(pb, type);
	return (pb);
}

/*
 *     (8)      (8)           (16)
 * +--------+---------+---------------------------+
 * | flags  | pkttype |       reserved            |
 * +--------+---------+---------------------------+
 */

void
SRMv2_BufferPool::build_cmnhdr(pktbuf *pb, unsigned char type)
{
	srmv2_cmnhdr *sh = (srmv2_cmnhdr *)pb->data;
	sh->flags = (SRMv2_VERSION << 6);
	sh->type  = type;
	sh->ipaddr = htonl(LookupLocalAddr());

	pb->dp = pb->data + sizeof(srmv2_cmnhdr);
	pb->len = sizeof(srmv2_cmnhdr);
	return;
}


/*
 *                    (32)
 * +--------+---------+---------------------------+
 * |               source ID (0-31)               |
 * +--------+---------+---------------------------+
 * |               source ID (32-63)              |
 * +--------+---------+---------------------------+
 * |               container ID                   |
 * +--------+---------+---------------------------+
 *
 * sourceID is obtained by the exclusive OR of the two halves
 * of the 16 B quantity from md5_digest(). See source-srmv2.cc
 * for more details.
 *
 * This is the source, container ID w.r.t the original source.
 * Hence, RREQ packets never carry the SRMv2 source ID of the
 * requesting source.
 */

void
SRMv2_BufferPool::build_conhdr(pktbuf *pb, int *srcid, u_int32_t cid)
{
	srmv2_conhdr *sh   = (srmv2_conhdr *)pb->dp;
	for (int i = 0; i < 4; i ++)
		sh->srcid[i]  = htonl(srcid[i]);
	sh->cid       = cid;

	pb->dp += sizeof(srmv2_conhdr);
	pb->len += sizeof(srmv2_conhdr);
	return;
}


/*
 *                    (32)
 * +--------+---------+---------------------------+
 * |               source ID (0-31)               |
 * +--------+---------+---------------------------+
 * |               source ID (32-63)              |
 * +--------+---------+---------------------------+
 */
void
SRMv2_BufferPool::build_srchdr(pktbuf *pb, int *srcid)
{
	int* sid   = (int *)pb->dp;
	for (int i = 0; i < 4; i ++)
		sid[i]  = htonl(srcid[i]);
	pb->dp += 4 * sizeof(int);
	pb->len += 4 * sizeof(int);
	return;
}


/*
 *     (8)      (8)           (16)
 * +-+------+---------+---------------------------+
 * |F| resv | adutype |      adulen/fraglen       |
 * +-+------+---------+---------------------------+
 * |              seq# (or adu#)                  |
 * +--------+---------+---------------------------+
 * |               adu timestamp (0-31)           |
 * +--------+---------+---------------------------+
 * |               adu timestamp (32-63)          |
 * +--------+---------+---------------------------+
 */

void
SRMv2_BufferPool::build_aduhdr(pktbuf *pb, u_int8_t atype, u_int16_t alen,
			  u_int32_t seqno, ntp64 timestamp, u_int8_t frag)
{
	srmv2_aduhdr *sh   = (srmv2_aduhdr *)pb->dp;
	sh->flags        = frag << 7;
	sh->atype        = atype;
	sh->alen         = htons(alen);
	sh->seqno        = htonl(seqno);

	sh->ts[0]  = htonl(timestamp.lower);
	sh->ts[1]  = htonl(timestamp.upper);

	pb->dp  += sizeof(srmv2_aduhdr);
	pb->len += sizeof(srmv2_aduhdr);
	return;
}

/*  (1)                (31)
 * +---+---------+---------------------------+
 * | M |         byte  offset into adu#      |
 * +---+---------+---------------------------+
 */
void
SRMv2_BufferPool::build_fraghdr(pktbuf *pb, u_int8_t more, u_int32_t offset)
{
	srmv2_fraghdr *sh = (srmv2_fraghdr *)pb->dp;
	if (offset >> 31) {
		printf("oversized fragment! This can't happen.\n");
		abort();
	}
	*sh = offset;
	if (more == 1)
		*sh  |= 0x80000000;
	*sh = htonl(*sh);
	pb->dp  += sizeof(srmv2_fraghdr);
	pb->len += sizeof(srmv2_fraghdr);
	return;
}



/*                      (32)
 * +-+-----+---------+---------------------+
 * |  count (#RREQs) |        unused       |
 * +-+-----+---------+---------------------+
 *
 *
 *
 *                    (32)
 * +--------+---------+---------------------------+
 * |               start adu seqno                |
 * +--------+---------+---------------------------+
 * |               start offset (Bytes)           |
 * +--------+---------+---------------------------+
 * |                 end adu seqno                |
 * +--------+---------+---------------------------+
 * |                 end offset (Bytes)           |
 * +--------+---------+---------------------------+
 *
 * ==> {left, ... right - 1} are missing.
 */

void
SRMv2_BufferPool::build_rreqhdr(pktbuf *pb, unsigned int ss, unsigned int sbytes,
				unsigned int es, unsigned int ebytes)
{
	srmv2_rreqhdr *rh = (srmv2_rreqhdr *)pb->dp;
	rh->ss          = htonl(ss);
	rh->sbytes      = htonl(sbytes);
	rh->es          = htonl(es);
	rh->ebytes      = htonl(ebytes);

	pb->dp  += sizeof(srmv2_rreqhdr);
	pb->len += sizeof(srmv2_rreqhdr);
}


/*
 *                    (32)
 * ==> {sbytes, sbytes + 1, ... ebytes - 1} are missing.
 *     This header appears if (ss == es) in the rreq header
 */


/*
 * SRMv2_ANNOUNCE :
 *                      (32)
 * +--------+---------+---------------------------+
 * |               container ID                   |
 * +--------+---------+---------------------------+
 * |                signature (last adu#)         |
 * +--------+---------+---------------------------+
 * |                #bytes in last adu            |
 * +--------+---------+---------------------------+
 */
void
SRMv2_BufferPool::build_announcehdr(pktbuf *pb, srmv2_announcehdr *salist, int count)
{
	srmv2_announcehdr *al = (srmv2_announcehdr *)pb->dp;
	int i;

	for (i = 0; i < count && salist[i].cid != SRMv2_ERROR; i++) {
		al[i].cid = htonl(salist[i].cid);
		al[i].sign = htonl(salist[i].sign);
		al[i].ebytes = htonl(salist[i].ebytes);
	}
	pb->dp  += i * sizeof(srmv2_announcehdr);
	pb->len += i * sizeof(srmv2_announcehdr);
}


int
SRMv2_BufferPool::command(int argc, const char*const* argv)
{
	if (argc == 3) {
		/*
		 * Associate an SRMv2 source with each BufferPool
		 * object so that src ID field can be filled in
		 * appropriately.
		 */
		if (strcmp(argv[1], "srmv2-source") == 0) {
			source_ = (SRMv2_Source*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
	}
	return (BufferPool::command(argc, argv));
}

