# cam-client.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#----
# Todd Hodes (hodes@cs)
# arrows from Gordon Chaffee
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/camera-client/cam-client.tcl,v 1.13 2002/02/03 04:21:39 lim Exp $


#
# A GUI for access to a remote camera
#
Class CameraUI

#
# init method takes frame/window parent for gui
# and unicast address of camera
#
CameraUI public init {w addrspec} {
    $self instvar al_ zoomSpeed_ panSpeed_ tiltSpeed_ \
	    presets_ pending_ setPresets_ remain_quiet_

    set remain_quiet_ 1
    set pending_ -1
    array set presets_ ""
    set zoomSpeed_ -1
    set panSpeed_ -1
    set tiltSpeed_ -1
    set setPresets_ 0

    # if addrspec includes a name, turn it into IP addr
    set firstchar [string index $addrspec 0]
    if [string match \[a-zA-Z\] $firstchar] {
	set n [lindex [split $addrspec "/"] 0]
	set p [lindex [split $addrspec "/"] 1]
	set s [gethostbyname $n]
	if { $s == "" } {
	    puts "cannot find address for '$n'"
	    exit
	}
	set addrspec $s/$p
    }

    set al_ [new UDPChannel/CamCl $addrspec $self]
    $self build_gui $w
    update
    set remain_quiet_ 0
}

CameraUI public destroy {} {
    $self instvar al_
    delete $al_
    eval [list $self] next
}



CameraUI private build_gui {w} {
    $self instvar al_ scales_ w_

    set w $w.camFrame
	set w_ $w
    frame $w
    pack $w

    # build bottom informational label
    set il $w.infolabel
    frame $il -relief groove
    label $il.l -text "Camera Controller"
    pack $il -side bottom -fill x -expand 1
    pack $il.l -in $il -side bottom -fill x -expand 1

    set t $w.camera
    set cameraParent $w

    frame $t
    pack $t -in $w
    set width 120
    set height 80
    set winwidth 124
    set uparrow " -6 -9	 -6 -23	 -14 -23  0 -35	 14 -23	 6 -23	 6 -9  -6 -9"
    set dnarrow " -6  9	 -6  23	 -14  23  0  35	 14  23	 6  23	 6  9  -6  9"

    set ltarrow " -9 -6	 -23 -6	 -23 -14  -35 0	 -23 14	 -23 6	 -9 6  -9 -6"
    set rtarrow "  9 -6	  23 -6	  23 -14   35 0	  23 14	  23 6	  9 6	9 -6"

    set ziarrow " 54 -9	 54 -23	 46 -23	 60 -35	 74 -23	 66 -23 66 -9  54 -9"
    set zoarrow " 56  0	 56  10	 50  10	 60  15	 70  10	 64  10 64  0  56  0"

    set hhalf [expr $height/2]
    set c $t.c
    set region [list -40 -$hhalf 80 $hhalf]
    canvas $c -height $height -width $width -borderwidth 0 \
	    -scrollregion $region

    set p_up [eval $c create polygon $uparrow -tags \{up arrow\}]
    set p_dn [eval $c create polygon $dnarrow -tags \{dn arrow\}]
    set p_lt [eval $c create polygon $ltarrow -tags \{lt arrow\}]
    set p_rt [eval $c create polygon $rtarrow -tags \{rt arrow\}]
    set p_zi [eval $c create polygon $ziarrow -tags \{zi arrow\}]
    set p_zo [eval $c create polygon $zoarrow -tags \{zo arrow\}]

    $c scale all 0 0 .8 .8

    set t_pos [$c create text 0 -33 -text "position" -anchor c]

    set t_zoom [$c create text 50 -33 -text "zoom" -anchor c]
    #	       -font "-Adobe-Helvetica-Bold-R-Normal--*-80-*"

    $c bind arrow <Any-Enter> "$self itemEnter $c"
    $c bind arrow <Any-Leave> "$self itemLeave $c"

    set pressCmd "$self itemPress $c"
    set releaseCmd "$self itemRelease $c"
    $c bind up <ButtonPress-1>	 "$pressCmd;   $al_ send move_up"
    $c bind up <ButtonRelease-1> "$releaseCmd; $al_ send move_stop"
    $c bind dn <ButtonPress-1>	 "$pressCmd;   $al_ send move_down"
    $c bind dn <ButtonRelease-1> "$releaseCmd; $al_ send move_stop"
    $c bind lt <ButtonPress-1>	 "$pressCmd;   $al_ send move_left"
    $c bind lt <ButtonRelease-1> "$releaseCmd; $al_ send move_stop"
    $c bind rt <ButtonPress-1>	 "$pressCmd;   $al_ send move_right"
    $c bind rt <ButtonRelease-1> "$releaseCmd; $al_ send move_stop"
    $c bind zi <ButtonPress-1>	 "$pressCmd;   $al_ send zoom_in"
    $c bind zi <ButtonRelease-1> "$releaseCmd; $al_ send zoom_stop"
    $c bind zo <ButtonPress-1>	 "$pressCmd;   $al_ send zoom_out"
    $c bind zo <ButtonRelease-1> "$releaseCmd; $al_ send zoom_stop"
    pack $c -side top

    # ---- scales
    frame $t.right
    pack $t.right -side right
    set scales_ $t

    frame $t.pan
    scale $t.pan.sc -command "$self set_speed p" -showvalue 0 \
	    -from 1 -to 100 -orient horizontal \
	    -font "-*-*-*-*-*-*-8-*-*-*-*-*-*-*" -width 10
    #$t.pan.sc set 5
    label $t.pan.lab -text "pan"
    pack $t.pan.sc $t.pan.lab -in $t.pan -side right

    frame $t.tilt
    scale $t.tilt.sc -command "$self set_speed t" -showvalue 0 \
	    -from 1 -to 100 -orient horizontal \
	    -font "-*-*-*-*-*-*-8-*-*-*-*-*-*-*" -width 10
    #$t.tilt.sc set 1
    label $t.tilt.lab -text "tilt"
    pack $t.tilt.sc $t.tilt.lab -in $t.tilt -side right

    frame $t.zoom
    scale $t.zoom.sc -command "$self set_speed z" -showvalue 0 \
	    -from 1 -to 100 -orient horizontal \
	    -font "-*-*-*-*-*-*-8-*-*-*-*-*-*-*" -width 10
    #$t.zoom.sc set 50
    label $t.zoom.lab -text "zoom"
    pack $t.zoom.sc $t.zoom.lab -in $t.zoom -side right

    pack $t.pan $t.tilt $t.zoom -in $t.right -anchor e

    # ---- presets
    frame $t.presets
    pack $t.presets -in $t.right
    set tpre $t.presets
    checkbutton $tpre.setButton -command "$self toggle_preset_set" -text "set"
    pack $tpre.setButton -side bottom -anchor e
    foreach i {1 2 3 4} {
		button $tpre.b$i -text $i -command "$self presetButtonInvoke $i"
		pack  $tpre.b$i -side left -in $tpre
    }

    bind . <q> exit
}

CameraUI private hide_gui {} {
	$self instvar w_
	pack forget $w_
}

CameraUI private show_gui {} {
	$self instvar w_
	pack $w_
}

CameraUI private toggle_preset_set {} {
    $self instvar setPresets_

    if {$setPresets_} {
	set setPresets_ 0
    } else {
	set setPresets_ 1
    }
}

#
# update UI from server message
#
# format of received announcement is:
# z: <zoomSpeed> t: <tiltSpeed> p: <panSp> pre: <presetList>
# [preset list entries are "name" "coords"]
#
CameraUI private receive_update {data} {
    $self instvar scales_ zoomSpeed_ panSpeed_ tiltSpeed_ presets_ pending_

    # filter data from other clients
    if {[scan $data "z: %d t: %d p: %d pre: " z t p] != 3} {return}

    set pres [lrange $data 7 end]
    array set presets_ $pres

    if {$pending_ != -1} {
	# a more current update is pending -- throw away these
	return
    }
    scan $data "z: %d t: %d p: %d pre: " z t p
    if {$zoomSpeed_ != $z} {
	set zoomSpeed_ $z
	set s $scales_.zoom.sc
	catch {$s set $z}
    }
    if {$tiltSpeed_ != $t} {
	set tiltSpeed_ $t
	set s $scales_.tilt.sc
	catch {$s set $t}
    }
    if {$panSpeed_ != $p} {
	set panSpeed_ $p
	set s $scales_.pan.sc
	catch {$s set $p}
    }

}

# either sets a preset or tells the server to go to the preset
# <br>
# side effect: asks camera for current settings --
# reply is received and interpreted in the recv method.
#<br>
# reply contains actual preset data (rather than just numeric index)
# to preserve presets over client and server crashes
#
CameraUI private presetButtonInvoke {num} {
    $self instvar setPresets_ presets_ al_

    if {$setPresets_} {
	$al_ send "set_preset $num"
    } elseif {[array names presets_ $num] == ""} {
	puts "No preset set..."
    } else {
	$al_ send "goto_preset $presets_($num)"
	#puts "goto_preset $presets_($num)"
    }
}

# holds onto speed updates until slider activity quiets.
#
CameraUI private set_speed {param perc} {
    $self instvar pending_ zoomSpeed_ panSpeed_ tiltSpeed_ remain_quiet_
    switch $param {
	z {set zoomSpeed_ $perc}
	p {set panSpeed_ $perc}
	t {set tiltSpeed_ $perc}
    }

    if {$remain_quiet_} {
	return
    }

    if {$pending_ != -1} {
	after cancel $pending_
    }
    set pending_ [after 300 $self announce_speeds]
}

CameraUI private announce_speeds {} {
    $self instvar al_ pending_ zoomSpeed_ panSpeed_ tiltSpeed_
    $al_ send "set_zoom_speed $zoomSpeed_"
    $al_ send "set_pan_speed $panSpeed_"
    $al_ send "set_tilt_speed $tiltSpeed_"
    set pending_ -1
}

CameraUI private itemPress {c} {
    set fill [lindex [$c itemconfig current -fill] 4]
    $c itemconfig current -fill blue
}

CameraUI private itemRelease {c} {
    $c itemconfig current -fill grey
}

CameraUI private itemEnter {c} {
    $c itemconfig current -fill gray
}

CameraUI private itemLeave {c} {
    $c itemconfig current -fill black
}


#-----------------------------------------------------


import UDPChannel


# Camera RPC-style interface component
#
Class UDPChannel/CamCl -superclass UDPChannel

UDPChannel/CamCl public init {addrSpec parent {mtu 1500}} {
    eval [list $self] next $addrSpec $mtu

    $self instvar parent_
    set parent_ $parent

    # get the current speed settings and presets
    $self send "update_state"
}


# Receive a msg from the Camera Server.
#
UDPChannel/CamCl private recv {addr port data size} {
    #puts "Msg: $addr/$port \[$size\]: $data"

    $self instvar parent_
    $parent_ receive_update $data
}

