# datasrc.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#---------------------------------------------------------------------------
# DataSrc
#
Class DataSrc

DataSrc instproc init {}  {
        # defaults to zero, can be set later
        $self set firstEId_ 0
        $self reset
        $self set running_ 0
        $self set now_ 0
        # should set these values same as animation ui
        $self set granularity_ 2e-3
        $self set refreshTime_ 100
        $self set requestR_ ""
        $self set requestId_ 0
}

# this id is used by Atobj_rcvr to (de)mux among different animations
DataSrc instproc setAnmId {anmId} {
        $self set anmId_ $anmId
}

# set nextEId to firstEId-1 so that the next item we will inspect is firstEId
DataSrc instproc reset {} {
        $self set nextEId_ [$self set firstEId_]
}

DataSrc instproc setNextEId {eventId} {
        $self set nextEId_ $eventId
}

DataSrc instproc attach_view {view} {
        $self instvar view_
        set view_ $view
}

DataSrc instproc attach_rcvr {atobj_rcvr} {
        $self set atobjRcvr_ $atobj_rcvr
}

DataSrc instproc mintime {} {
        # start at 0 initially
        return 0
}

DataSrc instproc maxtime {} {
        return ""
}

DataSrc instproc handle {eId event} {
        $self instvar view_
        set type [lindex $event 0]
#        DbgOut $type $event
        if {$type=="END"} {
                $self set endTime_ [lindex $event 2]
                return
        } elseif {$type=="INIT"} {
                # note: we ignore some of the values, they are there so that
                #       the structure is consistent with the other events...
                foreach {t sT eT fStatic lStatic mintime maxtime bbox} $event {
                        $self set lastStatic_ $lStatic
                        $self setNextEId [expr $lStatic + 1]
                        $self set mintime_ $mintime
                        $self set maxtime_ $maxtime
                        $self set now_ $mintime
                        $view_ setBBox $bbox
                        # make view aware of the change in bbox
                        $view_ resize
                }
#                puts stderr "INIT received $event"
                return
        }
        # send up to view by default
        return [$view_ handle $eId $event]
}

#
# There are 2 versions of advance, one for the dead reckoning stage/local
# case, where we want to advance to a point in time, but don't know how much
# to move to. We try and advance as much as possible, but if we encounter
# events that are not here yet, we stop, request for one event, and wait.
#
# In the second case (advance_to_eId) we know the limits to move to
# (e.g. signal comes thru SRM that specifies dependency). In such case, we
# stop also, but look thru the range and request for a range of data.
#
DataSrc instproc advance {time} {
        $self instvar view_ firstEId_ events_

        set nextEId [$self nextEId]

        if [info exists events_($nextEId)] {
                DbgOut advance: t:$time evST: [$self eventST $nextEId] \
                                nextEId: $nextEId st=[$self eventST $nextEId]
        }

        if {$time > [$self maxtime]} return

        while {1} {
                set event [$self get_event $nextEId]
                # DbgOut "event $nextEId is $event"
                if {[llength $event]>0} {
                        # skip over unwanted events
                        if {[$self inRange $nextEId $time]} {
                                DbgOut T=$time ET=[$self eventET $nextEId]
                                $self handle $nextEId $event
                        }
                } else {
                        # since we don't know how much to advance to the given
                        # time, we stop at the first unseen event (note that
                        # get_event will trigger the repair request, so in the
                        # worst case, we proceed one event by one event)
                        #
                        # note: this could be due to EOF as well...
                        return 0
                }
                if {[$self eventST $nextEId] > $time} {
                        break
                }
                incr nextEId
        }
        $self setNextEId $nextEId
        return 1
}

# same as advance, but this time we advance to an eId
# (see comments on advance)
DataSrc instproc advance_to_eId {tgtEId t} {
        $self instvar view_
        set nextEId [$self nextEId]

        if {$nextEId > $tgtEId} return
        set gotAll 1
        DbgOut atoEId cEId=$nextEId tgtEId=$tgtEId t=$t
        while {1} {
                set event [$self get_event $nextEId]
#                DbgOut "atoEID: $nextEId"
                if {[llength $event]>0} {
                        # skip over unwanted events
                        if {[$self inRange $nextEId $t]} {
#                                DbgOut T=$t ET=[$self eventET $nextEId]
                                $self handle $nextEId $event
                        }
                } else {
                        # no data, view_ will get called when repair is here
                        set gotAll 0
                        DbgOut "Don't have data $nextEId"
                }
                if {$nextEId >= $tgtEId} {
                        break
                }
                incr nextEId
        }
        $self setNextEId $tgtEId
        return $gotAll
}

# move the stream to minEventId, and play all active objects til tTgt
DataSrc instproc scan {minEId maxEId tTgt} {
#        DbgOut "ds scan $tTgt $minEvId $maxEvId"
        $self set activeR_ [list $minEId $maxEId]
        $self setNextEId $minEId
        # REVIEW: should check start and ending time for the events
        #         for now, treat it as advance
        if ![$self advance_to_eId $maxEId $tTgt] {
                $self next_request
        }
        # send requests here?
}

DataSrc instproc nextEId {} {
        return [$self set nextEId_]
}

DataSrc instproc recv_event {args} {
        # get rid of firs level of braces
        set args [lindex $args 0]
        set eId [removeFirst args]
        $self instvar events_
        if ![info exists events_($eId)] {
                set events_($eId) [lindex $args 0]
#                DbgOut "ds194:event $eId set to $events_($eId)"
#                $self notify_received $eId
        }
        # Is initialization packet and we got it for the first time,
        # send it up...
        $self instvar lastStatic_
        if {$eId == 1 && ![info exists lastStatic_] } {
                $self handle $eId $events_($eId)
                $self setNextEId 1
                $self advance_to_eId $lastStatic_ 0.0
                return
        }
        if {[info exists lastStatic_] && $eId <= $lastStatic_} {
                $self handle $eId $events_($eId)
                return
        }
}

DataSrc instproc sched_render {msg} {
        $self instvar renderId_ refreshTime_
        if [info exists renderId_] {
                after cancel $renderId_
        }
        set renderId_ [after $refreshTime_ [concat $self render $msg]]
}

DataSrc instproc recv_ctrl {msg} {
        set msg [lindex $msg 0]
        set ctype [removeFirst msg]
        switch -- $ctype {
                s {
                        # whole message is 'c s time'
                        # (see NamUI attach_hooks)
                        $self set endTime_ $msg
                }
                r {
                        # whole message is 'c r t min max'
                        # see DataSrc/Local
                        DbgOut recv r: $msg
                        $self sched_render $msg
                }
        }
}

# REVIEW: should be possible to handle more than one item in one packet...
DataSrc instproc recv_ADU {args} {
        $self instvar events_ view_
#        DbgOut "rec_ADU: $args"
        set pktType [removeFirst args]
        switch -- $pktType {
                # event
                e {
                        $self recv_event $args
                }
                # control messages: run or stop
                c {
                        $self recv_ctrl $args
                }
        }
}

#-- callback --#
#  returns true if request is still needed, false otherwise
DataSrc instproc update_request {requestId} {
        $self instvar requestR_ atobjRcvr_ requestId_ anmId_
        set newR [$self find_requestR]
        set s [lindex $newR 0]
        set e [lindex $newR 1]
        set sR [lindex $requestR_ 0]
        set eR [lindex $requestR_ 1]

        # if not overlap, we shift onto a new range (cancel_request will
        # call next_request to update the range)
        if {$eR < $s || $sR > $e} {
                $atobjRcvr_ cancel_request $requestId_ $anmId_ 1
                return 0
        }
        # overlapping, just update the request
        set requestR_ $newR
        return 1
}

#-- callback --#
DataSrc instproc fill_request {requestId len} {
        # we only keep a single request at a time
        $self instvar requestId_ requestR_
        if {$requestId != $requestId_} {
                DbgOut f-requ $requestId ignored
                return ""
        }
        DbgOut "fill_req rid:$requestId l:$len ret:$requestR_"
        return $requestR_
}

#--- callback --#
# called when an external datasrc wants the data #
DataSrc instproc handle_request {args} {
        set requestR $args
        if {$args==""}  return
        DbgOut "r req $requestR"
        $self instvar replies_ anmId_ atobjRcvr_ events_
        # only try to answer the reply we have all the data
        set haveAll 1
        for {set i [lindex $requestR 0]} {$i<=[lindex $requestR 1]} {incr i} {
                if ![info exists events_($i)] {
                        set haveAll 0
                        break
                }
        }
        if {!$haveAll} {
                DbgOut ignoring request: $requestR
                return
        }

        # don't duplicate replies!
        foreach a [array names replies_] {
                set rp $replies_($a)
                if {[lindex $requestR 0] <= [lindex $rp 1] && \
                                [lindex $requestR 1] >= [lindex $rp 0]} {
                        DbgOut ignoring request (overlap): $requestR
                        return
                }
        }
        $self instvar anmId_
        set replyId [$atobjRcvr_ sched_reply $anmId_]
        DbgOut "reply: $replyId scheduled for ( $requestR )"
        set replies_($replyId) $requestR
}

DataSrc instproc find_requestR {} {
        # REVIEW: should check thru all the list of 'future' advance events
        #         so that we will still send those requests.
        #    right now we only maintain one active range at a time

        $self instvar activeR_ events_ lastStatic_ staticR_ view_

        set aR $activeR_
        # Always want static
        # review: should allow multiple requests
        if $staticR_ {
                if ![info exists lastStatic_] {
                        set aR [list 1 1]
                } else {
                        set aR [list 2 $lastStatic_]
                }
        }

        DbgOut ar:$aR
        for {set i [lindex $aR 0]} {$i<=[lindex $aR 1]} {incr i} {
                if {![info exists events_($i)] && ![info exists s]} {
                        # first event
                        set s $i
                        continue
                }
                if {[info exists events_($i)] && [info exists s]} {
                        break
                }
        }
        set e [expr $i - 1]
        # got everyting in the activeRange
        if ![info exists s] {
                if $staticR_ {
#                        puts stderr "end static!"
                        set staticR_ 0
                        $view_ refresh [$self set now_]
                        eval $self scan $activeR_ [$self set now_]
                }
                return ""
        }
        return [list $s $e]
}

# called when activeR_ changes, or when repairs comes in
# returns whether current event cancelled
DataSrc instproc next_request {} {
        $self instvar requestR_ atobjRcvr_ requestId_ anmId_
        set newR [$self find_requestR]
        DbgOut "nr: newR=$newR, requestR=$requestR_"

        set s [lindex $newR 0]
        set e [lindex $newR 1]
        set eR [lindex $requestR_ 1]
        set sR [lindex $requestR_ 0]
        DbgOut nr: $eR $sR
        # if previous request range does not overlap this new range at
        # all, cancel it.
        # note: we set wantcallback to 1 so that next_request will
        #       not get called recursively.
        if {"$newR"=="" || ($eR < $s || $sR > $e)} {
                if {"$requestR_"!=""} {
                        $atobjRcvr_ cancel_request $requestId_ $anmId_ 0
                        $self set requestId_ 0
                }
        }
        # new request or overlaps
        set requestR_ $newR
        DbgOut nr: rr: $requestR_ rId: $requestId_
        if {"$requestR_"!="" && ($requestId_ == 0)} {
                set requestId_ [$atobjRcvr_ sched_request $anmId_]
                DbgOut req $requestId_ scheduled
        }
}

#-- callback (used from tcl as well) --#
# cancel THE current request
DataSrc instproc cancel_request {requestId} {
        $self instvar requestId_ requestR_
        DbgOut "$requestId"  canncelled
        DbgAssert [expr $requestId == [$self set requestId_]]
        set requestR_ ""
        set requestId_ 0
        $self next_request
}

#-- callback --#
DataSrc instproc cancel_reply {replyId} {
        $self instvar replies_ axedRepairs_
        DbgAssert [info exists replies_($replyId)]
        unset replies_($replyId)
        unset axedRepairs_($replyId)
        DbgOut "reply $replyId cancelled"
}

#-- callback --#
#
# Reply format: {$sId $eId} event_{sId} event_{sId+1} .... event_{eId}
#
DataSrc instproc fill_reply {replyId len} {
        $self instvar replies_ events_ atobjRcvr_ anmId_ axedRepairs_
        if ![info exists replies_($replyId)] return
        if [info exists axedRepairs_($replyId)] return
        set rRange $replies_($replyId)
        # -2 for braces
        set maxlen [expr $len - [string length $rRange] - 2]
        set reply ""
        for {set i [lindex $rRange 0]} {$i <= [lindex $rRange 1]} {incr i} {
                set newItem [list $i $events_($i)]
                if {[string length $reply] + [string len $newItem] + 4 < \
                                $maxlen  } {
                        lappend reply [list $i $events_($i)]
                } else {
                        break
                }
        }
        incr i -1
        # one event should never be larger than a packet, for now...
        DbgAssert [expr $i >= [lindex $rRange 0]]
        if {$i == [lindex $rRange 1]} {
                # note that unset of replies_ elt is done at call back
                # REVIEW: keep the reply for a while to suppress...
                if ![info exists axedRepairs_($replyId)] {
                        set axedRepairs_($replyId) [after 100 "$atobjRcvr_ \
                                        cancel_reply $replyId $anmId_"]
                }
        } else {
                # shorten the range
                set replies_($replyId) [list [expr $i + 1] [lindex $rRange 1]]
                DbgAssert [expr [lindex $replies_($replyId) 1] >= \
                                [lindex $replies_($replyId) 0]]
        }
        set reply [linsert $reply 0 [list [lindex $rRange 0] $i]]
        DbgOut "sending reply: [lindex $reply 0]"
        return $reply
}

#-- callback --#
# if this fullfills request partially, backoff,
# if fullfills entirely, cancel
DataSrc instproc handle_reply {args} {
        DbgOut r rpy: [lindex $args 0]
        $self instvar requestR_ requestId_ atobjRcvr_ anmId_
        set rRange [removeFirst args]
        set sId [lindex $rRange 0]
        set eId [lindex $rRange 1]
        if {"$requestR_"!=""} {
                set sReq [lindex $requestR_ 0]
                set eReq [lindex $requestR_ 1]
                if {$sId <= $eReq && $eId >= $sReq} {
                        # overlap
                        DbgOut "backing off request"
                        $atobjRcvr_ backoff_request $requestId_
                        set sR $sReq
                        set eR $eReq
                        if {$sId <= $sReq} {
                                # overlap on the left
                                set sR [expr $eId + 1]
                        }
                        if {$eId >= $eReq} {
                                # overlap on the right
                                set eR [expr $sId - 1]
                        }
                        if {$eR < $sR} {
                                # note that we will send the next request
                                #      in the cancel_request callback
                                $atobjRcvr_ cancel_request $requestId_ $anmId_ 1
                        }
                }
        }
        $self instvar events_ lastStatic_
        foreach e $args {
                set eId [lindex $e 0]
#                DbgOut h rpy $eId
                # already got it. ignore duplicate
                if [info exists events_($eId)] {
#                        DbgOut "h rpy: evt $eId duplicated, ignored"
                        continue
                }
                $self recv_event $e
                if {[info exists lastStatic_] && $eId <=$lastStatic_} {
                        # recv_event will send it up.
#                        DbgOut "h rpy: evt $eId static, ignored (handled by r evt)"
                        continue
                }
                if [$self inRange $eId [$self set now_]] {
                        $self handle $eId $events_($eId)
                } else {
#                        DbgOut "h rpy: evt $eId ignored "
                }
        }
}

# for now, the session announcement is a control msg
DataSrc instproc fill_sa {} {
        $self instvar view_
        return [list c [list r [$self set now_] \
                        [$view_ oldestEId] \
                        [$view_ newestEId]]]
}

# for now, the session announcement is a control msg
DataSrc instproc handle_sa {args} {
#        DbgOut SA: $args
        $self recv_ctrl [list [lindex $args 1]]
}

# if the datasrc did not get the event, request for repair, and
# return NULL
DataSrc instproc get_event {eventId} {
        $self instvar events_
        if ![info exists events_($eventId)] {
                return ""
        }
        return $events_($eventId)
}

#-- hook ----#
DataSrc instproc hook_updateTime {t} {}

#-- hook ----#
DataSrc instproc hook_stop {t} {}

DataSrc instproc setRefresh {refreshTime} {
        $self set refreshTime_ $refreshTime
}

DataSrc instproc setGran {gran} {
        $self set granularity_ $gran
}

# this is called for normal frame by frame movement
DataSrc instproc render_next {t} {
        $self instvar objects_ damage_ datasrc_ view_

        $self advance $t
        $view_ update $t
        $view_ refresh $t
}

DataSrc instproc setTime {t} {
        $self set now_ $t
}

DataSrc instproc next_frame {} {
        $self instvar now_ granularity_ refreshTime_ running_ endTime_ \
                        renderId_
        DbgOut dsnf r= $running_ n=$now_ e=$endTime_
        if {$now_ < $endTime_} {
                $self hook_updateTime $now_
                if $running_ {
                        set now_ [expr $now_ + $granularity_]
                        $self render_next $now_
                        if [info exists renderId_] {
                                after cancel $renderId_
                        }
                        set renderId_ [after $refreshTime_ "$self next_frame"]
                }
        } else {
                $self hook_stop $now_
        }
}

# moves to a different place in the stream
DataSrc instproc render {t minEId maxEId} {
        # DbgOut "in render: $minEId $maxEId $t"
        $self instvar now_ view_ renderId_

        set now_ $t
        $self hook_updateTime $now_

        # flush unused data
        $view_ update $t
        # {min,max}EId could be null if the page is blank.
        if {$minEId!="" && $maxEId!=""} {
                $self scan $minEId $maxEId $t
        }
        $view_ refresh $t
}

DataSrc instproc now {} {
        return [$self set now_]
}

DataSrc instproc isRunning {} {
        return [$self set running_]
}

DataSrc instproc setRunning {shouldRun {runPeriod {}}} {
        $self instvar running_ endTime_ now_
        if {"$runPeriod"!=""} {
                set endTime_ [expr $now_ + $runPeriod]
        }
        if {$running_ != $shouldRun} {
                set running_ $shouldRun
                if $shouldRun {
                        $self next_frame
                }
        }
}

DataSrc instproc setup {} {
        $self instvar staticR_ events_ activeR_
        if ![info exists events_(1)] {
                set activeR_ [list 1 1]
                set staticR_ 1
                $self next_request
        }
}
