# application.tcl --
#
#       The Application class is the base abstraction for the main program of
#       an application built from mash components.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/common/application.tcl,v 1.49 2002/02/03 04:25:43 lim Exp $


import Log

#
# The Application class is the base abstraction for the main program
# of an application built from mash components.  There is one application
# object per process and it handles the argument parsing, the default
# configuration options, and formatting of error messages.
# <p>
# Warning: this object is intended to offer its subclasses some common
# methods for reuse as a mode of convenience, but in general it is safer
# to put common functionality in other objects, so as not to restrict
# all programs to be subclassed from Application.  For example,
# programmers shouldn't be forced to subclass from Application to write
# a simple mashlet.  Likewise, other mash objects should not depend on
# the existence of an Application object, since that would restrict
# their applicabilty to use in programs that are subclassed from
# Application.
#
Class Application

#
# The Application constructor initializes the application object.
# The <i>name</i> argument is a string that identifies the
# application, i.e., the name of the application.
# The first character of <i>name</i> must be lower case
# or results are undefined.
#
Application public init name {
	$self next
	$self instvar name_ class_
	#
	# If tk is available, set the name and class fields for
	# interacting with the tk database.  Otherwise, we shouldn't
	# need them... FIXME actually rtp needs the toolname? this should
	# be set by the app subclass in the config object
	#
	set name_ $name

	$self add_option appname $name
	Log set name_ $name
	set class_ [string toupper [string index $name_ 0]][string \
		range $name_ 1 end]
	#
	# Register the app name with tk,
	# and catch the error in case tk is not compiled in
	#
	catch "tk appname $name"

	Application set instance_ $self
}

#
# Return a pointer to the sole instance of the application object.
# Only one such instance is allowed per process.
#
Application proc instance {} {
	return [Application set instance_]
}

Application proc name {} {
	return [[Application instance] set name_]
}

Application proc class {} {
	return [[Application instance] set class_]
}

#
# Convenient place to put hook to create toplevel windows
# FIXME put this elsewhere...?
#
Application proc toplevel w {
	Application instvar visual_ colormap_
	if [info exists visual_] {
		toplevel $w -class [Application class] \
			-visual $visual_ -colormap $colormap_
	} else {
		toplevel $w -class [Application class]
	}
}

#FIXME
global font
set font(helvetica10) {
	normal--*-100-75-75-*-*-*-*
	normal--10-*-*-*-*-*-*-*
	normal--11-*-*-*-*-*-*-*
	normal--*-100-*-*-*-*-*-*
	normal--*-*-*-*-*-*-*-*
}
set font(helvetica12) {
	normal--*-120-75-75-*-*-*-*
	normal--12-*-*-*-*-*-*-*
	normal--14-*-*-*-*-*-*-*
	normal--*-120-*-*-*-*-*-*
	normal--*-*-*-*-*-*-*-*
}
set font(helvetica14) {
	normal--*-140-75-75-*-*-*-*
	normal--14-*-*-*-*-*-*-*
	normal--*-140-*-*-*-*-*-*
	normal--*-*-*-*-*-*-*-*
}
set font(times14) {
	normal--*-140-75-75-*-*-*-*
	normal--14-*-*-*-*-*-*-*
	normal--*-140-*-*-*-*-*-*
	normal--*-*-*-*-*-*-*-*
}

#
# This method has been moved to FontInitializer, though not all apps
# have been updated to use that object yet.  Also, MBapp (a subclass of
# Application) redefines this method depending which version of tcl is
# being used.
#
Application instproc search_font { foundry style weight points slant } {
	global font tcl_version tcl_platform

 	if {$tcl_version >= 8} {
 		if {$slant == "r"} {
 			set slant ""
 		} elseif {$slant == "o"} {
 			set slant "italic"
 		}
		if {$weight == "medium"} {
			set weight ""
		}
		# make points negative since we use pixel size
 		return "$style -$points $weight $slant"
 	}

	foreach f $font($style$points) {
		set fname -$foundry-$style-$weight-$slant-$f
		if [havefont $fname] {
			return $fname
		}
	}
	$self instvar name_
	puts stderr "$name_: can't find $weight $fname font (using fixed)"
	if ![havefont fixed] {
		puts stderr "$name_: can't find fixed font"
		exit 1
	}
	return fixed
}

#
# Called to set up local configuration options like those
# stored in a preference file. FIXME: get rid of this.
#
Application public init_local {} {
	$self instvar name_
# $name_.tcl was causing a problem with spurious files
# with the same name.
	set f ~/.$name_.tcl
	if [file exists $f] {
		uplevel #0 "source $f"
	}
	set script [$self resource startupScript]
	if { $script != "" } {
		uplevel #0 "source $script"
	}
}

# default user_hook is a nop -- users should re-define this in their
# ~/.{vic,vat}.tcl to (re-)define procedures to change or augment
# the tools behavior.

Application instproc user_hook {} {
}

#
# Every object by default has hooks to look up configuration options.
# Any class can overide the "get_option" method, or it can
# install a pointer to the options object (in the field options_)
# by passing "-optionsFrom $object" to "new"
#
Object instproc options {} {
	$self instvar options_
	if ![info exists options_] {
		#
		# If there is no default configuration
		# object, create it.
		#
		Object instvar options_
		if ![info exists options_] {
			set options_ [new Configuration]
			global tcl_platform
			if {"$tcl_platform(platform)"=="windows"} {
				$options_ add_default \
					background SystemButtonFace
				$options_ add_default \
					infoHighlightColor SystemHighlightText
			}
		}
	}

	# With this, objects that are not derived from Application will still have an "appname" resource.
	$options_ add_default appname mash

	return $options_
}

Object instproc optionsFrom o {
	$self set options_ $o
}

#
# Class method that allows classes to declare default configuration
# options on a per-class basis, e.g., by given a "-configuration"
# option where the Class is defined.
#
Class instproc configuration a {
 	$self instvar options_
	if ![info exists options_] {
		set options_ [new Configuration]
	}
	foreach { option value } $a {
		$options_ add_default $option $value
	}
}

Object instproc get_option r {
	set v [[$self options] get_option $r]
	if { $v != "" } {
		return $v
	}
	#
	# Option not found.  Look through this object's class
	# hierarchy to see if it can be found there.
	# Note that this checks Object::options_ since
	# $self::options_ might point somewhere else
	# (i.e., to  custom object)
	#
	set cl [$self info class]
	foreach cl "$cl [$cl info heritage]" {
		$cl instvar options_
		if [info exists options_] {
			set v [$options_ get_option $r]
			if { $v != "" } {
				return $v
			}
		}
	}
	return ""
}

# FIXME backward compat
Object instproc resource r {
	return [$self get_option $r]
}

Object instproc add_option { r v } {
	return [[$self options] add_option $r $v]
}

Object instproc add_default { r v } {
	return [[$self options] add_default $r $v]
}

#
# Returns 0 if the value of the option <i>r</i> is false
# and returns 1 if the value is true.
# (fyi: "" is considered false )
#
Object instproc yesno r {
	set v [$self get_option $r]
	if [string match \[0-9\]* $v] {
		return $v
	}
	if [string match \[tT\]* $v] {
		return 1
	}
	return 0
}

Object instproc debug s {
	if [$self yesno debug] {
		Log warn $s
	}
}

Object instproc warn s {
	Log warn $s
}

Object instproc fatal s {
	Log fatal $s
}

#
# The base class abstraction for housing and maintaining configuration
# options.  This is a replacement (and sort of a front-end) for the
# Tk options database, which has been far too difficult to make it
# do what we need.  Any class can create a configuration object
# and attach options info to it.  In turn, the object can be attached
# to any OTcl object using the optionsFrom method.  Once installed
# in some object, say o, then the get_option and add_options methods
# dispatched to $o are diverted to the configuration object.
# This approach allows us to embed multiple configuration databases
# in a single application.
#
Class Configuration

#
# Return the option named <i>r</i> stored in this object.
# If a binding was established with <i>add_option</i> return
# that entry, otherwise return the binding established via
# <i>add_default</i>.  If no binding exists, return an empty string.
#
Configuration public get_option r {
	$self instvar table_ default_
	if [info exists table_($r)] {
		return $table_($r)
	}
	if [info exists default_($r)] {
		return $default_($r)
	}
	return ""
}

#
# Assign the value <i>v</i> to the option named <i>r</i>
# in this configuration object.  This does not affect the
# tk options parameters (as was the convention in the old
# mash code base).  Now, instead, you must set a tk option
# explicitly with the global "option" command.
#
Configuration public add_option { r v } {
	$self instvar table_
	set table_($r) $v
}

#
# Assign the value <i>v</i> to the default for the option named <i>r</i>
# in this configuration object.
#
Configuration public add_default { r v } {
	$self set default_($r) $v
}

#
# Register a command-line option with the command-line argument parser
# embedded in this application.  By registering the command-line option
# indicated by the <i>flag</i> argument, the add_option method on
# the named Configuration <i>object</i> will be invoked with the
# corresponding <i>option</i> argument and the run-time value
# encountered.  For example:
#
#<pre>
# 	$app register_option $o -p port
#</pre>
#
# will cause
#
# <pre>
#	$o add_option port 1200
# </pre>
#
# to be called when
#
# <pre>
# $app parse_args</i>
# </pre>
#
# is called with the string "-p 1200" somewhere in the argument list.
#
Configuration public register_option  { flag option args } {
	$self instvar arg_option_ usage_ arg_option_default_
	set arg_option_($flag) $option
	if { [lindex $args 0] == "-default" } {
		set arg_option_default_($flag) [lindex $args 1]
		set args [lrange $args 2 end]
	}
	set usage_($flag) $args
}

Configuration public register_boolean_option  { flag option args } {
	$self instvar arg_bool_ arg_bool_val_
	set arg_bool_($flag) $option
	if { $args == "" } {
		set args 1
	}
	set arg_bool_val_($flag) $args
}

#
# Similar to <i>register_option</i>, but allows the option to
# be specified multiple times on the command line, with all
# arguments put in to a list.
# For example:
#
# <pre>
#    $o register_list_option -map rtpmap
# </pre>
#
# with the argument vector
#
# <pre>
#   ... -map 26:jpeg -map 31:h261 ...
# </pre>
#
# will result in the <i>rtpmap</i> option containing the
# following Tcl list: {26:jpeg 31:h261}.
#
Configuration public register_list_option {flag option args} {
	$self instvar arg_list_option_
	set arg_list_option_($flag) $option
	set usage_($flag) $args
}

#
# Return true iff the first string of list <i>argv</i> is a
# command option (i.e., begins with a dash).
#
Configuration private is_arg argv {
	if { $argv != "" } {
		return [string match -* [lindex $argv 0]]
	}
	return 0
}

#
# Parse and process the command options in the list of
# arguments given by <i>argv</i>.  The rules for parsing
# arguments are set up by previous calls to <i>register_option</i>.
#
# Returns the remaining command arguments after stripping
# and processing all the options.
#
# FIXME - actually, if there is an unrecognized option, the "usage" message is
#   printed and the app exits
#
Configuration instproc parse_args argv {
	$self instvar arg_resource_ bool_resource_
	$self instvar arg_option_ arg_bool_ arg_bool_val_ arg_list_option_ \
			arg_option_default_

	if { [info exists arg_resource_] || [info exists bool_resource_] } {
		puts stderr "your application class needs to be fixed"
		exit 1
	}

	while 1 {
		if ![$self is_arg $argv] {
			break
		}
		set arg [lindex $argv 0]
		set argv [lrange $argv 1 end]
		set val [lindex $argv 0]
		if { $arg == "-help" } {
			$self usage
			exit
		}
		if { $arg == "-X" } {
			set L [split $val =]
			if { [llength $L] != 2 } {
				puts stderr "malformed -X argument"
				exit 1
			}
			$self add_option [lindex $L 0] [lindex $L 1]
			set argv [lrange $argv 1 end]
			continue
		}

		set fatal_msg ""
		if [info exists arg_option_($arg)] {
			if { [llength $argv] > 0 && \
					[string index $val 0]!="-" } {
				$self add_option $arg_option_($arg) $val
				set argv [lrange $argv 1 end]
				continue
			}
			set fatal_msg "must be followed by an argument"
		}
		if [info exists arg_bool_($arg)] {
			$self add_option $arg_bool_($arg) $arg_bool_val_($arg)
			continue
		}
		if [info exists arg_list_option_($arg)] {
			if { [llength $argv] > 0 || \
					[string index $val 0]!="-" } {
				set o $arg_list_option_($arg)
				set l [$self get_option $o]
				lappend l $val
				$self add_option $o $l
				set argv [lrange $argv 1 end]
				continue
			}
			set fatal_msg "must be followed by an argument"
		}
		# FIXME - caller should handle
		#   the rest of argv should be passed back
		$self usage
		$self fatal "unknown/invalid command option: $arg ($fatal_msg)"
	}
	return $argv
}

# The default 'usage' statement.  Simply prints out the information of the
# argument database returned by Configuration::arg_info.
Configuration public usage {} {
	set display_args_on_single_line 0

	if { $display_args_on_single_line } {
		puts "usage: [Application name] [join [$self arg_info]]"
	} else {
		puts "usage: [Application name]"
		foreach arg [$self arg_info] {
			puts $arg
		}
	}
}

# Present a human readable version of the argument database.  In particular,
# identify optional and required arguments and their default values if
# applicable.  The procedure returns a list with two elements: a list of the
# optional arguments and a list of the required arguments.  The format is
# consistent with standard 'usage' statements with the default values
# apprearing in parentheses.  Note that if the all arguments to an
# application are specified with switches, this information is
# sufficient for a 'usage' statement.  Otherwise, an application should
# derive its own usage statement and use this procedure as part of the
# error message.
Configuration private arg_info {} {
	$self instvar arg_option_ arg_bool_ usage_

	foreach arg [array names arg_option_] {
		set r $arg_option_($arg)
		# has default?
		set d [$self get_option $r]
		if { $d != "" || $usage_($arg) != "required"} {
			lappend opt "\[$arg $r ($d)\]"
		} else {
			lappend req "$arg $r"
		}
	}

	foreach arg [array names arg_bool_] {
		set r $arg_bool_($arg)
		# has default?
		set d [$self get_option $r]
		if { $d != "" } {
		        lappend opt "\[$arg ($d)\]"
		} else {
			lappend opt "\[$arg\]"
		}
	}

	if [info exists opt] {
		if [info exists req] {
			return [concat $opt $req]
		} else {
			return $opt
		}
	} else {
		if [info exists req] {
			return $req
		} else {
			return ""
		}
	}
}

#
# Loads the user preferences profile from persistent storage.
# As with parse args, this method uses the argument info that
# was previously registered with the register_option method.
#
Configuration public load_preferences suffixList {
    global env
    if {![info exists env(HOME)]} {
        new ErrorWindow {Your HOME environment variable must be set.}
        exit 1
    }
	set mash [file join $env(HOME) .mash]
	if {[file isdirectory $mash]} {
		$self load_file $mash/prefs
		foreach suffix $suffixList {
			$self load_file $mash/prefs-$suffix
		}
	}
}

#
# Load a preference file into the configuration database.
# The format of the file is one configuration option per line.
# The first word is the key and the remaining words are the value.
#
Configuration private load_file fname {
	if ![file readable $fname] {
		return
	}
	set f [open $fname r]
	set count 0
	while 1 {
		incr count
		if [eof $f] {
			close $f
			return
		}
		set line [string trim [gets $f]]
		#FIXME dangerous -- should factor out common piece here.

		# first check if this is a blank line or a comment
		if { $line == {} || [string index $line 0]=="#" } {
			continue
		}

		set colon [string first ":" $line]
		if { $colon==-1 } {
			# could not find a colon; must be a file in the old
			# format; ignore this line and output a warning

			puts stderr "Invalid line $count in $fname:\
					Must be of the form \"key: value\""
			continue
		}

		set option [string trim [string range $line 0 [expr $colon-1]]]
		set value [string trim [string range $line \
				[expr $colon+1] end]]

		## set option [lindex $line 0]
		## set value [lrange $line 1 end]

		#FIXME need to find the option in the right data base!
		$self add_option $option $value
	}
}


Configuration public open_preferences { suffix {mode w} } {
    global env
    if {![info exists env(HOME)]} {
        new ErrorWindow {Your HOME environment variable must be set.}
        exit 1
    }
	set mash [file join $env(HOME) .mash]
	if {![file exists $mash]} {
		file mkdir $mash
	}
	set f [open $mash/prefs-$suffix $mode 0644]
	return $f
}


Configuration public write_preference { file key value } {
	puts $file "$key: $value"
}


Configuration public close_preferences { file } {
	close $file
}


import ErrorWindow
