# ui-dcbroadcast.tcl --
#
#       UI for broadcast pane.  Handles drag/drop events which cause
#       broadcast/clearing to occur.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import CDcUIVideo

Class CDcUIBroadcastFrame

CDcUIBroadcastFrame instproc init { appDc uiMain winFrame } {
    $self instvar m_appDc
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_iFrameCounter
    $self instvar m_luiBroadcast
    $self instvar m_agentBroadcast

    $self next

    # initialize member variables
    set m_iFrameCounter 0
    set m_luiBroadcast ""

    # copy arguments to member variables
    set m_appDc $appDc
    set m_uiMain $uiMain
    set m_winFrame $winFrame
    set m_agentBroadcast [$m_appDc get_bcast_agent]

    # build the ui
    $self BuildUI
}

CDcUIBroadcastFrame instproc BuildUI {} {
    $self instvar m_winFrame

    # the layout of this frame goes like this:
    # Inside the main frame, there are two frames
    # the top frame has a label in it
    # the bottom frame has a column of broadcasts

    # the top frame
    set f [[$self options] get_option smallfont]
    label $m_winFrame.title -text "Broadcast" -relief ridge -font $f
    pack $m_winFrame.title -side top -fill x

    # the bottom frame
    frame $m_winFrame.broadcast -relief ridge -borderwidth 2
    pack $m_winFrame.broadcast -side top -fill both
}

CDcUIBroadcastFrame public DropPreview { dragNDrop uiPreview x y } {
    set source [$uiPreview set m_source]
    set service [$uiPreview set m_service]

    return [$self DropSource $dragNDrop $source $service $x $y]
}

CDcUIBroadcastFrame public DropThumbnail { dragNDrop uiThumbnail x y } {
    set source [$uiThumbnail set m_source]
    set service [$uiThumbnail set m_service]

    return [$self DropSource $dragNDrop $source $service $x $y]
}

CDcUIBroadcastFrame public DropSource { dragNDrop source service x y } {
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_luiBroadcast
    $self instvar m_agentBroadcast

    # first check if it was dropped within the main window
    if { ![$m_uiMain is_in_window $m_winFrame $x $y] } {
	return 0
    }

    # next we'll check if the source was dropped in a broadcast window
    foreach uiBroadcast $m_luiBroadcast {

	set win [$uiBroadcast set m_winFrame]
	if { ![$m_uiMain is_in_window $win $x $y] } {
	    continue
	}

	# if it's in the broadcast window then we should switch the source
	# for this window
	$uiBroadcast SwitchSource $source $service
	return 1
    }

    # Else we'll have to create a new broadcast window and add it
    $self instvar m_appDc
    $self instvar m_uiMain
    $self instvar m_iFrameCounter

    # create the frame for the window
    set winFrame [frame $m_winFrame.$m_iFrameCounter]
    pack $m_winFrame.$m_iFrameCounter -side top

    # increment the frame counter
    set m_iFrameCounter [expr "$m_iFrameCounter + 1"]

    set uiBroadcast [new CDcUIBroadcast $m_appDc $m_uiMain $self \
	    $winFrame $m_agentBroadcast $source $service]

    # add it to the list of review windows
    lappend m_luiBroadcast $uiBroadcast

    return 1
}

CDcUIBroadcastFrame public RemoveBroadcast { uiBroadcast } {
    $self instvar m_luiBroadcast

    # get the broadcast window
    set winFrame [$uiBroadcast set m_winFrame]

    # remote it from the list of broadcasts
    set luiNewBroadcast ""
    foreach uiBroadcastTemp $m_luiBroadcast {
	if { $uiBroadcast != $uiBroadcastTemp } {
	    lappend luiNewBroadcast $uiBroadcastTemp
	}
    }
    set m_luiBroadcast $luiNewBroadcast

    # destory the broadcast
    delete $uiBroadcast 

    # get the frame for this broadcast
    destroy $winFrame
}

CDcUIBroadcastFrame public getBroadcastInfo {} {
    $self instvar m_luiBroadcast

    set arr(windows) ""
    foreach uiBroadcast $m_luiBroadcast {
	lappend arr(windows) $uiBroadcast
	set source [$uiBroadcast set m_source]
   	set inetAddr [$source addr]
	set hostname [lookup_host_name $inetAddr]
	set hostname [string tolower $hostname]
	set arr($uiBroadcast,hostname) $hostname
	set arr($uiBroadcast,addr) $inetAddr
    }
    set retList [array get arr]
    return $retList
}


#
# Return the CDcUIPreview object containing the source 
#
CDcUIBroadcastFrame public GetUIVideoWithSource { source } {
    $self instvar m_luiBroadcast
    foreach p $m_luiBroadcast {
	if {[$p GetSource] == $source} {
	    return $p
	}
    }
    return ""
}



Class CDcUIBroadcast -superclass CDcUIVideo

CDcUIBroadcast public init { appDc uiMain uiBroadcastFrame winFrame \
	agentBroadcast source service} {

    $self next $uiMain $winFrame $source $service

    # member variables
    $self instvar m_appDc
    $self instvar m_uiBroadcastFrame
    $self instvar m_dragNDrop
    $self instvar m_winFrame
    $self instvar m_agentBroadcast
    $self instvar m_iOutSrcId

    # store away input
    set m_appDc $appDc
    set m_uiBroadcastFrame $uiBroadcastFrame
    set m_winFrame $winFrame
    set m_agentBroadcast $agentBroadcast

    set m_dragNDrop 0

    # step the packet switcher
    set m_iOutSrcId [$m_agentBroadcast map_source [$source srcid]]

    $self BuildUI
}

CDcUIBroadcast public destroy { } {
    $self instvar m_agentBroadcast
    $self instvar m_iOutSrcId
    $self instvar m_source

    # unmap the outgoing video
    $m_agentBroadcast unmap_source [$m_source srcid] $m_iOutSrcId

    $self next
}

CDcUIBroadcast public StartVideo { } {
    $self next

    $self instvar m_winFrame

    # create and deal with the drag n drop object
    set m_dragNDrop [new DragNDrop $m_winFrame.frame.video.video \
	    "$self DropBroadcast"]
}

CDcUIBroadcast public SwitchSource { source service } {
    $self instvar m_agentBroadcast
    $self instvar m_iOutSrcId
    $self instvar m_source

    # now let's deal with the video agent and the switcher
    $m_agentBroadcast unmap_source [$m_source srcid] $m_iOutSrcId
    $m_agentBroadcast map_source [$source srcid] $m_iOutSrcId


    # first let's deal with the video object
    $self SwitchVideo $source
    $self SwitchService $service
}

CDcUIBroadcast public DropBroadcast { dragNDrop x y } {
    $self instvar m_uiMain
    $self instvar m_uiBroadcastFrame

    # get the main window for the broadcast frame
    set winFrame [$m_uiBroadcastFrame set m_winFrame]

    # check if it was dropped within the main window
    if { ![$m_uiMain is_in_window $winFrame $x $y] } {
	$m_uiBroadcastFrame RemoveBroadcast $self
	return
    }

    $dragNDrop zoom_back
}
