# application-replayproxy.tcl --
#
#       Created to talk to MARS to provide replay.  Also provides replay UI to
#       DC video window
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Application
import SSAC_Client
import CServiceManager
import CService


Class CReplayProxyApplication -superclass { Application Observer }

##############################################################################
#
# CReplayProxyApplication instproc InitArgs { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Registers all options before the command line argument is parsed
#
##############################################################################
CReplayProxyApplication instproc InitArgs { options } {

    # for the active services
    $options register_option -sas optSpecAS
    $options register_option -ComPort optComPort
}

##############################################################################
#
# CReplayProxyApplication instproc InitResources { options } {
#
# Input:
# options - the options object
#
# Output:
# none
#
# Description:
# Gives defaults for options
#
##############################################################################
CReplayProxyApplication instproc InitResources { options } {
    # for the active services
    $options add_default optSpecAS 224.4.5.24/50000/31
    $options add_default optComPort 0
}


##############################################################################
#
# CReplayProxyApplication instproc init { argv } {
#
# Input:
# argv - command line input
#
# Output:
# none
#
# Description:
# Constructor for the object.
#
##############################################################################
CReplayProxyApplication instproc init { argv } {
    $self next ReplayProxy

    # initialize some variables
    $self instvar m_inetClientAddress
    $self instvar m_iClientPort
    $self instvar m_liSourceId
    $self instvar m_szPlayPauseState
    $self instvar m_iOffset
    $self instvar m_iPreviousOffset
    $self instvar m_timer
    $self instvar m_tmStart
    $self instvar m_tmEnd

    set m_inetClientAddress ""
    set m_iClientPort ""
    set m_liSourceId ""
    set m_szPlayPauseState "PLAY"
    set m_iOffset 0
    set m_iPreviousOffset 600000
    set m_timer ""
    set m_tmStart 0
    set m_tmEnd 0


    # Initiailization of options
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options
    $options load_preferences "archival ReplayProxy"
    set argv [$options parse_args $argv]


    # setup the service manager stuff
    $self instvar m_serviceManager
    $self instvar m_lService

    set iPort [$options get_option optComPort]
    set m_serviceManager [new CServiceManager "ServiceApp" "Replay" $iPort]
    $m_serviceManager Attach $self

    set m_lService ""

    # setup the ssac client
    $self instvar m_ssacClient
    $self instvar m_specSsacAS

    set m_specSsacAS [$options get_option optSpecAS]
    set m_ssacClient [new SSAC_Client $m_specSsacAS -]
    $m_ssacClient attach_observer $self

    $self StartReplay $argv
}


##############################################################################
#
# CReplayProxyApplication instproc destroy { }
#
# Input:
# none
#
# Output:
# none
#
# Description:
# destructor
#
##############################################################################
CReplayProxyApplication instproc destroy { } {
    $self instvar m_timer

    if { m_timer != "" } {
	after cancel $m_timer
    }
}



##############################################################################
#
# CReplayProxyApplication instproc NewConnection { service }
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
CReplayProxyApplication instproc NewConnection { service } {
    # this member should never be called.
    puts "Replay Proxy got a new connection: exiting"
    exit

    return 0
}


##############################################################################
#
# CReplayProxyApplication public StartReplay { arguments } {
#
# Input:
# arguments - the arguments that are sent by the client.  It should hold
#   the address and port of the remote client as well as all the information
#   to start a new replay
#
# Output:
# none
#
# Description:
#
##############################################################################
CReplayProxyApplication public StartReplay { arguments } {
    $self instvar m_ssacClient
    $self instvar m_specSsacAS
    $self instvar m_inetClientAddr
    $self instvar m_iClientPort

    # parse out the arguments
    set inetMStudioAddr [lindex $arguments 0]
    set iMStudioPort [lindex $arguments 1]
    set specVideo "$inetMStudioAddr/$iMStudioPort"
    set m_inetClientAddr [lindex $arguments 2]
    set m_iClientPort [lindex $arguments 3]
    set szCatalogName [lindex $arguments 4]
    set specAudio [lindex $arguments 5]

    # Start filling out the description fields
    set description ""
    append description "START_DESCR\n"
    append description "file: $szCatalogName\n"
    append description "server: $m_specSsacAS\n"
    append description "inetBroadcastAddr: video:$specVideo\n"
    append description "inetBroadcastAddr: audio:$specAudio\n"
    append description "END_DESCR\n"

    # now get the information from the catalog file and append it to the end
    if [catch "open $szCatalogName r" fileId] {
	puts stderr " Can't open $szCatalogName: $fileId"
	exit
    }

    append description [read $fileId]

    # tell the server to start the video
    $m_ssacClient open $description 0
}


##############################################################################
#
# CReplayProxyApplication public activate { tmStart tmEnd }
#
# Input:
# tmStart - the start time of the replay
# tmEnd - the end time of the replay
#
# Output:
# none
#
# Description:
# This is a callback from the client-ssac object which tells you the start
# and end time for the replay archive.  The main function is to change the
# slider to have the correct start and end times
#
##############################################################################
CReplayProxyApplication public activate { tmStart tmEnd } {
    $self instvar m_tmStart
    $self instvar m_tmEnd

    set m_tmStart $tmStart
    set m_tmEnd $tmEnd
}


##############################################################################
#
# CReplayProxyApplication public play_update { state }
#
# Input:
# state - the state of the replay either play or pause
#
# Output:
# none
#
# Description:
# if the state is pause, then it's actually playing and vice-versa.
#
##############################################################################
CReplayProxyApplication public play_update { state } {
    $self instvar m_lService
    $self instvar m_szPlayPauseState

    if { $m_szPlayPauseState != $state } {
	set m_szPlayPauseState $state

	$self UpdateAllClients
    }
}

##############################################################################
#
# CReplayProxyApplication public update_title { szSessionInfo1 \
#	szSessionInfo2 iDuration }
#
# Input:
# szSessionInfo1 - session info
# szSessionInfo2 - session info
# iDuration - the duration of the video
#
# Output:
# none
#
# Description:
# if the state is pause, then it's actually playing and vice-versa.
#
##############################################################################
CReplayProxyApplication public update_title { szSessionInfo1 \
	szSessionInfo2 iDuration } {
}


##############################################################################
#
# CReplayProxyApplication public set_offset { iOffset }
#
# Input:
# iOffset - the new offset that the replay is on
#
# Output:
# none
#
# Description:
# if the state is pause, then it's actually playing and vice-versa.
#
##############################################################################
CReplayProxyApplication public set_offset { iOffset } {
    $self instvar m_iPreviousOffset
    $self instvar m_iOffset
    $self instvar m_tmStart
    $self instvar m_tmEnd

    set m_iOffset $iOffset

    set duration [expr $m_tmEnd - $m_tmStart]
    if { [expr abs($iOffset - $m_iPreviousOffset) > int($duration/20)] } {
	set m_iPreviousOffset $iOffset
	$self UpdateAllClients
    }
}

##############################################################################
#
# CReplayProxyApplication public ssac_reset { szMedia szAddress }
#
# Input:
# szMedia - the type of media that the this address is broadcasting
# szAddress - the address of the broadcast
#
# Output:
# none
#
# Description:
# This function will use the inputted information to start the video service
# connection back to the client.  However, the source id's area also needed.
# The source id's are all in the ssac client object but we don't have a good
# interface to get them.  Changing this callback function would adversely
# affect other programs using this object.  So a hack will be made.
#
##############################################################################
CReplayProxyApplication public ssac_reset { szMedia szAddress } {
    $self instvar m_ssacClient
    $self instvar m_liSourceId
    $self instvar m_lService
    $self instvar m_inetClientAddr
    $self instvar m_iClientPort
    $self instvar m_serviceManager
    $self instvar m_lService

    # I'm only interested in video streams right now
    if { $szMedia != "video" } {
	return
    }

    # I'm breaking the abstraction layer for the ssac.  I'm going to dig deep
    # into it's data structure to get to the source id's for the archive
    # stream being sent.
    set lSessionList [$m_ssacClient set sesslist_]

    foreach sess $lSessionList {
	if { [$m_ssacClient set session_($sess,Address)] == $szAddress && \
		[$m_ssacClient set session_($sess,Media)] == $szMedia } {
	    set liSourceId [$m_ssacClient set session_($sess,SourceId)]
	    break
	}
    }

    foreach iSourceId $liSourceId {
	# see if we've already dealt with this source id
	if { [lsearch -exact $m_liSourceId $iSourceId] == -1 } {
	    lappend m_liSourceId $iSourceId

	    set service [$m_serviceManager NewService $m_inetClientAddr \
		    $m_iClientPort "VideoService" $iSourceId]

	    $service MapMessage "GET_UI_CONTROL" $self "GetUIControl"
	    $service MapMessage "UPDATE_CONTROL_UI" $self "UpdateControlUI"
	    $service MapMessage "PLAY_PAUSE" $self "PlayPauseCmd"
	    $service MapMessage "MOVE_SLIDER" $self "MoveSliderCmd"
	    $service MapMessage "CLOSE_LINK" $self "CloseService"
	    $service MapMessage "GET_INFO" $self GetInfo

	    lappend m_lService $service

	}
    }
}

##############################################################################
#
# CReplayProxyApplication private GetUIControl { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the ui commands that will be evaluated to update the ui
#
# Output:
# none
#
# Description:
# send the control ui commands back to the client
#
##############################################################################
CReplayProxyApplication private GetUIControl { service arguments } {
    $self instvar m_szPlayPauseState
    $self instvar m_iOffset
    $self instvar m_tmStart
    $self instvar m_tmEnd

    set cmd ""

    # for unique variable names
    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    # the play/pause button
    append cmd "global playPause\$__name \n"
    append cmd "set playPause\$__name $m_szPlayPauseState \n"
    append cmd "button \$winFrame.playPauseButton \
	    -textvariable playPause\$__name  \
	    -command \"\$service Send PLAY_PAUSE \\\"\\\" \" \n"

    append cmd "pack \$winFrame.playPauseButton -fill y -side right \n"

    # the progress slider
    append cmd "global slider\$__name \n"
    append cmd "set slider\$__name [expr int($m_iOffset)] \n"
    append cmd "scale \$winFrame.slider -orient horizontal \
	    -from 0 -to [expr int($m_tmEnd - $m_tmStart)] \
	    -tickinterval [expr int(0.49 * ($m_tmEnd - $m_tmStart))] \
	    -showvalue true \
	    -variable slider\$__name \
	    -length 200 \
	    -command \"\$service Send MOVE_SLIDER\" \n"
    append cmd "pack \$winFrame.slider -side left -fill x -expand 1 \n"


    $service Send CONTROL_UI $cmd
}

##############################################################################
#
# CReplayProxyApplication private UpdateControlUI { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the ui commands that will be evaluated to update the ui
#
# Output:
# none
#
# Description:
#
##############################################################################
CReplayProxyApplication private UpdateControlUI { service arguments } {
    $self instvar m_szPlayPauseState
    $self instvar m_iOffset

    set cmd ""

    # for unique variable names
    append cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    append cmd "global playPause\$__name \n"
    append cmd "set playPause\$__name $m_szPlayPauseState \n"

    append cmd "global slider\$__name \n"
    append cmd "set slider\$__name [expr int($m_iOffset)] \n"

    $service Send UPDATE_CONTROL_UI $cmd
}


##############################################################################
#
# CReplayProxyApplication private UpdateAllClients { }
#
# Input:
# none
#
# Output:
# none
#
# Description:
# this will update all the clients' ui.
#
##############################################################################
CReplayProxyApplication private UpdateAllClients { } {
    $self instvar m_lService

    foreach service $m_lService {
	$self UpdateControlUI $service ""
    }
}


##############################################################################
#
# CReplayProxyApplication private UpdateOffset { }
#
# Input:
# none
#
# Output:
# none
#
# Description:
# update the offset
#
##############################################################################
CReplayProxyApplication private UpdateOffset { } {
    $self instvar m_timer
    $self instvar m_iOffset
    $self instvar m_szPlayPauseState

    if { $m_szPlayPauseState == "PAUSE" } {
	incr m_iOffset 2
	$self UpdateAllClients
    }

    set m_timer [after 2000 "$self UpdateOffset"]
}


##############################################################################
#
# CReplayProxyApplication private PlayPauseCmd { service arguments }
#
# Input:
# service - the service that called this function
# arguments - the ui commands that will be evaluated to update the ui
#
# Output:
# none
#
# Description:
# called by the client when the user presses the play/pause button
#
##############################################################################
CReplayProxyApplication private PlayPauseCmd { service arguments } {
    $self instvar m_ssacClient
    $self instvar m_szPlayPauseState
    $self instvar m_iOffset

    if { $m_szPlayPauseState == "PLAY" } {
	$m_ssacClient play $m_iOffset
    } else {
	$m_ssacClient pause
    }


}


##############################################################################
#
# CReplayProxyApplication private MoveSliderCmd { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the new offset to move to
#
# Output:
# none
#
# Description:
# called by the client when the user moves the slider
#
##############################################################################
CReplayProxyApplication private MoveSliderCmd { service arguments } {
    $self instvar m_ssacClient
    $self instvar m_szPlayPauseState
    $self instvar m_iOffset

    # set the new offset
    set m_iOffset [expr int($arguments)]

    # move position in the archive only if we're playing - ie. with pause
    # showing.  This is cause once we move, we'll be in the play mode.
    if { $m_szPlayPauseState == "PAUSE" } {
	$m_ssacClient send PLAY [expr int($arguments)] 1
    }
}


##############################################################################
#
# CReplayProxyApplication public CloseService { service arguments } {
#
# Input:
# service - the service that called this function
# arguments - the arguments associated with this call
#
# Output:
# none
#
# Description:
# This is called by the remote service client closes the service link.
#
##############################################################################
CReplayProxyApplication public CloseService { service arguments } {
    exit
}

# the string argument to UPDATE_INFO is put in the DC display
CReplayProxyApplication public GetInfo { service } {
    $service Send UPDATE_INFO ""
}

