# source.tcl --
#
#       Defines Program object, which is a single event described by one
#       of more SDP messages.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import SDPParser

# An instance of Program is a single event described by one or more
# SDP messages.  Events that do not use layered transmission have
# just a single SDPMessage.  A Program that does use layered
# transmission consists of a list of SDPMessage objects.
Class Program

# Initialize a new Program object.  Argument list should contain a
# (potentially incomplete) list of SDPMessage objects that are part
# of this program.
Program public init {args} {
    $self next

    $self set complete_ 0
    $self set msgs_ {}
    foreach m $args {
	$self message $m
    }
}

#
Program public destroy {} {
	$self instvar msgs_
	foreach msg $msgs_ {
		delete $msg
	}
	$self next
}

# Returns a boolean value that indicates whether the SDPMessage
# objects that are part of this Program completely specify the
# program (i.e., whether announcements for all layers have been
# received).
Program public complete {} {
    return [$self set complete_]
}

# Returns the SDPMessage that announces the base layers for this
# program (which also contains all other information such as the
# i=, s=, etc.. fields)
Program public base {} {
    $self instvar msgs_
    if {[llength $msgs_] < 1} {
	return ""
    }
    return [lindex $msgs_ 0]
}

# Like the <i>have_attr</i> method on an SDPMessage, returns true
# if the attribute <i>a</i> is defined on any of the SDPMessages
# that make up this Program.
Program public have_attr {a} {
    $self instvar msgs_
    foreach m $msgs_ {
	if [$m have_attr $a] {
	    return 1
	}
    }
    return 0
}

# Like the <i>attr_value</i> method on an SDPMessage, returns the
# value of the attribute <i>a</i> if it is defined on any of the
# SDPMessages that make up this Program.
Program public attr_value {a} {
    $self instvar msgs_
    foreach m $msgs_ {
	set v [$m attr_value $a]
	if {$v != ""} {
	    return $v
	}
    }
    return ""
}

# Like the <i>have_field</i> method on an SDPMessage, returns true
# if the field <i>f</i> is present in any of the SDPMessages that
# make up this Program.
Program public have_field {f} {
    $self instvar msgs_
    foreach m $msgs_ {
	if [$m have_field $f] {
	    return 1
	}
    }
    return 0
}

# Like the <i>field_value</i> method on an SDPMessage, returns the
# value of the field <i>f</i> if it is present in any of the
# SDPMessages that make up this Program.
Program public field_value {f} {
    $self instvar msgs_
    foreach m $msgs_ {
	set v [$m field_value $f]
	if {$v != ""} {
	    return $v
	}
    }
    return ""
}

# Same as the <i>unique_key</i> method on an SDPMessage, returns
# a string that serves as a unique key for this program.
Program public unique_key {} {
    set b [$self base]
    if {$b == ""} { return "" }
    return [$b unique_key]
}

# Parse the string <i>attr</i>, which should be an SDP layers
# attribute (i.e., the string "1-2/3" from the line
# "a=layers:1-2/3" in an SDP message).  The variables <i>begin</i>,
# <i>end</i>, and <i>total</i> in the calling frame are set to
# the first layer number, the last layer number, and the total
# number of layers, respectively.
Program private parse_layers {attr begin end total} {
    upvar $begin b $end e $total t

    set l [split $attr "/"]
    set len [llength $l]
    if {$len < 1 || $len > 2} {
	$self warn "Malformed layers attribute \"$t\""
	return 1
    }
    if {$len == 2} {
	set t [lindex $l 1]
    } else {
	set t ""
    }

    set layers [lindex $l 0]

    set bounds [split $layers "-"]
    set b [lindex $bounds 0]
    set l [llength $bounds]
    if {$l == 1} {
	set e $b
    } elseif {$l == 2 } {
	set e [lindex $bounds 1]
    } else {
	$self warn "Malformed layers attribute \"$t\""
	return 1
    }
    return 0
}

# Add the SDPMessage <i>msg</i> to this Program.  Returns a boolean
# value that is true if this message changes the Program in some way
# (e.g., if it completes the descriptions of all layers or if it is
# an update to an SDPMessage that was previously received).
#
# FIXME this doesn't handle two layered streams of the same media type
# (e.g., two layered video streams in a single program)
Program public message {msg} {
    $self instvar msgs_ complete_

    set len [llength $msgs_]
    if {$len == 0} {
	# first message (i.e., we are being called from constructor)
	set msgs_ [list $msg]
	set complete_ 1
	foreach m [$msg set allmedia_] {
	    if [$m have_attr layers] {
		set a [$m attr_value layers]
		if [$self parse_layers $a begin end total] {
		    continue
		}

		if {$begin > 0} {
		    set complete_ 0
		}
		if {$total != "" && $end < [expr $total-1]} {
		    set complete_ 0
		}
	    }
	}
	return 1
    }

    # multiple messages with the same o= tuple.  If no "layers"
    # attributes are present then this should be an update.
    set layered 0
    foreach m [$msg set allmedia_] {
	if [$m have_attr layers] {
	    set layered 1
	    break
	}
    }
    if {$layered == 0} {
	if {$len == 1} {
	    return [$self update $msg 0]
	} else {
	    $self warn "Got apparent layered announcement without layers attribute"
	    return 0
	}
    }

    # multiple messages with "layers" attribute present -- this may
    # be a refresh of a set of layers we already have an announcement
    # for or it may describe a set of layers we don't know about yet.
    #
    # FIXME assumes only a single media type has layers attributes.
    foreach m [$msg set allmedia_] {
	if [$m have_attr layers] {
	    break
	}
    }
    set type [$m set mediatype_]

    if [$self parse_layers [$m attr_value layers] begin end total] {
	#ignore bogus message
	delete $msg
	return
    }

    set position end
    set i 0
    while {$i < $len} {
	set a [[[lindex $msgs_ $i] media $type] attr_value layers]
	if [$self parse_layers $a begin2 end2 total2] {
	    #should never happen
	    $self fatal "have message is msgs_ with bogus layers attr"
	}
	#FIXME should check for overlap
	if {$begin == $begin2 && $end == $end2} {
	    return [$self update $msg $i]
	}
	if {$begin < $begin2} {
	    set position $i
	    break
	}
	incr i
    }
    set msgs_ [linsert $msgs_ $position $msg]
    incr len

    # check if the message is "complete".  That is, it total is
    # specified, do we have all layers.  If total is not specified,
    # do we (not) have any gaps?
    set next 0
    set i 0
    while {$i < $len} {
	set a [[[lindex $msgs_ $i] media $type] attr_value layers]
	if [$self parse_layers $a begin end total2] {
	    #should never happen
	    $self fatal "have message is msgs_ with bogus layers attr"
	}
	if {$begin == $next} {
	    set next [expr $end+1]
	} else {
	    return 0
	}
	incr i
    }
    if {$total != "" && $next != $total} {
	return 0
    }
    set complete_ 1
    return 1
}

# Used internally.  Replaces the message in the <i>i</i>th position
# in the list of messages with <i>msg</i>
Program private update {msg i} {
    $self instvar msgs_
    set old [lindex $msgs_ $i]
    set oldversion [lindex [$old field_value o] 2]
    set newversion [lindex [$msg field_value o] 2]
    if {$newversion > $oldversion} {
	set msgs_ [lreplace $msgs_ $i $i $msg]
	delete $old
	return 1
    } else {
	delete $msg
	return 0
    }
}

# ProgramSource in an abstract base class.  Any method for retrieving
# a program description (e.g., SAP) should be implemented in a class
# that inherits from ProgramSource.  Since it is an abstract base class,
# no ProgramSource object is instantiated, but some functionality common
# to all derived classes is provided.

Class ProgramSource

# Sets up the initial state for a ProgramSource object.
# <i>rcvr</i> is an object that will receive program announcements
# as they are received.  It should implement methods <i>addsource</i>,
# <i>addprog</i>, <i>removeprog</i>, and <i>updateprog</i>.
# See NsdrUI for an example.
ProgramSource public init {rcvr} {
    $self set rcvr_ $rcvr
    $rcvr addsource $self
    $self set sdp_ [new SDPParser 0]
}

# Invoked from a derived class when an announcement is received.
# <i>data</i> is a string containing an SDP message.
ProgramSource public recv {data} {
    $self process [string trim $data]
}

# Parses the string <i>data</i> as SDP.  Updates this source's list
# of programs as appropriate.
ProgramSource private process {data} {
    $self instvar sdp_ rcvr_

    set progs {}
    foreach msg [$sdp_ parse $data] {
	set o [$msg unique_key]

	$self instvar progs_
	if ![info exists progs_($o)] {
	    set p [new Program $msg]
	    set o [$msg unique_key]
	    $self set progs_($o) $p

	    if [$p complete] {
		$rcvr_ addprog $self $p
	    }
	} else {
	    set p $progs_($o)
	    set wascomplete [$p complete]
	    if [$p message $msg] {
		if {$wascomplete == 0} {
		    $self instvar rcvr_
		    $rcvr_ addprog $self $p
		} else {
		    $rcvr_ updateprog $self $p
		}
	    }
	}
	lappend progs $p
    }
    return $progs
}

# Set the timeout for the Program <i>p</i> to be <i>timeout</i>
# seconds.  If a timeout was previously set for this program, it
# is cancelled and replace by this one.
ProgramSource public timeout {p timeout} {
    $self instvar timeouts_
    set o [$p unique_key]
    if [info exists timeouts_($o)] {
	after cancel $timeouts_($o)
    }
    set timeouts_($o) [after [expr 1000*$timeout] "$self remove $p"]
}

# Notifies the receiver attached to this source that the Program
# <i>p</i> should be removed.
ProgramSource public remove {p} {
    $self instvar progs_ rcvr_

    set o [$p unique_key]
    if {![info exists progs_($o)] || $progs_($o) != $p} {
	$self warn "inconsistency in ProgramSource::remove"
	return
    }

    $rcvr_ removeprog $self $p
    delete $p
    unset progs_($o)
}

# If the instvar <i>cache_</i> is set, try to read cached program
# announcements from that file and display them in the user interface.
ProgramSource private readcache {} {
    $self instvar cache_
    if {![info exists cache_] || ![file readable $cache_]} {
	return
    }
    set fp [open $cache_ r]
    set progs [$self process [read $fp]]
    close $fp

    foreach p $progs {
	# set initial timeout to be 30 mins
	$self timeout $p 1800
    }
}

# Notify this object that it is about to be destroyed.
ProgramSource public shutdown {} {
    $self writecache
}

# If the instvar <i>cache_</i> is set, try to write the
# information about all present programs in to that file
# so that it can later be read by <i>readcache</i>.
ProgramSource private writecache {} {
    $self instvar cache_
    if ![info exists cache_] { return }

    if ![file isdirectory [file dirname $cache_]] {
	file mkdir [file dirname $cache_]
    }

    if [catch {set fp [open $cache_ w]} m] {
	$self warn "couldn't open cache file $cache_ for writing: $m"
	return
    }

    $self instvar progs_
    foreach o [array names progs_] {
	foreach m [$progs_($o) set msgs_] {
	    puts $fp [$m set msgtext_]
	}
    }
    close $fp
}

# Writes the program cache every <i>time</i> seconds.
ProgramSource public periodic-writecache {time} {
    $self writecache
    after [expr 1000 * $time] "$self periodic-writecache $time"
}

