# client-rtsp.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/rtsp/client-rtsp.tcl,v 1.15 2002/02/03 04:29:24 lim Exp $

Import enable
import mashutils SDPParser TCP

# Client side processing and parsing of the RTSP archive control protocol

Class RTSP_Client

#
# RTSP client classes
#
RTSP_Client instproc init_vars { app } {
	$self instvar rtsp_ session_

	set rtsp_(version) "RTSP/1.0"
	set rtsp_(seqno) 0
	set rtsp_(state) IDLE
	set rtsp_(application) $app
	set session_(uri_) ""
	set session_(mlist) [$rtsp_(application) get_option defaultmediaList]
	set session_(setup,mcnt)  [llength $session_(mlist)]
	set session_(setup,ccnt) 0

	$self init_stat
	$self init_perf
	$self init_tokens
	# Seed the random number generator heuristically.
        random 0

	# If a uri is passed to the application through the -server option,
	# start the tools immediately.
	return
}

RTSP_Client instproc init_stat { } {
	$self instvar statmsgs_

	set statmsgs_("Continue") 100
	set statmsgs_("OK") 200
	set statmsgs_("Created") 201
	set statmsgs_("Accepted") 202
	set statmsgs_("Non-Authoritative\ Information") 203
	set statmsgs_("No\ Content") 204
	set statmsgs_("Reset\ Content") 205
	set statmsgs_("Partial\ Content") 206
	set statmsgs_("Multiple\ Choices") 300
	set statmsgs_("Moved\ Permanently") 301
	set statmsgs_("Moved\ Temporarily") 302
	set statmsgs_("Bad\ Request") 400
	set statmsgs_("Unauthorized") 401
	set statmsgs_("Payment\ Required") 402
	set statmsgs_("Forbidden") 403
	set statmsgs_("Not\ Found") 404
	set statmsgs_("Method\ Not\ Allowed") 405
	set statmsgs_("Not\ Acceptable") 406
	set statmsgs_("Proxy\ Authentication\ Required") 407
	set statmsgs_("Request\ Time-out") 408
	set statmsgs_("Conflict") 409
	set statmsgs_("Gone") 410
	set statmsgs_("Length\ Required") 411
	set statmsgs_("Precondition\ Failed") 412
	set statmsgs_("Request\ Entity\ Too\ Large") 413
	set statmsgs_("Request-URI\ Too\ Large") 414
	set statmsgs_("Unsupported\ Media\ Type") 415
	set statmsgs_("Bad\ Extension") 420
	set statmsgs_("Invalid\ Parameter") 450
	set statmsgs_("Parameter\ Not\ Understood") 451
	set statmsgs_("Conference\ Not\ Found") 452
	set statmsgs_("Not\ Enough\ Bandwidth") 453
	set statmsgs_("Session\ Not\ Found") 454
	set statmsgs_("Method\ Not\ Valid\ In\ This\ State") 455
	set statmsgs_("Header\ Field\ Not\ Valid\ for\ Resource") 456
	set statmsgs_("Invalid\ Range") 457
	set statmsgs_("Parameter\ Is\ Read-Only") 458
	set statmsgs_("Internal\ Server\ Error") 500
	set statmsgs_("Not\ Implemented") 501
	set statmsgs_("Bad\ Gateway") 502
	set statmsgs_("Service\ Unavailable") 503
	set statmsgs_("Gateway\ Time-out") 504
	set statmsgs_("RTSP\ Version\ Not\ Supported") 505
	set statmsgs_("Extended\ Error:") 911
}


RTSP_Client instproc init_perf { } {
	$self instvar perf_
	set perf_(empty) 0
	set perf_(reordered) 0
}


RTSP_Client instproc init_tokens { } {
	$self instvar rtsp_
	$self instvar token_


	set token_(SETUP)       "SETUP"
	set token_(REDIRECT)    "REDIRECT"
	set token_(PLAY)        "PLAY"
	set token_(PAUSE)       "PAUSE"
	set token_(SESSION)     "SESSION"
	set token_(RECORD)      "RECORD"
	set token_(EXT_METHOD)  "EXT-"

	set token_(HELLO)       "OPTIONS"
	set token_(GET)         "DESCRIBE"
	set token_(GET_PARAM)   "GET_PARAMETER"
	set token_(SET_PARAM)   "SET_PARAMETER"
	set token_(CLOSE)       "TEARDOWN"
	set rtsp_(tknlist) {"REDIRECT" "PLAY" "PAUSE" "SESSION" "RECORD" "EXT-" \
			"OPTIONS" "DESCRIBE" "GET_PARAMETER" "SET_PARAMETER" "TEARDOWN"}
}


Class RTSP_Socket -superclass TCP/Client

RTSP_Socket public init p {
	$self next
	$self set parser_ $p
	$self set buffer_ ""
}

#
# Called from TCP module when we receive data on our socket, which
# consists of rtsp commands from the server that we have connected to.
#
RTSP_Socket public recv s {
	$self instvar buffer_ parser_
	set buffer_ "$buffer_\n$s"
	#
	# Check for blank line then hand to parser.
	#
	if { [string trim $s] == "" } {
		$parser_ parse_message $buffer_
		set buffer_ ""
		#FIXME parent class should do this (but not in parse message)
		#$self close
	}
}

#RTSP_Client instproc set_uri { u } {
#	$self instvar session_
#	set session_(uri_) $u
#}


RTSP_Client instproc reset-vars { } {
	$self instvar rtsp_ session_

	if {[info exists session_(uri_)] && [string length $session_(uri_)]} {
		$self send "CLOSE"
		$rtsp_(application) set-offset 0
	}
	set rtsp_(seqno) 0
	set rtsp_(state) IDLE

	# Stop timer updates
	$rtsp_(application) stop
	# Clean up session_ variables
	unset session_
	set session_(mlist) [$rtsp_(application) get_option defaultmediaList]
	set session_(setup,mcnt)  [llength $session_(mlist)]
	set session_(setup,ccnt) 0
}


RTSP_Client instproc open { url desc } {

	$self instvar session_ rtsp_

	$self reset-vars
	set session_(uri_) $url
	set err [$self parse_url $url]
	if { $err != "" } {
		return $err
	}
	# Parsed tokens stored in session_

	if {![info exists rtsp_(port)] || $rtsp_(port) == ""} {
		set port_ 12124
	}
	$self state_machine

	return ""
}

RTSP_Client instproc state_machine { } {
	$self instvar rtsp_

	set rtsp_(state) IDLE
	set rtsp_(seqno) 0
	$self send "GET"
}

#
# Construct and send messages of type 'type'
#
RTSP_Client instproc send { type {mtype ""} } {

	global tcl_platform

	$self instvar rtsp_ session_
	$self instvar statmsgs_ token_

	if [info exists token_($type)] {
		set tkn $token_($type)
	} else {
		puts "RTSP client : No token $type"
		return
	}
	if [info exists session_(ret_status)] {
		set code $statmsgs_(\"$session_(ret_status)\")
	} else {
		set code " "
	}
	set date "Date: [clock format [clock seconds] -format {%d %b %Y %H:%M:%S GMT} -gmt true]"


	switch -exact -- $type {
		"SETUP" {
#			puts "Address = $session_($mtype,addr)"
			if {![info exists session_($mtype,addr)]} {
				puts "send: No address available for media type $mtype"
				return -1
			} else {
				incr rtsp_(seqno)
				set session_($mtype,cseq) $rtsp_(seqno)

				$self alloc "multicast" $mtype
				set msg "$tkn $session_($mtype,addr) $rtsp_(version)\nCSeq: $rtsp_(seqno)\n"
				if {[info exists session_(Session)]} {
					set msg "$msg\Session: $session_(Session)\n"
				}
				set msg "$msg\Transport: RTP/AVP;multicast;destination=$session_($mtype,daddr);"
				#set msg "$msg\Transport: RTP/AVP;unicast;destination=[localaddr];"
				set msg "$msg\port=$session_($mtype,dport);ttl=$session_($mtype,dttl)\r\n\r\n"
			}
		}
		"HELLO" {
			incr rtsp_(seqno)
			set msg "$tkn * $rtsp_(version)\nCSeq: $rtsp_(seqno)\nUser-Agent: mash RTSP/0.1alpha [set tcl_platform(os)]\r\n\r\n"
		}

		"HELLO_RESPONSE" {
			incr rtsp_(seqno)
			set msg "$rtsp_(version) $code\nCSeq: $rtsp_(seqno) $smsg_\n$date\r\n\r\n"
		}
		"CLOSE" {
			incr rtsp_(seqno)
			set msg "$tkn $rtsp_(version)\nSession: $session_(Session)\nCSeq: $rtsp_(seqno)\r\n\r\n"
		}
		"GET" {
			incr rtsp_(seqno)
			set msg "$tkn $session_(uri_) $rtsp_(version)\nCSeq: $rtsp_(seqno)\r\n\r\n"
		}
		"PAUSE" {
			incr rtsp_(seqno)
			set msg "$tkn $session_(uri_) $rtsp_(version)\nSession: $session_(Session)\nCSeq: $rtsp_(seqno)\r\n\r\n"
		}
		"PLAY" {
			incr rtsp_(seqno)
			set offset [$rtsp_(application) offset]
			set msg "$tkn $session_(uri_) $rtsp_(version)"
			set msg "$msg\nCSeq: $rtsp_(seqno)\nSession: $session_(Session)"
			set msg "$msg\nRange: npt=[expr int($session_(starttime_) + $offset)]\-\r\n\r\n"
		}
		default {
			puts "send: default type unknown"
			return
		}
	}

	$self instvar socket_
	if ![info exists socket_] {
		set socket_ [new RTSP_Socket $self]
		$socket_ open $rtsp_(addr) $rtsp_(port)
	}
	$socket_ sendline $msg
puts stderr "===SEND===\n$msg\n===EOB==="
}


# FIXME: Address allocation
# What happens if the port gets allocated to another
# process in the meanwhile?
# What happens if two different servers re-use the
# same multicast address but have overlapped regions?


RTSP_Client instproc alloc_port { mtype } {
	# Random number U[8192, 16384]
	set r01 [expr [random]/double(0x7fffffff)]
	return [expr round(8392 + $r01 * 8192)]
}


RTSP_Client instproc alloc_mcast_addr { mtype } {
	set lo1 round([expr [random]/double(0x7fffffff) * 256])
	set lo2 round([expr [random]/double(0x7fffffff) * 256])
	return "224.2.[expr round($lo1)].[expr round($lo2)]"
}

RTSP_Client instproc alloc { type mtype } {
	$self instvar session_ rtsp_

	if {$type == "unicast"} {
#		set session_($mtype,daddr) 128.32.130.20
		set session_($mtype,daddr) 127.0.0.1
		set session_($mtype,dttl)  [$rtsp_(application) get_option $mtype\TTL]
	} else {
		switch -- $mtype {
			video {
				set session_($mtype,daddr) 224.8.8.1
			}
			audio {
				set session_($mtype,daddr) 224.8.8.2
			}
			mediaboard {
				set session_($mtype,daddr) 224.8.8.3
			}
		}
		#set session_($mtype,daddr) [$self alloc_mcast_addr $mtype]

		  set session_($mtype,dttl)  [$rtsp_(application) get_option $mtype\TTL]
	}
	##set session_($mtype,dport) [$self alloc_port $mtype]
	set session_($mtype,dport) 8000
	return
}

RTSP_Client instproc parse_url { url } {

	$self instvar rtsp_

	# Check to see if  the URL is valid
	if [expr [string length $url] <= 0] {
		return "Invalid URL.\n Example: rtsp://rose.cs.berkeley.edu:12124"
	}
	if [expr [string first "rtsp://" $url] == 0] {
		set url [string range $url [string length "rtsp://"] end]
	}
	# Split at the first occurrence of '/'
	set sIndex [string first "/" $url]
	set lIndex [string length $url]
	set rtsp_(addr) [string range $url 0 [expr $sIndex - 1]]
	set rtsp_(path) [string range $url $sIndex end]

	set sIndex [string first ":" $rtsp_(addr)]
	set rtsp_(port) [string range $rtsp_(addr) [expr $sIndex + 1] end]
	set rtsp_(addr) [string range $rtsp_(addr) 0 [expr $sIndex -1]]
	if {$rtsp_(addr) == ""} {
		return "Invalid URL.\n Example: rtsp://rose.cs.berkeley.edu:12124"
	}
	return ""
}

## Server send messages terminated by "\n\n"
RTSP_Client instproc valid_message {message} {
	return "[expr [string first "\n\n" $message]]"
}

RTSP_Client instproc parse_message { message } {
	$self instvar session_ perf_
	$self instvar parse_ rtsp_
	puts "Received\n$message"

	if [info exists session_(Transport)] {
		unset session_(Transport)
	}
	if {[string match [string trim $message] ""]} {
		incr perf_(empty)
		return 0
	}
	puts "Checking ..."
	set is_resp [$self valid_response $message]
	if {!$is_resp} {
		set op [lindex $parse_(parts) 0]
		set r  [$self valid_method $op]
		if {!$r} {
			puts "parser: Invalid message received"
			return 0
		}
	}

	# Throw away unexpected packets.
	# Since we only support TCP now, we don't handle
	# reordered packets.
	if {$session_(CSeq) != $rtsp_(seqno)} {
#		puts "Reordered packets"
		incr perf_(reordered)
		return 0
	}
	puts "Doing handle_event ...."
	if {[$self handle_event $message $session_(status)]==0} {
		unset session_
		$self send "CLOSE"
		set rtsp_(state) IDLE
		set rtsp_(seqno) 0
	}
}

#
# The setup state has a self-loop.
#
RTSP_Client instproc setup { } {
	$self instvar session_ rtsp_

	if {$session_(setup,ccnt) < [expr $session_(setup,mcnt)]} {
		set media [lindex $session_(mlist) $session_(setup,ccnt)]
		$self send "SETUP" [lindex $session_(mlist) $session_(setup,ccnt)]
		incr session_(setup,ccnt)
	} else {
		set rtsp_(state) READY
		$self play
	}
}

#
# Wait until the client enters the state named by <i>state</i>.
# FIXME If this never happens, call will block forever.
#
RTSP_Client public wait state {
	$self instvar rtsp_
	while { $state != $rtsp_(state) } {
		vwait rtsp_(state)
	}
}


RTSP_Client instproc handle_event { message opcode } {
	$self instvar session_ rtsp_

	switch -exact -- $rtsp_(state) {
		IDLE {
			switch -- $opcode {
				"OK" {
					# SDP announcement piggybacked on "OK"
					# response from server.
					puts "before parse_desc"
					if {[$self parse_desc $message] == 1} {
						set rtsp_(state) INIT
						$self setup
						return 1
					} else {
						puts "Invalid SDP description received.\nSession is perhaps not available at this location."
						return 0
					}
				}
			}
		}
		INIT {
			switch -- $opcode {
				"OK" {
					foreach media $session_(mlist) {
						if {[expr $session_(code) < 203] && \
								[info exists session_($media,cseq)] &&\
								[expr $session_(CSeq) == $session_($media,cseq)]} {
							$self start_tool $media
							unset session_(Transport)
						}
					}
					$self setup
					return 1
				}
			}
		}
		READY {
			switch -- $opcode {
				"OK" {
					if {[expr $session_(code) < 203]} {
						set session_(state) PLAY
						$rtsp_(application) start-timer
						return 1
					} else {
						return 0
					}
				}
			}
		}
		PLAY {
			switch -exact -- $opcode {
				"OK" {
					if {![expr $session_(code) < 203]} {
						puts "RTSP_Client: Play request failed"
						return 0
					}
					$rtsp_(application) activate_slider
				}
			}
		}
		PAUSE {
			switch -exact -- $opcode {
				"OK" {
					if {![expr $session_(code) < 203]} {
						puts "RTSP_Client: Pause request failed"
						return 0
					}
				}
			}
		}
		default {
			puts "State unknown $rtsp_(state)"
			return 0
		}
	}
}

RTSP_Client instproc parse_address { } {

	$self instvar session_
 	set fields [split $session_(Transport) ";"]
	set l [llength $fields]
	for {set i 0} {$i < $l} {incr i} {
		set phrase [string trim [lindex $fields $i]]
		regexp {([a-zA-Z]+)(=)(.+)} $phrase all token mid rest
		if {[info exists token]} {
			set session_($token) [string trim [string trimright $rest ";"]]
		}
	}
}


RTSP_Client instproc start_tool {mediatype} {
	$self instvar session_ rtsp_

	$self parse_address
	set fmt [$self get_option audioFormat]
	switch -- $mediatype {
		video {
			set address "$session_(destination)/$session_(port)/$session_(ttl)"
			puts "Starting VIC..."
			$rtsp_(application) start_vic $address
		}
		audio {
			set address "$session_(destination)/$session_(port)/PCM/$session_(ttl)"
			puts "Starting VAT..."
			$rtsp_(application) start_vat $address
		}
		mediaboard {
			puts "Not starting Mediaboard**"
		}
	}
}


RTSP_Client instproc valid_method {opcode} {
	$self instvar rtsp_
	foreach t $rtsp_(tknlist) {
		if {[string match $t $opcode]} {
			return 1
		}
	}
	return 0
}

RTSP_Client instproc valid_response {m} {
	$self instvar session_

	#
	# strip any leading/trailing whitespace (including returns)
	# then replace all multiple newline sequences with a single newline
	#
	set m [string trim $m]
	regsub -all "(\r|\n)+" $m "\n" message
	set lines [split $message "\n"]
	set fields [split [lindex $lines 0]]

	set session_(version) [string trim [lindex $fields 0]]
	if  ![string match "RTSP/1.0" $session_(version)] {
		puts "Incorrect version string $session_(version)"
		return 0
	}
	# The numeric return code (200) and the status "OK" are redundant
	set session_(code) [lindex $fields 1]
	set session_(status) [lindex $fields 2]
	set l [llength $lines]
	for {set i 1} {$i < $l} {incr i} {
		if ![string match [lindex $lines $i] ""] {
			regexp {([a-zA-Z]+)(:)(.+)} [lindex $lines $i] all token mid rest
			if {$token != ""} {
				set session_($token) $rest
			}
		}
	}
	return 1
}

RTSP_Client instproc playOrPause { {stop "" } } {
	$self instvar rtsp_
	if {$stop == "STOP"} {
		set rtsp_(state) "PLAY"
	}
	if {$rtsp_(state) == "PLAY"}  {
		$self pause
	} else {
		$self play
	}
}

RTSP_Client instproc play { } {
	$self instvar rtsp_
	if { $rtsp_(state) != "PLAY" } {
		set rtsp_(state) PLAY
		$rtsp_(application) play-update "Pause"
		$self send PLAY
	}
}

RTSP_Client instproc pause { } {
	$self instvar rtsp_
	if { $rtsp_(state) != "PAUSE" } {
		set rtsp_(state) PAUSE
		$rtsp_(application) play-update "Play"
		$self send PAUSE
	}
}

RTSP_Client instproc starttime { } {
	$self instvar session_
	return $session_(starttime_)
}

RTSP_Client instproc endtime { } {
	$self instvar session_
	return $session_(endtime_)
}

RTSP_Client instproc duration { } {
	$self instvar session_
	return [expr $session_(endtime_) - $session_(starttime_)]
}


RTSP_Client instproc elapsed { time } {
	$self instvar session_
	return [duration_readable $time]
}

RTSP_Client instproc exit { } {
	return
}

RTSP_Client instproc parse_desc { message } {

	$self instvar session_ rtsp_
	regsub -all "\r\n" $message "\n" message
	regsub -all "\n\n" $message "\n" message
	regsub -all "\n\n" $message "\n" message

	set lines [split $message "\n"]
	set l [llength $lines]
	for {set i 0} {$i < $l} {incr i} {
		set prev [lindex $lines [expr $i - 1]]
		set phrase [lindex $lines $i]
		if {[expr [string first "Content-Length:" $phrase] >= 0] && \
				[expr [string last "application/sdp" $prev] >= 0]}  {
#			puts " **** \n$prev\n$phrase"
			break
		}
	}
	set alen [string trim [lindex [split $phrase ":"] 1]]
	if {$alen <= 0} {
		return 0
	}
	set announcement [join [lrange $lines [expr $i+1] end] "\n"]
	set p [new SDPParser]
	puts "Parsing message"
# Right now, we only expect 1 message
	set message [lindex [$p parse $announcement] 0]
	set td  [lindex [$message set alltimedes_] 0]
	set props "version_ creator_ createtime_ session_name_ session_info_"
# Need to put in uri_ right now, server sends a wrong uri_ consistently
	set times "starttime_ endtime_"

	foreach pr $props {
		set session_($pr) [$message set $pr]
	}
	foreach t $times {
		set session_($t) [$td set $t]
	}
# Convert NTP time to UNIX and format
	set session_(createtime_) \
			"[clock format [ntp_to_unix $session_(createtime_)]]"

# FIXME
#	set session_(recorded) [expr !$desc_(starttime_)]
# If start time is 0, session is recorded, not live
# This has changed now! This should be set in a separate 'a' field

	set v [$message media video]
	set a [$message media audio]
	set m [$message media mediaboard]
	set session_(mlist) ""
	if {$v != ""} {
		lappend session_(mlist) "video"
	}
	if {$a != ""} {
		lappend session_(mlist) "audio"
	}
	if {$m != ""} {
		lappend session_(mlist) "mediaboard"
		set session_(mediaboard,addr) [$m set caddr_]
		puts "**Appended**"
	}
	set session_(setup,mcnt) [llength $session_(mlist)]
	set session_(video,addr) [$v set caddr_]
	set session_(audio,addr) [$a set caddr_]
	set session_(length_) [expr $session_(endtime_) - $session_(starttime_)]

	set session_(duration_) [duration_readable $session_(length_)]
	$rtsp_(application) update-title $session_(session_info1_) $session_(session_info2_) $session_(duration_)
#	puts "Info: $session_(session_info_)"
#	puts "Name: $session_(session_name_)"
#	$rtsp_(application) update-title $session_(session_name_) $session_(session_info_) $session_(duration_)
	return 1
}
