# serial.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/serial/serial.tcl,v 1.13 2002/02/03 04:29:47 lim Exp $


Class SerialChannel

# superclass for things that talk over a serial port --
# replaces the in-the-core implementation (misc/serial.cc)
# (with this tcl-only version) for portability
#
# understands the option 'serialdevicename'
#
SerialChannel public init { args } {
    $self instvar reply_ device_ baud_ parity_ stopBits_ byteBits_ fd_

    if [catch {$self get_option serialdevicename} sdn] { set sdn "" }
    if {$sdn == ""} { set sdn "/dev/cuac00" }

    set device_ $sdn
    set baud_ 9600
    set parity_ n
    set stopBits_ 1
    set byteBits_ 8
    set reply_ ""
    set fd_ ""

    $self open
}

# set the device; changes not "commited" until a new open() is called
SerialChannel public device {args} {
    $self instvar device_
    if {$args!={}} {
	set device_ $args
    }
    return $device_
}

# set the baud; changes not "commited" until a new open() is called
SerialChannel public baud {args} {
    $self instvar baud_
    if {$args!={}} {
	set baud_ $args
    }
    return $baud_
}

# set the parity; changes not "commited" until a new open() is called
SerialChannel public parity {args} {
    $self instvar parity_
    if {$args!={}} {
	switch $args {
	    none { set newP n }
	    even { set newP e }
	    odd { set newP o }
	    default {
		puts "Bad parity: $args"
		set newP $parity_
	    }
	}
	set parity_ $newP
    }
    return $parity_
}

# set the stop bits; changes not "commited" until a new open() is called
SerialChannel public stopBits {args} {
    $self instvar stopBits_
    if {$args!={}} {
	if {($stopBits_ < 0) || ($stopBits_ > 2)} {
	    puts "Bad stop bits: $args"
	    return
	}
	set stopBits_ $args
    }
    return $stopBits_
}

# set the data (byte) bits; changes not "commited" until open() is re-called
SerialChannel public byteBits {args} {
    $self instvar byteBits_
    if {$args!={}} {
	if {($byteBits_ < 5) || ($byteBits_ > 8)} {
	    puts "Bad data (byte) bits: $args"
	    return
	}
	set byteBits_ $args
    }
    return $byteBits_
}

# open serial device with current parameters.
# (also closes the previously opened dev)
SerialChannel public open {} {
    $self instvar fd_ device_ baud_ parity_ byteBits_ stopBits_
    if {$device_ == ""} { return }
    if {$fd_ != ""} {
	$self close
    }
    set fd_ [open $device_ r+]
    #puts "fconfigure $fd_ -mode $baud_,$parity_,$byteBits_,$stopBits_"
    fconfigure $fd_ -mode "$baud_,$parity_,$byteBits_,$stopBits_"
    fconfigure $fd_ -translation binary
    fconfigure $fd_ -blocking false
}

# close the current serial device
SerialChannel public close {} {
    $self instvar fd_
    if {$fd_ != ""} { close $fd_ }
    set fd_ ""
}

# Send message, encoded in ASCII form for use from tcl.
# Format is space-separated bytes, where '0xZZ'
# will be converted to the equivalent binary data and other data
# will passed as is.  "  " (three spaces) is a single space character,
# 0x00 is NULL, and "0 x Z Z" will become the literals "0xZZ", so no
# explicit escape character is needed. e.g.
#  "M A S H   0 x A f 0x0A 0x0d" --> "MASH 0xAf\n\r"
#
SerialChannel public send {msg} {
    $self instvar fd_

    set msgString ""
    set formatString ""
    set len [string length $msg]
    set i 0
    while {$i < $len} {
	set char [string index $msg $i]

	if {$char != 0} {
	    if {$char == " "} {
		# literal space
		append formatString "H2"
		append msgString "20 "
		incr i 2
	    } else {
		# ascii
		append formatString "A"
		append msgString "$char "
		incr i 2
	    }
	} else {
	    if {[string index $msg [expr $i+1]] == " "} {
		# literal '0', not hex beginning
		append formatString "A"
		append msgString "$char "
		incr i 2
	    } else {
		# "0xHH"
		if {[string index $msg [expr $i+1]] != "x"} {
		    puts "Improperly formatted msg!"
		    return
		}
		append formatString "H2"
		append msgString "[string range $msg [expr $i+2] [expr $i+3]] "
		incr i 5
	    }
	}
	#puts "end - $i: $formatString ; $msgString"
    }
    set binaryMsg [eval binary format $formatString $msgString]
    puts -nonewline $fd_ $binaryMsg
    flush $fd_

    ### FIXME hmmm..... how much should I delay here??!?!?
    after 1000 $self receive
}

# get data from the device
SerialChannel private receive {} {
    $self instvar reply_ fd_

    set tmp3 "initially not empty"
    while {$tmp3 != ""} {
	set tmp [read $fd_]
	binary scan $tmp "c*" tmp2
	set tmp3 ""
	foreach i $tmp2 {
	    append tmp3 [expr ( $i + 0x100 ) % 0x100] " "
	}

	#if {$reply_ != "" } {
	#    puts "replacing previous reply: '[to_ascii [$self set reply_]]'"
	#}
	#puts "receive(): reply in ascii = '[to_ascii $tmp3]'"
	append reply_ $tmp3
    }
    return $reply_
}

# get the current "reply" data from the device, which might be cached
SerialChannel public getReply {} {
    $self instvar reply_

    # if empty, try to see if a new reply is ready by un-read
    ##if {$reply_ == ""} { $self receive }
    $self receive
    return $reply_
}

# clear the cached "reply" data
SerialChannel public clearReply {} {
    $self instvar reply_
    set reply_ ""
}



#
# get reply, converted to ascii.  Do not use with binary data:
# will fail at the first 0x00.
# (can be used for convenience in cases where the serial device
# speaks an ascii protocol)
#
SerialChannel public getReply_ascii {} {

    set newReply ""
    set reply [$self getReply]

    set len [llength $reply]
    for {set i 0} {$i < $len} {incr i} {
	append newReply [format "%c" [lindex $reply $i]]
    }
    return $newReply
}


#
# get reply, converted to decimal
#
SerialChannel public getReply_decimal {} {

    # no conversion necessary ...
    set reply [$self getReply]
    return $reply
}

#
# convert reply to hex
# (1/26/00 -- added suffixed space, need to check if this breaks anything)
#
SerialChannel public getReply_hex {} {

    set newReply ""
    set reply [$self getReply]

    set len [llength $reply]
    for {set i 0} {$i < $len} {incr i} {
	append newReply [format "0x%x " [lindex $reply $i]]
    }
    return $newReply
}


##
proc to_ascii {decMsg} {
    set newReply ""
    set reply $decMsg
    set len [llength $reply]
    for {set i 0} {$i < $len} {incr i} {
	append newReply [format "%c" [lindex $reply $i]]
    }
    return $newReply
}

