# ui-audio.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $header: /usr/src/mash/repository/mash/mash-1/tcl/vat/ui-audio.tcl,v 1.13 1997/12/23 05:31:51 mccanne Exp $


import Observable Configuration CoordinationBus

#
# The AudioArbiter is used to negotiate with other processes for control
# over the host's audio device.  Multiple applications might want to
# simultaneously access the underlying audio hardware, but many such
# devices cannot be shared because of the operating system's contraints
# on the API.  Thus, we have explicit methods for obtaining and
# releasing the underlying hardware resources.
# <p>
#
# The AudioArbiter is also an Observable so an Observer, e.g. a UI, can watch
# its status (i.e. Do we have the device? Did we just get it so we
# should raise the window?, etc.)
#
Class AudioArbiter -superclass {CoordinationBus Observable} -configuration {
	idleDropTime 20
	defaultPriority 100
}

#
# Create an arbiter to represent the given AudioAgent, <i>agent</i>.
# Register CoordinationBus messages to suit the following device bus API: <br>
# <dd> $cb send audio-demand $pid
# <dd> $cb send audio-request $pid $pri
# <dd> $cb send audio-release $pid
#
AudioArbiter public init agent {
	# the default coordbus channel is 0
	# so, we get the base channel for the global stuff
	$self next
	$self instvar agent_ activity_ id_ priority_ hold_
	set agent_ $agent
	set hold_ 0

	$self register audio-demand someone_demands
	$self register audio-request someone_requests
	$self register audio-release someone_released

	set activity_ 0
	set priority_ [$self get_option defaultPriority]
	set id_ [after 5000 "$self timeout"]
}

#
# Cancel the "timeout".
#
AudioArbiter instproc destroy {} {
	$self instvar id_
	after cancel $id_
	$self next
}

#
# Set the priority, <i>p</i>, that this arbiter has over the control of the audio device.
#
AudioArbiter instproc set-pri p {
	$self instvar priority_
	set priority_ $p
}

#
# Close the audio device.
#
AudioArbiter public release {} {
	$self instvar agent_
	$agent_ release
	$self indicator_update
}

#
# Callback when someone requests the audio.  If we have the audio, it's
# not pinned, and it's unmuted, we must relinquish it to anyone with higher
# priority.  #
#
AudioArbiter public someone_requests { pid pri } {
	$self instvar agent_ priority_ hold_
	global unmuted outputMutebutton
	if { [$agent_ have_audio] && !$hold_ && ($pri > $priority_) } {

#FIXME FIX unmuted ref
#		|| !$unmuted($outputMutebutton))} {
		#
		# we have the audio, it's not pinned, and someone
		# with a high priority wants it or we have it muted.
		# give it up to them.
		#
		$self give_it_up $pid
	}
}

#
# Close the audio device, and advertise that it's
# been released to the conference bus.
#
AudioArbiter private give_it_up pid {
	$self release
	$self send audio-release $pid
}

#
# Callback when someone demands the audio
# If we have it, honor request.
#
AudioArbiter public someone_demands pid {
	$self instvar agent_
	if { [$agent_ have_audio] } {
		#
		# we have the audio and someone demanded it.
		# give it up.
		#
		$self give_it_up $pid
	}
}

#
# Callback when someone releases the audio to process <i>pid</i>.  If
# the audio was released to us, i.e. if the pid matches, try to grab it.
#
AudioArbiter public someone_released pid {
	if { $pid == [pid] } {
		#
		# someone released the audio to us.
		# try to grab it -- if we fail, somebody else
		# got it and give up (shouldn't happen because
		# pids are unique).
		#
		$self grab
		$self notify_observers arbiter_snatch
	}
}

#
# Update the title bar indicator to show whether or
# not we have the device open.
#
AudioArbiter instproc indicator_update { } {
	$self instvar agent_ activity_ hold_
	if [$agent_ have_audio] {
		$self notify_observers arbiter_have 1
		#
		# when we're not running off the audio clock, our time base
		# can shift pretty badly (especially on PC's where gettimeofday
		# accuracy is low).  so we reset all our clock offsets whenever
		# we recover the audio device.
		#
		$agent_ reset_source_offsets
	} else {
		$self notify_observers arbiter_have 0
		set hold_ 0
	}
	set activity_ [$agent_ unix_time]
}

#
# Attempt to open and intialize the underlying audio hardware.  After
# <i>grab</i> is called, the AudioAgent <i>haveaudio</i> method can be
# queried to see if the device was successfully opened.  Once obtained,
# the device can be released with the AudioAgent <i>release</i> method.
#
AudioArbiter public grab {} {
	$self instvar agent_
	$agent_ obtain
	$self indicator_update
}

#
# Ask for the audio, but don't demand it, because it
# might be in use for more important things.  First,
# just try to open the device directly.  If that fails,
# ask the vat that has it to give it up via the conference bus.
#
AudioArbiter public request {} {
	# first just try to see if we can get it
	$self instvar agent_ priority_
	$agent_ obtain
	if [$agent_ have_audio] {
		$self indicator_update
	} else {
		$self send audio-request [pid] $priority_
	}
}

#
# Demand the audio.  Whoever has it will hear us, close the
# device, and eventually notify us back over the conference bus.
#
AudioArbiter public demand {} {
	$self instvar agent_
	$agent_ obtain
	if [$agent_ have_audio] {
		$self indicator_update
	} else {
		$self send audio-demand [pid]
	}
}

#
# Demand the audio device if <i>v</i> is true and the AudioAgent does not have_audio.
#
AudioArbiter public hold v {
	$self instvar hold_ agent_
	set hold_ $v
	if { $hold_ && ![$agent_ have_audio] } {
		$self demand
	}
}

#
# Periodically check the status of the AudioAgent, releasing the audio
# device if the AudioAgent has been inactive for given amount of time.
#
AudioArbiter private timeout {} {
	$self instvar activity_ agent_ id_ hold_

	if { [$agent_ have_audio] && !$hold_ } {
		if [$agent_ is_active] {
			$agent_ clear_active
			set activity_ [$agent_ unix_time]
		} else {
			set r [$self get_option idleDropTime]
			if { $r && [$agent_ unix_time] - $activity_ > \
			    $r } {
				$self give_it_up 0
			}
		}
	}
	set id_ [after 5000 "$self timeout"]
}
