# desc.tcl --
#
#       Takes in Broadcast Description file and evaluates rules and executes
#       commands as needed.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import 405Client DcClient
import BroadcastDescription_Recorder

# use the Tcl namespace facility to prevent conflicts
namespace eval BroadcastDescription {
    variable self
    variable enabled
    variable goButton
    variable goWidget
    variable sourceButton
    variable sourceWidget
    variable broadcastClock
    variable 405Control
    variable DcControl
    variable RecorderControl
    variable VdControl
    variable debugging
    variable passed
    variable errorMsg
}


Class BroadcastDescription

#
# see parseSpec function to see what should be in spec
#
BroadcastDescription instproc init {base spec} {
    $self instvar baseFrame_ specFrame_ vdHandle_ descriptionFile_
    $self instvar dcHost_ dcPort_
    $self instvar recorderHost_ recorderPort_

    $self parseSpec $spec

    # change this to 0 to turn off debugging stuff
    set BroadcastDescription::debugging 0

    # so that the callback can refer to this object
    set BroadcastDescription::self $self
    # init global variables
    set BroadcastDescription::enabled 0
    set BroadcastDescription::broadcastClock -1
    
    $self initUI $base
    $self initRules
    # set up handles for descriptions to use
    set BroadcastDescription::RecorderControl [new BroadcastDescription_Recorder $recorderHost_ $recorderPort_]
    set BroadcastDescription::405Control [new 405Client]
    if {[catch {set BroadcastDescription::DcControl [new DcClient $dcHost_ $dcPort_]}]} {
	puts stderr "Cannot connect to DC.  Exiting."
	exit
    }
    set BroadcastDescription::VdControl [new BroadcastDescription_VD $vdHandle_]
}

BroadcastDescription instproc destroy {} {
    if {[catch {BroadcastSpecification::cleanUp}]} {
    }
}

BroadcastDescription instproc parseSpec {spec} {
    $self instvar descriptionFile_
    $self instvar dcHost_ dcPort_
    $self instvar recorderHost_ recorderPort_
    $self instvar vdHandle_

    set descriptionFile_ [lindex $spec 0]
    set dcHost_ [lindex $spec 1]
    set dcPort_ [lindex $spec 2]
    set recorderHost_ [lindex $spec 3]
    set recorderPort_ [lindex $spec 4]
    set vdHandle_ [lindex $spec 5]
}

BroadcastDescription instproc initUI {base} {
    $self instvar baseFrame_ specFrame_

    set baseFrame_ $base
    
    set curFrame $base.enabled
    frame $curFrame
    pack $curFrame -side top -fill x
    checkbutton $curFrame.enabledButton -text "Enabled" -variable BroadcastDescription::enabled
    pack $curFrame.enabledButton -side left
    
    set specFrame_ "$base.specFrame"
    frame $specFrame_ -borderwidth 3 -relief sunken
    pack $specFrame_ -side top -fill x -padx 3 -pady 2 -expand true

    # new base is the controls frame underneath
    set base $base.controls
    frame $base
    pack $base -side top -fill x

    set curFrame $base.sourceGo
    frame $curFrame
    pack $curFrame -side top -fill x

    # this command starts the broadcast description, evaling rules and
    #   incrementing the broadcast clock
    set BroadcastDescription::goButton [button $curFrame.go -text "Go" -command "$self go" -width 8]
    set BroadcastDescription::goWidget $curFrame.go
    pack $curFrame.go -side left -fill x -padx 3

    if {$BroadcastDescription::debugging} {
	button $base.printRules -text "print rules" -command "$self printRules"
	pack $base.printRules -side top
	button $base.evalRules -text "eval rules" -command "$self evalRules 1"
	pack $base.evalRules -side top
	button $base.incrTime -text "incr time" -command "$self incrTime"
	pack $base.incrTime -side top
    }
}

BroadcastDescription instproc sourceFile {} {
    $self instvar descriptionFile_ specFrame_

    if {$descriptionFile_ != ""} {
	if {[catch {open $descriptionFile_ "r"} result]} {
	    # the file doesn't exist
	    set msg "Error: couldn't open description file: $descriptionFile_"
	    tk_messageBox -message $msg -type ok -title "BroadcastDescription: Error" -icon "error"
	    return -1
	} else {
	    close $result
	}

	set cmd "if \{\[catch \{source $descriptionFile_\} BroadcastDescription::errorMsg\]\} \{\n"
	set cmd "$cmd set ::BroadcastDescription::passed 0\n"
	set cmd "$cmd \} else \{\n"
	set cmd "$cmd set ::BroadcastDescription::passed 1\n"
	set cmd "$cmd \}"
	# have to use "after" so sourced at the global level
	after 1 $cmd
	# make sure the source command is complete before proceeding
	set x 0
	after 1 "set x 1"
	vwait x

	if {$BroadcastDescription::passed} {
	    if {[catch {BroadcastSpecification::broadcastSpecification $specFrame_} result]} {
		set msg "Error: BroadcastSpecification::broadcastSpecification not defined in $descriptionFile_ or some other error"
		tk_messageBox -message $msg -type ok -title "BroadcastDescription: Error" -icon "error"
		puts stdout "error is: $result"
		return -1
	    }
	} else {
	    set msg "Error: $descriptionFile_ failed: $BroadcastDescription::errorMsg"
	    tk_messageBox -message $msg -type ok -title "BroadcastDescription: Error" -icon "error"
	    return -1
	}

    } else {
	set msg "Error: no description file specified"
	tk_messageBox -message $msg -type ok -title "BroadcastDescription: Error" -icon "error"
	return -1
    }
    return 1
}

BroadcastDescription instproc go {} {
    $self instvar specFrame_ baseFrame_

    set result [$self sourceFile]

    if {$result == -1} {
	# there was some error, do nothing
	return ""
    }

    $self enable
        
    #$BroadcastDescription::goButton configure -state disabled
    pack unpack $BroadcastDescription::goWidget

    if {!$BroadcastDescription::debugging} {
	after 1000 "$self timeout"
    }
}

BroadcastDescription public enable {} {
    set BroadcastDescription::enabled 1
}

BroadcastDescription public disable {} {
    set BroadcastDescription::enabled 0
}

BroadcastDescription instproc initRules {} {
    $self instvar rules

    set rules(numRules) 0
    set rules(nextIndex) 0
    set rules(indices) [list]
}

BroadcastDescription instproc printRules {} {
    $self instvar rules

    puts stdout "printing rules, indices is $rules(indices)"
    foreach index $rules(indices) {
	puts stdout "rule: $index"
	puts stdout "\tcheck: $rules($index,check)"
	puts stdout "\tcmd: $rules($index,cmd)"
	puts stdout "\ttransitionOnly: $rules($index,transitionOnly)"
	puts stdout "\tenabled: $rules($index,enabled)"
	puts stdout "\tlastState: $rules($index,lastState)"
    }
}

BroadcastDescription instproc evalRules {{verbose 0}} {
    $self instvar rules

    if {$verbose} {
	puts stdout "evaling rules, indices is $rules(indices)"
    }
    foreach index $rules(indices) {
	if {$verbose} {
	    puts stdout "rule: $index"
	    puts stdout "\tcheck: $rules($index,check)"
	    puts stdout "\tcmd: $rules($index,cmd)"
	    puts stdout "\ttransitionOnly: $rules($index,transitionOnly)"
	    puts stdout "\tenabled: $rules($index,enabled)"
	    puts stdout "\tlastState: $rules($index,lastState)"
	}
	if {[expr $rules($index,check)]} {
	    if {$BroadcastDescription::enabled} {
		set doAction $rules($index,enabled)
		if {$rules($index,transitionOnly)} {
		    # check if it was false previously
		    if {$rules($index,lastState) != "false"} {
			# it was true last time, no transition
			set doAction 0
		    }
		}
		if {$doAction} {
		    if {[catch {eval $rules($index,cmd)} result]} {
			puts stderr "Warning: cmd: \"$rules($index,cmd)\" failed: $result"
		    }
		}
	    }
	    set rules($index,lastState) "true"
	} else {
	    set rules($index,lastState) "false"
	}
	if {$verbose} {
	    puts stdout "\tresult: $rules($index,lastState)"
	}
    }
}

BroadcastDescription instproc registerWhen {check cmd transitionOnly enabled} {
    $self instvar rules

    if {$BroadcastDescription::debugging} {
	puts stdout "registering rule: \"$check\" \"$cmd\""
    }

    set rules(numRules) [expr $rules(numRules) + 1]
    set index $rules(nextIndex)
    set rules(nextIndex) [expr $index + 1]
    lappend rules(indices) $index
    set rules($index,check) $check
    set rules($index,cmd) $cmd
    set rules($index,lastState) false
    set rules($index,transitionOnly) $transitionOnly
    set rules($index,enabled) $enabled

    return $index
}

BroadcastDescription instproc incrTime {} {
    set BroadcastDescription::broadcastClock [expr $BroadcastDescription::broadcastClock + 1]

    if {$BroadcastDescription::debugging} {
	puts stdout "clock = $BroadcastDescription::broadcastClock"
    }
}

BroadcastDescription instproc timeout {} {
    # do the after statement first so we won't slip off timing by the amount
    #   of time it takes to process everything (assume we'll be done before
    #   the next callback comes back)
    after 1000 "$self timeout"

    set BroadcastDescription::broadcastClock [expr $BroadcastDescription::broadcastClock + 1]

    if {$BroadcastDescription::debugging} {
	puts stdout "clock = $BroadcastDescription::broadcastClock"
    }

    $self evalRules
}

# return 1 if the enable/disable was successful
# return 0 otherwise
BroadcastDescription instproc enableRule {index newState} {
    $self instvar rules

    if {[info exists rules($index,enabled)]} {
	set rules($index,enabled) $newState
	return 1
    }
    return 0
}

# returns the id of this rule, which is just the index
#
# the rule is disabled when it is created
#
proc BroadcastDescription_when {check cmd {transitionOnly 1}} {
    return [$BroadcastDescription::self registerWhen $check $cmd $transitionOnly 0]
}

# newState is 0 to disable
# newState is 1 to enable
proc BroadcastDescription_enableRule {index newState} {
    return [$BroadcastDescription::self enableRule $index $newState]
}

# FIXME - there should be a way to unregister a rule so it isn't evaluated
#    every time; might not be worth it to do since the real implementation
#    should be like nsync
