/*
 * controller-erik.cc --
 *
 *      Modification of the Controller class which allows a stereo audio
 *      to be split into left and right and passed to separate transducers.
 *
 * Copyright (c) 1991-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
"@(#) $Header: /usr/mash/src/repository/mash/mash-1/vd/audio/controller-erik.cc,v 1.13 2002/02/03 04:19:42 lim Exp $";

#define AUDIO_SPS 8000	/* audio samples per second (used to convert
			 * playout delay from seconds to bytes of buffer).
			 * This is the sample rate used by the audio
			 * hardware.  If this rate is different than
			 * the sample rate of some network audio format,
			 * sample rate conversion has to be done between
			 * the encoder/decoder objects & here.  Note
			 * that sample rate conversion is a very compute
			 * intensive operation & there isn't currently
			 * any support for it in vat.  Also note that
			 * there are many control variables expressed
			 * in terms of a 160 sample (20ms at 8KHz)
			 * audio frame size and some of these need to
			 * be changed if the AUDIO_SPS is changed.
			 */
#define SS_GRANULARITY 1440	/* sample stream sizes are always rounded
				 * to some integer multiple of this number.
				 * It should be set to the least common
				 * multiple of the possible *output* audio
				 * frame sizes (see comments below).
				 * In our case, possible frame sizes are
				 * 160 & 180 samples.
				 */
/*
 * following is timeout interval (in ms) when vat does not have
 * the audio & is running off a system timer instead.  This interval
 * must be the same as the audio read blocksize (AUDIO_FRAMESIZE /
 * AUDIO_SPS * 1000 ms) and must be < 1 sec.
 */
#define FRAME_TIME (AUDIO_FRAMESIZE * 1000 / AUDIO_SPS)

#include "config.h"
#include "sys-time.h"
#include "audio.h"
#include "ss.h"
#include "controller-erik.h"
#include "encoder.h"
#include "mulaw.h"
#include "tclcl.h"
#include "ntp-time.h"
#include "transducer.h"

#define METER_UPDATE_FREQ 3

// <otcl> Class AudioControllerErik

/*
 * <otcl> Class AudioControllerErik/FullDuplex -superclass AudioController
 * AudioController/FullDuplex specifically handles full duplex
 * devices.
 */
static class ControllerErikClass : public TclClass {
public:
  ControllerErikClass() : TclClass("AudioControllerErik/FullDuplex") {}
  TclObject* create(int, const char*const*) {
    return (new ControllerErik);
  }
} controller_erik_class;

ControllerErik::ControllerErik() :
  audio_(0),
#ifdef WIN32
  /* windows95 has trouble with 20ms events */
  timer_interval_(FRAME_TIME * 4),
#else
  timer_interval_(FRAME_TIME),
#endif
  tsec_(0),
  tusec_(0),
  ostate_(TALK_TAIL + TALK_LEAD),
  test_tone_(0),
  left_meter_(0),
  right_meter_(0),
  talk_thresh_(0),
  meter_update_(METER_UPDATE_FREQ),
  active_(0),
  blksize_(0),
  left_blk(0),
  right_blk(0),
  leftStream_(0),
  rightStream_(0)
{

//  printf("ControllerErik::ControllerErik: starting\n");

  bind("max_playout_", (int*)&max_playout_);

//  printf("ControllerErik::ControllerErik: bound tcl variables\n");

//  printf("ControllerErik::ControllerErik: finishing\n");
}

// FIXME - not sure this is really needed; It never seems to be called
void ControllerErik::update(Observable*)
{

//  printf("ControllerErik::update: called\n");

  if (!audio_->haveaudio()) {
    timeval tv;
    ::gettimeofday(&tv, 0);
    tsec_ = tv.tv_sec;
    tusec_ = tv.tv_usec;
    /* Reset the meters and force a redraw. */
    if (left_meter_ != 0) {
      left_meter_->set(0.);
    }
    if (right_meter_ != 0) {
      right_meter_->set(0.);
    }
    msched(timer_interval_);
  } else {
    cancel();
  }
}

void ControllerErik::DoAudio()
{
  double left_level, right_level;

  //printf("ControllerErik::DoAudio: called\n");

  mixaudio();

  if (audio_->RMuted()) {
    left_level = 0.0;
    right_level = 0.0;
  } else {
    leftStream_->Compute();
    rightStream_->Compute();
    left_level = leftStream_->Mean();
    right_level = rightStream_->Mean();
  }
  /*
   * Update the meters.  We control the rate with METER_UPDATE_FREQ
   * to cut down on CPU load from the X window updates.
   */
  if(--meter_update_ <= 0)
  {
    if(left_meter_ != 0)
    {
      left_meter_->set(left_level);
    }
    if(right_meter_ != 0)
    {
      right_meter_->set(right_level);
    }
    meter_update_ = METER_UPDATE_FREQ;
  }
}

void ControllerErik::DoTimer()
{
  /* Advance to next audio frame */
  leftStream_->Advance();
  rightStream_->Advance();
}

void ControllerErik::audio_handle()
{
  DoAudio();
  DoTimer();
}

/*
 * Called when we don't have the audio device.  Normally, our time
 * base comes from the audio device's sample clock, so when we don't
 * have the device open, we revert to timers.
 */
void ControllerErik::timeout()
{
  /*
   * Use get time of day and keep track of the current hard time
   * in the tsec_/tusec_ variables.  Tk timers are an unreliable
   * time base.  We use them to dispatch an event here then call
   * gettimeofday to see how many times we really should have
   * been called in the intervening period.
	 */
  timeval tv;
  ::gettimeofday(&tv, 0);
  u_int u = (u_int)tv.tv_usec;
  u_int s = (u_int)tv.tv_sec;
  if (s > tsec_ + 3) {
    /*
     * We're way behind.  Most likely we were suspended and
     * then resumed.  Instead of trying to catch up, just resync.
     */
    tusec_ = u;
    tsec_ = s;
  }
  while ((int(tusec_ - u) <= 0 && s == tsec_) || int(tsec_ - s) < 0) {
    DoTimer();
    tusec_ += 1000 * FRAME_TIME;
    while (tusec_ >= 1000000) {
      tusec_ -= 1000000;
      ++tsec_;
    }
  }
  msched(timer_interval_);
}

void ControllerErik::mixaudio()
{
  u_char* blk = 0;
  unsigned int x;
  unsigned int index = 0;
  unsigned int y;

  for(y = 0; y < 2; ++y)
  {
    // we do 2 reads because each read returns 160 samples, 80 left, 80 right,
    //   but the SampleStreams expect 160 samples
    blk = audio_->Read();
    // this assumes that blksize_ MOD 2 is 0 (it should be)
    for(x = 0; x < blksize_; x++)
    {
      left_blk[index] = blk[x++];
      right_blk[index] = blk[x];
/*
      printf("left:\t%d\t\t", left_blk[index]);
      printf("right:\t%d\n", right_blk[index]);
*/
/*
      // full off = 127
      left_blk[index] = 127;
      right_blk[index] = 127;
      x++;
*/
      index++;
    }
  }

  if(!(index == 160))
  {
    fprintf(stderr, "!!!!!Warning - index != 160\n");
  }

  leftStream_->Mix(0, left_blk, blksize_);
  rightStream_->Mix(0, right_blk, blksize_);
}

int ControllerErik::command(int argc, const char*const* argv)
{
  Tcl& tcl = Tcl::instance();
  if (argc == 2) {
    /*
     * <otcl> AudioControllerErik public ntp_time {}
     * Return the current system time as the middle 32-bit
     * of a standard 64-bit NTP timestamp.
     */
    if (strcmp(argv[1], "ntp_time") == 0) {
      sprintf(tcl.buffer(), "%u", ntptime());
      tcl.result(tcl.buffer());
      return (TCL_OK);
    }
    /*
     * <otcl> AudioControllerErik public unix_time {}
     * Return the number of seconds since Jan 1, 1970 UTC.
     */
    if (strcmp(argv[1], "unix_time") == 0) {
      sprintf(tcl.buffer(), "%ld", unixtime().tv_sec);
      tcl.result(tcl.buffer());
      return (TCL_OK);
    }
    /*
     * <otcl> AudioControllerErik public media-time {}
     * Return the current time according to the audio
     * sample clock of the underlying device.
     * This value is a 32-bit number that wraps
     * and is not guaranteed to have a well-defined
     * start value.
     */

  if (strcmp(argv[1], "media-time") == 0) {
      sprintf(tcl.buffer(), "%u", leftStream_->Clock());
      tcl.result(tcl.buffer());
      return (TCL_OK);
    }

    /*
     * <otcl> AudioControllerErik public active v
     * Return or set the active status depending
     * on whether the <i>v</i> argument is present.
     * The active status is automatically set to 1
     * whenever the AudioControllerErik sends audio
     * packets to the network.  It is never automatically
     * cleared.  Instead, the calling tcl script is
     * responsible for clearing it if it wants to
     * make use of this hook.  If $v$ is not present,
     * the current status is returned.
     */
    if (strcmp(argv[1], "active") == 0) {
      tcl.result(active_ ? "1" : "0");
      return (TCL_OK);
    }

    /* FIXME agc doesn't really work yet. */
    if (strcmp(argv[1], "agc-input") == 0) {
      sprintf(tcl.buffer(), "%d", leftStream_->AGCLevel() / 10 - 10);
      tcl.result(tcl.buffer());
      return (TCL_OK);
    }
  } else if (argc == 3) {
    /*
		 * <otcl> AudioControllerErik public audio o
		 * Install the Audio object named by the <i>o</i>
		 * argument in this controller.  Only one audio
		 * object can be attached at any given time.
		 */
    if (strcmp(argv[1], "audio") == 0) {
      audio_ = (Audio*)TclObject::lookup(argv[2]);
      if(audio_ == NULL)
      {
	fprintf(stderr, "ControllerErik::command: no audio object found\n");
	abort();
      }
      audio_->attach(this);
      audio_->handler(this);
      blksize_ = audio_->GetBlockSize();
      left_blk = new u_char[blksize_];
      right_blk = new u_char[blksize_];
      leftStream_ = new SampleStream(blksize_, max_playout_,
			 (TALK_LEAD+1)*blksize_, 0);
      rightStream_ = new SampleStream(blksize_, max_playout_,
			 (TALK_LEAD+1)*blksize_, 0);
      leftStream_->ssthresh(20);
      rightStream_->ssthresh(20);
      update(audio_);
      return (TCL_OK);
    }


    /*
		 * <otcl> AudioControllerErik public test_tone type
		 * Arrange for an audio test-tone to be generated
		 * to the audio output. <i>type</i> indicates
		 * which test-tone to use and may be one of:
		 * <ul>
		 * <li> none - no test tone
		 * <li> low - 6dBm sine wave
		 * <li> med - 0dBm sine wave
		 * <li> max - full scale sine wave
		 * </ul>
		 */
    if (strcmp(argv[1], "test_tone") == 0) {
      if (strcmp(argv[2], "low") == 0)
	test_tone_ = 1;
      else if (strcmp(argv[2], "med") == 0)
	test_tone_ = 2;
      else if (strcmp(argv[2], "max") == 0)
	test_tone_ = 3;
      else
	test_tone_ = 0;
      return (TCL_OK);
    }
    /*
		 * <otcl> AudioControllerErik public output-meter meter
		 * Install the Transducer object named by the <i>meter</i>
		 * argument as the output level monitoring device
		 * in this controller.  Only one output-meter
		 * can be attached at any given time.
		 * Before each audio frame is written to the encoder object,
		 * it's average power level is computed and this value
		 * is conveyed to the output-meter by calling
		 * the Transducer's set method.  A Transducer
		 * is an object that tranforms some series of
		 * real-valued numbers into some UI element.
		 */

    if (strcmp(argv[1], "left-meter") == 0) {
      left_meter_ = (Transducer*)TclObject::lookup(argv[2]);
      return (TCL_OK);
    }

    if (strcmp(argv[1], "right-meter") == 0) {
      right_meter_ = (Transducer*)TclObject::lookup(argv[2]);
      return (TCL_OK);
    }




    /*
		 * <otcl> AudioControllerErik public silence-thresh level
		 * Set the threshold for the silence suppression algorithm
		 * used by the AudioController to <i>level</i>.
		 * FIXME: need units of level.  Should be probably be dB.
		 */
    if (strcmp(argv[1], "silence-thresh") == 0) {
      int thresh = atoi(argv[2]);
      leftStream_->ssthresh(thresh);
      rightStream_->ssthresh(thresh);
      return (TCL_OK);
    }
    /*
		 * <otcl> AudioControllerErik public talk-thresh level
		 * Set the amount of additional thresholding that
		 * the ControllerErik applies to the silence suppression
		 * when the speaker is not talking.  This means
		 * there's a slight cliff to get over to start
		 * talking, but once the mike is active, it's
		 * harder for it to turn off.  This tends
		 * to do a good job at eliminating noise spikes
		 * (like typing) and artifical drop-outs in mid-sentence.
		 * FIXME: need units of level.  Should be probably be dB.
		 */
    if (strcmp(argv[1], "talk-thresh") == 0) {
      talk_thresh_ = atoi(argv[2]);
      return (TCL_OK);
    }


		/* FIXME agc doesn't work */
		if (strcmp(argv[1], "agc-input") == 0) {
			int level = atoi(argv[2]);
			level = 10 * (level + 10);
			if(leftStream_)
			{
			  leftStream_->SetAGCLevel(level);
			  rightStream_->SetAGCLevel(level);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "agc-input-enable") == 0) {
			leftStream_->DoAGC(atoi(argv[2]));
			rightStream_->DoAGC(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "active") == 0) {
			active_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}
