/*
 * Copyright (c) 1997 Carnegie Mellon University. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is hereby granted (including for commercial or
 * for-profit use), provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative
 * works, or modified versions, and any portions thereof, and that
 * both notices appear in supporting documentation, and that credit
 * is given to Carnegie Mellon University in all publications reporting
 * on direct or indirect use of this code or its derivatives.
 *
 * THIS SOFTWARE IS EXPERIMENTAL AND IS KNOWN TO HAVE BUGS, SOME OF
 * WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON PROVIDES THIS
 * SOFTWARE IN ITS ``AS IS'' CONDITION, AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Carnegie Mellon encourages (but does not require) users of this
 * software to return any improvements or extensions that they make,
 * and to grant Carnegie Mellon the rights to redistribute these
 * changes without encumbrance.
 */


#ifdef BSD_KERNEL
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#else
#include <stdlib.h>
#include <stdio.h>  /* yc-a */
#include <string.h>
#include "hfsc_list.h"
#endif

#include "hfsc_err.h"
#include "hfsc_def.h"
#include "hfsc_decl.h"
#include "hfsc_fun.h"

char *ErrMsg[] = {
/* #define ERR_MEM_ALLOC   0 */
	"Memory Allocation Error !\n",
/* #define ERR_MEM_REALLOC 1 */
	"Memory Re-allocation Error !\n",
/* #define ERR_INCOMP_SC   2 */
	"Incompatible service curves !\n",
/* #define ERR_INCOMPATIBLE_SC 3 */
	"Incompatible service curves !\n",
/* #define ERR_INVALID_POINT   4 */
	"Wrong point coordinates !\n",
/* #define ERR_INVALID_SEG     5 */
	"Invalid segment !\n",
/* #define ERR_INVALID_POINTER 6 */
	"Invalid pointer !\n",
/* #define ERR_INVALID_DATA    7 */
	"Invalid data !\n",
/* #define ERR_FILE_OPEN       8 */
	"Error on file open !\n",
/* #define  ERR_ADMISSION_FAILURE 9 */
	"Admission control has failed !\n",
/* #define ERR_EMPTY_QUEUES  10 */
	"All queues are empty !\n",
/* #define ERR_EMPTY_QUEUE       11 */
	"Empty queue !\n",
/* #define ERR_INVALID_NODE_TYPE 12 */
	"Invalid node type !\n",
/* #define ERR_INVALID_STRING_ID 13 */
	"Invalid string identifier !\n",
/* #define ERR_INVALID_ID        14 */
	"Invalide identifier !\n",
/* #define ERR_INVALID_SEG1      15 */
	"The segment should !be infinite to the right ! \n",
/* #define ERR_NOT_SESSION       16 */
	"Invalid session !\n",
/* #define ERR_SES_STAT_UNCHANGED 17 */
	"Session status does not change !\n",
/* #define ERR_INVALID_YVAL      18 */
	"Invalid Y value !\n",
/* #define ERR_INVALID_XVAL      19 */
	"Invalid X value !\n"
};

/* extern double clock;  yc-d: no use */

/****************************************************************
 *
 *  NAME: newHFSCData
 *
 *  DESCRIPTION: 
 *    Allocate memory for HFSC data structure
 *
 *  PARAMETERS:
 *
 ************************************************************************/

HFSCStruct *newHFSCData(ClassItem *proot)
{
	HFSCStruct *pHFSC;
#define TLEN 5
	static u_int       relLogUnits[TLEN] = {0, 2, 3, 3, 4};  

#ifdef BSD_KERNEL
	if (!(pHFSC = (HFSCStruct *)malloc(sizeof(HFSCStruct), M_TEMP, M_NOWAIT)))
		hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
	bzero(pHFSC, sizeof(HFSCStruct));
#else
	if ((pHFSC = (HFSCStruct *)calloc(1, sizeof(HFSCStruct))) == NULL)
		hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif
  
	pHFSC->proot = proot;
  
	/* 
	 * allocate and initialize the data structure to maintain the 
	 * eligible requests
	 */
	pHFSC->pReqQueue = allocCalHeapStruct(TLEN, relLogUnits);
#undef TLEN  

	return pHFSC;
}
 

/****************************************************************
 *
 *  NAME: newClassItem
 *
 *  DESCRIPTION: Allocate memory for link-sharing item and initialize
 *  it
 *
 *  PARAMETERS:
 *    id   - item id
 *    type - item type
 *    psc  - service curve
 *
 *  RETURN:
 *    - item
 *
 *  Note: (yhchu)
 *    This function is used by the HFSC simulator only
 *
 ************************************************************************/

ClassItem *newClassItem(int id, unsigned int type, ServiceCurve *psc)
{
	ClassItem *plsi;
#define TLEN 4
#ifdef HFSC_CALENDAR
	static u_int       relLogUnits[TLEN] = {0, 3, 3, 3};  
#endif
  
	if ((plsi = (ClassItem *)calloc(1, sizeof(ClassItem))) == NULL)
		hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);

	plsi->id        = id;
	plsi->type      = type;
	plsi->pService  = psc;
	plsi->pDeadline = newRTSC(0, 0, 0, 0, 0);
	plsi->pVirtual  = newRTSC(0, 0, 0, 0, 0);
	plsi->pEligible = newRTSC(0, 0, 0, 0, 0);
	plsi->pTotalSC  = newGSC();

#ifdef FAST1
	plsi->idleTime = coarseClock();
	printf("with FAST1 optimization...\n");
#endif

	if ((plsi->pFairReq = (ReqItem *)calloc(1, sizeof(ReqItem))) == NULL)
		hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
	plsi->pFairReq->pClass = plsi;

	if ((plsi->pGuarantReq = (ReqItem *)calloc(1, sizeof(ReqItem))) == NULL)
		hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
	plsi->pGuarantReq->pClass = plsi;

	if (type == LEAF) {
		new_lst(plsi->dataQueue);
		init_list(plsi->dataQueue);
	} else {
		plsi->dataQueue = 0;
	}

	if (!type) 
		/* 
		 * this is an internal class; create multi-level calendar 
		 * queue to maintain the fair requests of children classes
		 */
#ifdef HFSC_CALENDAR
		plsi->pReqQueue = allocCalPriorQueue(TLEN, relLogUnits);
#else
	plsi->pReqQueue = newHeap();
#endif
	return plsi;
}
 

/************************************************************************
 *
 *  NAME: insertClassItem
 *
 *  DESCRIPTION: 
 *   Insert a new link-sharing item
 *
 *  PARAMETERS:
 *    plsRoot - root of the hierarchy 
 *    pls     - link-sharing item
 *
 *  RETURN:
 *    - plsRoot
 *
 ************************************************************************/

#ifdef STOICA_SIM
ClassItem *insertClassItem(ClassItem *plsRoot, int pid, ClassItem *pls)
{
	ClassItem *plsParent, *plsCrt;
  
	if (!plsRoot) {
		if (pls->id) {
			hfsc_panic(ErrMsg[ERR_INVALID_ID]);
		} else {
			return pls;
                }
        }

	plsParent = getClassItem(plsRoot, pid);

	/* find the proper place to insert new node */
	if (!(plsCrt = plsParent->pchildList)) {
		plsParent->pchildList = pls;
		pls->parent = plsParent;
		addSCtoGSC(plsParent->pTotalSC, pls->pService);
		if (!IsGSCLessThanSC(plsParent->pTotalSC, plsParent->pService))
			hfsc_panic(ErrMsg[ERR_ADMISSION_FAILURE]);
		return plsRoot;
	} 
  
	if (pls->id < plsCrt->id) {
		pls->next = plsParent->pchildList;
		plsParent->pchildList = pls;
		addSCtoGSC(plsParent->pTotalSC, pls->pService);
		if (!IsGSCLessThanSC(plsParent->pTotalSC, plsParent->pService))
			hfsc_panic(ErrMsg[ERR_ADMISSION_FAILURE]);
		pls->parent = plsParent;
		return plsRoot;
	}

	for (; plsCrt->next && pls->id < plsCrt->next->id; 
	     plsCrt = plsCrt->next);
      
	pls->next = plsCrt->next;
	plsCrt->next = pls;
	pls->parent  = plsParent;
	addSCtoGSC(plsParent->pTotalSC, pls->pService);
	if (!IsGSCLessThanSC(plsParent->pTotalSC, plsParent->pService))
		hfsc_panic(ErrMsg[ERR_ADMISSION_FAILURE]);

	return plsRoot;
}  
#endif /* STOICA_SIM */

ClassItem *insertClassItem(ClassItem *plsRoot, int pid, ClassItem *pls)
{
	ClassItem *plsParent, *plsCrt;
  
	if (!plsRoot) {
		if (pls->id) {
			hfsc_panic(ErrMsg[ERR_INVALID_ID]);
		} else {
			return pls;
                }
        }

	plsParent = getClassItem(plsRoot, pid);
	plsCrt = plsParent->pchildList;
	/* find the proper place to insert new node */
	if (!plsCrt) {
		plsParent->pchildList = pls;
		pls->parent = plsParent;
	} else if (pls->id < plsCrt->id) {
		pls->next = plsParent->pchildList;
		plsParent->pchildList = pls;
	} else {
		/* Find the correct position based on ID */
		for ( ; plsCrt->next && pls->id < plsCrt->next->id; plsCrt = plsCrt->next);
		pls->next = plsCrt->next;
		plsCrt->next = pls;
	}

	pls->parent  = plsParent;
	/* FIXME We should not hfsc_panic. Some better error handling. */
	addSCtoGSC(plsParent->pTotalSC, pls->pService);
	if (!IsGSCLessThanSC(plsParent->pTotalSC, plsParent->pService))
		hfsc_panic(ErrMsg[ERR_ADMISSION_FAILURE]);
	return plsRoot;
}

/************************************************************************
 *
 *  NAME: updateHierarchy
 *
 *  DESCRIPTION: 
 *   Update the virtual time and the total amount of work on 
 *   the path from leaf to roote
 *
 ************************************************************************/


void updateHierarchy(ClassItem *ps, u_int len)
{

	for (; ps->parent; ps = ps->parent) {

		/* yc-a: the following code deals with totalWork wrap-around problem
		 * totalWork is 32 bit integer, and wraps around fairly quickly.
		 * The trick is to "cut" the virtual service curve, where
		 * y=ps->totalWork, and move to y-axis (y=0), and then
		 * set totalWork to be 0
		 */
		if ((int)ps->totalWork > (int)MAX_WORK) {
			// printf("reset totalWork, totalWork=%d\n", ps->totalWork);
			cutRTSC(ps->pVirtual, mult(ps->totalWork));
			ps->totalWork = 0;
		} 

		ps->totalWork += len;


		/* yc: Because we reset totalWork before it wraps, totalWork should
		 * never be less than 0
		 */
		if ((int)ps->totalWork < (int) 0) {
			hfsc_panic("totalwork wrapped!!!\n");
		}

		ps->vt = getRTSC_x(ps->pVirtual, mult(ps->totalWork));
		changeReq(ps->parent->pReqQueue, ps->pFairReq->idx, ps->vt);
	}
}

    
/************************************************************************
 *
 *  NAME: changeSessionStatus
 *
 *  DESCRIPTION: 
 *    Change the session status (togle between active/pasive) and
 *    update all the related service curves
 *
 ************************************************************************/

void changeSessionStatus(HFSCStruct  *phData, ClassItem *ps, unsigned int bussy)
{
	ReqItem *pReq;
#ifdef FAST1 /* FAST1 -- speedup the algorithm */ 
	int crtTime;
#endif

	/* check whether ps represents indeed a session */
	if (ps->pchildList) 
		hfsc_panic(ErrMsg[ERR_NOT_SESSION]);

	if ((bussy && ps->nBussy) || (!bussy && !ps->nBussy)) {
		hfsc_panic(ErrMsg[ERR_SES_STAT_UNCHANGED]);
	}

	if (bussy) {
		/* the session has just became active */
		/* update deadline curve */
#ifdef BSD_KERNEL
		minRTSC(ps->pDeadline, ps->pService, coarseClock(phData, 0), 0);
#else
		minRTSC(ps->pDeadline, ps->pService, coarseClock(), 0); 
#endif

		/* update eligible curve */
		if (ps->pService->m1 < ps->pService->m2) {
			/* yc-m: removed ... ps->pDeadline->x + ps->pDeadline->y; */
			ps->pEligible->x = ps->pDeadline->x; 
			ps->pEligible->m2 = ps->pEligible->m1 = ps->pService->m2;
			ps->pEligible->y = ps->pEligible->d = 0;
		} else 
			memcpy((char *)ps->pEligible, 
			       (char *)ps->pDeadline, sizeof(RTServiceCurve));
	} else {
		/* the session has just became passive */
		/* update deadline curve */
		/* yc: FIXME no need to reset ? */
		cutRTSC(ps->pDeadline, mult(ps->cWork));
		ps->cWork = 0;
	}  

	for (; ps->parent; ps = ps->parent) {

#ifdef FAST1 /* FAST1 -- speedup the algorithm */ 
		crtTime = coarseClock();
#endif

		if (bussy) {
			(ps->nBussy)++;
			if (ps->nBussy > 1)
				return;
			/* the session has just become active */
			/* compute new virtual time */

			/* basically you have to set vt to 0 for all parent classes when you
			 * change the session status 
			 */

#ifdef FAST1
			if ((ps->idleTime - ps->parent->idleTime) <= 0) {
				ps->vt = 0;
				minRTSC(ps->pVirtual, ps->pService, ps->vt, 0);
				ps->pVirtual->x = 0;
			}
#endif

			pReq   = getReq(ps->parent->pReqQueue);
			if (pReq) 
				ps->vt = max(((ClassItem *)pReq->pClass)->vt, ps->vt);
      
#ifndef FAST1
			else {
				ClassItem *p1;
				for (p1 = ps->parent->pchildList; p1; p1 = p1->next) {
					p1->vt = 0;          
					minRTSC(p1->pVirtual, p1->pService, p1->vt, 0);	  
					p1->pVirtual->x = 0;
				}
			}
#endif
  
			/* enqueue request */
			ps->pFairReq->d = ps->vt;
			enqueueReq(ps->parent->pReqQueue, ps->pFairReq);
			/* update virtual curve */
			minRTSC(ps->pVirtual, ps->pService, ps->vt, 0);

		} else {
			ps->nBussy--;
			if (ps->nBussy > 0)
				return;      
#ifdef FAST1
			ps->idleTime = crtTime;
#endif
			/* the session has just become passive */
			/* recompute virtual curve*/
			removeReq(ps->parent->pReqQueue, ps->pFairReq->idx);
#ifdef FAST1
			/* take care of root */
			if (!(ps->parent->parent) && (ps->parent->pReqQueue))
				ps->parent->idleTime = crtTime;;
#endif
			/* yc-d: FIXME no need to reset ? no reset => vt out of sync*/
			cutRTSC(ps->pVirtual, mult(ps->totalWork)); 
			ps->totalWork = 0;
		}
	}
}
    

/************************************************************************
 *
 *  NAME: selSesByVT
 *
 *  DESCRIPTION: 
 *   Select the session to send the next packet
 *
 *  PARAMETERS:
 *    psRoot - the root of the hierarchy
 *    
 *  RETURN:
 *    session to send the next packet
 *
 ************************************************************************/

ClassItem *selSesByVT(ClassItem *psRoot) 
{
	ClassItem   *ps = psRoot;
	ReqItem *pReq;

	while (ps->pchildList) {
		pReq = getReq(ps->pReqQueue);
		if (!pReq) 
			return NULL;
		ps = (ClassItem *)pReq->pClass;
	}

	return ps;
}  


/************************************************************************
 *
 *  NAME: getClassItem
 *
 *  DESCRIPTION: 
 *   ??
 *
 *  PARAMETERS:
 *    pls - ??
 *    id  - ??
 *
 *  RETURN:
 *    - ??
 *
 ************************************************************************/

ClassItem *getClassItem(ClassItem *pls, int id)
{
	ClassItem *ptemp;

  
	while (pls) {
		if (pls->id == id)
			return pls;
		if (pls->pchildList) {
			ptemp = getClassItem(pls->pchildList, id);			
			if (ptemp) return ptemp;
		}
		pls = pls->next;
	}
	return NULL;
}

