#===========================================================================
# DpTcl.tpl -- (BS)
#
#       Implements a tclDP compatible portable RPC library atop of the 
#       Tcl 7.5 socket command.
#
# Copyright (c) 1996 Eolas Technologies, Inc.
# Copyright (c) 1996 Computerized Processes Unlimited, Inc.
# Copyright (c) 1996 Mark Roseman
# Copyright (c) 1996 Brian Smith
# all rights reserved.
#
# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software and its documentation for any purpose, provided
# that existing copyright notices are retained in all copies and that this
# notice is included verbatim in any distributions. No written agreement,
# license, or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors
# and need not follow the licensing terms described here, provided that
# the new terms are clearly indicated on the first page of each file where
# they apply.
# 
# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
# FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
# ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
# DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# 
# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.
# 
# RESTRICTED RIGHTS: Use, duplication or disclosure by the government
# is subject to the restrictions as set forth in subparagraph (c) (1) (ii)
# of the Rights in Technical Data and Computer Software Clause as DFARS
# 252.227-7013 and FAR 52.227-19.
# 
#
#---------------------------------------------------------------------------

# -- validate appropriate execution environment
if {[catch {info tclversion} tclversion] || $tclversion < 7.5} {
    return -code error "DpTcl requires Tcl7.5 / Tk4.1 or later releases"
} else {unset tclversion}

package provide Dp 3.5

#----------
# -- create a client RPC connection
proc dp_MakeRPCClient {host port {checkProc ""}} {
    global _rpc
    if {[catch {socket $host $port} sock]} {
        return -code error "dp_MakeRPCClient:  $sock"
    }

    set _rpc(${sock}::state) idle
    set _rpc(${sock}::RPCdisabled) 0
    set _rpc(${sock}::closehooks) ""
    set _rpc(${sock}::checkhook) $checkProc
    set _rpc(${sock}::safeinterp) [interp create -safe];
    interp eval $_rpc(${sock}::safeinterp) {proc unknown {cmd args} { \
	    return [eval _myrpc_checkhandler $cmd $args];}};
    interp alias $_rpc(${sock}::safeinterp) _myrpc_checkhandler {} _myrpc_checkhandler $sock;
    interp eval $_rpc(${sock}::safeinterp) {rename proc ""};

    set _rpc(${sock}::isClient) 1
    set _rpc(${sock}::isServer) 0
    dp_atexit prepend close $sock
    catch {fconfigure $sock -blocking 0 -buffering none}
    catch {fileevent $sock readable [list _myrpc_readable $sock]}
    return $sock
}

#----------
# -- create an RPC server and make it available for client connections
proc dp_MakeRPCServer {{port 0} {logincmd ""} {checkcmd ""} {closecmd ""}} {
    global _rpc

    if {[catch {socket -server [list _myrpc_accept $port] $port} sock]} {
        return -code error "dp_MakeRPCServer:  $sock"
    }

    if {$checkcmd == "none"} {
        set checkcmd ""
    }
    if {$closecmd == "none"} {
        set closecmd ""
    }
    if {$logincmd == "none"} {
	set logincmd "";
    }
    
    set _rpc(listen${port}::checkhook) $checkcmd
    set _rpc(listen${port}::closehook) $closecmd
    set _rpc(listen${port}::loginhook) $logincmd

    if {$port == 0} {
        set port [lindex [fconfigure $sock -sockname] 2]
    }

    # -- if no access control list set, let everyone in
    return $port
}

#----------
# -- send an asynchronous RPC Tcl command (don't wait for result)
proc dp_RDO {sock args} {
    set ceTemplate {
       if [catch {%s} dp_rv] {
            dp_RDO $dp_rpcFile set errorInfo "$errorInfo\n\twhile remotely executing\n%s"
            dp_RDO $dp_rpcFile eval "%s {$dp_rv}"
        } else {
            dp_RDO $dp_rpcFile eval "%s {$dp_rv}"
        }
    }

    set eTemplate {
        if [catch {%s} dp_rv] {
            dp_RDO $dp_rpcFile set errorInfo "$errorInfo\n\twhile remotely executing\n%s"
            dp_RDO $dp_rpcFile eval "%s {$dp_rv}"
        }
    }

    set cTemplate {
        set dp_rv [%s]; 
        dp_RDO $dp_rpcFile eval "%s {$dp_rv}"
    }

    set onerrorPresent [lsearch -exact $args -onerror]
    if {$onerrorPresent == -1} {
        set onerrorPresent 0
    } else {
        set onerror [lindex $args [expr {$onerrorPresent + 1}]]
        set args [concat \
            [lrange $args 0 [expr {$onerrorPresent - 1}]] \
            [lrange $args [expr {$onerrorPresent + 2}] end] \
        ]
        set onerrorPresent 1
    }

    set callbackPresent [lsearch -exact $args -callback]
    if {$callbackPresent == -1} {
        set callbackPresent 0
    } else {
        set callback [lindex $args [expr {$callbackPresent + 1}]]
        set args [concat \
            [lrange $args 0 [expr {$callbackPresent - 1}]] \
            [lrange $args [expr {$callbackPresent + 2}] end] \
        ]
        set callbackPresent 1
    }

    if {$onerrorPresent && $callbackPresent} {
	# Both onerror & callback specified.
	set command [format $ceTemplate $args $args $onerror $callback]
    } elseif {$onerrorPresent} {
	# Onerror specififed
	set command [format $eTemplate $args $args $onerror]
    } elseif {$callbackPresent} {
	# Just callback specified.
	set command [format $cTemplate $args $callback]
    } else {
	# No callbacks specified. 
	set command $args
    }
    
    if {[catch {eof $sock} eofflag] || $eofflag} {
        return -code error "dp_RDO:  socket $sock is not open"
    }
    catch {fconfigure $sock -blocking 1 -buffering full}
    catch {puts -nonewline $sock [format {%6d} [string length $command]]}
    catch {puts -nonewline $sock "d \{$command\}"}
    if {[catch {flush $sock}]} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
	return
    }
    catch {fconfigure $sock -blocking 0 -buffering none}
    return {}
}

#----------
# -- send a Tcl command to remote server, retrieve result of remote execution
proc dp_RPC {sock args} {
    global _rpc
    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return -code error "dp_RPC:  socket $sock is not open"
    }

    catch {fconfigure $sock -blocking 1 -buffering full}
    catch {puts -nonewline $sock [format {%6d} [string length $args]]}
    catch {puts -nonewline $sock "e \{$args\}"}
    if {[catch {flush $sock}]} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return {}
    }

    if {[catch {eof $sock} eofflag] || $eofflag} {
        return -code error "dp_RPC:  socket $sock is not open"
    }
    set _rpc(${sock}::RPCdisabled) 1
    while {[string compare $_rpc(${sock}::state) answered]} {
        catch {fconfigure $sock -blocking 1 -buffering full}
        _myrpc_readable $sock
    }

    catch {fconfigure $sock -blocking 0 -buffering none}
    set _rpc(${sock}::RPCdisabled) 0
    set _rpc(${sock}::state) idle
    if {[string compare $_rpc(${sock}::type) r]} {
        ##
        ## Type must be x, for error,
        ##
        foreach {results info code} $_rpc(${sock}::buffer) {break}
        return -code error -errorinfo $info -errorcode $code $results
    } else {
        ##
        ## Type must be r, for results,
        ##
        return $_rpc(${sock}::buffer)
    }
}

#----------
# -- cleanly close an RPC_connection on both sides from the client
proc dp_CloseRPC {sock} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
}

#----------
# -- client cancels pending RPC operations at server end
proc dp_Cancel {args} {
    # -- no-op, for now
}

#----------
# -- set a command to check incoming Tcl command requests of the server
proc dp_SetCheckCmd {sock args} {
    global _rpc
    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return -code error "dp_SetCheckCmd:  socket $sock is not open"
    }
    set _rpc(${sock}::checkhook) $args
}

#----------
# -- server function to maintain access control list
proc dp_Host {host} {
    global _rpc
    # -- validate host argument as being well formed IP address
    set opcode [string index $host 0]
    if {$opcode != "+" && $opcode != "-"} {
        return -code error "dp_Host usage: dp_Host \[+\|-\]ipaddress"
    }
    if {[string length $host] == 1} {
        append host "*.*.*.*"
    }
    set iplist [string range $host 1 [expr [string length $host] - 1]]
    set iplist [split $iplist "."]
    if {[llength $iplist] != 4} {
        # -- assume non-ip hostname given...can't handle it so return
        return {}
    }
    foreach ipitem $iplist {
        if {$ipitem != "*"} {
            if {[catch {expr $ipitem * 1}] || $ipitem > 255} {
                # -- assume non-ip hostname given,..can't handle so return
                return {}
            }
        }
    }
    # -- create the acl list, enable universal access, add modifier 
    if {![info exists _rpc(acl)]} {
        lappend _rpc(acl) [list + * * * *]
    }
    lappend _rpc(acl) "$opcode $iplist"
    return {}
}

#----------
# -- define commands to be executed just prior to really exiting
proc dp_atexit {option args} {
    global _rpc
    if {![info exists _rpc(atexit)]} {
        # -- create exit callbacks, replace exit command to invoke them
        rename exit dp_atexit_really_exit
        set _rpc(atexit) ""
        uplevel #0 {proc exit {{code 0}} {
            global _rpc
            while {1} {
                if {[catch {set _rpc(atexit)} _rpc(atexit)]} {
                    break
                }
                if {[llength $_rpc(atexit)] <= 0} {
                    break
                }
                set callback [lindex $_rpc(atexit) 0]
                set _rpc(atexit) [lrange $_rpc(atexit) 1 end]
                catch {uplevel #0 "$callback"}
            }
            catch {unset _rpc(atexit)}
            catch {dp_atexit_really_exit $code}
        }   
        }
    }
    switch -exact -- $option {
        set {
            set _rpc(atexit) [split $args]
        }
        appendUnique {
            lappend _rpc(atexit) $args
        }
        append {
            lappend _rpc(atexit) $args
        }
        prepend {
            set _rpc(atexit) [linsert $_rpc(atexit) 0 $args] 
        }
        insert {
            set _rpc(atexit) [linsert $_rpc(atexit) 0 $args
        }
        delete {}
        clear {
            set _rpc(atexit) ""
        }
        list {
            return $_rpc(atexit)
        }
        default {
            return -code error "dp_atexit: unrecognized option \[$option\]"
	}
    }
    return $_rpc(atexit)
}

#----------
# -- register callbacks to RPC channel to execute just before channel closes
proc dp_atclose {sock option args} {
    global _rpc
    if {![info exists _rpc(${sock}::isClient)]} {
        return
    }
    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return -code error "dp_atclose:  socket $sock is not open"
    }
    switch -exact -- $option {
        set {
            set _rpc(${sock}::closehooks) $args
        }
        append {
            lappend _rpc(${sock}::closehooks) $args
        }
        appendUnique {
            lappend _rpc(${sock}::closehooks) $args
        }
        prepend {
            set _rpc(${sock}::closehooks) [linsert $_rpc(${sock}::closehooks) 0 $args] 
        }
        insert {
            set _rpc(${sock}::closehooks) [linsert $_rpc(${sock}::closehooks) 0 $args] 
        }
        delete {}
        clear {
            set _rpc(${sock}::closehooks) ""
        }
        list {
            return $_rpc(${sock}::closehooks)
        }
        default {
            return -code error "dp_atclose: unrecognized option \[$option\]"
        }
    }
    return $_rpc(${sock}::closehooks)
}

# -- replacement close function to ensure RPC close callbacks are run
if {![llength [info commands dp_atclose_really_close]]} {
    rename close dp_atclose_really_close
    proc close {sock} {
        global _rpc
        if {"" == [array names _rpc ${sock}::*]} {
            dp_atclose_really_close $sock
            return {}
        }
        if {![info exists _rpc(${sock}::isServer)]} {
            dp_atclose_really_close $sock
            return {}
        }
        if {$_rpc(${sock}::isClient)} {
            catch {dp_RDO $sock close [dp_RPC $sock set dp_rpcFile]}
        }
        fileevent $sock readable ""
        foreach i $_rpc(${sock}::closehooks) {
            catch {uplevel #0 $i}
	}
        dp_atclose_really_close $sock
        after 10 _myrpc_remove_client $sock
    }
}    
    
#----------
# -- return RPC channel identifier
proc rpcFile {} {
    global myrpc_channel
    if {[info exists myrpc_channel]} {
        return $myrpc_channel
    } else {
        return {}
    }
}

#----------
# -- INTERNAL:  Server accepts a client connection
proc _myrpc_accept {listener sock addr port} {
    global _rpc

#puts stdout "accept: listener=$listener sock=$sock addr=$addr port=$port"
    
    if {[info exists _rpc(acl)] && [llength $_rpc(acl)] > 1} {
        set cip [split $addr "."]
        set allowed 1
        foreach ip $_rpc(acl) {
            set opcode [lindex $ip 0]
            set ip [lrange $ip 1 4]
            set j 0
            for {set i 0} {$i<4} {incr i} {
                if {[lindex $ip $i] == "*" || \
                        [lindex $ip $i] == [lindex $cip $i]} {
                    incr j
                }
            }
            if {$j == 4} {
                if {$opcode == "-"} {
                    set allowed 0
                } else {
                    set allowed 1
                }
            }
        }
        if {!$allowed} {
            puts stderr "RPC connection from $addr refused"
		catch {fileevent $sock readable {}}
		catch {close $sock}
            return
        }
    }
    if {$_rpc(listen${listener}::loginhook) != ""} {
	if {[catch "$_rpc(listen${listener}::loginhook) $addr"]} {
	    puts stderr "RPC connection from $addr refused"
	    catch {fileevent $sock readable {}}
	    catch {close $sock}
	    return;
	}
    }

    set _rpc(${sock}::state) idle
    set _rpc(${sock}::closehooks) $_rpc(listen${listener}::closehook)
    set _rpc(${sock}::checkhook) $_rpc(listen${listener}::checkhook)
    set _rpc(${sock}::safeinterp) [interp create -safe];
    interp eval $_rpc(${sock}::safeinterp) {proc unknown {cmd args} { \
	    return [eval _myrpc_checkhandler $cmd $args];}};
    interp alias $_rpc(${sock}::safeinterp) _myrpc_checkhandler {} _myrpc_checkhandler $sock;
    interp eval $_rpc(${sock}::safeinterp) {rename proc ""};

    set _rpc(${sock}::listener) $listener
    set _rpc(${sock}::clientip) $addr
    set _rpc(${sock}::ipport) $port
    set _rpc(${sock}::RPCdisabled) 0
    set _rpc(${sock}::isClient) 0
    set _rpc(${sock}::isServer) 1
    dp_atexit prepend close $sock
    if {[eof $sock]} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return -code error "dp_MakeRPCServer:  socket $sock is not open"
    }
    catch {fconfigure $sock -blocking no -buffering none}
    catch {fileevent $sock readable [list _myrpc_readable $sock]}
}

#----------
# -- INTERNAL:  Client or Server interupt processing for new data on the
#               RPC channel
proc _myrpc_readable {sock} {
    global _rpc

    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
    switch $_rpc(${sock}::state) {
        idle {
            set _rpc(${sock}::state) readhdr
            set _rpc(${sock}::buffer) ""
            set _rpc(${sock}::toread) 6
            _myrpc_readhdr $sock
        }
        readhdr {
            _myrpc_readhdr $sock
        }
        readmsg {
            _myrpc_readmsg $sock
        }
    }
}

#----------
# -- INTERNAL:  read metadata component of message received over RPC channel
proc _myrpc_readhdr {sock} {
    global _rpc
    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
    if {[catch {read $sock $_rpc(${sock}::toread)} result]} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
    if {$result == ""} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
    append _rpc(${sock}::buffer) $result
    incr _rpc(${sock}::toread) [expr -[string length $result]]
    if {$_rpc(${sock}::toread) == 0} {
        set _rpc(${sock}::state) readmsg
 	if {$_rpc(${sock}::isServer)} { 
            set _rpc(${sock}::toread) [expr $_rpc(${sock}::buffer) + 4] 
	}
        if {$_rpc(${sock}::isClient)} { 
            set _rpc(${sock}::toread) [expr $_rpc(${sock}::buffer) + 4] 
        }
	set _rpc(${sock}::buffer) ""
    }
    if {[catch {eof $sock} eofflag] || $eofflag} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
}

#----------
# -- INTERNAL:  read/check/execute Tcl command/result component of message 
#               received over RPC channel
proc _myrpc_readmsg {sock} {
    global _rpc errorInfo errorCode
    global myrpc_channel dp_rpcFile 
    if {[catch {read $sock $_rpc(${sock}::toread)} result]} {
	catch {fileevent $sock readable {}}
	catch {close $sock}
        return
    }
    if {$result == ""} {
        # close $sock
        return
    }
    append _rpc(${sock}::buffer) $result
    incr _rpc(${sock}::toread) [expr -[string length $result]]
    if {$_rpc(${sock}::toread) == 0} {
        set dp_rpcFile $sock
        set myrpc_channel $sock
        set _rpc(${sock}::state) idle
        set _rpc(${sock}::type) [lindex $_rpc(${sock}::buffer) 0]
        switch -exact $_rpc(${sock}::type) {
            x {
                ##
                ## Error result return 
                ##
                set _rpc(${sock}::state) answered
                set _rpc(${sock}::buffer) [lrange $_rpc(${sock}::buffer) 1 end]
                return
            }
            r {
                ##
                ## Normal result return 
                ##
                set _rpc(${sock}::state) answered
                set _rpc(${sock}::buffer) [lindex $_rpc(${sock}::buffer) 1]
                return
            }
            e {
                ##
                ## Rpc call
                ##
                if {$_rpc(${sock}::RPCdisabled)} {
                    set blockingState [fconfigure $sock -blocking]
                    set bufferState [fconfigure $sock -buffering]
                    set _rpc(${sock}::outbuf) [list x {dp_RPC: deadlock detected, RPC aborted} {} {}]
                    set _rpc(${sock}::outlen) [string length _rpc(${sock}::outbuf)]
                    set hdr [format "%6d" [expr $_rpc(${sock}::outlen)-4]]
                    if {[catch {eof $sock} eofflag] || $eofflag} {
        		catch {fileevent $sock readable {}}
        		catch {close $sock}
                    }
                    catch {fconfigure $sock -blocking 1 -buffering full}
                    if {[catch {puts -nonewline $sock $hdr}]} {
			catch {fileevent $sock readable {}}
			catch {close $sock}
        	    }
                    if {[catch {flush $sock}]} {
			catch {fileevent $sock readable {}}
			catch {close $sock}
        	    }
                    _myrpc_writeresult $sock
                    catch {fconfigure $sock -blocking $blockingState -buffering $bufferState}
                    return                    
                }
            }
            d {
                ##
                ## Rdo call
                ##
            }
        }
        # -- if no checking proc or checking proc does not error out, eval
        # -- the command
        set status 1
        set _rpc(${sock}::outdone) 0
 	set cmd [lindex $_rpc(${sock}::buffer) 1]

	if {![string compare $_rpc(${sock}::type) d]} {
                    interp eval $_rpc(${sock}::safeinterp) " \
		    after 1 \{                              \
		    set myrpc_channel $myrpc_channel;       \
		    set dp_rpcFile $dp_rpcFile;             \
		    catch {$cmd};                           \
		    catch {unset myrpc_channel};            \
		    catch {unset dp_rpcFile};               \
		    \}"
	    return
	}

	if {[catch {interp eval $_rpc(${sock}::safeinterp) $cmd} result]} {
	    set _rpc(${sock}::outbuf) [list x $result $errorInfo $errorCode]
	} else {
	    set _rpc(${sock}::outbuf) [list r $result]
	}
	set _rpc(${sock}::outlen) [string length $_rpc(${sock}::outbuf)]

        if {![string compare $_rpc(${sock}::type) e]} {
            if {[catch {eof $sock} eofflag] || $eofflag} {
		catch {fileevent $sock readable {}}
		catch {close $sock}
		return
            }
            catch {fconfigure $sock -blocking 1 -buffering full}
            set hdr [format "%6d" [expr $_rpc(${sock}::outlen)-4]]
            if {[catch {puts -nonewline $sock $hdr}]} {
		catch {fileevent $sock readable {}}
		catch {close $sock}
		return
            }
            if {[catch {flush $sock}]} {
		catch {fileevent $sock readable {}}
		catch {close $sock}
		return
            }
            _myrpc_writeresult $sock
            catch {fconfigure $sock -blocking 0 -buffering none}
        }
        catch {unset myrpc_channel}
        catch {unset dp_rpcFile}
    }
}

#----------
# -- write result of command back to client based on writable event
proc _myrpc_writeresult {sock} {
    global _rpc
    while {!$_rpc(${sock}::outdone)} {
        if {$_rpc(${sock}::outlen) <= 0} {
            set _rpc(${sock}::outdone) 1
#            update idletasks
            return
        }
        if {$_rpc(${sock}::outlen) > 4096} {
            set len 4096
        } else {
            set len $_rpc(${sock}::outlen)
        }
        set packet "[string range $_rpc(${sock}::outbuf) 0 [expr $len - 1]]"
        set _rpc(${sock}::outbuf) "[string range $_rpc(${sock}::outbuf) $len [expr $_rpc(${sock}::outlen) - 1]]"
        incr _rpc(${sock}::outlen) -[set len]
        if {[catch {eof $sock} eofflag] || $eofflag} {
	    catch {fileevent $sock readable {}}
	    catch {close $sock}
            return
        }
        if {[catch {puts -nonewline $sock $packet}]} {
	    catch {fileevent $sock readable {}}
	    catch {close $sock}
            return
        }
        if {[catch {flush $sock}]} {
	    catch {fileevent $sock readable {}}
	    catch {close $sock}
            return
        }
        if {[catch {eof $sock} eofflag] || $eofflag} {
	    catch {fileevent $sock readable {}}
	    catch {close $sock}
            return
        }
#        update idletasks
    }
}

#----------
# -- get rid of a client's entry
proc _myrpc_remove_client {sock} {
    global _rpc
    foreach item [array names _rpc "${sock}::*"] {
        catch {unset _rpc($item)}
    } 
}

proc _myrpc_checkhandler {sock cmd args} {
    global _rpc;

    if {$_rpc(${sock}::checkhook) == ""} {
	interp alias $_rpc(${sock}::safeinterp) $cmd {} $cmd;
	return [eval $cmd $args];
    } elseif {[$_rpc(${sock}::checkhook) $cmd] == 1} {
	interp alias $_rpc(${sock}::safeinterp) $cmd {} $cmd;
	return [eval $cmd $args];
    } else {
	error "The command $cmd is disallowed";
    }
}

#----------
# -- report background errors on the server
if {[info procs bgerror] == ""} {
    proc bgerror {args} {
        global errorInfo errorCode
        puts stderr "Background error: $args"
        puts stderr "\t$errorInfo"
        puts stderr "errorCode = \[$errorCode\]"
        return {}
    }
}





