# Copyright (C) 2000-2001 The OpenRPG Project
#
#    openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: mapper/fog.py
# Author: OpenRPG Team
#
# Description: Maintenance of data structures required for FoW
#

from base import *
from random import Random
from region import *
from orpg.minidom import Element
import traceback

COURSE = 10



class fog_line(protectable_attributes):
    def __init__(self,outline):
        protectable_attributes.__init__(self)
        self._protected_attr = ["outline"]
        self.outline=outline
        #self._clean_all_attr()

    def set_fog_props(self,str):
        self.outline=str

    def points_to_elements( self, points=None ):
        result = []
        if points == None:
            return result

        for pairs in string.split( points, ';' ):
            # print 'pair :',pairs
            pair = string.split( pairs, ',' )
            # result += '<point x="%s" y="%s"/>' % ( pair[0], pair[1] )
            p = Element( "point" )
            p.setAttribute( "x", pair[0] )
            p.setAttribute( "y", pair[1] )
            result.append( p )
        return result

    def toxml(self,action="update",preserve_changed=0):
        xml_str = ""
        if preserve_changed:
            original_changed = self._changed_attr()

        localOutline = self.outline
        # print 'localOutline',localOutline
        if localOutline != None and localOutline != "all" and localOutline != "none":
            localOutline = "points"

        # print 'localOutline',localOutline
        elem = Element( "poly" )

        if action == "del":
            elem.setAttribute( "action", action )
            elem.setAttribute( "outline", localOutline )
            if localOutline == 'points':
                list = self.points_to_elements( self.outline )
                for p in list:
                    elem.appendChild( p )
            str = elem.toxml()
            elem.unlink()
            return str
            # xml_str = "<poly action='del'"
            # xml_str += " outline='" + localOutline+ "'"
            # if localOutline == 'points':
            #     xml_str += self.points_to_elements( self.outline )
            # xml_str +="/>"
            # return xml_str

        if action == "new":
            self._dirty_all_attr()

        changed = self._changed_attr()

        if changed:
            # xml_str = "<poly"
            # xml_str += " action='" + action + "'"
            elem.setAttribute( "action", action )

            for a in changed.keys():
                if a == "outline" and not (localOutline is None):
                    elem.setAttribute( "outline", localOutline )
                    # xml_str += " outline='" + localOutline + "'"
                    if localOutline == 'points':
                        list = self.points_to_elements( self.outline )
                        for p in list:
                            elem.appendChild( p )
                    #     xml_str += self.points_to_elements( self.outline )
            # xml_str += "/>"
            xml_str = elem.toxml()
            # elem.unlink()
        self._clean_all_attr()

        if preserve_changed:
            for a in original_changed.keys():
                self._dirty_attr(a)
        elem.unlink()
        return xml_str


class fog_layer(layer_base):
    def __init__(self, canvas):
        layer_base.__init__(self)
        self.canvas = canvas

        self.color = wxColor(128,128,128)
        self.fogregion = wxRegion()
        self.fogregion.Clear()
        self.fog_bmp = None
        self.width = 0
        self.height = 0
        self.use_fog = 0
        self.last_role =""
#        self._protected_attr=["fog_bmp","width","height","use_fog","color"]

    def clear(self):
        self.fogregion.Clear()
        self.use_fog=1
        self.del_area("all")
        self.recompute_fog()

    def remove_fog(self):
        self.fogregion.Clear()
        self.use_fog=0
        self.add_area("none")
        self.fog_bmp = None

    def showall(self):
        if self.fogregion.IsEmpty():
            self.fogregion=wxRegion(0,0,self.width,self.height)
        else:
            self.fogregion.Union(0,0,self.width,self.height)
        self.use_fog=1
        self.add_area("all")
        self.fill_fog()

    def resize(self,size):
        try:
            if self.width==size[0] and self.height==size[1]:
                return
            self.recompute_fog()
        except:
            pass


    def recompute_fog(self):
        if not self.use_fog:
            return
        size = self.canvas.size
        self.width=size[0]/COURSE+1
        self.height=size[1]/COURSE+1
        self.fog_bmp=wxEmptyBitmap(self.width+2,self.height+2)
        self.fill_fog()

    def fill_fog(self):
        if not self.use_fog:
            return
        mdc = wxMemoryDC()
        mdc.BeginDrawing()
        mdc.SetOptimization(true)
        mdc.SelectObject(self.fog_bmp)
        mdc.SetPen(wxTRANSPARENT_PEN)
        if (self.canvas.frame.session.role == "GM"):
            color = self.color
        else:
            color = wxBLACK

        self.last_role=self.canvas.frame.session.role
        mdc.SetBrush(wxBrush(color,wxSOLID))
        mdc.DestroyClippingRegion()
        mdc.DrawRectangle(0,0,self.width+2,self.height+2)
        mdc.SetBrush(wxBrush(wxWHITE,wxSOLID))
        if not self.fogregion.IsEmpty():
            mdc.SetClippingRegionAsRegion(self.fogregion)
            mdc.DrawRectangle(0,0,self.width+2,self.height+2)
        mdc.SelectObject(wxNullBitmap)
        del mdc


    def draw(self,dc,topleft,size):
        if (self.fog_bmp == None):
            return
        if not self.use_fog:
            return
        if self.last_role!=self.canvas.frame.session.role:
            self.fill_fog()

        sc = dc.GetUserScale()
        bmp = wxEmptyBitmap(size[0],size[1])
        mdc=wxMemoryDC()
        mdc.BeginDrawing()
        mdc.SelectObject(bmp)
        mdc.SetOptimization(true)
        mdc.SetPen(wxTRANSPARENT_PEN)
        mdc.SetBrush(wxBrush(wxWHITE,wxSOLID))
        mdc.DrawRectangle(0,0,size[0],size[1])



        srct=[int(topleft[0]/(sc[0]*COURSE)),int(topleft[1]/(sc[1]*COURSE))]
        srcsz=[int((int(size[0]/COURSE+1)*COURSE)/(sc[0]*COURSE))+2,
               int((int(size[1]/COURSE+1)*COURSE)/(sc[1]*COURSE))+2]
        if (srct[0]+srcsz[0]>self.width):
            srcsz[0]=self.width-srct[0]
        if (srct[1]+srcsz[1]>self.height):
            srcsz[1]=self.height-srct[1]

        img=wxImageFromBitmap(self.fog_bmp).GetSubImage(
            wxRect(srct[0],srct[1],srcsz[0],srcsz[1]))
        img.Rescale(srcsz[0]*COURSE*sc[0],srcsz[1]*COURSE*sc[1])
        fog=wxBitmapFromImage(img)
        mdc.SetDeviceOrigin(-topleft[0],-topleft[1])
        mdc.DrawBitmap(fog,srct[0]*COURSE*sc[0],srct[1]*COURSE*sc[1])
        mdc.SetDeviceOrigin(0,0)

        mdc.SetUserScale(1,1)
        mdc.EndDrawing()
        dc.SetUserScale(1,1)
        dc.Blit(topleft[0],topleft[1],size[0],size[1],mdc,0,0,wxAND)
        dc.SetUserScale(sc[0],sc[1])
        mdc.SelectObject(wxNullBitmap)
        del mdc


    def createregn2(self,polyline,mode,show):
        regn = self.scan_convert(polyline)
        area = ""
        for i in polyline:
            if (area != ""):
                area += ";"
            area += str(i.X) + "," + str(i.Y)
        if mode == 'new':
            if (self.fogregion.IsEmpty()):
                self.fogregion=regn
            else:
                self.fogregion.UnionRegion(regn)
            self.add_area(area,show)
        else:
            if not self.fogregion.IsEmpty():
                self.fogregion.SubtractRegion(regn)
            self.del_area(area,show)

    def createregn(self,polyline,mode,show="Yes"):
        self.createregn2(polyline,mode,show)
        self.fill_fog()

    def scan_convert(self,polypt):
        regn = wxRegion()
        regn.Clear()
        list=IRegion().scan_convert(polypt)
        for i in list:
            if regn.IsEmpty():
                regn=wxRegion(i.left,i.y,i.right+1-i.left,1)
            else:
                regn.Union(i.left,i.y,i.right+1-i.left,1)
        return regn

    def add_area(self,area="",show="Yes"):
        poly = fog_line(area)
        poly._dirty_all_attr()
        xml_str = "<map><fog>"
        xml_str += poly.toxml("new")
        xml_str += "</fog></map>"
        if show == "Yes":
            self.canvas.frame.session.send(xml_str)

    def del_area(self,area="",show="Yes"):

        poly = fog_line(area)
        poly._dirty_all_attr()
        xml_str = "<map><fog>"
        xml_str += poly.toxml("del")
        xml_str += "</fog></map>"
        if show == "Yes":
            self.canvas.frame.session.send(xml_str)

    def toxml(self,action="update"):
        if not (self.use_fog):
            return ""
        attributes = ""

        if action == "new":
            self._dirty_all_attr()

        changed = self._changed_attr()

        if changed:
            for a in changed.keys():
                if a == "serial_number":
                    attributes += " serial='" + str(self.serial_number) + "'"
                    self._clean_attr("serial_number")

        fog_string = ""
        ri = wxRegionIterator(self.fogregion)

        if not (ri.HaveRects()):
            fog_string=fog_line("all").toxml("del")

        while ri.HaveRects():
            x1=ri.GetX()
            x2=x1+ri.GetW()-1
            y1=ri.GetY()
            y2=y1+ri.GetH()-1
            poly=fog_line(str(x1)+","+str(y1)+";"+
                          str(x2)+","+str(y1)+";"+
                          str(x2)+","+str(y2)+";"+
                          str(x1)+","+str(y2))
            fog_string += poly.toxml(action)
            ri.Next()

        if fog_string or changed:
            s = "<fog"
            s += attributes
            if fog_string:
                s += ">"
                s += fog_string
                s += "</fog>"
            else:
                s+="/>"

            return s

        else:
            return ""

    def takedom(self,xml_dom):
        # Fixed the issue with the fog layer that would cause it to render improperly if you loaded a map from a file or
        # entered a room with pre excisting fog information set. Also cleaned up the code abit to use code over insted of
        # new code that preformed the same task. This should help speed up bug fixes in the future.
        #                                                   Dj Gilcrease 11/30/03

        try:
            if not self.use_fog:
                self.use_fog=1
                self.recompute_fog()
            serial_number = xml_dom.getAttribute('serial')
            if serial_number <> "":
                self.serial_number = int(serial_number)
                self._clean_attr("serial_number")


            children = xml_dom._get_childNodes()
            for l in children:
                action = l.getAttribute("action")
                outline = l.getAttribute("outline")
                if (outline=="all"):
                    polyline=[IPoint().make(0,0),IPoint().make(self.width-1,0),
                              IPoint().make(self.width-1,self.height-1),
                              IPoint().make(0,self.height-1)]
                elif (outline=="none"):
                    polyline=[]
                    self.use_fog=0
                    self.fog_bmp=None
                else:
                    polyline=[]
                    lastx = None
                    lasty = None
                    list = l._get_childNodes()
                    for point in list:
                        x = point.getAttribute( "x" )
                        y = point.getAttribute( "y" )
                        if (x!=lastx or y!=lasty):
                            polyline.append( IPoint().make( int(x), int(y) ) )
                        lastx=x
                        lasty=y

                print polyline;
                if (len(polyline)>1):
                    self.createregn2(polyline,action,"No")
            self.fill_fog()
        except:
            print "Exception: Source -> fog_layer.takedom()"
            traceback.print_exc()

