/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* 
 * hresult.h - COM-compatible error handling for UNIX
 * Notes: Under Win32, define DEFINE_HRESULT if you also need HRESULT defined.
 */

#ifndef __hresult_h__
#define __hresult_h__

#ifdef WIN32

# if defined(DEFINE_HRESULT) || defined(WIN32_LEAN_AND_MEAN)
    typedef long HRESULT;
# endif

#else

typedef long HRESULT;

/* This is not a handle to anything, but is merely a 32-bit value
   with several fields encoded in the value.  The parts of an 
   HRESULT are shown below.  
  
    HRESULTs are 32 bit values layed out as follows:
  
     3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
     1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
    +-+-+-+-+-+---------------------+-------------------------------+
    |S|R|C|N|r|    Facility         |               Code            |
    +-+-+-+-+-+---------------------+-------------------------------+
  
    where
  
        S - Severity - indicates success/fail
  
            0 - Success
            1 - Fail (COERROR)
  
        R - reserved portion of the facility code, corresponds to NT's
                second severity bit.
  
        C - reserved portion of the facility code, corresponds to NT's
                C field.
  
        N - reserved portion of the facility code. Used to indicate a
                mapped NT status value.
  
        r - reserved portion of the facility code. Reserved for internal
                use. Used to indicate HRESULT values that are not status
                values, but are instead message ids for display strings.
  
        Facility - is the facility code
  
        Code - is the facility's status code
*/

/* Severity values */
#define SEVERITY_SUCCESS    0
#define SEVERITY_ERROR      1

/* Facility values */
#define FACILITY_WINDOWS_CE              24
#define FACILITY_WINDOWS                 8
#define FACILITY_URT                     19
#define FACILITY_UMI                     22
#define FACILITY_SXS                     23
#define FACILITY_STORAGE                 3
#define FACILITY_STATE_MANAGEMENT        34
#define FACILITY_SSPI                    9
#define FACILITY_SCARD                   16
#define FACILITY_SETUPAPI                15
#define FACILITY_SECURITY                9
#define FACILITY_RPC                     1
#define FACILITY_WIN32                   7
#define FACILITY_CONTROL                 10
#define FACILITY_NULL                    0
#define FACILITY_METADIRECTORY           35
#define FACILITY_MSMQ                    14
#define FACILITY_MEDIASERVER             13
#define FACILITY_INTERNET                12
#define FACILITY_ITF                     4
#define FACILITY_HTTP                    25
#define FACILITY_DPLAY                   21
#define FACILITY_DISPATCH                2
#define FACILITY_CONFIGURATION           33
#define FACILITY_COMPLUS                 17
#define FACILITY_CERT                    11
#define FACILITY_BACKGROUNDCOPY          32
#define FACILITY_ACS                     20
#define FACILITY_AAF                     18

/*
   Generic test for success on any status value (non-negative numbers
   indicate success).
*/

#define SUCCEEDED(Status) ((HRESULT)(Status) >= 0)

/*
   and the inverse
*/

#define FAILED(Status) ((HRESULT)(Status)<0)


/*
   Generic test for error on any status value.
*/

#define IS_ERROR(Status) ((unsigned long)(Status) >> 31 == SEVERITY_ERROR)

/*
   Return the code
*/

#define HRESULT_CODE(hr)    ((hr) & 0xFFFF)

/*
    Return the facility
*/

#define HRESULT_FACILITY(hr)  (((hr) >> 16) & 0x1fff)

/*
    Return the severity
*/

#define HRESULT_SEVERITY(hr)  (((hr) >> 31) & 0x1)

/*
   Create an HRESULT value from component pieces
*/

#define MAKE_HRESULT(sev,fac,code) \
    ((HRESULT)(((unsigned long)(sev)<<31) | ((unsigned long)(fac)<<16) | ((unsigned long)(code))) )


/* Facility NULL errors that may be needed. */
#define E_UNEXPECTED                     0x8000FFFFL
#define E_NOTIMPL                        0x80004001L
#define E_OUTOFMEMORY                    0x8007000EL
#define E_INVALIDARG                     0x80070057L
#define E_NOINTERFACE                    0x80004002L
#define E_POINTER                        0x80004003L
#define E_HANDLE                         0x80070006L
#define E_ABORT                          0x80004004L
#define E_FAIL                           0x80004005L
#define E_ACCESSDENIED                   0x80070005L

#endif /* WIN32 */

#ifndef NOERROR 
#define NOERROR 0
#endif
#ifndef S_OK
#define S_OK                0L
#endif
#ifndef S_FALSE
#define S_FALSE             1L
#endif

/* Codes from 0x8000 - 0x8FFF in FACILITY_ITF are reserved for OpenSAML */

#define SAML_E_FIRST MAKE_HRESULT(SEVERITY_ERROR,FACILITY_ITF,0x8000)
#define SAML_E_LAST MAKE_HRESULT(SEVERITY_ERROR,FACILITY_ITF,0x8FFF)

#define SAML_S_FIRST MAKE_HRESULT(SEVERITY_SUCCESS,FACILITY_ITF,0x8000)
#define SAML_S_LAST MAKE_HRESULT(SEVERITY_SUCCESS,FACILITY_ITF,0x8FFF)

/* Specific code definitions */

#define SAML_E_UNSPECIFIED              (SAML_E_FIRST + 0L)
#define SAML_E_MALFORMED                (SAML_E_FIRST + 1L)

#endif
