/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* QName.cpp - QName wrapper class

   Scott Cantor
   5/6/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


saml::QName::QName(const XMLCh* nsURI, const XMLCh* localName)
{
    if (!localName || *localName==chNull)
        throw SAMLException("QName() requires local name");
#ifdef HAVE_GOOD_STL
    if (nsURI)
        m_namespace=nsURI;
    m_localName=localName;
#else
    m_namespace=XMLString::replicate(nsURI);
    m_localName=XMLString::replicate(localName);
#endif
}

#ifndef HAVE_GOOD_STL
saml::QName::QName(const saml::QName& src)
{
    m_namespace=XMLString::replicate(src.getNamespaceURI());
    m_localName=XMLString::replicate(src.getLocalName());
}

saml::QName& saml::QName::operator=(const saml::QName& src)
{
    m_namespace=XMLString::replicate(src.getNamespaceURI());
    m_localName=XMLString::replicate(src.getLocalName());
    return *this;
}

SAML_EXPORTS bool saml::operator==(const saml::QName& op1, const saml::QName& op2)
{
    return (!XMLString::compareString(op1.getNamespaceURI(),op2.getNamespaceURI()) &&
            !XMLString::compareString(op1.getLocalName(),op2.getLocalName()));
}
#endif

saml::QName::~QName()
{
#ifndef HAVE_GOOD_STL
    XMLString::release(&m_namespace);
    XMLString::release(&m_localName);
#endif
}

saml::QName* saml::QName::getQNameAttribute(DOMElement* e, const XMLCh* nsURI, const XMLCh* localName)
{
    const XMLCh* qval = e->getAttributeNS(nsURI,localName);
    if (!qval || !*qval)
        return NULL;
    return new saml::QName(saml::QName::getNamespaceForQName(qval,e),qval+XMLString::indexOf(qval,chColon) + 1);
}

saml::QName* saml::QName::getQNameTextNode(DOMText* t)
{
    const XMLCh* qval = t->getNodeValue();
    DOMNode* n = t->getParentNode();
    if (!qval || !*qval || !n || n->getNodeType()!=DOMNode::ELEMENT_NODE)
        return NULL;
    return new saml::QName(saml::QName::getNamespaceForQName(qval,static_cast<DOMElement*>(n)),
                            qval+XMLString::indexOf(qval,chColon) + 1);
}

const XMLCh* saml::QName::getNamespaceForQName(const XMLCh* qname, DOMElement* e)
{
    int i;
    if (qname && (i=XMLString::indexOf(qname,chColon))>0)
    {
        XMLCh* prefix=new XMLCh[i+1];
        XMLString::subString(prefix,qname,0,i);
        prefix[i]=chNull;
        const XMLCh* ret=getNamespaceForPrefix(prefix,e);
        delete[] prefix;
        return ret;
    }
    return getNamespaceForPrefix(NULL,e);
}

const XMLCh* saml::QName::getNamespaceForPrefix(const XMLCh* prefix, DOMElement* e)
{
    DOMNode* n = e;
    const XMLCh* ns = NULL;

    if (prefix)
    {
        if (!XMLString::compareString(prefix,L(xml)))
            return XML::XML_NS;
        else if (!XMLString::compareString(prefix,L(xmlns)))
            return XML::XMLNS_NS;
    }

    while ((!ns || !*ns) && n && n->getNodeType() == DOMNode::ELEMENT_NODE)
    {
        ns = (static_cast<DOMElement*>(n))->getAttributeNS(XML::XMLNS_NS,prefix ? prefix : L(xmlns));
        n = n->getParentNode();
    }
    return ns;
}

bool saml::operator<(const saml::QName& op1, const saml::QName& op2)
{
    int i=XMLString::compareString(op1.getNamespaceURI(),op2.getNamespaceURI());
    if (i<0)
        return true;
    else if (i==0)
        return (XMLString::compareString(op1.getLocalName(),op2.getLocalName())<0);
    else
        return false;
}
